// ============================================================================
// Textfile reader/writer class
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cTxtFile;

interface

uses SysUtils, Classes;

const
  TEXTFILE_READ   = 'r';
  TEXTFILE_WRITE  = 'w';
  TEXTFILE_APPEND = 'a';

type
  TTextfile = class
    protected
      Filename : String;
      OpenMode : Char;
      Stream   : TStream;
      procedure AutoDetectEOL;
      procedure Open;
      procedure Close;
    public
      UseUnixEOL: Boolean;
      property StreamDirect: TStream read Stream;

      function  IsEndOfLine( c: Char ): Boolean;
      function  EndOfFile: Boolean;
      function  ReadLine: String;
      procedure WriteLine( T: String );
      function  ReadBuf ( var Buffer; Count: LongInt ): LongInt;
      function  WriteBuf( var Buffer; Count: LongInt ): LongInt;

      constructor Create( AFilename: String; AOpenMode: Char ); virtual;
      destructor Destroy; override;
  end;

function ConvertLFtoCRLF( T: String ): String;
function ConvertCRLFtoLF( T: String ): String;

implementation

// ---------------------------------------------------------------- Tools -----

function ConvertLFtoCRLF( T: String ): String;
var  i: Integer;
begin
     Result := '';
     for i:=1 to length(T) do begin
        if T[i]=#10 then Result:=Result+#13;
        Result := Result + T[i];
     end;
end;

function ConvertCRLFtoLF( T: String ): String;
var  i: Integer;
begin
     Result := '';
     for i:=1 to length(T) do begin
        if T[i]<>#13 then Result:=Result+T[i];
     end;
end;

// ------------------------------------------------------------ TTextfile -----

procedure TTextfile.AutoDetectEOL;
var  SavePos: LongInt;
     c: Char;
begin
     UseUnixEOL := False;

     SavePos := Stream.Position;
     Stream.Position := 0;

     while Stream.Position<Stream.Size do begin
        Stream.Read( c, 1 );
        if c=#10 then begin UseUnixEOL:=True; break; end;
        if c=#13 then break;
     end;
     
     Stream.Position := SavePos;
end;

function TTextfile.EndOfFile: Boolean;
begin
     if Assigned(Stream) then Result:=(Stream.Position>=Stream.Size)
                         else Result:=True;
end;

function TTextfile.IsEndOfLine( c: Char ): Boolean;
begin
     Result := True;
     if EndOfFile then exit;

     if UseUnixEOL then begin
        if ( PChar(TMemoryStream(Stream).Memory) + Stream.Position )^ = #10 then exit;
     end else begin
        if ( PChar(TMemoryStream(Stream).Memory) + Stream.Position )^ = #13 then exit;
     end;

     Result := False;
end;

function TTextfile.ReadLine: String;
var  p: PChar;
begin
     Result := '';
     p := PChar(TMemoryStream(Stream).Memory) + Stream.Position;
     while not EndOfFile do begin
        if IsEndOfLine(p^) then begin
           Stream.Position := Stream.Position + 1; // skip EOL-char
           if not(EndOfFile or UseUnixEOL) then begin // skip (CR+)LF
              if (p+1)^=#10 then Stream.Position:=Stream.Position+1;
           end;
           break;
        end;
        Result := Result + p^;
        Stream.Position := Stream.Position + 1;
        inc( p );
     end;
end;

procedure TTextfile.WriteLine( T: String );
begin
     if not Assigned(Stream) then exit;
     if UseUnixEOL then T:=T+#10 else T:=T+#13#10;
     Stream.Write( T[1], length(T) );
end;

function TTextfile.ReadBuf ( var Buffer; Count: LongInt ): LongInt;
begin
     if Assigned(Stream) then begin
        Result := Stream.Read( Buffer, Count );
     end else begin
        Result := 0;
     end;
end;

function TTextfile.WriteBuf( var Buffer; Count: LongInt ): LongInt;
begin
     if Assigned(Stream) then begin
        Result := Stream.Write( Buffer, Count );
     end else begin
        Result := 0;
     end;
end;

procedure TTextfile.Open;
begin
     if (OpenMode<>TEXTFILE_WRITE ) and
        (OpenMode<>TEXTFILE_APPEND) then OpenMode:=TEXTFILE_READ;

     if OpenMode=TEXTFILE_READ then begin
        Stream := TMemoryStream.Create;
        TMemoryStream(Stream).LoadFromFile( Filename );
        AutoDetectEOL;
     end else begin
        if (OpenMode=TEXTFILE_APPEND) and FileExists(Filename) then begin
           Stream := TFileStream.Create( Filename, fmOpenReadWrite );
           AutoDetectEOL;
           Stream.Position := Stream.Size;
        end else begin
           Stream := TFileStream.Create( Filename, fmCreate or fmOpenWrite );
        end;
     end;
end;

procedure TTextfile.Close;
begin
     if Assigned(Stream) then begin Stream.Free; Stream:=nil; end;
end;

constructor TTextfile.Create( AFilename: String; AOpenMode: Char );
begin
     inherited Create;
     Filename   := AFilename;
     OpenMode   := AOpenMode;
     Stream     := nil;
     UseUnixEOL := False;
     Open;
end;

destructor TTextfile.Destroy;
begin
     Close;
     inherited Destroy;
end;

end.
