// ============================================================================
// Messagefile reader/writer class
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cMsgFile;

interface

uses SysUtils, Windows, Classes, cTxtFile, BigStr8;

const
  MSGFILE_READ   = TEXTFILE_READ;
  MSGFILE_WRITE  = TEXTFILE_WRITE;
  MSGFILE_APPEND = TEXTFILE_APPEND;

  MSGFMT_UNKNOWN = 0;  // unknown format (causes immediate .EndOfFile=True, if
                       // format is not set manually after .Create)
  MSGFMT_SINGLE  = 1;  // single message in one file
  MSGFMT_MBOX    = 2;  // multiple messages (mbox-format)
  MSGFMT_RNEWS   = 3;  // multiple messages (rnews-format)
  MSGFMT_TXT2MSG = 99; // Special: converts plain-text into dummy-message
  RFC_DAY_NAMES  = 'MonTueWedThuFriSatSun';
  RFC_MONTH_NAMES= 'JanFebMarAprMayJunJulAugSepOctNovDec';


type
  TMessagefile = class
    protected
      Filename  : String;
      OpenMode  : Char;
      Textfile  : TTextfile;
      ByNumList : TList;
      ByNumDone : Boolean;
      OptForteMBox : Boolean;  //HRR -fa
      procedure Open;
      procedure Close;
      procedure AutoDetectFormat;
      function GetByNumCount: Integer;
      function GetByNumMessage( Index: Integer ): String;
      function GetForteMBox:Boolean;                   //HRR: -fa
      procedure SetForteMBox(ParaAgentMBox:Boolean);   //HRR: -fa
    public
      UseUnixEOL: Boolean;
      MessageFormat: Integer;

      property ByNumCount: Integer read GetByNumCount;
      property ByNumMessage[ Index: Integer ]: String read GetByNumMessage;
      property ForteMBox:Boolean read GetForteMBox write SetForteMBox;     //HRR: -fa

      function EndOfFile: Boolean;
      function ReadMessage: String;
      procedure WriteMessage( T: String );

      constructor Create( AFilename: String; AOpenMode: Char);
      destructor Destroy; override;
  end;

function ConvertCRLFtoMBOX( T: String; LFonly: Boolean ): String;
function ConvertCRLFtoRNEWS( T: String ): String;

implementation

// ---------------------------------------------------------------- Tools -----

function ConvertCRLFtoRNEWS( T: String ): String;
var  i: Integer;
begin
     T := ConvertCRLFtoLF( T );
     i := length( T );
     Result := '#! rnews ' + inttostr(i) + #10 + T;
end;

function ConvertCRLFtoMBOX( T: String; LFonly: Boolean ): String;
var  H: String;
     i: Integer;
     TS: TStringList;
begin
     TS := TStringList.Create;
     TS.Text := T;
     H := 'From dummy@dummy Sun Jul 11 00:00:00 1999';
     for i:=0 to TS.Count-1 do begin
        if copy(TS[i],1,5)='From ' then begin
           TS[i] := '>' + TS[i];
        end;
     end;
     if LFonly then Result := H + #10 + ConvertCRLFtoLF(TS.Text)
               else Result := H + #13#10 + TS.Text;
     TS.Free;
end;

// --------------------------------------------------------- TMessagefile -----

function TMessagefile.EndOfFile: Boolean;
begin
     if Assigned(Textfile) then begin
        Result := Textfile.EndOfFile or (MessageFormat=MSGFMT_UNKNOWN);
     end else begin
        Result := True;
     end;
end;

procedure TMessagefile.AutoDetectFormat;
var  SavePos: LongInt;
     s: String;
     c: Char;
begin
     UseUnixEOL := Textfile.UseUnixEOL;
     MessageFormat := MSGFMT_UNKNOWN;

     SavePos := TextFile.StreamDirect.Position;
     TextFile.StreamDirect.Position := 0;

     s := '';
     while TextFile.StreamDirect.Position<TextFile.StreamDirect.Size do begin
        TextFile.StreamDirect.Read( c, 1 );

        if (c=#10) or (c=#13) then begin
           if s<>'' then begin
              if copy(s,1,3)='#! ' then begin
                 if LowerCase(copy(s,1,9))='#! rnews ' then begin
                    MessageFormat := MSGFMT_RNEWS;
                    break;
                 end else begin
                    // ignore
                 end;
              end else begin
                 if copy(s,1,5)='From ' then begin
                    MessageFormat := MSGFMT_MBOX;
                 end else begin
                    if Pos( ':', s )>1 then MessageFormat := MSGFMT_SINGLE;
                 end;
                 break;
              end;
              s := '';
           end;
        end else begin
           s := s + c;
        end;
     end;

     TextFile.StreamDirect.Position := SavePos;
end;

function TMessagefile.GetByNumCount: Integer;
var  SavePos: LongInt;
begin
     Result := 0;
     if not Assigned(ByNumList) then exit;

     if not ByNumDone then begin
        SavePos := Textfile.StreamDirect.Position;
        while not ByNumDone do ReadMessage;
        Textfile.StreamDirect.Position := SavePos;
     end;

     Result := ByNumList.Count;
end;

function TMessagefile.GetByNumMessage( Index: Integer ): String;
var  SavePos: LongInt;
begin
     Result := '';
     if (Index<0) or (Index>=ByNumCount) then exit;

     SavePos := Textfile.StreamDirect.Position;
     Textfile.StreamDirect.Position := LongInt( ByNumList[Index] );
     Result := ReadMessage;
     Textfile.StreamDirect.Position := SavePos;
end;

function TMessagefile.ReadMessage: String;  //HRR -fa
var StartPos: LongInt;

   procedure ReadOneFile;
   var s8 : tBigString8;    //HRR
   begin
    //while not(EndOfFile) do Result := Result + Textfile.ReadLine + #13#10;
    s8:=tBigString8.create;                         //HRR Start
    while not(EndOfFile) do
     s8.addstring(Textfile.ReadLine + #13#10)
    ;
    result:=result+s8.wert;
    s8.free;                                        //HRR Ende
   end;

   procedure ReadOneMBox;
   var  s: String;
        p: LongInt;
        s8 : tBigString8;                           //HRR
   begin
        repeat
           s := Textfile.ReadLine;
           if (copy(s,1,5)='From ') and
               ((not ForteMBox) or (pos(':',s)>0)) then break;  //HRR -fa
        until EndOfFile;
        if EndOfFile then exit;
        s8:=tBigString8.Create;                      //HRR
        while not(EndOfFile) do begin
           p := Textfile.StreamDirect.Position;
           s := Textfile.ReadLine;
           if (copy(s,1,5)='From ') and
               ((not ForteMBox) or (pos(':',s)>0)) then begin  //HRR -fa
              Textfile.StreamDirect.Position := p;
              break;
           end;
           if copy(s,1,6)='>From ' then System.Delete(s,1,1);
           //Result := Result + s + #13#10;
           s8.addstring(s+#13#10);                   //HRR
        end;
        Result:=Result+s8.Wert;                      //HRR
        s8.Free;                                     //HRR
   end;

   procedure ReadOneRNews;
   var  s: String;
        l: LongInt;
   begin
        repeat
           s := Textfile.ReadLine;
           if LowerCase(copy(s,1,9))='#! rnews ' then break;
        until EndOfFile;
        if EndOfFile then exit;
        l := strtoint( copy(s,10,10) );
        SetLength( s, l );
        TextFile.ReadBuf( s[1], l );
        if UseUnixEOL then Result:=ConvertLFtoCRLF(s)
                      else Result:=s;
   end;

   procedure TextToMsg;
   const Counter: LongInt = 0;
   var   Header: String;
         OT: DWORD;
         FT: TFileTime;
         ST: TSystemTime;
   begin
        Randomize;
        inc( Counter );

        OT := FileAge(Filename);
        DosDateTimeToFileTime( (OT and $FFFF0000) shr 16,OT and $0000FFFF,FT );
        LocalFileTimeToFileTime( FT, FT );
        FileTimeToSystemTime( FT, ST );

        Header := 'Message-ID: <' + FormatDateTime('yyyymmddhhnnss',Now) + '.'
                                  + inttostr( Counter ) + '@txt2msg.r'
                                  + inttostr( Random(MaxLongInt) ) + '.invalid>'
                                  + #13#10
                + 'Subject: [FILE] ' + ExtractFilename(Filename) + ' ('
                                     + ExtractFilepath(Filename) + ')'
                                     + #13#10
                + 'From: "' + ExtractFilename(Filename)
                            + '" <dummy@txt2msg.invalid>'
                            + #13#10
                + 'Date: ' + copy(RFC_DAY_NAMES,ST.wDayOfWeek*3+1,3)
                           + ', ' + Format( '%.2d', [ST.wDay] ) + ' '
                           + copy(RFC_MONTH_NAMES,ST.wMonth*3-2,3)
                           + ' ' + Format( '%.4d %.2d:%.2d:%.2d GMT',
                                 [ST.wYear, ST.wHour, ST.wMinute, ST.wSecond] )
                           + #13#10
                + 'Newsgroups: invalid' + #13#10;

        Result := Header + #13#10 {Header-/Body-sep.} + Result;
   end;

begin
     Result := '';
     if EndOfFile or (MessageFormat=MSGFMT_UNKNOWN) then begin
        ByNumDone := True;
        exit;
     end;

     StartPos := Textfile.StreamDirect.Position;

     case MessageFormat of
        MSGFMT_SINGLE : ReadOneFile;
        MSGFMT_MBOX   : ReadOneMBox;
        MSGFMT_RNEWS  : ReadOneRNews;
        MSGFMT_TXT2MSG: begin ReadOneFile; TextToMsg; end;
     end;

     if not ByNumDone then begin
        if EndOfFile then begin
           ByNumDone := True
        end else begin
           if Result<>'' then ByNumList.Add( Pointer( StartPos ) );
        end;
     end;
end;

procedure TMessagefile.WriteMessage( T: String );

   procedure WriteOneFile;
   begin
        if UseUnixEOL then T:=ConvertCRLFtoLF(T);
        Textfile.WriteBuf( T[1], length(T) );
   end;

   procedure WriteOneMBox;
   begin
        T := ConvertCRLFtoMBOX( T, UseUnixEOL );
        Textfile.WriteBuf( T[1], length(T) );
   end;

   procedure WriteOneRNews;
   begin
        T := ConvertCRLFtoRNEWS( T );
        Textfile.WriteBuf( T[1], length(T) );
   end;

begin
     if MessageFormat=MSGFMT_UNKNOWN then exit;

     case MessageFormat of
        MSGFMT_SINGLE: WriteOneFile;
        MSGFMT_MBOX  : WriteOneMBox;
        MSGFMT_RNEWS : WriteOneRNews;
     end;
end;

procedure TMessagefile.Open;
begin
     if (OpenMode<>MSGFILE_WRITE ) and
        (OpenMode<>MSGFILE_APPEND) then OpenMode:=MSGFILE_READ;

     if OpenMode=MSGFILE_READ then begin
        Textfile := TTextfile.Create( Filename, TEXTFILE_READ );
        AutoDetectFormat;
     end else begin
        if (OpenMode=TEXTFILE_APPEND) and FileExists(Filename) then begin
           Textfile := TTextfile.Create( Filename, TEXTFILE_APPEND );
           AutoDetectFormat;
        end else begin
           Textfile := TTextfile.Create( Filename, TEXTFILE_WRITE );
        end;
     end;

     ByNumList := TList.Create;
     ByNumDone := False;
end;

procedure TMessagefile.Close;
begin
     if Assigned(Textfile ) then begin Textfile.Free;  Textfile :=nil; end;
     if Assigned(ByNumList) then begin ByNumList.Free; ByNumList:=nil; end;
end;

//HRR: -fa start
function tMessagefile.GetForteMBox:Boolean;
begin
 result:=OptForteMBox;
end;

procedure tMessageFile.SetForteMBox(ParaAgentMBox:Boolean);
begin
 OptForteMBox := ParaAgentMBox
end;
//HRR: -fa ende

constructor TMessagefile.Create( AFilename: String; AOpenMode: Char);
begin
     inherited Create;
     Filename   := AFilename;
     OpenMode   := AOpenMode;
     Textfile   := nil;
     UseUnixEOL := False;
     MessageFormat := MSGFMT_UNKNOWN;
     ByNumList  := nil;
     ByNumDone  := False;
     ForteMBox  := False;   //HRR -fa
     Open;
end;

destructor TMessagefile.Destroy;
begin
     Close;
     inherited Destroy;
end;

end.
