// ============================================================================
// Dynamically loaded functions for services
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uWinSvc;

{$ALIGN ON}
{$MINENUMSIZE 4}

interface

uses Windows;

const
   ERROR_FAILED_SERVICE_CONTROLLER_CONNECT = 1063;

   // controls
   SERVICE_CONTROL_STOP           = $00000001;
   SERVICE_CONTROL_PAUSE          = $00000002;
   SERVICE_CONTROL_CONTINUE       = $00000003;
   SERVICE_CONTROL_INTERROGATE    = $00000004;
   SERVICE_CONTROL_SHUTDOWN       = $00000005;

   // service State -- for currentState
   SERVICE_STOPPED                = $00000001;
   SERVICE_START_PENDING          = $00000002;
   SERVICE_STOP_PENDING           = $00000003;
   SERVICE_RUNNING                = $00000004;
   SERVICE_CONTINUE_PENDING       = $00000005;
   SERVICE_PAUSE_PENDING          = $00000006;
   SERVICE_PAUSED                 = $00000007;

   // controls accepted  (bit mask)
   SERVICE_ACCEPT_STOP            = $00000001;
   SERVICE_ACCEPT_PAUSE_CONTINUE  = $00000002;
   SERVICE_ACCEPT_SHUTDOWN        = $00000004;

   // service control manager object specific access types
   SC_MANAGER_CONNECT             = $0001;
   SC_MANAGER_CREATE_SERVICE      = $0002;
   SC_MANAGER_ENUMERATE_SERVICE   = $0004;
   SC_MANAGER_LOCK                = $0008;
   SC_MANAGER_QUERY_LOCK_STATUS   = $0010;
   SC_MANAGER_MODIFY_BOOT_CONFIG  = $0020;

   SC_MANAGER_ALL_ACCESS          = STANDARD_RIGHTS_REQUIRED or
                                    SC_MANAGER_CONNECT or
                                    SC_MANAGER_CREATE_SERVICE or
                                    SC_MANAGER_ENUMERATE_SERVICE or
                                    SC_MANAGER_LOCK or
                                    SC_MANAGER_QUERY_LOCK_STATUS or
                                    SC_MANAGER_MODIFY_BOOT_CONFIG;

   // service object specific access type
   SERVICE_QUERY_CONFIG           = $0001;
   SERVICE_CHANGE_CONFIG          = $0002;
   SERVICE_QUERY_STATUS           = $0004;
   SERVICE_ENUMERATE_DEPENDENTS   = $0008;
   SERVICE_START                  = $0010;
   SERVICE_STOP                   = $0020;
   SERVICE_PAUSE_CONTINUE         = $0040;
   SERVICE_INTERROGATE            = $0080;
   SERVICE_USER_DEFINED_CONTROL   = $0100;
 
   SERVICE_ALL_ACCESS             = STANDARD_RIGHTS_REQUIRED or
                                    SERVICE_QUERY_CONFIG or
                                    SERVICE_CHANGE_CONFIG or
                                    SERVICE_QUERY_STATUS or
                                    SERVICE_ENUMERATE_DEPENDENTS or
                                    SERVICE_START or
                                    SERVICE_STOP or
                                    SERVICE_PAUSE_CONTINUE or
                                    SERVICE_INTERROGATE or
                                    SERVICE_USER_DEFINED_CONTROL;


   // service types (bit mask)
   SERVICE_KERNEL_DRIVER          = $00000001;
   SERVICE_FILE_SYSTEM_DRIVER     = $00000002;
   SERVICE_ADAPTER                = $00000004;
   SERVICE_RECOGNIZER_DRIVER      = $00000008;

   SERVICE_DRIVER                 = SERVICE_KERNEL_DRIVER or
                                    SERVICE_FILE_SYSTEM_DRIVER or
                                    SERVICE_RECOGNIZER_DRIVER;

   SERVICE_WIN32_OWN_PROCESS      = $00000010;
   SERVICE_WIN32_SHARE_PROCESS    = $00000020;
   SERVICE_WIN32                  = SERVICE_WIN32_OWN_PROCESS or
                                    SERVICE_WIN32_SHARE_PROCESS;

   SERVICE_INTERACTIVE_PROCESS    = $00000100;

   SERVICE_TYPE_ALL               = SERVICE_WIN32 or
                                    SERVICE_ADAPTER or
                                    SERVICE_DRIVER or
                                    SERVICE_INTERACTIVE_PROCESS;

   // start type
   SERVICE_BOOT_START             = $00000000;
   SERVICE_SYSTEM_START           = $00000001;
   SERVICE_AUTO_START             = $00000002;
   SERVICE_DEMAND_START           = $00000003;
   SERVICE_DISABLED               = $00000004;

   // error control type
   SERVICE_ERROR_IGNORE           = $00000000;
   SERVICE_ERROR_NORMAL           = $00000001;
   SERVICE_ERROR_SEVERE           = $00000002;
   SERVICE_ERROR_CRITICAL         = $00000003;

type
   // handle types
   SC_HANDLE = THandle;
   LPSC_HANDLE = ^SC_HANDLE;
   SERVICE_STATUS_HANDLE = DWORD;

   // pointer to string pointer
   PLPSTRA  = ^PAnsiChar;
   PLPWSTRW = ^PWideChar;
   PLPSTR   = PLPSTRA;

   // service status structure
   PServiceStatus = ^TServiceStatus;
   TServiceStatus = record
      dwServiceType: DWORD;
      dwCurrentState: DWORD;
      dwControlsAccepted: DWORD;
      dwWin32ExitCode: DWORD;
      dwServiceSpecificExitCode: DWORD;
      dwCheckPoint: DWORD;
      dwWaitHint: DWORD;
   end;

   // function prototype for the service main function
   TServiceMainFunction = TFarProc;

   // service start table
   PServiceTableEntry = ^TServiceTableEntry;
   TServiceTableEntry = record
      lpServiceName: PAnsiChar;
      lpServiceProc: TServiceMainFunction;
   end;

   // prototype for the service control handler function
   THandlerFunction = TFarProc;

type
   // API function prototypes
   TCloseServiceHandle = function (
      hSCObject: SC_HANDLE ): BOOL; stdcall;
   TControlService = function (
      hService: SC_HANDLE; dwControl: DWORD;
      var lpServiceStatus: TServiceStatus ): BOOL; stdcall;
   TCreateService = function (
      hSCManager: SC_HANDLE; lpServiceName, lpDisplayName: PChar;
      dwDesiredAccess, dwServiceType, dwStartType, dwErrorControl: DWORD;
      lpBinaryPathName, lpLoadOrderGroup: PChar; lpdwTagId: LPDWORD;
      lpDependencies, lpServiceStartName, lpPassword: PChar
      ): SC_HANDLE; stdcall;
   TChangeServiceConfig = function (
      hService: SC_HANDLE; dwServiceType, dwStartType, dwErrorControl: DWORD;
      lpBinaryPathName, lpLoadOrderGroup: PAnsiChar; lpdwTagId: LPDWORD;
      lpDependencies, lpServiceStartName, lpPassword, lpDisplayName: PAnsiChar
      ): BOOL; stdcall;
   TDeleteService = function (
       hService: SC_HANDLE ): BOOL; stdcall;
   TOpenSCManager = function (
       lpMachineName, lpDatabaseName: PChar; dwDesiredAccess: DWORD
       ): SC_HANDLE; stdcall;
   TOpenService = function (
      hSCManager: SC_HANDLE; lpServiceName: PChar; dwDesiredAccess: DWORD
      ): SC_HANDLE; stdcall;
   TQueryServiceStatus = function (
      hService: SC_HANDLE; var lpServiceStatus: TServiceStatus
      ): BOOL; stdcall;
   TRegisterServiceCtrlHandler = function (
      lpServiceName: PChar; lpHandlerProc: ThandlerFunction
      ): SERVICE_STATUS_HANDLE; stdcall;
   TSetServiceStatus = function (
      hServiceStatus: SERVICE_STATUS_HANDLE;
      var lpServiceStatus: TServiceStatus ): BOOL; stdcall;
   TStartServiceCtrlDispatcher = function (
      var lpServiceStartTable: TServiceTableEntry ): BOOL; stdcall;
   TStartService = function (
      hService: SC_HANDLE; dwNumServiceArgs: DWORD;
      var lpServiceArgVectors: PChar ): BOOL; stdcall;

var
   // API functions
   CloseServiceHandle        : TCloseServiceHandle         = nil;
   ControlService            : TControlService             = nil;
   CreateService             : TCreateService              = nil;
   ChangeServiceConfig       : TChangeServiceConfig        = nil;
   DeleteService             : TDeleteService              = nil;
   OpenSCManager             : TOpenSCManager              = nil;
   OpenService               : TOpenService                = nil;
   QueryServiceStatus        : TQueryServiceStatus         = nil;
   RegisterServiceCtrlHandler: TRegisterServiceCtrlHandler = nil;
   SetServiceStatus          : TSetServiceStatus           = nil;
   StartServiceCtrlDispatcher: TStartServiceCtrlDispatcher = nil;
   StartService              : TStartService               = nil;

function IsWinSvcAvailable: Boolean;
function IsPlatformNT: Boolean;


implementation

const
   sADVAPI32 = 'advapi32.dll';

var
   hADVAPI32: HModule = 0;


function IsWinSvcAvailable: Boolean;
begin
   Result := ( hADVAPI32 <> 0 );
end;

function IsPlatformNT: Boolean;
var  VerInfo: TOSVersionInfo;
begin
   VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
   GetVersionEx( VerInfo );
   Result := ( VerInfo.dwPlatformId = VER_PLATFORM_WIN32_NT );
end;

procedure Load_ADVAPI32;
begin
   hADVAPI32 := LoadLibrary( sADVAPI32 );
   if hADVAPI32 < 32 then begin
      hADVAPI32 := 0;
      exit;
   end;

   @CloseServiceHandle         := GetProcAddress( hADVAPI32, 'CloseServiceHandle' );
   @ControlService             := GetProcAddress( hADVAPI32, 'ControlService' );
   @CreateService              := GetProcAddress( hADVAPI32, 'CreateServiceA' );
   @ChangeServiceConfig        := GetProcAddress( hADVAPI32, 'ChangeServiceConfigA' );
   @DeleteService              := GetProcAddress( hADVAPI32, 'DeleteService' );
   @OpenSCManager              := GetProcAddress( hADVAPI32, 'OpenSCManagerA' );
   @OpenService                := GetProcAddress( hADVAPI32, 'OpenServiceA' );
   @QueryServiceStatus         := GetProcAddress( hADVAPI32, 'QueryServiceStatus' );
   @RegisterServiceCtrlHandler := GetProcAddress( hADVAPI32, 'RegisterServiceCtrlHandlerA' );
   @SetServiceStatus           := GetProcAddress( hADVAPI32, 'SetServiceStatus' );
   @StartServiceCtrlDispatcher := GetProcAddress( hADVAPI32, 'StartServiceCtrlDispatcherA' );
   @StartService               := GetProcAddress( hADVAPI32, 'StartServiceA' );
end;


initialization
   if IsPlatformNT then Load_ADVAPI32;

finalization
   if hADVAPI32 <> 0 then FreeLibrary( hADVAPI32 );

end.

