// ============================================================================
// WinSock-related tools.
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uWinSock; // WinSock-related tools

// ----------------------------------------------------------------------------
// Contains WinSock-related tools
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

function nAddrToStr( nAddr: LongInt ): String;
function hAddrToStr( hAddr: LongInt ): String;
function StrTohAddr( const ipStr: String ): LongInt;
function LookupLocalHostName: String;
function LookupLocalHostAddr: LongInt;
function LookupHostName( HostAddr: LongInt ): String;
function LookupHostAddr( Hostname: String ): LongInt;
function LookupServicePort( Servicename: String; Default: Integer = 0 ): LongInt;
function IsLocalHost( nAddr: LongInt ): Boolean;
function WinsockErrTxt( ErrNo: LongInt ): String;

// ----------------------------------------------------------------------------

implementation

uses SysUtils, WinSock, uTools;

function nAddrToStr( nAddr: LongInt ): String;
begin
     Result := hAddrToStr( ntohl( nAddr ) );
end;

function hAddrToStr( hAddr: LongInt ): String;
begin
     Result := inttostr( (hAddr shr 24)         ) + '.'
             + inttostr( (hAddr shr 16) and $ff ) + '.'
             + inttostr( (hAddr shr 8 ) and $ff ) + '.'
             + inttostr( (hAddr       ) and $ff );
end;

function StrTohAddr( const ipStr: String ): LongInt;
var  P: TParser;
     s: String;
begin
   Result := 0;
   if ipStr = '' then exit;
   if not( ipStr[1] in ['0'..'9'] ) then exit;

   P := TParser.Create;
   try
      P.Parse( ipStr, '.' );
      s := P.sPart( 0, '' );  Result := Result or ( (strtointdef(s,0) and $FF) shl 24 );
      s := P.sPart( 1, '' );  Result := Result or ( (strtointdef(s,0) and $FF) shl 16 );
      s := P.sPart( 2, '' );  Result := Result or ( (strtointdef(s,0) and $FF) shl  8 );
      s := P.sPart( 3, '' );  Result := Result or ( (strtointdef(s,0) and $FF)        );
   finally P.Free end;
end;

function LookupLocalHostName: String;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
begin
     Result:='localhost';
     GetMem( HostNam, SIZE );

     try
        if GetHostName( HostNam, SIZE ) = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then Result := String( HostEnt^.h_name );
           end;
        end;
     finally
       FreeMem( HostNam, SIZE );
     end;
end;

function LookupLocalHostAddr: LongInt;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
begin
     Result := $7F000001; // 127.0.0.1
     GetMem( HostNam, SIZE );

     try
        if GetHostName( HostNam, SIZE ) = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then begin
                 Result := LongInt( Pointer( HostEnt^.h_addr_list^ )^ );
              end;
           end;
        end;
     finally
        FreeMem( HostNam, SIZE );
     end;
end;

function IsLocalHost( nAddr: LongInt ): Boolean;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
     AddrList: Pointer;
begin
     Result := False;
     if nAddr=$0100007f then begin Result:=True; exit; end;

     GetMem( HostNam, SIZE );
     try
        if GetHostName( HostNam, SIZE ) = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then begin
                 try
                    AddrList := Pointer( HostEnt^.h_addr_list^ );
                    while (LongInt(AddrList)<>0) and (LongInt(AddrList^)<>0) do begin
                       if LongInt( AddrList^ )=nAddr then begin Result:=True; break; end;
                       AddrList := Pointer( LongInt(AddrList) + sizeof(LongInt) );
                    end;
                 except
                 end;
              end;
           end;
        end;
     finally
        FreeMem( HostNam, SIZE );
     end;
end;

function LookupHostAddr( Hostname: String ): LongInt;
var  HostEnt: PHostEnt;
begin
     Result := 0;
     if HostName='' then exit;

     try
        Result := inet_addr( PChar(HostName) );
        if Result=LongInt(INADDR_NONE) then begin
           HostEnt := GetHostByName( PChar(HostName) );
           if HostEnt<>nil then begin
              Result := LongInt( Pointer( HostEnt^.h_addr_list^ )^ );
           end else begin
              Result := 0; 
           end;
        end;
     except
        Result := 0;
     end;
end;

function LookupHostName( HostAddr: LongInt ): String;
var  HostEnt: PHostEnt;
begin
     Result := '[' + nAddrToStr( HostAddr ) + ']';

     try
        HostEnt := GetHostByAddr( @HostAddr, 4, PF_INET );
        if HostEnt<>nil then Result := String( HostEnt^.h_name );
     except
     end;
end;

function LookupServicePort( Servicename: String; Default: Integer = 0 ): LongInt;
var  ServEnt: PServEnt;
begin
     Result := Default;
     if Servicename = '' then exit;

     if Servicename[1] in ['0'..'9'] then begin
        Result := strtointdef( Servicename, Default );
        exit;
     end;

     if          LowerCase( Servicename ) = 'nntp'   then begin
        Result := 119; exit
     end else if LowerCase( Servicename ) = 'pop3'   then begin
        Result := 110; exit;
     end else if LowerCase( Servicename ) = 'smtp'   then begin
        Result := 25; exit;
     end else if LowerCase( Servicename ) = 'domain' then begin
        Result := 53; exit;
     end;

     try
        ServEnt := getservbyname( PChar(Servicename), nil );
        if ServEnt <> nil then Result := ntohs( ServEnt^.s_port );
     except
        Result := Default;
     end;
end;

function WinsockErrTxt( ErrNo: LongInt ): String;
begin
     case ErrNo of
        10004: Result := 'WSAEINTR';    
        10009: Result := 'WSAEBADF';
        10013: Result := 'WSEACCES';
        10014: Result := 'WSAEFAULT';
        10022: Result := 'WSAEINVAL';
        10024: Result := 'WSAEMFILE';
        10035: Result := 'WSAEWOULDBLOCK';
        10036: Result := 'WSAEINPROGRESS';
        10037: Result := 'WSAEALREADY';
        10038: Result := 'WSAENOTSOCK';
        10039: Result := 'WSAEDESTADDRREQ';
        10040: Result := 'WSAEMSGSIZE';
        10041: Result := 'WSAEPROTOTYPE';
        10042: Result := 'WSAENOPROTOOPT';
        10043: Result := 'WSAEPROTONOSUPPORT';
        10044: Result := 'WSAESOCKTNOSUPPORT';
        10045: Result := 'WSAEOPNOTSUPP';
        10046: Result := 'WSAEPFNOSUPPORT';
        10047: Result := 'WSAEAFNOSUPPORT';
        10048: Result := 'WSAEADDRINUSE';
        10049: Result := 'WSAEADDRNOTAVAIL';
        10050: Result := 'WSAENETDOWN';
        10051: Result := 'WSAENETUNREACH';
        10052: Result := 'WSAENETRESET';
        10053: Result := 'WSAECONNABORTED';
        10054: Result := 'WSAECONNRESET';
        10055: Result := 'WSAENOBUFS';
        10056: Result := 'WSAEISCONN';
        10057: Result := 'WSAENOTCONN';
        10058: Result := 'WSAESHUTDOWN';
        10059: Result := 'WSAETOOMANYREFS';
        10060: Result := 'WSAETIMEDOUT';
        10061: Result := 'WSAECONNREFUSED';
        10062: Result := 'WSAELOOP';
        10063: Result := 'WSAENAMETOOLONG';
        10064: Result := 'WSAEHOSTDOWN';
        10065: Result := 'WSAEHOSTUNREACH';
        10091: Result := 'WSASYSNOTREADY';
        10092: Result := 'WSAVERNOTSUPPORTED';
        10093: Result := 'WSANOTINITIALISED';
        10101: Result := 'WSAEDISCON';
        11001: Result := 'WSAHOST_NOT_FOUND';
        11002: Result := 'WSATRY_AGAIN';
        11003: Result := 'WSANO_RECOVERY';
        11004: Result := 'WSANO_DATA';
        else Result:='';
     end;
end;

// ----------------------------------------------------------------------------

end.
