// ============================================================================
// Various tools and functions
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uTools; // Various tools and functions

// ----------------------------------------------------------------------------
// Contains a really horrible, historically grown mix of various tools and
// functions ...
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

{$R-}
{$Q-}

uses SysUtils, Classes, Windows, SyncObjs;

type
   TParser = class( TStringList )
      public
        function sPart( PartNo: Integer; const DefVal: String ): String;
        function iPart( PartNo: Integer; const DefVal: Integer ): Integer;
        function i64Part( PartNo: Integer; const DefVal: Int64 ): Int64;
        procedure Parse( const ALine: String; ASplit: Char ); // deprecated
        procedure SplitWhSpace( const ALine: String );
        procedure SplitChar( const ALine: String; ASplit: Char );
   end;

   TStringListEx = class( TStringList )
      public
         procedure AppendToFile( const FileName: string );
   end;

   // TStringList_ExactMatch: Like TStringList, but uses CompareStr instead of
   // AnsiCompareText, so "a" and "A" will be different entries here.
   TStringList_ExactMatch = class( TStringListEx )
      public
         function Find(const S: string; var Index: Integer): Boolean; override;
         function IndexOf(const S: string): Integer; override;
         procedure Sort; override;
   end;

   // TStringList_NoAnsi: Like TStringList, but uses CompareText instead of
   // AnsiCompareText, so "a" and "A" are the same, but "" and "" will be
   // different entries here. Significantly faster if appropriate.
   TStringList_NoAnsi = class( TStringListEx )
      public
         function Find(const S: string; var Index: Integer): Boolean; override;
         function IndexOf(const S: string): Integer; override;
         procedure Sort; override;
   end;

   // TExpireStrings: A list of unique strings that expire after a given
   // number of seconds.
   TExpireStrings = class
      private
         FList          : TStringList;
         FDefaultSeconds: Integer;
         procedure expireStrings;
      public
         function Add( const s: String ): Boolean; overload;
         function Add( const s: String;
                       const refreshTime: Boolean ): Boolean; overload;
         function Add( const s: String;
                       const expireSeconds: Integer ): Boolean; overload;
         function Add( const s: String;
                       const expireSeconds: Integer;
                       const refreshTime: Boolean ): Boolean; overload;
         function Contains( const s : String ): Boolean;
         procedure Remove( const s : String );

         constructor Create( const defaultExpireSeconds: Integer );
         destructor Destroy; override;
   end;

   TKeyValueCache = class
      private
         FFilename: String;
         FChanged : Boolean;
         FLock    : TCriticalSection;
         FItems   : TStringList;
         FMaxItems: Integer;

         procedure loadCache;
         procedure saveCache;

      public
         function  Get( const Key: String ): String;
         procedure Add( const Key, Value: String );

         procedure Clear;
         procedure Flush;
         function  Purge( const MaxAgeMinutes: Integer ): Integer;

         constructor Create( const AFilename: String;
                             const AMaxItems: Integer );
         destructor Destroy; override;
   end;


function strtoint( const s: String ) : Integer;
function sgn( i: Integer ): Integer;
function CountChar( c: Char; const s: String ): Integer;
function ExtractStr( const SourceStr: String;
                     Separator: Char;
                     Index: Integer ) : String; overload;
function ExtractStr( const SourceStr: String;
                     Separator: Char;
                     Index: Integer;
                     const Default: String ) : String; overload;
function TrimCrLf( s: String ): String;
function TrimWhSpace( const s : String ) : String;
function PosWhSpace( const s : String ) : Integer;
function EndsWith( const s: String; const test: String; ignoreCase: Boolean = True ): Boolean;

function ArgsSplitChar( const Line: String;
                        const Args: TStringList;
                        SplitChar: Char;
                        FillupArgs: Integer = 0 ): Integer; overload;
function ArgsSplitChar( const Line: String;
                        const Args: TStringList;
                        SplitChar: Char;
                        DQuoted: Boolean;
                        FillupArgs: Integer = 0 ): Integer; overload;
function ArgsSplitChars( const Line: String;
                         const Args: TStringList;
                         SplitChars: Array of Char;
                         FillupArgs: Integer = 0 ): Integer;
function ArgsWhSpace( const Line: String;
                      const Args: TStringList;
                      FillupArgs: Integer = 0 ): Integer;
function ArgsWhSpaceDQuoted( const Line: String;
                             const Args: TStringList;
                             FillupArgs: Integer = 0 ): Integer;
function DQuoteStr( const Str: String ): String;
function DQuoteStrFix( const Str: String ): String;
function UnDQuoteStr( const Str: String ): String;
function NextTopLine( var Lines: String ): String;
function NextSepPart( var Line: String; const Separator: Char ): String;
function NextWhSpacePart( var Line: String ): String;
function TextToRfcWireFormat( const OrgText: String ): String;

procedure AddChar( var s: String; const c: Char ); // s := s + c

procedure StreamToString( const Strm: TStream; var Result: String );
procedure StreamToFile( const Strm: TStream; const FileName: String );
procedure StreamWriteLn( const Strm: TStream; const Line: String );

function iif( AValue: Boolean; const AThen: String;  const AElse: String = '' ): String;  overload;
function iif( AValue: Boolean; const AThen: Integer; const AElse: Integer ): Integer; overload;

function FilterNameOfFrom( const F: String ) : String;
procedure ForceFileExists( const Filename: String );
function LikeRadix32( l : Longint ): String;
function MIDGenerator( FQDN: String; LocalSuffix: String = '' ): String;
function UIDLGenerator: String;
function GetExeVersion : String;
function GetFileInfo( const Filename: String ): String;
function Winsock2Installed( Silent: Boolean = False ): Boolean;
function GetWindowsPath: String;
function GetSystemPath: String;
function IsWindowsNT: Boolean;
function GetWinVerInfo: String;
function ReplaceChar( s: String; oldc, newc: Char ): String;
function GetFileSize( const Filename: String ): LongInt;
function WildMat( pText: PChar; pPattern: PChar ): Boolean;
function IsDomain( const s: String ): Boolean;
function IsNewsgroup( const s: String ): Boolean;
function IsMessageId( const s: String ): Boolean;
function IsFilename( const c: Char ): Boolean;
function GetMyVersionInfo : String;
function GetMyStringFileInfo( const StringName, DefResult: String ): String;
function FileExistsByPattern( const SearchFile: String ): Boolean;
function DirectoryExists(const Name: string): Boolean;
function ForceDirectories(Dir: string): Boolean;
function WinErrMsg( ErrorCode: DWORD ): String; overload;
function WinErrMsg: String; overload;

function ExtractEnvelopeAddr( const EnvAddr: String ): String;
function ExtractMailAddr( const MailAddr: String ): String;
function ExtractMailDomain( const MailAddr: String ): String;

// ----------------------------------------------------------------------------

implementation

uses cPCRE, uCRC32, uDateTime;

// -------------------------------------------------------------- TParser -----

function TParser.sPart( PartNo: Integer; const DefVal: String ): String;
begin
     if (PartNo>=0) and (PartNo<Count) then Result:=Strings[PartNo]
                                       else Result:=DefVal;
end;

function TParser.iPart( PartNo: Integer; const DefVal: Integer ): Integer;
begin
     try
        if (PartNo>=0) and (PartNo<Count) then Result:=strtoint(Strings[PartNo])
                                          else Result:=DefVal;
     except
        Result := DefVal;
     end;
end;

function TParser.i64Part( PartNo: Integer; const DefVal: Int64 ): Int64;
begin
     try
        if (PartNo>=0) and (PartNo<Count) then Result:=strtoint64(Strings[PartNo])
                                          else Result:=DefVal;
     except
        Result := DefVal;
     end;
end;

procedure TParser.Parse( const ALine: String; ASplit: Char ); // deprecated
begin
   if ASplit = ' ' then ArgsWhSpace  ( ALine, Self )
                   else ArgsSplitChar( ALine, Self, ASplit );
end;

procedure TParser.SplitWhSpace( const ALine: String );
begin
   ArgsWhSpace( ALine, Self );
end;

procedure TParser.SplitChar( const ALine: String; ASplit: Char );
begin
   ArgsSplitChar( ALine, Self, ASplit );
end;


// ----------------------------------------------- TStringList_ExactMatch -----

function TStringList_ExactMatch.Find(const S: string; var Index: Integer): Boolean;
var  L, H, I, C: Integer;
begin
   Result := False;
   L := 0;
   H := GetCount - 1;
   while L <= H do begin
      I := ( L + H ) shr 1;
      C := CompareStr( Get(I), S );
      if C < 0 then
         L := I + 1
      else begin
         H := I - 1;
         if C = 0 then begin
            Result := True;
            if Duplicates <> dupAccept then L := I;
         end;
      end;
   end;
   Index := L;
end;

function TStringList_ExactMatch.IndexOf( const S: string ): Integer;
begin
   if not Sorted then begin
      for Result := GetCount - 1 downto 0 do
         if CompareStr( Get(Result), S ) = 0 then exit;
      Result := -1;
   end else begin
      if not Find( S, Result ) then Result := -1;
   end;
end;

function StringListCompare_ExactMatch( List: TStringList; I1, I2: Integer ): Integer;
begin
   Result := CompareStr( List[I1], List[I2] );
end;

procedure TStringList_ExactMatch.Sort;
begin
   CustomSort( StringListCompare_ExactMatch );
end;

// --------------------------------------------------- TStringList_NoAnsi -----

function TStringList_NoAnsi.Find(const S: string; var Index: Integer): Boolean;
var  L, H, I, C: Integer;
begin
   Result := False;
   L := 0;
   H := GetCount - 1;
   while L <= H do begin
      I := ( L + H ) shr 1;
      C := CompareText( Get(I), S );
      if C < 0 then
         L := I + 1
      else begin
         H := I - 1;
         if C = 0 then begin
            Result := True;
            if Duplicates <> dupAccept then L := I;
         end;
      end;
   end;
   Index := L;
end;

function TStringList_NoAnsi.IndexOf( const S: string ): Integer;
begin
   if not Sorted then begin
      for Result := GetCount - 1 downto 0 do
         if CompareText( Get(Result), S ) = 0 then exit;
      Result := -1;
   end else begin
      if not Find( S, Result ) then Result := -1;
   end;
end;

function StringListCompare_NoAnsi( List: TStringList; I1, I2: Integer ): Integer;
begin
   Result := CompareText( List[I1], List[I2] );
end;

procedure TStringList_NoAnsi.Sort;
begin
   CustomSort( StringListCompare_NoAnsi );
end;

// -------------------------------------------------------- TStringListEx -----

procedure TStringListEx.AppendToFile( const FileName: string );
var  Stream: TStream;
begin
   if FileExists(FileName) then begin
      Stream := TFileStream.Create( FileName, fmOpenWrite );
   end else begin
      Stream := TFileStream.Create( FileName, fmCreate );
   end;

   try
      Stream.Seek( 0, soFromEnd );
      SaveToStream( Stream );
   finally
      Stream.Free;
   end
end;

// ---------------------------------------------------- TExpireStringList -----

constructor TExpireStrings.Create(const defaultExpireSeconds: Integer);
begin
   inherited Create;

   FList            := TStringList.Create;
   FList.Sorted     := True;
   FList.Duplicates := dupIgnore;
   
   FDefaultSeconds  := defaultExpireSeconds;
end;

destructor TExpireStrings.Destroy;
begin
   if Assigned( FList ) then FreeAndNil( FList );
   inherited Destroy;
end;

procedure TExpireStrings.expireStrings;
var  i, expiredUntil: Integer;
begin
   expiredUntil := DateTimeToUnixTime( NowGMT );
   for i := FList.Count - 1 downto 0 do begin
      if Integer( FList.Objects[i] ) <= expiredUntil then FList.Delete( i );
   end;
end;

function TExpireStrings.Add( const s: String ): Boolean;
begin
   Result := Add( s, FDefaultSeconds, False );
end;

function TExpireStrings.Add( const s: String;
                             const expireSeconds: Integer ): Boolean;
begin
   Result := Add( s, expireSeconds, False );
end;

function TExpireStrings.Add( const s: String;
                             const refreshTime: Boolean ): Boolean;
begin
   Result := Add( s, FDefaultSeconds, refreshTime );
end;

function TExpireStrings.Add( const s: String;
                             const expireSeconds: Integer;
                             const refreshTime: Boolean ): Boolean;
var  i, expireTime: Integer;
begin
   // remove expired strings
   expireStrings;

   // add string with expire time
   expireTime := DateTimeToUnixTime( NowGMT ) + expireSeconds;
   i := FList.IndexOf( s );
   if i < 0 then begin
      Result := True;  // ok, new string added
      FList.AddObject( s, Pointer( expireTime ) );
   end else begin
      Result := False; // already in list
      if refreshTime then FList.Objects[ i ] := Pointer( expireTime );
   end;
end;

function TExpireStrings.Contains( const s : String ): Boolean;
begin
   expireStrings;
   Result := ( FList.IndexOf( s ) >= 0 );
end;

procedure TExpireStrings.Remove( const s: String );
var  i: Integer;
begin
   i := FList.IndexOf( s );
   if i >= 0 then FList.Delete( i );
end;

// ------------------------------------------------------- TKeyValueCache -----

constructor TKeyValueCache.Create( const AFilename: String;
                                   const AMaxItems: Integer );
begin
   inherited Create;
   FFilename := AFilename;
   FMaxItems := AMaxItems;
   FChanged  := False;
   FItems    := TStringList.Create;
   FLock     := TCriticalSection.Create;
   if length( FFilename ) > 0 then loadCache;
end;

destructor TKeyValueCache.Destroy;
begin
   if Assigned( FItems ) then begin
      if FChanged then Flush;
      FreeAndNil( FItems );
   end;
   if Assigned( FLock ) then FreeAndNil( FLock );
   inherited Destroy;
end;


procedure TKeyValueCache.Add( const Key, Value: String );
var  i: Integer;
begin
   FLock.Enter;
   try
      i := FItems.IndexOfName( Key );
      if i >= 0 then begin
         FItems[i] := Key + '=' + Value;
         FItems.Objects[i] := Pointer( DateTimeToUnixTime(Now) );
      end else begin
         FItems.AddObject( Key + '=' + Value,
                           Pointer( DateTimeToUnixTime(Now) ) );
         while FItems.Count > FMaxItems do FItems.Delete(0);
      end;
      FChanged := True;
   finally FLock.Leave end;
end;

function TKeyValueCache.Get( const Key: String ): String;
begin
   FLock.Enter;
   try
      Result := FItems.Values[ Key ];
   finally FLock.Leave end;
end;

procedure TKeyValueCache.Clear;
begin
   FLock.Enter;
   try
      FItems.Clear;
      FChanged := True;
   finally FLock.Leave end;
end;

procedure TKeyValueCache.Flush;
begin
   FLock.Enter;
   try
      if FChanged then begin
         if length( FFilename ) > 0 then saveCache
                                    else FChanged := False;
      end;
   finally FLock.Leave end;
end;

function TKeyValueCache.Purge( const MaxAgeMinutes: Integer ): Integer;
var  i, t, tNow: Integer;
begin
   FLock.Enter;
   try
      tNow := DateTimeToUnixTime( Now );
      for i := FItems.Count - 1 downto 0 do begin
         t := Integer( FItems.Objects[i] );
         if ( (tNow - t) div 60 ) > MaxAgeMinutes then begin
            FItems.Delete( i );
            FChanged := True;
         end;
      end;
      Flush;
      Result := FItems.Count;
   finally FLock.Leave end;
end;


procedure TKeyValueCache.loadCache;
var  i, t: Integer;
     s: String;
     sl: TStringList;
begin
   FLock.Enter;
   try

      sl := TStringList.Create;
      try

         if FileExists( FFilename ) then sl.LoadFromFile( FFilename );

         Clear;
         for i := 0 to sl.Count - 1 do begin
            t := strtointdef( '$' + copy( sl[i], 1, 8 ), 0 );
            s := copy( sl[i], 10, MaxInt );
            FItems.AddObject( s, Pointer(t) );
         end;
         FChanged := False;

      finally sl.Free end;

   finally FLock.Leave end;
end;

procedure TKeyValueCache.saveCache;
var  i: Integer;
     sl: TStringList;
begin
   FLock.Enter;
   try

      sl := TStringList.Create;
      try
         for i := 0 to FItems.Count - 1 do begin
            sl.Add( inttohex( Integer( FItems.Objects[i] ), 8 )
                  + ':' + FItems[i] );
         end;

         sl.SaveToFile( FFilename );
         FChanged := False;

      finally sl.Free end;

   finally FLock.Leave end;
end;

// ----------------------------------------------------------------------------

function strtoint( const s: String ) : Integer;
begin
   Result := strtointdef( s, 0 );
end;

function sgn( i: Integer ): Integer;
begin
   if i<0 then Result:=-1 else if i>0 then Result:=+1 else Result:=0;
end;

function CountChar( c: Char; const s: String ): Integer;
var  i: Integer;
begin
   Result := 0;
   for i := 1 to length(s) do begin
      if s[i] = c then Inc( Result );
   end;
end;

function ExtractStr( const SourceStr: String;
                     Separator: Char;
                     Index: Integer ) : String;
var  pb, pe: PChar;
begin
   SetLength( Result, 0 );
   pb := PChar( SourceStr );

   while pb^ <> #0 do begin
      pe := StrScan( pb, Separator );
      if not Assigned( pe ) then pe := strend( pb );

      if Index = 0 then begin
         SetString( Result, pb, pe - pb );
         exit;
      end;

      pb := pe + 1;
      dec( Index );
   end;
end;

function ExtractStr( const SourceStr: String;
                     Separator: Char;
                     Index: Integer;
                     const Default: String ) : String;
begin
   Result := ExtractStr( SourceStr, Separator, Index );
   if length(Result) = 0 then Result := Default;
end;

function TrimCrLf( s: String ): String;
begin
   if copy(s,length(s),1)=#10 then System.Delete(s,length(s),1);
   if copy(s,length(s),1)=#13 then System.Delete(s,length(s),1);
   Result := s;
end;

function TrimWhSpace( const s : String ) : String;
var  pb, pe: PChar;
begin
   SetLength( Result, 0 );

   pb := PChar( s );
   while pb^ in [#9,' '] do inc(pb);
   if pb^=#0 then exit;

   pe := strend( pb ) - 1;
   while pe^ in [#9,' '] do dec(pe);

   // Result := copy( pb, 1, pe-pb+1 );
   SetString( Result, pb, pe-pb+1 );
end;

function PosWhSpace( const s : String ) : Integer;
var  j: Integer;
begin
     Result := Pos( ' ', s );
     if Result=0 then begin
        Result := Pos( #9, s );
     end else begin
        j := Pos( #9, s );
        if (j>0) and (j<Result) then Result:=j;
     end;
end;

function EndsWith( const s: String; const test: String; ignoreCase: Boolean = True ): Boolean;
var  h: String;
begin
   Result := False;
   if length(s) < length(test) then exit;

   h := copy( s, length(s)-length(test)+1, length(test) );
   if ignoreCase then begin
      Result := AnsiSameText( h, test );
   end else begin
      Result := (h = test);
   end;
end;

function ArgsSplitChar( const Line: String;
                        const Args: TStringList;
                        SplitChar: Char;
                        FillupArgs: Integer = 0 ): Integer;
begin
   Result := ArgsSplitChar( Line, Args, SplitChar, False, FillupArgs );
end;

function ArgsSplitChar( const Line: String;
                        const Args: TStringList;
                        SplitChar: Char;
                        DQuoted: Boolean;
                        FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     s  : String;
     isq: Boolean;
begin
   Args.Clear;
   pch := PChar( Line );
   isq := False;

   while pch^<>#0 do begin

      s := '';

      while pch^ <> #0 do begin
         if DQuoted then begin
            if pch^ = '"' then isq := not isq;
            if (pch^ = SplitChar) and not isq then break;
         end else begin
            if pch^ = SplitChar then break;
         end;

         s := s + pch^;
         inc( pch );
      end;

      Args.AddObject( s, Pointer(1) );

      if pch^<>#0 then inc( pch );
   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

function ArgsSplitChars( const Line: String;
                         const Args: TStringList;
                         SplitChars: Array of Char;
                         FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     c  : Char;
     s  : String;
     i  : Integer;
begin
   Args.Clear;
   pch := PChar( Line );

   while pch^<>#0 do begin

      s := '';
      c := #0;

      while pch^ <> #0 do begin
         for i := low( SplitChars ) to high( SplitChars ) do begin
            c := SplitChars[i];
            if pch^ = c then break;
         end;

         if pch^ = c then break;
         s := s + pch^;
         inc( pch );
      end;

      Args.AddObject( s, Pointer(1) );

      if pch^ <> #0 then inc( pch );

   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

function ArgsWhSpace( const Line: String;
                      const Args: TStringList;
                      FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     s  : String;
begin
   Args.Clear;

   pch := PChar( Line );
   while pch^ in [ #9, ' ' ] do inc( pch );

   while pch^<>#0 do begin
      s := '';
      while not( pch^ in [ #0, #9, ' ' ] ) do begin
         s := s + pch^;
         inc( pch );
      end;
      Args.AddObject( s, Pointer(1) );
      while pch^ in [ #9, ' ' ] do inc( pch );
   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

function ArgsWhSpaceDQuoted( const Line: String;
                             const Args: TStringList;
                             FillupArgs: Integer = 0 ): Integer;
var  pch: PChar;
     s  : String;
     isq: Boolean;
begin
   Args.Clear;

   pch := PChar( Line );
   while pch^ in [ #9, ' ' ] do inc( pch );

   while pch^<>#0 do begin
      s := '';
      
      if pch^='"' then begin
         isq := True;
         inc( pch );
      end else begin
         isq := False;
      end;

      while pch^ <> #0 do begin
         case pch^ of
            #9, ' ': if isq then begin
                        s := s + pch^;
                     end else begin
                        inc( pch );
                        break;
                     end;
            '"'    : if (pch+1)^='"' then begin
                        s := s + '"';
                        inc( pch );
                     end else begin
                        inc( pch );
                        break; 
                     end;
            else     s := s + pch^;
         end;
         inc( pch );
      end;

      Args.AddObject( s, Pointer(1) );
      while pch^ in [ #9, ' ' ] do inc( pch );
   end;

   Result := Args.Count;
   while Args.Count<FillupArgs do Args.AddObject( '', nil );
end;

function DQuoteStr( const Str: String ): String;
var  i: Integer;
begin
   i := Pos( '"', Str ) + Pos( ' ', Str ) + Pos( #9, Str );
   if i = 0 then begin
      Result := Str;
   end else begin
      Result := '"' + StringReplace( Str, '"', '""', [rfReplaceAll] ) + '"';
   end;
end;

function DQuoteStrFix( const Str: String ): String;
var  i: Integer;
begin
   i := Pos( '"', Str ) + Pos( ' ', Str ) + Pos( #9, Str );
   if i = 0 then begin
      Result := Str;
      if length( Result ) = 0 then Result := '""';
   end else begin
      Result := '"' + StringReplace( Str, '"', '""', [rfReplaceAll] ) + '"';
   end;
end;

function UnDQuoteStr( const Str: String ): String;
begin
   Result := Str;
   if copy( Str, 1, 1 ) = '"' then begin
      System.Delete( Result, 1, 1 );
      if copy(Result,length(Result),1)='"' then SetLength(Result,length(Result)-1);
      Result := StringReplace( Result, '""', '"', [rfReplaceAll] );
   end;
end;

function NextTopLine( var Lines: String ): String;
var  i: Integer;
begin
   i := Pos( #13#10, Lines );
   if i = 0 then begin
      Result := Lines;
      Lines  := '';
   end else begin
      Result := copy( Lines, 1, i-1 );
      System.Delete( Lines, 1, i+1 );
   end;
end;

function NextSepPart( var Line: String; const Separator: Char ): String;
var  i: Integer;
begin
   i := Pos( Separator, Line );
   if i = 0 then begin
      Result := Line;
      Line := '';
   end else begin
      Result := copy( Line, 1, i-1 );
      System.Delete( Line, 1, i );
   end;
end;

function NextWhSpacePart( var Line: String ): String;
var  i: Integer;
begin
   i := PosWhSpace( Line );
   if i = 0 then begin
      Result := Line;
      Line := '';
   end else begin
      Result := copy( Line, 1, i-1 );
      System.Delete( Line, 1, i );
      Line := TrimWhSpace( Line );
   end;
end;

function TextToRfcWireFormat( const OrgText: String ): String;
// convert given text to wire format (CRLF separators and quoted '.')
var  iptr: PChar;
     oidx, olen: Integer;
     NewLine: Boolean;

   procedure OutGrow;
   // grow output buffer to store next character and an additional one
   begin
      if oidx + 1 >= olen then begin
         inc( olen, 1024 );
         SetLength( Result, olen );
      end;
   end;

   procedure OutChar( ch: Char );
   // store character in output buffer and increment out-index
   begin
      Result[ oidx ] := ch;
      inc( oidx );
   end;

begin
   if length( OrgText ) = 0 then begin
      SetLength( Result, 0 );
      exit;
   end;

   oidx := 1;
   olen := length(OrgText) + 1024;
   SetLength( Result, olen );

   iptr := PChar( OrgText );
   NewLine := True;

   while iptr^ <> #0 do begin

      case iptr^ of
         #13:  if (iptr+1)^ = #10 then begin
                  // skip, will be handled by #10
               end else begin
                  OutChar( #13 );
                  NewLine := False;
               end;
         #10:  begin
                  OutChar( #13 );
                  OutChar( #10 );
                  NewLine := True;
               end;
         '.':  begin
                  OutChar( '.' );
                  if NewLine then begin
                     OutChar( '.' ); // double dot at beginning of line
                     NewLine := False;
                  end;
               end;
         else  begin
                  OutChar( iptr^ );
                  NewLine := False;
               end;
      end;

      OutGrow;

      inc( iptr );
   end;

   if not NewLine then begin
      OutGrow;
      OutChar( #13 );
      OutChar( #10 );
   end;

   SetLength( Result, oidx - 1 );
end;

procedure AddChar( var s: String; const c: Char ); // s := s + c
begin
   SetLength( s, length(s)+1 );
   s[ length(s) ] := c;
end;

procedure StreamToString( const Strm: TStream; var Result: String );
var  OldPos: Integer;
begin
   SetLength( Result, Strm.Size );
   if Strm.Size > 0 then begin
      OldPos := Strm.Position;
      try
         Strm.Position := 0;
         Strm.Read( Result[1], Strm.Size );
      finally Strm.Position := OldPos end;
   end;
end;

procedure StreamToFile( const Strm: TStream; const FileName: String );
var  FileStrm: TFileStream;
     OldPos: Integer;
begin
   FileStrm := TFileStream.Create( FileName, fmCreate );
   try
      OldPos := Strm.Position;
      try
         Strm.Position := 0;
         FileStrm.CopyFrom( Strm, Strm.Size );
      finally Strm.Position := OldPos end;
   finally FileStrm.Free end;
end;

procedure StreamWriteLn( const Strm: TStream; const Line: String );
const CRLF = #13#10;
begin
   if length(Line) > 0 then Strm.Write( Line[1], length(Line) );
   Strm.Write( CRLF[1], 2 );
end;

function iif( AValue: Boolean; const AThen: String; const AElse: String = '' ): String;
begin
   if AValue then Result := AThen else Result := AElse;
end;

function iif( AValue: Boolean; const AThen: Integer; const AElse: Integer ): Integer;
begin
   if AValue then Result := AThen else Result := AElse;
end;

function FilterNameOfFrom( const F: String ) : String;
var  j : Integer;
     s : String;
begin
     s := F;
     j := Pos( '(', s );
     if j>0 then begin
        System.Delete( s, 1, j );
        j := Pos( ')', s );
        if j>0 then s:=copy(s,1,j-1);
     end else begin
        // From: "xxx" <yyy@zzz>
        j := Pos( '<', s );
        if j>=5 then s := Trim( copy(s,1,j-1) );
        if copy(s,1,1)='"' then System.Delete(s,1,1);
        if copy(s,length(s),1)='"' then System.Delete(s,length(s),1);
     end;

     s := Trim( s );
     if s='' then s:=F;
     Result := s;
end;

procedure ForceFileExists( const Filename: String );
var  STM : TFileStream;
begin
     try
        ForceDirectories( ExtractFilePath(Filename) );
        if not FileExists(Filename) then begin
           STM := TFileStream.Create( Filename, fmCreate );
           STM.Free;
        end;
     except
     end;
end;

function LikeRadix32( l : Longint ): String;
const ALPHABET = '0123456789abcdefghijklmnopqrstuv';
var  p : Longint;
begin
     Result := '';
     repeat
        p := l and $1F;
        l := l shr 5;
        Result := copy(ALPHABET,p+1,1) + Result;
     until l=0;
end;

var
   MIDCounter: Integer = 0;
   MIDOldTime: Integer = 0;

function MIDGenerator( FQDN: String; LocalSuffix: String = '' ): String;
const OFFSET = 673416000;
var   NewDateTime: LongInt;
      i: LongInt;
begin
     NewDateTime := DateTimeToUnixTime( Now ) - OFFSET;

     if NewDateTime=MIDOldTime then begin
        inc( MIDCounter )
     end else begin
        MIDCounter := 1;
        MIDOldTime := NewDateTime;
     end;

     if FQDN='' then FQDN := 'FQDN-not-set';
     if Pos( '.', FQDN ) = 0 then begin
        i := NewDateTime XOR LongInt(GetCurrentThreadID) XOR MIDCounter;
        FQDN := FQDN
              + '.h' + lowercase( inttohex( StrToCRC32(inttostr(i)), 8 ) )
              + '.' + 'invalid';
     end;

     Result := '<'
             + LikeRadix32( NewDateTime ) + '.'
             + LikeRadix32( GetCurrentThreadID ) + '.'
             + inttostr( MIDCounter )
             + LocalSuffix + '@' + FQDN + '>';
end;

function UIDLGenerator: String;
var  j: Integer;
begin
     Result := MIDGenerator( 'dum.my' );
     System.Delete( Result, 1, 1 ); // "<"
     j := Pos( '@', Result ); // "@dum.my>"
     if j>0 then Result:=copy(Result,1,j-1);
end;

function GetExeVersion : String;
var  vlen, dw: DWord;
     vstr    : Pointer;
     p       : Pointer;
begin
     {Versionsnummer ermitteln}
     Result := '?.?.?.?';

     vlen := GetFileVersionInfoSize( PChar(ParamStr(0)), dw );
     if vlen=0 then exit;

     GetMem( vstr, vlen + 1 );
     if GetFileVersionInfo( PChar(ParamStr(0)), dw, vlen, vstr ) then begin
        if VerQueryValue( vstr, '\', p, dw ) then begin
           with PVSFixedFileInfo(p)^ do begin
              Result := inttostr( hiword(dwFileVersionMS) ) + '.'
                      + inttostr( loword(dwFileVersionMS) );
              if dwFileVersionLS <> 0 then begin
                 Result := Result + '.' + inttostr( hiword(dwFileVersionLS) )
                                  + '.' + inttostr( loword(dwFileVersionLS) );
              end;
           end;
        end;
     end;

     FreeMem( vstr, vlen + 1 );
end;

function GetFileInfo( const Filename: String ): String;
var  vlen, dw : DWord;
     vstr, p: Pointer;
     TS     : TSearchRec;
begin
     if FindFirst( Filename, faAnyFile, TS ) <> 0 then begin
        Result := '(file not found)';
        exit;
     end;

     Result := FormatDateTime( 'dd/mm/yyyy hh:nn', FileDateToDateTime(TS.Time) );
     Result := Result + ', ' + inttostr( TS.Size );
     SysUtils.FindClose( TS );

     vlen := GetFileVersionInfoSize( PChar(Filename), dw );
     if vlen=0 then exit;

     GetMem( vstr, vlen + 1 );
     if GetFileVersionInfo( PChar(Filename), dw, vlen, vstr ) then begin
        if VerQueryValue( vstr, '\', p, dw ) then begin
           with PVSFixedFileInfo(p)^ do begin
               Result := Result + ', ' + inttostr( hiword(dwFileVersionMS) )
                                + '.'  + inttostr( loword(dwFileVersionMS) )
                                + '.'  + inttostr( hiword(dwFileVersionLS) )
                                + '.'  + inttostr( loword(dwFileVersionLS) );
           end;
        end;
     end;
     FreeMem( vstr, vlen + 1 );
end;

function Winsock2Installed( Silent: Boolean = False ): Boolean;
var  VerInfo: TOSVersionInfo;
     ws2_32_dll, vernr: String;
     vlen, dw : DWord;
     vstr, p: Pointer;
begin
   VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
   GetVersionEx( VerInfo );
   Result := ( VerInfo.dwPlatformId = VER_PLATFORM_WIN32_NT );
   if Result then exit;

   ws2_32_dll := GetSystemPath + 'ws2_32.dll';
   if FileExists( ws2_32_dll ) then begin
      vlen := GetFileVersionInfoSize( PChar(ws2_32_dll), dw );
      if vlen=0 then exit;
      GetMem( vstr, vlen + 1 );
      if GetFileVersionInfo( PChar(ws2_32_dll), dw, vlen, vstr ) then begin
         if VerQueryValue( vstr, '\', p, dw ) then begin
            with PVSFixedFileInfo(p)^ do begin
                vernr := inttostr( hiword(dwFileVersionMS) )
                + '.'  + inttostr( loword(dwFileVersionMS) )
                + '.'  + inttostr( hiword(dwFileVersionLS) )
                + '.'  + inttostr( loword(dwFileVersionLS) );
            end;
            // all versions ok except a specific Win95-"beta"-version
            Result := ( vernr <> '4.10.0.1511' );
         end;
      end;
      FreeMem( vstr, vlen + 1 );
   end;
   if Result then exit;

   if Silent then exit;
   MessageBox( 0, 'This application requires Winsock2, which can be loaded '
                + 'from www.microsoft.com!',
               'Winsock2-Check', MB_OK or MB_ICONEXCLAMATION );
end;

function GetWindowsPath: String;
var  p: array[0..256] of Char;
begin
  GetWindowsDirectory( p, 256 );
  Result := String(p);
  if Result='' then exit;
  if Result[length(Result)]<>'\' then Result:=Result+'\';
end;

function GetSystemPath: String;
var  p: array[0..256] of Char;
begin
  GetSystemDirectory ( p, 256 );
  Result := String(p);
  if Result='' then exit;
  if Result[length(Result)]<>'\' then Result:=Result+'\';
end;

function IsWindowsNT: Boolean;
var  VerInfo: TOSVersionInfo;
begin
   VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
   GetVersionEx( VerInfo );
   Result := ( VerInfo.dwPlatformId = VER_PLATFORM_WIN32_NT );
end;

function GetWinVerInfo: String;
var  VerInfo: TOSVersionInfo;
     s, h   : String;
begin
     Result := '';

     VerInfo.dwOSVersionInfoSize := sizeof(VerInfo);
     GetVersionEx( VerInfo );

     with VerInfo do begin
        s := 'Windows ';

        case dwPlatformId of
           VER_PLATFORM_WIN32s:
              s := s + '32s';
           VER_PLATFORM_WIN32_WINDOWS:
              if dwMinorVersion<10 then begin
                 s := s + '95'; // 4.0
                 if (dwBuildNumber and $FFFF)>=1111 then s:=s+'-OSR2';
              end else if dwMinorVersion<90 then begin
                 s := s + '98'; // 4.10
                 if (dwBuildNumber and $FFFF)>=2222 then s:=s+'-SE';
              end else 
                 s := s + 'ME'; // 4.90
           VER_PLATFORM_WIN32_NT:
              if dwMajorVersion<=4 then
                 s := s + 'NT'   // 3.51/4.0
              else if (dwMajorVersion=5) and (dwMinorVersion=0) then
                 s := s + '2000' // 5.0
              else if dwMinorVersion > 1 then
                 s := s + '2003' // 5.2 = 2003
              else
                 s := s + 'XP';  // 5.1
           else
              s := s + '?PlatformID=' + inttostr(dwPlatformId) + '?';
        end;

        s := s + ' (Vr. ' + inttostr(dwMajorVersion)
               + '.' + inttostr(dwMinorVersion)
               + '.' + inttostr(dwBuildNumber and $FFFF);
        h := String( szCSDVersion );
        if h<>'' then s := s + ', ' + h;
        s := s + ')';
        Result := s;
     end;
end;

function ReplaceChar( s: String; oldc, newc: Char ): String;
var  i: Integer;
begin
     if oldc<>newc then begin
        for i:=1 to length(s) do begin
           if s[i]=oldc then s[i]:=newc;
        end;
     end;

     Result := s;
end;

function GetFileSize( const Filename: String ): LongInt;
var  SR: TSearchRec;
begin
     if SysUtils.FindFirst( Filename, faAnyFile, SR )=0 then begin
        Result := SR.Size;
     end else begin
        Result := 0;
     end;
     SysUtils.FindClose( SR );
end;

const
  WIMA_TRUE  = 1;
  WIMA_FALSE = 0;
  WIMA_ABORT = -1;

function DoWildMat( pText: PChar; pPattern: PChar ): Integer;
// Adapted from INN 2.0 (wildmat.c, rev. 1.2) by Rich $alz (<rsalz@osf.org>).
var  cLast: Char;
     iMatched, iReverse: Integer;
begin
     while ( pPattern^ <> #0 ) do begin

	if ( pText^ = #0 ) and ( pPattern^ <> '*' ) then begin
	    Result := WIMA_ABORT;
            exit;
        end;

        if ( pPattern^ = '\' ) then begin

           // Literal match with following character.
           inc( pPattern );
           if ( pText^ <> pPattern^ ) then begin
              Result := WIMA_FALSE;
              exit;
           end;

        end else begin

           case pPattern^ of

              '?': ; // Match anything.

              '*': begin
                      repeat
                         inc( pPattern );
                      until pPattern^<>'*'; // Consecutive stars act just like one.

                      if (pPattern^ = #0) then begin
                          // Trailing star matches everything.
                          Result := WIMA_TRUE;
                          exit;
                      end;

                      while (pText^<>#0) do begin
                         iMatched := DoWildMat( pText, pPattern );
                         inc( pText );
                         if iMatched<>WIMA_FALSE then begin
                            Result := iMatched;
                            exit;
                         end;
                      end;

                      Result := WIMA_ABORT;
                      exit;
                   end;

              '[': begin
                      if (pPattern+1)^='^' then begin
                         // Inverted character class.
                         iReverse := WIMA_TRUE;
                         inc(pPattern);
                      end else begin
                         iReverse := WIMA_FALSE;
                      end;

                      iMatched := WIMA_FALSE;

                      if ((pPattern+1)^ = ']') or ((pPattern+1)^ = '-') then begin
                         // special case: first char of class is ']' or '-'
                         inc(pPattern);
                         if (pPattern^ = pText^) then iMatched := WIMA_TRUE;
                      end;

                      cLast := pPattern^;
                      while ((pPattern+1)^<>#0) and (pPattern^<>']') do begin
                         inc(pPattern);

                         // "This next line requires a good C compiler." *sigh*
                         if (pPattern^='-') and ((pPattern+1)^<>']') then begin
                            // char-range
                            inc(pPattern);
                            if (pText^ <= pPattern^) and (pText^ >= cLast) then iMatched:=WIMA_TRUE;
                         end else begin
                            // single char
                            if pText^=pPattern^ then iMatched:=WIMA_TRUE;
                         end;

                         cLast := pPattern^;
                      end;

                      if (iMatched = iReverse) then begin
                         Result := WIMA_FALSE;
                         exit;
                      end;
                   end;

              else
                 // match char
                 if (pText^ <> pPattern^) then begin
                     Result := WIMA_FALSE;
                     exit;
                 end;
           end;
        end;

        inc( pText );
        inc( pPattern );

     end;

     if (pText^ = #0) then Result := WIMA_TRUE
                      else Result := WIMA_FALSE;
end;

function WildMat( pText: PChar; pPattern: PChar ): Boolean;
begin
     Result := ( DoWildMat(pText,pPattern) = WIMA_TRUE );
end;

function IsAtom( c: Char ): Boolean;
begin
     {
     atom        =  1*<any CHAR except specials, SPACE and CTLs>
     CHAR        =  <any ASCII character>        ; (  0-177,  0.-127.)
     CTL         =  <any ASCII control           ; (  0- 37,  0.- 31.)
                     character and DEL>          ; (    177,     127.)
     SPACE       =  <ASCII SP, space>            ; (     40,      32.)
     specials    =  "(" / ")" / "<" / ">" / "@"  ; Must be in quoted-
                 /  "," / ";" / ":" / "\" / <">  ;  string, to use
                 /  "." / "[" / "]"              ;  within a word.
     }
     Result := False;
     if (c>=#33) and (c<=#126) then begin
        if Pos(c,'()<>@,;:\".[]')=0 then Result:=True;
     end;
end;

function IsDomain( const s: String ): Boolean;
var  NeedAtom: Boolean;
     c: Char;
     i: Integer;
begin
     // Note: 'domain-literal' not supported here
     Result := True;
     NeedAtom := True;
     for i:=1 to length(s) do begin
        c := s[i];
        if NeedAtom then begin
           if IsAtom(c) then NeedAtom := False
                        else break;
        end else begin
           if c='.' then
              NeedAtom := True
           else
              if not IsAtom(c) then begin
                 Result:=False;
                 break;
              end;
        end;
     end;
     if NeedAtom then Result:=False;
end;

function IsNewsgroup( const s: String ): Boolean;   
var  p: PChar;
begin
   Result := False;
   if length( s ) = 0 then exit;
   
   p := PChar( s );
   if p^ = '.' then exit;

   while p^ <> #0 do begin
      if p^ = '.' then begin
         if (p+1)^ in [ #0, '.' ] then exit;
      end else begin
         if p^ in [ #0..#31, ' ' ,':', '!', '/', '\', '@', '<', '>', '|' ] then exit;
      end;
      inc( p );
   end;

   Result := True;
end;

function IsMessageId( const s: String ): Boolean;
begin
   Result := False;
   if length(s) < 3 then exit;
   if s[1] <> '<' then exit;
   if s[length(s)] <> '>' then exit;
   if PosWhSpace( s ) > 0 then exit;
   Result := True;
end;

function IsFilename( const c: Char ): Boolean;
begin
   Result := True;
   if (c >= 'a') and (c <= 'z') then exit;
   if (c >= 'A') and (c <= 'Z') then exit;
   if (c >= '0') and (c <= '9') then exit;
   if (c = '-') or (c = '_') or (c = '$') or (c = '!') then exit;
   Result := False;
end;

function GetMyVersionInfo : String;
var  vlen, dw: DWord;
     vstr, p : Pointer;
begin
     Result := '?.?';

     vlen := GetFileVersionInfoSize( PChar(ParamStr(0)), dw );
     if vlen=0 then exit;

     GetMem( vstr, vlen + 1 );
     if GetFileVersionInfo( PChar(ParamStr(0)), dw, vlen, vstr ) then begin
        if VerQueryValue( vstr, '\', p, dw ) then begin
           with PVSFixedFileInfo(p)^ do begin
              Result := 'Vr. ' + inttostr( hiword(dwProductVersionMS) ) + '.'
                               + inttostr( loword(dwProductVersionMS) );
              if dwFileVersionLS <> 0 then begin
                 Result := Result + ' (Build '
                     + inttostr( hiword(dwFileVersionMS) ) + '.'
                     + inttostr( loword(dwFileVersionMS) ) + '.'
                     + inttostr( hiword(dwFileVersionLS) ) + '.'
                     + inttostr( loword(dwFileVersionLS) ) + ')'
              end;
           end;
        end;
     end;

     FreeMem( vstr, vlen + 1 );
end;

function GetMyStringFileInfo( const StringName, DefResult: String ): String;
var  vlen, dw: DWORD;
     vstr, p : Pointer;
     tmp     : array[0..3] of Byte;
     LangID  : String;
begin
   Result := DefResult;
   vlen := GetFileVersionInfoSize( PChar(ParamStr(0)), dw );
   if vlen=0 then exit;

   GetMem( vstr, vlen + 1 );
   if GetFileVersionInfo( PChar(ParamStr(0)), dw, vlen, vstr ) then begin
      if VerQueryValue( vstr, '\VarFileInfo\Translation', p, dw ) then begin
         if dw>=4 then begin
            Move( p^, tmp[0], 4 );
            dw := tmp[2] or tmp[3] shl 8 or tmp[0] shl 16 or tmp[1] shl 24;
            LangID := inttohex( dw, 8 );
            if VerQueryValue( vstr, PChar('\StringFileInfo\'+LangID+'\'+StringName), p, dw ) then begin
               Result := String( PChar(p) )
            end
         end
      end
   end;
   FreeMem( vstr, vlen + 1 );
end;

function FileExistsByPattern( const SearchFile: String ): Boolean;
var  Handle  : DWORD;
     FindData: WIN32_FIND_DATA;
begin
   Result := False;
   Handle := FindFirstFile( PChar(SearchFile), FindData );
   if Handle <> INVALID_HANDLE_VALUE then try
      repeat
         if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY)=0 then begin
            Result := True;
            break;
         end;
      until not FindNextFile( Handle, FindData );
   finally
      FindClose( Handle );
   end;
end;

function DirectoryExists(const Name: string): Boolean;
// taken from FileCtrl to avoid including Forms unit
var
  Code: Integer;
begin
  Code := GetFileAttributes(PChar(Name));
  Result := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
end;

function ForceDirectories(Dir: string): Boolean; 
// taken from FileCtrl to avoid including Forms unit
begin
  Result := True;
  if Length(Dir) = 0 then
    raise Exception.Create('Directory could not be created');
  Dir := ExcludeTrailingBackslash(Dir);
  if (Length(Dir) < 3) or DirectoryExists(Dir)
    or (ExtractFilePath(Dir) = Dir) then Exit; // avoid 'xyz:\' problem.
  Result := ForceDirectories(ExtractFilePath(Dir)) and CreateDir(Dir);
end;

function WinErrMsg( ErrorCode: DWORD ): String; 
var  lpMsgBuf: Pointer;
begin
   Result := IntToStr( ErrorCode );

   if FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER or FORMAT_MESSAGE_FROM_SYSTEM,
                     nil, ErrorCode, 0, @lpMsgBuf, 0, nil ) <> 0 then begin
      Result := Result + ' ' + String( PChar(lpMsgBuf) );
      LocalFree( Cardinal(lpMsgBuf) );
   end else begin
      Result := Result + ' (no description available)';
   end;
end;

function WinErrMsg: String;
begin
   Result := WinErrMsg( GetLastError );
end;

function ExtractEnvelopeAddr( const EnvAddr: String ): String;
var  s: String;
     i: Integer;
begin
   Result := '';

   i := Pos( '<', EnvAddr );
   if i = 0 then exit;
   s := copy( EnvAddr, i, MaxInt );

   i := Pos( '>', s );
   if i = 0 then exit;
   SetLength( s, i );

   i := Pos( ':', s ); // remove source-route
   if i > 0 then System.Delete( s, 2, i-1 );

   Result := s;
end;

function ExtractMailAddr( const MailAddr: String ): String;
var  i: Integer;
begin
   Result := TrimWhSpace( MailAddr );

   i := Pos( '<', Result );

   if i > 0 then begin

      System.Delete( Result, 1, i );
      i := Pos( '>', Result );
      if i > 0 then SetLength( Result, i-1 );

   end else begin

      i := pos( '(', Result );
      if i > 0 then begin
         SetLength( Result, i-1 );
         while length( Result ) > 0 do begin
            if not( Result[ length(Result) ] in [ #9, ' ' ] ) then break;
            SetLength( Result, length(Result) - 1 );
         end;
      end;

   end;
end;

function ExtractMailDomain( const MailAddr: String ): String;
var  i: Integer;
begin
   Result := ExtractMailAddr( MailAddr );
   i := Pos( '@', Result );
   if i > 0 then Result := copy( Result, i+1, MaxInt );
   i := Pos( '>', Result );
   if i > 0 then SetLength( Result, i-1 );
end;

end.

