// ============================================================================
// Helper functions for SASL mechanisms
// Copyright (c) 2001, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uSASL; // Helper functions for SASL mechanisms

interface

{$INCLUDE Compiler.inc}

// 'CRAM-MD5', RFC 2195
function AUTH_CRAM_MD5_Encode ( const TimeStamp: String;
                                const Username, Password: String ): String;
function AUTH_CRAM_SHA1_Encode( const TimeStamp: String;
                                const Username, Password: String ): String;
function AUTH_CRAM_Decode ( const Code: String;
                            out   Username, KeyedPassword: String ): Boolean;

// 'PLAIN', RFC 2595
function AUTH_PLAIN_Encode( const Username, Password: String ): String;
function AUTH_PLAIN_Decode( const Code: String;
                            out   Username, Password: String ): Boolean;

implementation

uses uMD5, uSHA1, uEncoding;

function AUTH_CRAM_MD5_Encode( const TimeStamp: String;
                               const Username, Password: String ): String;
// 'CRAM-MD5', RFC 2195
var  s: String;
begin
   s := Username + ' ' + HMAC_MD5( TimeStamp, Password );
   Result := EncodeB64( s[1], length(s) );
end;

function AUTH_CRAM_SHA1_Encode( const TimeStamp: String;
                                const Username, Password: String ): String;
var  s: String;
begin
   s := Username + ' ' + HMAC_SHA1( TimeStamp, Password );
   Result := EncodeB64( s[1], length(s) );
end;

function AUTH_CRAM_Decode( const Code: String;
                           out Username, KeyedPassword: String ): Boolean;
// 'CRAM-MD5', RFC 2195
var  s: String;
     j: Integer;
begin
   Result := False;
   Username := '';
   KeyedPassword := '';

   s := DecodeB64( Code[1], length(Code) );
   j := Pos( ' ', s );
   if j=0 then exit;

   Username := copy( s, 1, j-1 );
   KeyedPassword := copy( s, j+1, 9999 );
   Result := True;
end;

function AUTH_PLAIN_Encode( const Username, Password: String ): String;
// 'PLAIN', RFC 2595
begin
   Result := { '' + } chr(0) + Username + chr(0) + Password;
   Result := EncodeB64( Result[1], length(Result) );
end;

function AUTH_PLAIN_Decode( const Code: String;
                            out   Username, Password: String ): Boolean;
// 'PLAIN', RFC 2595
var  s: String;
     j: Integer;
begin
   Result := False;
   Username := '';
   Password := '';

   s := DecodeB64( Code[1], length(Code) );
   if s='' then exit;

   if s[1] <> chr(0) then exit;
   System.Delete( s, 1, 1 );

   j := Pos( chr(0), s );
   if j=0 then exit;

   Username := copy( s, 1, j-1 );
   Password := copy( s, j+1, 999 );
   Result := True;
end;

end.
