// ============================================================================
// Dynamically loaded "IP Helper API" and related functions
// Copyright (c) 2002, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uIpHelper;

{$ALIGN ON}
{$MINENUMSIZE 4}

interface

uses SysUtils, Classes, Windows;

const
   MAX_HOSTNAME_LEN    = 128;
   MAX_DOMAIN_NAME_LEN = 128;
   MAX_SCOPE_ID_LEN    = 256;

type
   IP_ADDRESS_STRING = record
      S: array [0..15] of Char;
   end;

   IP_MASK_STRING = IP_ADDRESS_STRING;

   PIP_ADDR_STRING = ^IP_ADDR_STRING;
   IP_ADDR_STRING = record
      Next: PIP_ADDR_STRING;
      IpAddress: IP_ADDRESS_STRING;
      IpMask: IP_MASK_STRING;
      Context: DWORD;
   end;

   PFIXED_INFO = ^FIXED_INFO;
   FIXED_INFO = record
      HostName: array [0..MAX_HOSTNAME_LEN + 3] of Char;
      DomainName: array[0..MAX_DOMAIN_NAME_LEN + 3] of Char;
      CurrentDnsServer: PIP_ADDR_STRING;
      DnsServerList: IP_ADDR_STRING;
      NodeType: UINT;
      ScopeId: array [0..MAX_SCOPE_ID_LEN + 3] of Char;
      EnableRouting: UINT;
      EnableProxy: UINT;
      EnableDns: UINT;
   end;

   TGetNetworkParams = function(
      pFixedInfo: PFIXED_INFO; var pOutBufLen: ULONG ): DWORD; stdcall;

var
   GetNetworkParams: TGetNetworkParams = nil;

function IsIpHelperAvailable: Boolean;

function GetKnownDnsServer: String;
function GetKnownDnsServerList( const List: TStrings ): Boolean;
// Note: Above GetKnownDnsServer functions also check the registry for known
//       servers, so they can be used even if IP Helper API is not available.


implementation

uses Registry;

const
   sIPHLPAPI = 'iphlpapi.dll';

var
   hIPHLPAPI: HModule = 0;

function IsIpHelperAvailable: Boolean;
// Returns True if IP Helper API is available.
// Note: Even if available, some functions may fail with ERROR_NOT_SUPPORTED.
begin
   Result := ( hIPHLPAPI <> 0 );
end;

function GetKnownDnsServer: String;
// Returns 1st detected DNS server or '127.0.0.1' if none was found.
var  SL: TStringList;
begin
   Result := '127.0.0.1';
   SL := TStringList.Create;
   try
      if GetKnownDnsServerList( SL ) then Result := SL[0];
   finally SL.Free end;
end;

function GetKnownDnsServerList( const List: TStrings ): Boolean;
// Get list of known DNS servers and return True, if at least one was found.
// GetDnsServerList(nil) may be used to just check, if there are known ones.
// Note: .Objects[] is filled with source of info just for debugging purposes.

   procedure SrvAdd( const Srv: String; const Source: Integer );
   // add server to list, filter duplicates
   var  S: String;
   begin
      S := Trim( Srv );
      if length( S ) = 0 then exit;
      if Assigned( List ) then begin
         if List.IndexOf( S ) >= 0 then exit;
         List.AddObject( S, Pointer(Source) );
      end;
      Result := True;
   end;

   procedure SrvListAdd( const SrvList: String; const Source: Integer );
   // add SP separated list of servers
   var  i: Integer;
        L, S: String;
   begin
      L := Trim( SrvList );
      while length( L ) > 0 do begin
         i := Pos( ' ', L );
         if i = 0 then begin
            S := L;
            L := '';
         end else begin
            S := copy( L, 1, i-1 );
            System.Delete( L, 1, i );
         end;
         SrvAdd( S, Source );
      end;
   end;

const
   // default registry keys of TCP/IP settings on NT and 95 platforms
   RegTcpNT = 'SYSTEM\CurrentControlSet\Services\Tcpip\Parameters';
   RegTcp95 = 'SYSTEM\CurrentControlSet\Services\VxD\MSTCP';

var
   FixedInfo: PFIXED_INFO;
   BufLen   : ULONG;
   pIPAddr  : PIP_ADDR_STRING;
   L        : String;

begin
   Result := False;
   if Assigned( List ) then List.Clear;

   // 1.) Read by "IP Helper API"
   //     (not available on all systems, so DLL is only loaded dynamically)

   if IsIpHelperAvailable then begin
      FixedInfo := PFIXED_INFO( GlobalAlloc( GPTR, sizeof(FIXED_INFO) ) );
      BufLen := sizeof( FIXED_INFO );

      if GetNetworkParams(FixedInfo, BufLen) = ERROR_BUFFER_OVERFLOW then begin
         GlobalFree( Cardinal(FixedInfo) );
         FixedInfo := PFIXED_INFO( GlobalAlloc( GPTR, BufLen ) );
      end;

      if GetNetworkParams(FixedInfo, BufLen) = ERROR_SUCCESS then begin
         SrvAdd( FixedInfo.DnsServerList.IpAddress.S, 1 );
         pIPAddr := FixedInfo.DnsServerList.Next;
         while Assigned( pIPAddr ) do begin
            SrvAdd( pIPAddr.IpAddress.S, 1 );
            pIPAddr := pIPAddr.Next;
         end;
      end;
      GlobalFree( Cardinal(FixedInfo) );
   end;

   // 2.) Read from dynamic RAS entries in registry
   //     (only available while connected)         
   with TRegistry.Create( KEY_READ ) do try

      RootKey := HKEY_LOCAL_MACHINE;
      L := '';

      if OpenKey( RegTcpNT + '\Transient', False ) then begin
         L := ReadString( 'NameServer' );
      end else if OpenKey( RegTcp95 + '\Transient', False ) then begin
         L := ReadString( 'NameServer' );
      end;

      if L <> '' then SrvListAdd( L, 2 );

   finally Free end;

   // 3.) Read from TCP/IP settings in registry
   with TRegistry.Create( KEY_READ ) do try

      RootKey := HKEY_LOCAL_MACHINE;
      L := '';

      if OpenKey( RegTcpNT, False ) then begin
         L := ReadString( 'NameServer' );
         if L = '' then L := ReadString( 'DhcpNameServer' );
      end else if OpenKey( RegTcp95, False ) then begin
         L := ReadString( 'NameServer' );
         if L = '' then L := ReadString( 'DhcpNameServer' );
      end;

      if L <> '' then SrvListAdd( L, 3 );

   finally Free end;
end;

procedure Load_IPHLPAPI;
begin
   hIPHLPAPI := LoadLibrary( sIPHLPAPI );
   if hIPHLPAPI <= HINSTANCE_ERROR then begin
      hIPHLPAPI := 0;
      exit;
   end;

   @GetNetworkParams := GetProcAddress( hIPHLPAPI, 'GetNetworkParams' );

   if not Assigned( GetNetworkParams ) then begin
      FreeLibrary( hIPHLPAPI );
      hIPHLPAPI := 0;
   end;
end;


initialization
   Load_IPHLPAPI;

finalization
   if hIPHLPAPI <> 0 then FreeLibrary( hIPHLPAPI );

end.
