// ============================================================================
// Date and time related functions
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uDateTime; // Date/time related functions

// ----------------------------------------------------------------------------
// Contains functions related with date and time.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils;

const
   SECONDS_PER_MINUTE = 60;
   SECONDS_PER_HOUR   = 60 * SECONDS_PER_MINUTE;
   SECONDS_PER_DAY    = 24 * SECONDS_PER_HOUR;

type
   TUnixTime    = Integer;
   TTimeStamp   = String; // 'yyyymmddhhnnss'
   TLogfileTime = String; // 'yyyy-mm-dd hh:nn:ss'
   TRfcDateTime = String;
   TRfcTimezone = String;

   TTimeSteps   = ( stepSeconds, stepMinutes, stepHours,
                    stepDays, stepWeeks, stepMonths, stepYears );


function HoursToDateTime  ( const Hours  : Integer ): TDateTime;
function MinutesToDateTime( const Minutes: Integer ): TDateTime;
function SecondsToDateTime( const Seconds: Integer ): TDateTime;
function MinutesToDHM     ( const Minutes: Integer ): String;
function SecondsToDHMS    ( const Seconds: Integer ): String;
function DateTimeDiffToMinutes( const DiffDateTime: TDateTime ): Integer;
function DateTimeDiffToSeconds( const DiffDateTime: TDateTime ): Integer;

function  DateOfDateTime( const dt: TDateTime ): TDateTime;
function  TimeOfDateTime( const dt: TDateTime ): TDateTime;
procedure DateTimeSplit( const dt: TDateTime; out datePart, timePart: TDateTime );
function  DateTimeCombine( const datePart, timePart: TDateTime ): TDateTime;
function  DateTimeAdd( dt: TDateTime; steps: TTimeSteps; count: Integer ): TDateTime;
function  TimeStringToTime( const hhmmss: String ): TDateTime;
function  TimeToTimeString( const dt: TDateTime; const strLen: Integer ): String;

function NowBiasMinutes: Integer;
function NowBias: TDateTime; // GMT = Local + Bias
function NowGMT : TDateTime;
function NowRfcTimezone: TRfcTimeZone;

function DateTimeToUnixTime( const DateTime: TDateTime ): TUnixTime;
function UnixTimeToDateTime( const UnixTime: TUnixTime ): TDateTime;

function DateTimeToTimeStamp( const DateTime: TDateTime ): TTimeStamp;
function TimeStampToDateTime( const TimeStamp: TTimeStamp ): TDateTime;

function DateTimeToLogTime( const DateTime: TDateTime ): TLogfileTime;
function LogTimeToDateTime( const LogTime: TLogfileTime ): TDateTime;

function DateTimeGMTToRfcDateTime( const DateTime   : TDateTime;
                                   const RfcTimezone: TRfcTimezone ): TRfcDateTime;
function DateTimeLocalToRfcDateTime( const DateTime   : TDateTime;
                                     const RfcTimezone: TRfcTimezone ): TRfcDateTime;
function RfcDateTimeToDateTimeGMT( const RfcDateTime: TRfcDateTime; const ErrorDefault: TDateTime ): TDateTime; overload;
function RfcDateTimeToDateTimeGMT( const RfcDateTime: TRfcDateTime ): TDateTime; overload;

function RfcTimezoneToBiasMinutes( const RfcTimezone: TRfcTimezone ): Integer;
function BiasMinutesToRfcTimezone( const BiasMin: Integer ): TRfcTimeZone;
function DateTimeToRfcTimezone( const DateTime: TDateTime ): TRfcTimeZone;

function DateTimeToSystemTime( const DateTime: TDateTime ): TSystemTime;
function SystemTimeIsDaylight( const SystemTime: TSystemTime ): Boolean;
function DateTimeIsDaylight( const DateTime: TDateTime ): Boolean;
function DateTimeToBiasMinutes( const DateTime: TDateTime ): Integer;
function DateTimeGMTToLocal( const DateTime: TDateTime ): TDateTime;
function DateTimeLocalToGMT( const DateTime: TDateTime ): TDateTime;

// ----------------------------------------------------------------------------

implementation

uses uTools;

const
   TIME_ZONE_ID_UNKNOWN  = 0;
   TIME_ZONE_ID_STANDARD = 1;
   TIME_ZONE_ID_DAYLIGHT = 2;

   RFC_DAY_NAMES   = 'SunMonTueWedThuFriSat'; 
   RFC_MONTH_NAMES = 'JanFebMarAprMayJunJulAugSepOctNovDec';

// ----------------------------------------------------------------------------

function HoursToDateTime( const Hours: Integer ): TDateTime;
begin
   Result := ( Hours / 24.0 );
end;

function MinutesToDateTime( const Minutes: Integer ): TDateTime;
begin
   Result := ( Minutes / 60.0 / 24.0 );
end;

function SecondsToDateTime( const Seconds: Integer ): TDateTime;
begin
   Result := ( Seconds / 60.0 / 60.0 / 24.0 );
end;

function MinutesToDHM( const Minutes: Integer ): String;
var  i: Integer;
begin
   i := Minutes;

   Result := inttostr( i mod 60 ) + 'm';
   i := i div 60;
   if i=0 then exit;

   Result := inttostr( i mod 24 ) + 'h ' + Result;
   i := i div 24;
   if i=0 then exit;

   Result := inttostr( i ) + 'd ' + Result;
end;

function SecondsToDHMS( const Seconds: Integer ): String;
var  i: Integer;
begin
   i := Seconds;
   Result := inttostr( i mod 60 ) + 's';
   i := i div 60;
   if i=0 then exit;

   Result := inttostr( i mod 60 ) + 'm ' + Result;
   i := i div 60;
   if i=0 then exit;

   Result := inttostr( i mod 24 ) + 'h ' + Result;
   i := i div 24;
   if i=0 then exit;

   Result := inttostr( i ) + 'd ' + Result;
end;

function DateTimeDiffToSeconds( const DiffDateTime: TDateTime ): Integer;
begin
   try
      Result := Trunc( DiffDateTime * 24.0 * 60.0 * 60.0 );
   except
      Result := MaxInt;
   end;
end;

function DateTimeDiffToMinutes( const DiffDateTime: TDateTime ): Integer;
begin
   try
      Result := Trunc( DiffDateTime * 24.0 * 60.0 );
   except
      Result := MaxInt;
   end;
end;

function DateOfDateTime( const dt: TDateTime ): TDateTime;
begin
   Result := Trunc( dt );
end;

function TimeOfDateTime( const dt: TDateTime ): TDateTime;
begin
   Result := Frac( dt );
end;

procedure DateTimeSplit( const dt: TDateTime; out datePart, timePart: TDateTime );
begin
   datePart := DateOfDateTime( dt );
   timePart := TimeOfDateTime( dt );
end;

function DateTimeCombine( const datePart, timePart: TDateTime ): TDateTime;
begin
   Result := DateOfDateTime( datePart ) + TimeOfDateTime( timePart );
end;

function DateTimeAdd( dt: TDateTime; steps: TTimeSteps; count: Integer ): TDateTime;
var  yy, mm, dd: Word;
begin
   case steps of
      stepSeconds:
         dt := dt + SecondsToDateTime( count );
      stepMinutes:
         dt := dt + SecondsToDateTime( count * 60 );
      stepHours:
         dt := dt + SecondsToDateTime( count * 3600 );
      stepDays:
         dt := dt + ( 1.0 * count );
      stepWeeks:
         dt := dt + ( 7.0 * count );
      stepMonths:
         dt := IncMonth( dt, count );
      stepYears:
         begin
            DecodeDate( dt, yy, mm, dd );
            dt := TimeOfDateTime(dt) + EncodeDate( yy + count, mm, dd );
         end;
   end;

   Result := dt;
end;

function TimeStringToTime( const hhmmss: String ): TDateTime;
var  wh, wm, ws, ms: Word;
     sh, sm, ss: String;
begin
   sh := copy( hhmmss, 1, 2 ); // hh <sep> mm <sep> ss
   sm := copy( hhmmss, 4, 2 );
   ss := copy( hhmmss, 7, 2 );

   wh := strtointdef( sh, 0 );
   wm := strtointdef( sm, 0 );
   ws := strtointdef( ss, 0 );
   ms := 0;

   if ws > 59 then begin ws := 59; inc(wm); end;
   if wm > 59 then begin wm := 59; inc(wh); end;
   if wh > 23 then begin wh := 23; wm := 59; ws := 59; ms := 999; end;

   Result := EncodeTime( wh, wm, ws, ms );
end;

function TimeToTimeString( const dt: TDateTime; const strLen: Integer ): String;
var  wh, wm, ws, ms: Word;
begin
   DecodeTime( dt, wh, wm, ws, ms );
   Result := Format( '%.2d', [wh] );
   if strlen >=5 then Result := Result + ':' + Format( '%.2d', [wm] );
   if strlen >=8 then Result := Result + ':' + Format( '%.2d', [ws] );
end;

// ----------------------------------------------------------------------------

function NowBiasMinutes: Integer;
var   TZI      : TTimeZoneInformation;
      TZResult : Integer;
begin
   TZResult := GetTimeZoneInformation(TZI);

   case TZResult of
      TIME_ZONE_ID_UNKNOWN : Result := TZI.Bias;
      TIME_ZONE_ID_STANDARD: Result := TZI.Bias + TZI.StandardBias;
      TIME_ZONE_ID_DAYLIGHT: Result := TZI.Bias + TZI.DaylightBias;
      else                   Result := 0;
   end;
end;

function NowBias: TDateTime;
begin
   Result := MinutesToDateTime( NowBiasMinutes );
end;

function NowGMT : TDateTime;
begin
   Result := Now + NowBias;
end;

// ----------------------------------------------------------------------------

function DateTimeToUnixTime( const DateTime: TDateTime ): TUnixTime;
begin
   Result := Round( ( DateTime - EncodeDate(1970,1,1) ) * 86400 );
end;

function UnixTimeToDateTime( const UnixTime: TUnixTime ): TDateTime;
var  Days, Hours, Mins, Secs, h: Integer;
begin
   Secs := UnixTime;

   if Secs>=0 then begin
      Days := Secs div 86400;
      Secs := Secs mod 86400;
   end else begin
      h    := Secs and $1;
      Secs := Secs shr 1;

      Days := Secs div (86400 shr 1);
      Secs := Secs mod (86400 shr 1);

      Secs := (Secs shl 1) or h;
   end;

   Hours := Secs div 3600;  Secs := Secs mod 3600;
   Mins  := Secs div 60;    Secs := Secs mod 60;

   Result := EncodeDate(1970,1,1) + Days + EncodeTime(Hours,Mins,Secs,0);
end;

// ----------------------------------------------------------------------------

function DateTimeToTimeStamp( const DateTime: TDateTime ) : TTimeStamp;
begin
   Result := FormatDateTime( 'yyyymmddhhnnss', DateTime )
end;

function TimeStampToDateTime( const TimeStamp: TTimeStamp ) : TDateTime;
begin
   if length(TimeStamp)<>14 then begin
      Result := 0;
   end else begin
      try
         Result := EncodeDate( strtoint   ( copy(TimeStamp, 1,4)    ),
                               strtointdef( copy(TimeStamp, 5,2), 1 ),
                               strtointdef( copy(TimeStamp, 7,2), 1 ) )
                 + EncodeTime( strtointdef( copy(TimeStamp, 9,2), 0 ),
                               strtointdef( copy(TimeStamp,11,2), 0 ),
                               strtointdef( copy(TimeStamp,13,2), 0 ), 0 );
      except
         Result := 0;
      end;
   end;
end;

// ----------------------------------------------------------------------------

function DateTimeToLogTime( const DateTime: TDateTime ): TLogfileTime;
begin
   Result := FormatDateTime( 'yyyy"-"mm"-"dd hh":"nn":"ss', DateTime )
end;

function LogTimeToDateTime( const LogTime: TLogfileTime ): TDateTime;
begin
   if length(LogTime) <> 19 then begin
      Result := 0;
   end else begin
      try
         Result := EncodeDate( strtoint   ( copy(LogTime, 1,4)    ),
                               strtointdef( copy(LogTime, 6,2), 1 ),
                               strtointdef( copy(LogTime, 9,2), 1 ) )
                 + EncodeTime( strtointdef( copy(LogTime,12,2), 0 ),
                               strtointdef( copy(LogTime,15,2), 0 ),
                               strtointdef( copy(LogTime,18,2), 0 ), 0 );
      except
         Result := 0;
      end;
   end;
end;

// ----------------------------------------------------------------------------

function DateTimeGMTToRfcDateTime( const DateTime   : TDateTime;
                                   const RfcTimezone: TRfcTimezone ) : TRfcDateTime;
var  dDT     : TDateTime;
     sDT, sTZ: String;
     DOW, MOY: Integer;
begin
   if length(RfcTimezone) = 0 then sTZ := 'GMT' else sTZ := RfcTimezone;

   dDT := DateTime - MinutesToDateTime( RfcTimezoneToBiasMinutes( sTZ ) );
   sDT := FormatDateTime( 'dd"."mm"."yyyy hh":"nn":"ss', dDT );
   DOW := DayOfWeek( dDT );  // 1=Sun, ..., 7=Sat
   MOY := strtoint( copy( sDT, 4, 2 ) );

   // Date: Fri, 27 Mar 1998 12:12:50 +1300

   Result := copy( RFC_DAY_NAMES, DOW*3-2, 3 ) + ',' + ' '
           + copy( sDT, 1, 2 ) + ' '
           + copy( RFC_MONTH_NAMES, MOY*3-2, 3 ) + ' '
           + copy( sDT, 7, 4 ) + ' '
           + copy( sDT, 12, 8 ) + ' '
           + sTZ;
end;

function DateTimeLocalToRfcDateTime( const DateTime   : TDateTime;
                                     const RfcTimezone: TRfcTimezone ): TRfcDateTime;
begin
   Result := DateTimeGMTToRfcDateTime( DateTimeLocalToGMT(DateTime), RfcTimezone );
end;

function RfcDateTimeToDateTimeGMT( const RfcDateTime: TRfcDateTime ) : TDateTime;
const OldDefault = 29221.0; // =EncodeDate(1980,1,1)
begin
   Result := RfcDateTimeToDateTimeGMT( RfcDateTime, OldDefault );
end;

function RfcDateTimeToDateTimeGMT( const RfcDateTime: TRfcDateTime;
                                   const ErrorDefault: TDateTime ) : TDateTime;
var  s, h, tz : String;
     i, yyyy, mm, dd, hh, nn, ss : Integer;
begin
   s := TrimWhSpace( RfcDateTime );
   if s='' then begin Result:=ErrorDefault; exit; end;

   try
      // Date: Fri, 27 Mar 1998 12:12:50 +1300

      i := Pos( ',', s );
      if (i>0) and (i<10) then begin
         System.Delete( s, 1, i ); // "Tue,", "Tuesday,"
         s := TrimWhSpace(s);
      end;

      i := Pos(' ',s);
      dd := strtoint( copy(s,1,i-1) );
      System.Delete( s, 1, i );
      s := TrimWhSpace(s);

      i := Pos(' ',s);
      h := lowercase( copy(s,1,i-1) );
      mm := ( ( Pos(h,LowerCase(RFC_MONTH_NAMES)) - 1 ) div 3 ) + 1;
      System.Delete( s, 1, i );
      s := TrimWhSpace(s);

      i := Pos(' ',s);
      yyyy := strtoint( copy(s,1,i-1) );
      if yyyy<100 then begin
         if yyyy>=50 then yyyy:=yyyy+1900 else yyyy:=yyyy+2000;
      end;
      System.Delete( s, 1, i );
      s := TrimWhSpace(s);

      i := Pos(' ',s);
      if i=0 then begin
         h := s;
         tz := '';
      end else begin
         h := TrimWhSpace( copy(s,1,i-1) );
         tz := UpperCase( TrimWhSpace( copy(s,i+1,32) ) );
      end;

      i:=Pos(':',h); if i=2 then h:='0'+h;
      hh := strtoint( copy(h,1,2) );
      nn := strtoint( copy(h,4,2) );
      ss := strtoint( copy(h,7,2) );

      Result := EncodeDate( yyyy, mm, dd )
              + MinutesToDateTime( RfcTimezoneToBiasMinutes( tz ) ) // -> GMT
              + EncodeTime( hh, nn, ss, 0 );
   except
      Result := ErrorDefault
   end;
end;

// ----------------------------------------------------------------------------

function RfcTimezoneToBiasMinutes( const RfcTimeZone: TRfcTimeZone ): Integer;
var  sTZ: String;
begin
   Result := 0;
   if RfcTimeZone='' then exit;

   if RfcTimeZone[1] in [ '+', '-' ] then begin

      Result := strtointdef( copy(RfcTimeZone,2,2), 0 ) * 60
              + strtointdef( copy(RfcTimeZone,4,2), 0 );
      if (Result<0) or (Result>=24*60) then Result:=0;
      if RfcTimeZone[1]='+' then Result:=-Result;

   end else begin

      sTZ := UpperCase( RfcTimeZone );

      if      sTZ='GMT' then Result:=  0
      else if sTZ='UT'  then Result:=  0

      else if sTZ='EST' then Result:= -5*60
      else if sTZ='EDT' then Result:= -4*60
      else if sTZ='CST' then Result:= -6*60
      else if sTZ='CDT' then Result:= -5*60
      else if sTZ='MST' then Result:= -7*60
      else if sTZ='MDT' then Result:= -6*60
      else if sTZ='PST' then Result:= -8*60
      else if sTZ='PDT' then Result:= -7*60

      else if sTZ='A'   then Result:= -1*60
      else if sTZ='B'   then Result:= -2*60
      else if sTZ='C'   then Result:= -3*60
      else if sTZ='D'   then Result:= -4*60
      else if sTZ='E'   then Result:= -5*60
      else if sTZ='F'   then Result:= -6*60
      else if sTZ='G'   then Result:= -7*60
      else if sTZ='H'   then Result:= -8*60
      else if sTZ='I'   then Result:= -9*60
      else if sTZ='K'   then Result:=-10*60
      else if sTZ='L'   then Result:=-11*60
      else if sTZ='M'   then Result:=-12*60
      else if sTZ='N'   then Result:=  1*60
      else if sTZ='O'   then Result:=  2*60
      else if sTZ='P'   then Result:=  3*60
      else if sTZ='Q'   then Result:=  4*60
      else if sTZ='R'   then Result:=  5*60
      else if sTZ='S'   then Result:=  6*60
      else if sTZ='T'   then Result:=  7*60
      else if sTZ='U'   then Result:=  8*60
      else if sTZ='V'   then Result:=  9*60
      else if sTZ='W'   then Result:= 10*60
      else if sTZ='X'   then Result:= 11*60
      else if sTZ='Y'   then Result:= 12*60
      else if sTZ='Z'   then Result:=  0;

   end;
end;

function BiasMinutesToRfcTimezone( const BiasMin: Integer ): TRfcTimeZone;
begin
   if BiasMin = 0 then begin
      Result := 'GMT';
   end else begin
      if BiasMin < 0 then Result := '+'
                     else Result := '-';
      Result := Result + Format( '%.2d%.2d',
                                 [ abs(BiasMin) div 60, abs(BiasMin) mod 60 ] )
   end;
end;

function NowRfcTimezone: TRfcTimeZone;
begin
   Result := BiasMinutesToRfcTimezone( NowBiasMinutes );
end;

function DateTimeToRfcTimezone( const DateTime: TDateTime ): TRfcTimeZone;
begin
   Result := BiasMinutesToRfcTimezone( DateTimeToBiasMinutes( DateTime ) );
end;

// ----------------------------------------------------------------------------

function DateTimeToSystemTime( const DateTime: TDateTime ): TSystemTime;
begin
   with Result do begin
      DecodeDate( DateTime, wYear, wMonth, wDay );
      DecodeTime( DateTime, wHour, wMinute, wSecond, wMilliseconds );
      wDayOfWeek := DayOfWeek( DateTime ) - 1;
   end;
end;

function SystemTimeIsDaylight( const SystemTime: TSystemTime ): Boolean;
var  XX, DB, DE: TDateTime;
     y, m, d: Word;
     TZI: TTimeZoneInformation;
begin
   Result := False;

   GetTimeZoneInformation( TZI );

   if TZI.StandardDate.wMonth=0 then exit; // no Daylight-infos

   if TZI.StandardDate.wYear=0 then begin
      // day-in-month-format (wDayOfWeek, wDay)
      with TZI.DaylightDate do begin
         DB := EncodeDate( SystemTime.wYear, wMonth, 1 );
         while DayOfWeek(DB){Sun=1}-1<>wDayOfWeek{Sun=0} do DB:=DB+1;
         dec( wDay );
         while wDay>0 do begin
            XX := DB + 7;
            DecodeDate( XX, y, m, d );
            if m<>wMonth then break;
            DB := XX;
            dec( wDay );
         end;
      end;
      with TZI.StandardDate do begin
         DE := EncodeDate( SystemTime.wYear, wMonth, 1 );
         while DayOfWeek(DE){Sun=1}-1<>wDayOfWeek{Sun=0} do DE:=DE+1;
         dec( wDay );
         while wDay>0 do begin
            XX := DE + 7;
            DecodeDate( XX, y, m, d );
            if m<>wMonth then break;
            DE := XX;
            dec( wDay );
         end;
      end;
   end else begin
      // absolute format
      DB := EncodeDate( SystemTime.wYear, TZI.DaylightDate.wMonth, TZI.DaylightDate.wDay );
      DE := EncodeDate( SystemTime.wYear, TZI.StandardDate.wMonth, TZI.StandardDate.wDay );
   end;

   DB := DB + EncodeTime( TZI.DaylightDate.wHour,   TZI.DaylightDate.wMinute,
                          TZI.DaylightDate.wSecond, TZI.DaylightDate.wMilliSeconds );
   DE := DE + EncodeTime( TZI.StandardDate.wHour,   TZI.StandardDate.wMinute,
                          TZI.StandardDate.wSecond, TZI.StandardDate.wMilliSeconds );

   XX := EncodeDate( SystemTime.wYear, SystemTime.wMonth, SystemTime.wDay );
   if DB<DE then begin
      if (XX>=DB) and (XX<DE) then Result:=True;
   end else begin
      if (XX>=DB) or  (XX<DE) then Result:=True;
   end;
end;

function DateTimeIsDaylight( const DateTime: TDateTime ): Boolean;
begin
   Result := SystemTimeIsDaylight( DateTimeToSystemTime( DateTime ) );
end;

function DateTimeToBiasMinutes( const DateTime: TDateTime ): Integer;
var  TZI: TTimeZoneInformation;
begin
   GetTimeZoneInformation(TZI);
   if DateTimeIsDaylight( DateTime ) then begin
      Result := TZI.Bias + TZI.DayLightBias;
   end else begin
      Result := TZI.Bias + TZI.StandardBias;
   end;
end;

function DateTimeGMTToLocal( const DateTime: TDateTime ): TDateTime;
begin
   Result := DateTime - MinutesToDateTime( DateTimeToBiasMinutes(DateTime) );
end;

function DateTimeLocalToGMT( const DateTime: TDateTime ): TDateTime;
begin
   Result := DateTime + MinutesToDateTime( DateTimeToBiasMinutes(DateTime) );
end;

// ----------------------------------------------------------------------------

end.
