// ============================================================================
// Handler for installed Windows character sets
// Copyright (c) 2002, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uCharsetsW; 

interface

uses SysUtils, Classes, uCharsets;

type
   TWindowsCharsetHandler = class( TCustomCharsetHandler )
      private
         FCodePage: Integer;
         FFallback: TCustomCharsetHandler;
         
      public
         function To16  ( const In8  : String8;
                          out   Out16: String16 ): Boolean; override;
         function From16( const In16 : String16;
                          out   Out8 : String8  ): Boolean; override;

         constructor Create( const ACodePage: Integer;
                             const AFallback: TCustomCharsetHandler = nil );
   end;

procedure EnumInstalledWindowsCodePages( const CodePageNumbers: TStrings );

implementation

uses Windows;

var  WindowsCodePageList: TStringList;

function MyEnumCodePagesProc( lpCodePageString: PChar ): Boolean; stdcall;
begin
   WindowsCodePageList.Add( lpCodePageString );
   Result := True;
end;

procedure EnumInstalledWindowsCodePages( const CodePageNumbers: TStrings );
begin
   CodePageNumbers.Clear;

   WindowsCodePageList := TStringList.Create;
   try
      if EnumSystemCodePages( @MyEnumCodePagesProc, CP_INSTALLED ) then begin
         CodePageNumbers.Text := WindowsCodePageList.Text;
      end;
   finally WindowsCodePageList.Free end;
end;


{ TWindowsCharsetHandler }

constructor TWindowsCharsetHandler.Create( const ACodePage: Integer;
                                           const AFallback: TCustomCharsetHandler );
var  csn: String;
begin
   csn :=       'cp'       + inttostr(ACodePage)
        + ' ' + 'cp-'      + inttostr(ACodePage)
        + ' ' + 'windows-' + inttostr(ACodePage);
        
   case ACodePage of
        932: csn := csn + ' ' + 'shift_jis';
        936: csn := csn + ' ' + 'gb2312';
        949: csn := csn + ' ' + 'euc-kr';
        950: csn := csn + ' ' + 'big5';
      50220: csn := csn + ' ' + 'iso-2022-jp';
      50221: csn := csn + ' ' + 'iso-2022-jp';
      50222: csn := csn + ' ' + 'iso-2022-jp';
      50225: csn := csn + ' ' + 'iso-2022-kr';
   end;

   inherited Create( csn );
   
   FCodePage := ACodePage;
   FFallback := AFallback;
end;

function TWindowsCharsetHandler.From16( const In16: String16;
                                        out   Out8: String8 ): Boolean;
var  r: Integer;
begin
   SetLength( Out8, length(In16) * 2 );

   r := WideCharToMultiByte(
           FCodePage,         // code page
           0,                 // performance and mapping flags
           PWideChar( In16 ), // address of wide-character string
           length( In16 ),    // number of characters in string
           PAnsiChar( Out8 ), // address of buffer for new string
           length( Out8 ),    // size of buffer
           nil,               // address of default for unmappable characters
           nil                // address of flag set when default char. used
        );

   if r <> 0 then begin

      Result := True;
      SetLength( Out8, r );

   end else begin

      Result := False;
      if Assigned( FFallback ) then FFallback.From16( In16, Out8 );

   end;
end;

function TWindowsCharsetHandler.To16( const In8  : String8;
                                      out   Out16: String16 ): Boolean;
var  r: Integer;
begin
   SetLength( Out16, length(In8) * 2 );

   r := MultiByteToWideChar(
           FCodePage,          // code page
           0,                  // character-type options
           PAnsiChar( In8 ),   // address of string to map
           length( In8 ),      // number of characters in string
           PWideChar( Out16 ), // address of wide-character buffer
           length( Out16 )     // size of buffer
        );

   if r <> 0 then begin

      Result := True;
      SetLength( Out16, r );

   end else begin

      Result := False;
      if Assigned( FFallback ) then FFallback.To16( In8, Out16 );

   end;
end;

end.
