// ============================================================================
// Handler for Unicode character sets
// Copyright (c) 2002, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uCharsetsU;

interface

uses SysUtils, Classes, uCharsets, uUnicoding;

type
   TUTF7CharsetHandler = class( TCustomCharsetHandler )
      public
         function To16  ( const In8  : String8;
                          out   Out16: String16 ): Boolean; override;
         function From16( const In16 : String16;
                          out   Out8 : String8  ): Boolean; override;

         constructor Create;
   end;

   TUTF8CharsetHandler = class( TCustomCharsetHandler )
      public
         function To16  ( const In8  : String8;
                          out   Out16: String16 ): Boolean; override;
         function From16( const In16 : String16;
                          out   Out8 : String8  ): Boolean; override;

         constructor Create;
   end;

   TUTF16CharsetHandler = class( TCustomCharsetHandler )
      private
         FDefaultBOM: TUTF_ByteOrderMarks;
         FForceBOM  : Boolean;
      public
         function To16  ( const In8  : String8;
                          out   Out16: String16 ): Boolean; override;
         function From16( const In16 : String16;
                          out   Out8 : String8  ): Boolean; override;

         constructor Create;
   end;

   TUTF16BECharsetHandler = class( TUTF16CharsetHandler )
      public
         constructor Create;
   end;

   TUTF16LECharsetHandler = class( TUTF16CharsetHandler )
      public
         constructor Create;
   end;


implementation

{ TUTF7CharsetHandler }

constructor TUTF7CharsetHandler.Create;
begin
   inherited Create( 'UTF-7' );
   FBitsPerChar := 7;
end;

function TUTF7CharsetHandler.From16( const In16: String16;
                                     out   Out8: String8 ): Boolean;
begin
   Result := UTF16ToUTF7( In16, Out8 );
end;

function TUTF7CharsetHandler.To16( const In8  : String8;
                                   out   Out16: String16 ): Boolean;
begin
   Result := UTF7ToUTF16( In8, Out16, bomNative );
end;

{ TUTF8CharsetHandler }

constructor TUTF8CharsetHandler.Create;
begin
   inherited Create( 'UTF-8' );
end;

function TUTF8CharsetHandler.From16( const In16: String16;
                                     out   Out8: String8 ): Boolean;
begin
   Result := UTF16ToUTF8( In16, Out8 );
end;

function TUTF8CharsetHandler.To16( const In8  : String8;
                                   out   Out16: String16 ): Boolean;
begin
   Result := UTF8ToUTF16( In8, Out16, bomNative );
end;

{ TUTF16CharsetHandler }

constructor TUTF16CharsetHandler.Create;
begin
   inherited Create( 'UTF-16' );
   FBitsPerChar := 16;
   FDefaultBOM  := bomBE;
   FForceBOM    := False;
end;

function TUTF16CharsetHandler.From16( const In16: String16;
                                      out   Out8: String8 ): Boolean;
// note: Out8 will contain bytes of char16 here
var  pch: PWideChar;
     swp: Boolean;
     Words, Bytes, i: Integer;
begin
   Result := True;

   Words := length( In16 );
   Bytes := Words * 2;
   SetLength( Out8, Bytes );
   if Bytes = 0 then exit;

   Move( In16[1], Out8[1], Bytes );

   // fix byte order if not in In16's native order
   swp := ( UTF16_BOM[FDefaultBOM].c16 <> UTF16_BOM[bomNATIVE].c16 );
   if not FForceBOM then begin // UTF-16: don't swap, if 1st char is BOM
      if UTF16( In16[1] ) = UTF16_BOM[bomNative].c16 then swp := False;
   end;

   if swp then begin // swap byte order
      pch := @Out8[1];
      for i := 1 to Words do begin
         pch^ := UTF16Char( Swap( UTF16(pch^) ) );
         inc( pch );
      end;
   end;
end;

function TUTF16CharsetHandler.To16( const In8  : String8;
                                    out   Out16: String16 ): Boolean;
// note: In8 contains bytes of char16 here
var  pch: PWideChar;
     c16: UTF16;
     bom: TUTF_ByteOrderMarks;
     swp: Boolean;
     Bytes, Words, i: Integer;
begin
   Result := True;

   Bytes := length( In8 );
   Words := Bytes div 2;
   SetLength( Out16, Words );
   if Bytes = 0 then exit;

   Move( In8[1], Out16[1], Words * 2 );

   // fix byte order if not in Out16's native order
   if FForceBOM then begin // UTF-16BE/LE
      swp := ( UTF16_BOM[FDefaultBOM].c16 <> UTF16_BOM[bomNATIVE].c16 );
   end else begin // UTF-16
      c16 := UTF16( Out16[1] ); // 1st char
      bom := FDefaultBOM;
      if c16 = UTF16_BOM[bomBE].c16 then bom := bomBE;
      if c16 = UTF16_BOM[bomLE].c16 then bom := bomLE;
      swp := ( UTF16_BOM[bom].c16 <> UTF16_BOM[bomNATIVE].c16 );
   end;

   if swp then begin // swap byte order
      pch := @Out16[1];
      for i := 1 to Words do begin
         pch^ := UTF16Char( Swap( UTF16(pch^) ) );
         inc( pch );
      end;
   end;
end;

{ TUTF16BECharsetHandler }

constructor TUTF16BECharsetHandler.Create;
begin
   inherited Create;
   FNamesList   := 'UTF-16BE';
   FBitsPerChar := 16;
   FDefaultBOM  := bomBE;
   FForceBOM    := True;
end;

{ TUTF16LECharsetHandler }

constructor TUTF16LECharsetHandler.Create;
begin
   inherited Create;
   FNamesList   := 'UTF-16LE';
   FBitsPerChar := 16;
   FDefaultBOM  := bomLE;
   FForceBOM    := True;
end;

end.
