// ============================================================================
// Simple XML reader
// Copyright (c) 2001, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cXML;

interface

uses SysUtils, Classes;

type
   TXMLElementTypes = (
      XML_CONTAINER,  // fictitious element containing xml-document

      XML_PROLOG,     // <?xml ... ?>
      XML_PROCESSING, // <?name ... ?>
      XML_ELEMENT,    // <name ...>
      XML_TEXT,       // text
      XML_COMMENT,    // <!-- ... -->

      XML_ATTRIBUTE   // attribute (name+value) of XML_ELEMENT
   );

   TXMLElement    = class;
   TXMLElements   = class;

   TXMLObject = class
      private
         fOwner: TXMLElement;
      public
         property Owner: TXMLElement read fOwner;
         constructor Create( aOwner: TXMLElement );
   end;

   TXMLAttribute = class( TXMLObject )
      private
         fName : String;
         fValue: String;
         procedure SetValue( NewValue: String );
      public
         property Name : String read fName;
         property Value: String read fValue write SetValue;
         constructor Create(aOwner: TXMLElement; aName, aValue: String);
   end;

   TXMLAttributes = class( TXMLObject )
      private
         fList: TList;
         function GetCount: Integer;
         function GetItem(Index: Integer): TXMLAttribute;
         function GetAttr(AttrName: String): TXMLAttribute;
         function GetValue(AttrName: String): String;
         procedure SetValue(AttrName: String; NewValue: String);

      public
         property Count: Integer read GetCount;
         property Item[Index: Integer]: TXMLAttribute read GetItem;
         property Attribute[AttrName: String]: TXMLAttribute read GetAttr;
         property Value[AttrName: String]: String read GetValue write SetValue;

         procedure Clear;
         function Add( AttrName, AttrValue: String ): TXMLAttribute;

         constructor Create(aOwner: TXMLElement);
         destructor Destroy; override;
   end;

   TXMLElement = class( TXMLObject )
      private
         fElemText  : String;
         fElemType  : TXMLElementTypes;
         fElements  : TXMLElements;
         fAttributes: TXMLAttributes;
         function GetElement(aName: String): TXMLElement;
         function GetAttribute(aName: String): TXMLAttribute;

      public
         property ElemText  : String           read fElemText;
         property ElemType  : TXMLElementTypes read fElemType;
         property Elements  : TXMLElements     read fElements;
         property Attributes: TXMLAttributes   read fAttributes;
         property Element  [aName: String]: TXMLElement   read GetElement;
         property Attribute[aName: String]: TXMLAttribute read GetAttribute;

         procedure Clear; virtual;

         constructor Create(aOwner: TXMLElement; aText: String; aType: TXMLElementTypes);
         destructor Destroy; override;
   end;

   TXMLElements = class( TXMLObject )
      private
         fList: TList;
         function GetCount: Integer;
         function GetItem(Index: Integer): TXMLElement;
         function GetElement(aName: String): TXMLElement;

      public
         property Count: Integer read GetCount;
         property Item[Index: Integer]: TXMLElement read GetItem;
         property Element[aName: String]: TXMLElement read GetElement;

         procedure Clear;
         function Add(aType: TXMLElementTypes; aText: String): TXMLElement;

         constructor Create(aOwner: TXMLElement);
         destructor Destroy; override;
   end;

   TXMLDocument = class( TXMLElement )
      private
         fParseError: String;
         fIndent    : Integer;
         fCharset   : String;
         function GetProlog: TXMLElement;
         function GetRoot: TXMLElement;
         function GetValid: Boolean;
         function GetAsText: String;
         procedure SetAsText( Value: String );
      public
         property ParseError: String      read fParseError;
         property Prolog    : TXMLElement read GetProlog;
         property Root      : TXMLElement read GetRoot;
         property Valid     : Boolean     read GetValid;
         property Indent    : Integer     read fIndent write fIndent;
         property AsText    : String      read GetAsText write SetAsText;
         property CharSet   : String      read fCharset write fCharset;

         procedure Clear; override;
         procedure ClearAndCreate( RootElementName: String );

         procedure LoadFromFile( XMLFile: String);
         procedure SaveToFile( XMLFile: String );

         constructor Create;
         destructor Destroy; override;
   end;

function TrimMultiSpace( const s: String ): String;

implementation

uses uCharsets;

type
   TXMLParser = class
      public
         procedure Parse        ( XMLText: String; XMLDocument: TXMLDocument );
         procedure ParseFromFile( XMLFile: String; XMLDocument: TXMLDocument );
   end;

{ parser tools }

function StrToXml( s: String ): String;
var  p: PChar;
begin
   Result := '';
   p := PChar( s );
   while p^<>#0 do begin
      case p^ of
         '<' : Result := Result + '&lt;';
         '>' : Result := Result + '&gt;';
         '&' : Result := Result + '&amp;';
         '''': Result := Result + '&apos;';
         '"' : Result := Result + '&quot;';
         else Result := Result + p^;
      end;
      inc( p );
   end;
end;

function XmlToStr( s: String ): String;
var  p: PChar;
begin
   Result := '';
   p := PChar( s );
   while p^<>#0 do begin
      if          StrLComp( p, '&lt;',   4 )=0 then begin
         Result := Result + '<';
         inc( p, 4 );
      end else if StrLComp( p, '&gt;',   4 )=0 then begin
         Result := Result + '>';
         inc( p, 4 );
      end else if StrLComp( p, '&amp;',  5 )=0 then begin
         Result := Result + '&';
         inc( p, 5 );
      end else if StrLComp( p, '&apos;', 6 )=0 then begin
         Result := Result + '''';
         inc( p, 6 );
      end else if StrLComp( p, '&quot;', 6 )=0 then begin
         Result := Result + '"';
         inc( p, 6 );
      end else begin
         Result := Result + p^;
         inc( p );
      end;
   end;
end;


function IsSpace( c: Char ): Boolean;
begin
   Result := ( c in [#9, #10, #13, #32] );
end;

function TrimSpace( const s: String ): String;
begin
   Result := s;
   while Result<>'' do begin
      if IsSpace(Result[1]) then
         System.Delete( Result, 1, 1 )
      else if IsSpace(Result[length(Result)]) then
         System.Delete( Result, length(Result), 1 )
      else
         break;
   end;
end;

function TrimMultiSpace( const s: String ): String;
var  i: Integer;
begin
   Result := TrimSpace( s );
   i := length( Result ) - 1;
   while i>0 do begin
      if IsSpace(Result[i]) and IsSpace(Result[i+1]) then System.Delete(Result,i,1);
      dec( i );
   end;
end;

function IsLetter( c: Char ): Boolean;
begin
   Result := ( c in ['a'..'z','A'..'Z'] );
end;

function IsNameFirst( c: Char ): Boolean;
begin
   Result := IsLetter(c) or ( c in ['_',':'] );
end;

function IsNameChar( c: Char ): Boolean;
begin
   Result := IsNameFirst(c) or ( c in ['0'..'9','.','-','_',':'] );
end;

procedure SkipS( var P: PChar );
begin
   while P^<>#0 do begin
      if IsSpace(P^) then inc(P) else break;
   end;
end;

function GetName( var P: PChar ): String;
begin
   Result := '';
   SkipS( P );

   if not IsNameFirst(P^) then exit;
   Result := P^;
   inc(P);

   while P^<>#0 do begin
      if not IsNameChar(P^) then break;
      Result := Result + P^;
      inc( P );
   end;
end;

function GetLiteral( var P: PChar ): String;
var  QuoteCh: Char;
begin
   Result := '';
   SkipS( P );

   if      P^='"'  then QuoteCh:='"'
   else if P^='''' then QuoteCh:=''''
   else    exit;

   inc(P);

   while P^<>#0 do begin
      if P^=QuoteCh then begin inc( P ); break; end;
      Result := Result + P^;
      inc( P );
   end;
end;

function GetCharData( var P: PChar ): String;
begin
   Result := '';
   SkipS( P );

   while P^<>#0 do begin
      if P^='<' then break;
      Result := Result + P^;
      inc( P );
   end;

   while Result<>'' do begin
      if not IsSpace( Result[length(Result)] ) then break;
      System.Delete( Result, length(Result), 1 );
   end;
end;

function GetMarkup( var P: PChar ): String;
begin
   Result := '';
   SkipS( P );
   if P^ <> '<' then exit;

   while P^ <> #0 do begin
      Result := Result + P^;
      if P^ = '>' then begin
         if copy(Result,1,4) = '<!--' then begin
            if copy(Result,length(Result)-2,3)='-->' then begin
               inc(P);
               break;
            end
         end else if copy(Result,1,2) = '<?' then begin
            if copy(Result,length(Result)-1,2)='?>' then begin
               inc(P);
               break;
            end
         end else if copy(Result,1,3) = '<![' then begin
            if copy(Result,length(Result)-1,2)=']>' then begin
               inc(P);
               break;
            end
         end else begin
            inc(P);
            break;
         end;
      end;
      inc( P );
   end;
end;

function GetEntity( var P: PChar ): String;
begin
   Result := '';
   SkipS( P );
   if P^=#0 then exit;
   if P^='<' then Result := GetMarkup(P)
             else Result := GetCharData(P);
end;

function NameOfMarkup( Markup: String ): String;
var  P: PChar;
begin
   Result := '';

   if copy(Markup,1,1)<>'<' then exit;
   System.Delete( Markup, 1, 1 );
   if copy(Markup,1,1)='/' then System.Delete( Markup, 1, 1 );

   if copy(Markup,length(Markup),1)<>'>' then exit;
   System.Delete( Markup, length(Markup), 1 );

   P := PChar( Markup );
   Result := GetName( P );
end;

{ TXMLParser }

procedure TXMLParser.ParseFromFile( XMLFile: String; XMLDocument: TXMLDocument );
var  fXMLTxt: TStringList;
begin
   fXMLTxt := TStringList.Create;
   try
      XMLDocument.Clear;
      if FileExists( XMLFile ) then begin
         fXMLTxt.LoadFromFile( XMLFile );
      end;
      Parse( fXMLTxt.Text, XMLDocument );
   finally
      FreeAndNil( fXMLTxt );
   end;
end;

procedure TXMLParser.Parse( XMLText: String; XMLDocument: TXMLDocument );

   procedure SetError( ErrText: String; Location: String );
   begin
      with XMLDocument do begin
         if fParseError<>'' then exit;
         if length(Location)>64 then Location := copy(Location,1,63) + '...';
         if Location<>'' then Location := ' at "' + Location + '"';
         fParseError := ErrText + Location;
      end;
   end;

   function AddElement( Current: TXMLElement; Entity: String ): TXMLElement;
   var  EText, AName, AValue: String;
        EType: TXMLElementTypes;
        P: PChar;
   begin
      Result := nil;
      if not Assigned(Current) then exit;

      if (Entity[1]='<') and (Entity[length(Entity)]='>') then begin

         System.Delete( Entity, 1, 1 );
         System.Delete( Entity, length(Entity), 1 );

         EText := '';
         EType := XML_ELEMENT;

         if copy(Entity,1,1)='?' then begin
            EType := XML_PROCESSING; // or XML_PROLOG
            System.Delete( Entity, 1, 1 );
         end;

         P := PChar( Entity );
         EText := GetName( P );
         if EText='' then begin
            SetError( 'Invalid ENAME in <ENAME ANAME="AVAL" ...>', '<'+Entity+'>' );
            exit;
         end;

         if EText='xml'  then EType:=XML_PROLOG;
         // if EText[1]='!' then begin EType:=XML_COMMENT; EText:=XmlToStr(EText); end;

         Result := Current.Elements.Add( EType, EText );

         while P^<>#0 do begin
            SkipS( P );
            if P^=#0 then break;

            AName := GetName( P );
            if AName='' then begin
               SetError( 'Missing ANAME in <ENAME ANAME="AVAL" ...>', '<'+Entity+'>' );
               break;
            end;

            SkipS( P );
            if P^<>'=' then begin
               SetError( 'Missing "=" in <ENAME ANAME="AVAL" ...>', '<'+Entity+'>' );
               break;
            end;
            inc(P);

            SkipS( P );
            if not( P^ in ['"',''''] ) then begin
               SetError( 'Missing "AVAL"-string in <ENAME ANAME="AVAL" ...>', '<'+Entity+'>' );
               break;
            end;
            AValue := GetLiteral( P );

            Result.Attributes.Add( AName, XmlToStr( AValue ) );
         end;

      end;
   end;

   function ReadEntity( Current: TXMLElement; var P: PChar; WantEndMark: String ): Boolean;
   var  Entity: String;
   begin
      Result := True;
      if not Assigned(Current) then begin Result:=False; exit; end;

      while P^<>#0 do begin
         if XMLDocument.fParseError<>'' then begin Result:=False; break; end;
         
         Entity := GetEntity( P );

         if Entity='' then begin
            break;

         end else if copy(Entity,1,1)<>'<' then begin
            // text
            Current.Elements.Add( XML_TEXT, XmlToStr( Entity ) );

         end else if copy(Entity,1,2)='<?' then begin
            if (copy(Entity,length(Entity)-1,2)='?>') then begin
               // prolog, processing
               System.Delete( Entity, length(Entity)-1, 1 ); // '?>' => '>'
               AddElement( Current, Entity );
            end else begin
               SetError( 'Missing closing "?>"', Entity );
               break;
            end;

         end else if copy(Entity,1,4)='<!--' then begin
            if copy(Entity,length(Entity)-2,3)='-->' then begin
               // comment
               System.Delete( Entity, 1, 4 );
               System.Delete( Entity, length(Entity)-2, 3 );
               Current.Elements.Add( XML_COMMENT, XmlToStr( Entity ) );
            end else begin
               SetError( 'Missing end of comment ("-->")', Entity );
               break;
            end;

         end else if copy(Entity,length(Entity)-1,2)='/>' then begin
            // empty element
            System.Delete( Entity, length(Entity)-1, 1 );  // '/>' => '>'
            AddElement( Current, Entity );

         end else if copy(Entity,1,2)='</' then begin
            // closing element
            if NameOfMarkup(Entity)<>WantEndMark then begin
               SetError( '</'+WantEndMark+'> expected but '+Entity+' found', '' );
               Result := False;
            end;
            exit;

         end else if (copy(Entity,1,3)='<![') and (copy(Entity,length(Entity),1)='>') then begin
            // <![CDATA[ text ]]>
            System.Delete( Entity, 1, 3 );
            System.Delete( Entity, length(Entity), 1 );
            while (Entity <> '') and (copy( Entity, length(Entity), 1 ) = ']') do begin
               System.Delete( Entity, length(Entity), 1 );
            end;
            if UpperCase( copy( Entity, 1, 6 ) ) = 'CDATA[' then begin
               System.Delete( Entity, 1, 6 );
               Current.Elements.Add( XML_TEXT, XmlToStr( Trim(Entity) ) );
            end else begin
               Current.Elements.Add( XML_TEXT, XmlToStr( Trim(Entity) ) );
            end;

         end else if (copy(Entity,1,2)='<!') and (copy(Entity,length(Entity),1)='>') then begin
            System.Delete( Entity, 1, 1 );
            System.Delete( Entity, length(Entity), 1 );
            Current.Elements.Add( XML_COMMENT, XmlToStr( Entity ) );

         end else if (copy(Entity,1,1)='<') and (copy(Entity,length(Entity),1)='>') then begin
            // start element
            Result := ReadEntity( AddElement( Current, Entity ), P, NameOfMarkup(Entity) );

         end else begin
            SetError( 'Invalid entity', Entity );
            Result := False;
            exit;

         end;
      end;

      if WantEndMark<>'' then begin
         SetError( 'Missing </'+WantEndMark+'>', '' );
         Result := False;
      end;

   end;

var  P: PChar;
     i, np, nm: Integer;
     e: TXMLElement;
     a: TXMLAttribute;
begin
   XMLDocument.Clear;
   P := PChar( XMLText );
   ReadEntity( XMLDocument, P, '' );
   SkipS( P );
   if P^<>#0 then SetError( 'End of file expected', String(P) );

   XMLDocument.CharSet := 'UTF-8'; // CharSets.WindowsACPHandler.PreferredName;
   np := 0;
   nm := 0;
   for i:=0 to XMLDocument.Elements.Count-1 do begin
      case XMLDocument.Elements.Item[i].ElemType of
         XML_PROLOG:
            begin
               inc(np);
               if i<>0 then SetError( '<?xml> is not the first element of document', '' );
               e := XMLDocument.Elements.Item[i];
               a := e.Attribute[ 'encoding' ];
               if a <> nil then begin
                  if a.Value <> '' then XMLDocument.CharSet := a.Value;
               end;
            end;
         XML_ELEMENT: inc(nm);
         XML_TEXT:
            SetError( 'CharData at top level of document not allowed',
                      XMLDocument.Elements.Item[i].ElemText );
      end;
   end;
   if np>1 then SetError( 'Only one <?xml?> allowed', '' );
   if nm=0 then SetError( 'Missing root element of document', '' );
   if nm>1 then SetError( 'Only one root element allowed in document', '' );
end;

{ TXMLObject }

constructor TXMLObject.Create(aOwner: TXMLElement);
begin
   inherited Create;
   fOwner := aOwner;
end;

{ TXMLAttribute }

procedure TXMLAttribute.SetValue( NewValue: String );
begin
   fValue := NewValue;
end;

constructor TXMLAttribute.Create(aOwner: TXMLElement; aName, aValue: String);
begin
   inherited Create( aOwner );
   fName  := aName;
   fValue := aValue;
end;

{ TXMLAttributes }

constructor TXMLAttributes.Create(aOwner: TXMLElement);
begin
   inherited Create( aOwner );
   fList  := TList.Create;
end;

destructor TXMLAttributes.Destroy;
begin
   try
      if Assigned(fList) then begin
         Clear;
         FreeAndNil( fList );
      end;
   except end;
   inherited Destroy;
end;

procedure TXMLAttributes.Clear;
begin
   while fList.Count>0 do begin
      TXMLAttribute( fList[0] ).Free;
      fList.Delete( 0 );
   end;
end;

function TXMLAttributes.GetCount: Integer;
begin
   Result := fList.Count;
end;

function TXMLAttributes.GetItem(Index: Integer): TXMLAttribute;
begin
   Result := fList[Index];
end;

function TXMLAttributes.GetAttr(AttrName: String): TXMLAttribute;
var  i: Integer;
begin
   Result := nil;
   for i:=0 to Count-1 do begin
      if Item[i].Name=AttrName then begin
         Result := Item[i];
         break;
      end;
   end;
end;

function TXMLAttributes.GetValue(AttrName: String): String;
var  A: TXMLAttribute;
begin
   A := Attribute[AttrName];
   if Assigned(A) then Result := A.Value
                  else Result := '';
end;

procedure TXMLAttributes.SetValue(AttrName: String; NewValue: String);
var  A: TXMLAttribute;
begin
   A := Attribute[AttrName];
   if Assigned(A) then A.Value := NewValue
                  else Add( AttrName, NewValue );
end;

function TXMLAttributes.Add(AttrName, AttrValue: String): TXMLAttribute;
begin
   Result := TXMLAttribute.Create(Owner, AttrName, AttrValue);
   fList.Add( Result );
end;

{ TXMLElement }

constructor TXMLElement.Create(aOwner: TXMLElement; aText: String; aType: TXMLElementTypes);
begin
   inherited Create( aOwner );
   fElemText   := aText;
   fElemType   := aType;
   fElements   := TXMLElements.Create(Self);
   fAttributes := TXMLAttributes.Create(Self);
end;

destructor TXMLElement.Destroy;
begin
   try
      Clear;
      FreeAndNil( fElements   );
      FreeAndNil( fAttributes );
   except end;
   inherited Destroy;
end;

procedure TXMLElement.Clear;
begin
   fElements.Clear;
   fAttributes.Clear;
end;

function TXMLElement.GetElement(aName: String): TXMLElement;
var  i: Integer;
begin
   Result := nil;

   for i:=0 to fElements.Count-1 do begin
      if fElements.Item[i].ElemText=aName then begin
         Result := fElements.Item[i];
         break;
      end;
   end;
end;

function TXMLElement.GetAttribute(aName: String): TXMLAttribute;
var  i: Integer;
begin
   Result := nil;

   for i:=0 to fAttributes.Count-1 do begin
      if fAttributes.Item[i].Name=aName then begin
         Result := fAttributes.Item[i];
         break;
      end;
   end;
end;

{ TXMLElements }

constructor TXMLElements.Create(aOwner: TXMLElement);
begin
   inherited Create( aOwner );
   fList  := TList.Create;
end;

destructor TXMLElements.Destroy;
begin
   try FreeAndNil( fList ); except end;
   inherited Destroy;
end;

procedure TXMLElements.Clear;
begin
   while fList.Count>0 do begin
      TXMLElement( fList[0] ).Free;
      fList.Delete( 0 );
   end;
end;

function TXMLElements.GetCount: Integer;
begin
   Result := fList.Count;
end;

function TXMLElements.GetItem(Index: Integer): TXMLElement;
begin
   Result := fList[Index];
end;

function TXMLElements.Add( aType: TXMLElementTypes; aText: String ): TXMLElement;
begin
   Result := TXMLElement.Create( Owner, aText, aType );
   fList.Add( Result );
end;

function TXMLElements.GetElement(aName: String): TXMLElement;
var  i: Integer;
begin
   Result := nil;

   for i:=0 to Count-1 do begin
      if Item[i].ElemText=aName then begin
         Result := Item[i];
         break;
      end;
   end;
end;

{ TXMLDocument }

function TXMLDocument.GetProlog: TXMLElement;
var  i: Integer;
begin
   Result := nil;

   for i:=0 to Elements.Count-1 do begin
      if Elements.Item[i].ElemType=XML_PROLOG then begin
         Result := Elements.Item[i];
         break;
      end;
   end;
end;

function TXMLDocument.GetRoot: TXMLElement;
var  i: Integer;
begin
   Result := nil;

   for i:=0 to Elements.Count-1 do begin
      if Elements.Item[i].ElemType=XML_ELEMENT then begin
         Result := Elements.Item[i];
         break;
      end;
   end;
end;

function TXMLDocument.GetValid: Boolean;
begin
   Result := Assigned( Prolog ) and Assigned( Root );
end;

function TXMLDocument.GetAsText: String;
var  TS: TStringList;

   function xSp( i: Integer ): String;
   begin
      SetLength( Result, i );
      FillChar( Result[1], i, 32 );
   end;

   function xLastLine: String;
   begin
      if TS.Count=0 then Result := ''
                    else Result := TS[ TS.Count-1 ];
   end;

   function xAttr( Name: String; Node: TXMLElement ): String;
   var  i: Integer;
        s: String;
   begin
      Result := '<' + Name;
      for i:=0 to Node.Attributes.Count-1 do begin
         s := '"' + StrToXml( Node.Attributes.Item[i].Value ) + '"';
         Result := Result + ' ' + Node.Attributes.Item[i].Name + '=' + s;
      end;
      if Result[2]='?' then Result:=Result+'?';
      Result := Result + '>';
   end;

   procedure xAddText( Text: String; Depth: Integer );
   var  i: Integer;
        s: String;
   begin
      while Text<>'' do begin
         i := Pos( #10, Text );
         if i>0 then begin
            s := copy( Text, 1, i-1 );
            if copy(s,length(s),1)=#13 then System.Delete(s,length(s),1);
            System.Delete( Text, 1, i );
         end else begin
            s := Text;
            Text := '';
         end;

         TS.Add( xSp(Depth) + TrimMultiSpace(s) );
      end;
   end;

   procedure xAdd( Node: TXMLElement; Depth: Integer );
   var  i: Integer;
        s: String;
   begin
      case Node.ElemType of
         XML_PROLOG: begin
            TS.Add( xSp(Depth) + xAttr( '?' + Node.ElemText, Node ) );
            TS.Add( '' );
         end;

         XML_PROCESSING: begin
            TS.Add( xSp(Depth) + xAttr( '?' + Node.ElemText, Node ) );
         end;

         XML_ELEMENT: begin
            if (Depth div Indent<=2) and (xLastLine<>'') then TS.Add( '' );
            if Node.Elements.Count=0 then begin
               s := xAttr( Node.ElemText, Node );
               System.Insert( '/', s, length(s) );
               TS.Add( xSp(Depth) + s );
            end else begin
               TS.Add( xSp(Depth) + xAttr( Node.ElemText, Node ) );
               for i:=0 to Node.Elements.Count-1 do xAdd( Node.Elements.Item[i], Depth+Indent );
               TS.Add( xSp(Depth) + '</' + Node.ElemText + '>' );
            end;
            if (Depth div Indent<=1) and (xLastLine<>'') then TS.Add( '' );
         end;

         XML_TEXT: begin
            xAddText( StrToXml( Node.ElemText ), Depth );
         end;

         XML_COMMENT: begin
            xAddText( '<!-- ' + StrToXml( Node.ElemText ) + ' -->', Depth );
         end;
      end;
   end;

var  i: Integer;
begin
   Result := '';
   TS := TStringList.Create;
   try
      for i:=0 to Elements.Count-1 do xAdd( Elements.Item[i], 0 );
      Result := TS.Text;
   finally
      TS.Free;
   end;
end;

procedure TXMLDocument.SetAsText( Value: String );
var  XMLParser: TXMLParser;
begin
   XMLParser := TXMLParser.Create;
   try
      XMLParser.Parse( Value, Self );
   finally
      XMLParser.Free;
   end;
end;

procedure TXMLDocument.LoadFromFile(XMLFile: String);
var  XMLParser: TXMLParser;
begin
   XMLParser := TXMLParser.Create;
   try
      XMLParser.ParseFromFile( XMLFile, Self );
   finally
      XMLParser.Free;
   end;
end;

procedure TXMLDocument.SaveToFile( XMLFile: String );
var  TS: TStringList;
begin
   TS := TStringList.Create;
   try
      TS.Text := AsText;
      TS.SaveToFile( XMLFile );
   finally
      TS.Free;
   end;
end;

procedure TXMLDocument.Clear;
begin
   inherited Clear;
   fParseError := '';
end;

procedure TXMLDocument.ClearAndCreate( RootElementName: String );
begin
   Clear;
   with Elements.Add( XML_PROLOG, 'xml' ) do begin
      Attributes.Add( 'version',  '1.0'   );
      Attributes.Add( 'encoding', fCharset );
   end;
   Elements.Add( XML_ELEMENT, RootElementName );
end;

constructor TXMLDocument.Create;
begin
   inherited Create( Self, '<>', XML_CONTAINER );
   fCharset := 'UTF-8'; // 'ISO-8859-1';
   fIndent := 2;
   Clear;
end;

destructor TXMLDocument.Destroy;
begin
   inherited Destroy;
end;

end.
