// ============================================================================
// Various synchronization objects
// Copyright (c) 2001, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cSyncObjects; // Various synchronization objects

interface

{$INCLUDE Compiler.inc}

uses SysUtils, SyncObjs, Classes, Windows;

type
   TReaderWriterLock = class
      // Replacement for TMultiReadExclusiveWriteSynchronizer, which causes
      // deadlocks in certain situation, especially when promoting a reader
      // to a writer lock.
      // Note: Until a safe replacement is available, this class just treats
      //       reader locks the same as writer locks.
      private
         FLock: TCriticalSection;
         
      public
         procedure BeginRead;
         procedure EndRead;
         procedure BeginWrite;
         procedure EndWrite;

         constructor Create;
         destructor Destroy; override;
   end;

   TThreadStringList = class
      // TStringList with synchronized access like TThreadList.
      private
         FLock: TCriticalSection;
         FList: TStringList;
         
      public
         function  LockList: TStringList;
         procedure UnlockList;

         procedure Clear;
         function  Add( const Item: String ): Boolean;
         procedure Remove( const Item: String );

         constructor Create( ASorted: Boolean; ADuplicates: TDuplicates );
         destructor Destroy; override;
   end;

   TMultiReadExclusiveWriteStringList = class
      // TStringList which allows multiple threads to "read" concurrently but
      // prevents from "writing" concurrently. In this case, "write" means
      // changing the StringList in any way (.Strings[]:='new', Clear, Add,
      // Delete, Sort, ...), and it's your responsibility to access the list
      // with the appropriate BeginRead/BeginWrite method.
      private
         FLock: TReaderWriterLock;
         FList: TStringList;

      public
         function  BeginRead: TStringList;
         procedure EndRead;

         function  BeginWrite: TStringList;
         procedure EndWrite;

         constructor Create( ASorted: Boolean; ADuplicates: TDuplicates );
         destructor Destroy; override;
   end;

   TMutexObj = class
      private
         FHandle: Cardinal;

      public
         function  Acquire( dwMilliseconds: Cardinal ): Boolean;
         procedure Release;

         constructor Create( SecurityAttr: PSecurityAttributes;
                             InitialOwner: Boolean;
                             const Name: String );
         destructor  Destroy; override;
   end;

   TSemaphoreObj = class
      // Simple wrapper for a semaphore object.
      // Note: Changing .Count is not thread-safe (see notes at SetCount).
      private
         FHandle: Cardinal;
         FCount : Integer;
         FInUse : Integer;

         procedure SetCount( NewCount: Integer );

      public
         property Count: Integer read FCount write SetCount;
         property InUse: Integer read FInUse;

         function  Acquire( dwMilliseconds: Cardinal ): Boolean;
         procedure Release;

         constructor Create( InitialCount: Integer ); overload;
         constructor Create( SecurityAttr: PSecurityAttributes;
                             InitialCount: Integer;
                             MaximumCount: Integer;
                             const Name: String ); overload;
         destructor Destroy; override;        
   end;


implementation

{ TThreadStringList }

constructor TThreadStringList.Create;
begin
   inherited Create;
   FLock := TCriticalSection.Create;
   FList := TStringList.Create;
   FList.Sorted := ASorted;
   FList.Duplicates := ADuplicates;
end;

destructor TThreadStringList.Destroy;
begin
   FList.Free;
   FLock.Free;
   inherited Destroy;
end;

procedure TThreadStringList.Clear;
begin
   with LockList do try
      Clear;
   finally
      UnlockList;
   end;
end;

function TThreadStringList.Add( const Item: String ): Boolean;
begin
   with LockList do try
      Result := ( Duplicates <> dupIgnore) or ( IndexOf(Item) < 0 );
      if Result then Add( Item );
   finally
      UnlockList;
   end;
end;

procedure TThreadStringList.Remove( const Item: String );
var  i: Integer;
begin
   with LockList do try
      i := IndexOf( Item );
      if i>=0 then Delete( i );
   finally
      UnlockList;
   end;
end;

function TThreadStringList.LockList: TStringList;
begin
   FLock.Enter;
   Result := FList;
end;

procedure TThreadStringList.UnlockList;
begin
   FLock.Leave;
end;


{ TMultiReadExclusiveWriteStringList }

constructor TMultiReadExclusiveWriteStringList.Create(ASorted: Boolean;
  ADuplicates: TDuplicates);
begin
   inherited Create;
   FLock := TReaderWriterLock.Create;
   FList := TStringList.Create;
   FList.Sorted := ASorted;
   FList.Duplicates := ADuplicates;
end;

destructor TMultiReadExclusiveWriteStringList.Destroy;
begin
   FreeAndNil( FList );
   FLock.Free;
   inherited Destroy;
end;

function TMultiReadExclusiveWriteStringList.BeginRead: TStringList;
begin
   FLock.BeginRead;
   Result := FList;
end;

function TMultiReadExclusiveWriteStringList.BeginWrite: TStringList;
begin
   FLock.BeginWrite;
   Result := FList;
end;

procedure TMultiReadExclusiveWriteStringList.EndRead;
begin
   FLock.EndRead;
end;

procedure TMultiReadExclusiveWriteStringList.EndWrite;
begin
   FLock.EndWrite;
end;


{ TMutexObj }

constructor TMutexObj.Create( SecurityAttr: PSecurityAttributes;
                              InitialOwner: Boolean;
                              const Name: String);
begin
   inherited Create;
   FHandle := CreateMutex( SecurityAttr, InitialOwner, PChar(Name) );
end;

destructor TMutexObj.Destroy;
begin
   CloseHandle( FHandle );
   inherited;
end;

function TMutexObj.Acquire( dwMilliseconds: Cardinal ): Boolean;
begin
   Result := WaitForSingleObject( FHandle, dwMilliseconds ) = WAIT_OBJECT_0;
end;

procedure TMutexObj.Release;
begin
   ReleaseMutex( FHandle );
end;

{ TSemaphoreObj }

constructor TSemaphoreObj.Create( InitialCount: Integer );
begin
   inherited Create;
   FCount  := InitialCount;
   FInUse  := 0;
   FHandle := CreateSemaphore( nil, FCount, $7fffffff, nil );
end;

constructor TSemaphoreObj.Create( SecurityAttr: PSecurityAttributes;
                                  InitialCount: Integer;
                                  MaximumCount: Integer;
                                  const Name: String ); 
begin
   inherited Create;
   FCount  := InitialCount;
   FInUse  := 0;
   FHandle := CreateSemaphore( SecurityAttr,
                               InitialCount, MaximumCount,
                               PChar(Name) );
end;

destructor TSemaphoreObj.Destroy;
begin
   CloseHandle( FHandle );
   inherited;
end;

function TSemaphoreObj.Acquire( dwMilliseconds: Cardinal ): Boolean;
begin
   Result := WaitForSingleObject( FHandle, dwMilliseconds ) = WAIT_OBJECT_0;
   if Result then InterlockedIncrement( FInUse );
end;

procedure TSemaphoreObj.Release;
begin
   if ReleaseSemaphore( FHandle, 1, nil ) then InterlockedDecrement( FInUse );
end;

procedure TSemaphoreObj.SetCount(NewCount: Integer);
// Note:
// This procedure is not thread-safe, so you either have to synchronize access
// to it when calling or make sure by convention, that it is only called by one
// specific thread (e.g. only from App's main-thread).
begin
   if NewCount = FCount then exit;

   if NewCount > FCount then begin
      if ReleaseSemaphore( FHandle, NewCount - FCount, nil ) then begin
         FCount := NewCount;
      end;
   end else begin
      while ( FCount > 0 ) and ( NewCount < FCount ) do begin
         if WaitForSingleObject( FHandle, INFINITE ) = WAIT_OBJECT_0 then
            dec( FCount )
         else
            break;
      end;
   end;
end;


{ TReaderWriterLock }

constructor TReaderWriterLock.Create;
begin
   inherited Create;
   FLock := TCriticalSection.Create;
end;

destructor TReaderWriterLock.Destroy;
begin
   BeginWrite;
   FLock.Free;
   inherited Destroy;
end;

procedure TReaderWriterLock.BeginRead;
begin
   BeginWrite;
end;

procedure TReaderWriterLock.BeginWrite;
begin
   FLock.Enter;
end;

procedure TReaderWriterLock.EndRead;
begin
   EndWrite;
end;

procedure TReaderWriterLock.EndWrite;
begin
   FLock.Leave;
end;

end.
