// ============================================================================
// Base class for 64 bit streams.
// Copyright (c) 2002, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cStream64;

interface

uses SysUtils;

const
   // seek origins
   soFromBeginning64 = 0;
   soFromCurrent64   = 1;
   soFromEnd64       = 2;

type
   EStream64Error = class( Exception );

   TStream64 = class
      protected
         function  GetPosition: Int64;                 virtual;
         procedure SetPosition( const NewPos: Int64 ); virtual;
         function  GetSize: Int64;                     virtual;
         procedure SetSize( const NewSize: Int64 );    virtual;

      public
         property Position: Int64 read GetPosition write SetPosition;
         property Size    : Int64 read GetSize     write SetSize;

         function Seek ( const Offset: Int64    ): Int64; overload; virtual;
         function Seek ( const Offset: Int64;
                         const Origin: LongWord ): Int64; overload; virtual; abstract;

         function Read ( var   Buffer; const Count: LongWord ): LongWord; virtual; abstract;
         function Write( const Buffer; const Count: LongWord ): LongWord; virtual; abstract;

         procedure Flush; virtual;

         procedure Pad( const StartOffset, Length: Int64;
                        const ClearChar: Char = #0 ); virtual;
   end;

implementation

{ TStream64 }

function TStream64.GetPosition: Int64;
begin
   Result := Seek( 0, soFromCurrent64 );
end;

procedure TStream64.SetPosition( const NewPos: Int64 );
begin
   Seek( NewPos, soFromBeginning64 );
end;

function TStream64.GetSize: Int64;
var  OldPos: Int64;
begin
  OldPos := Seek( 0, soFromCurrent64 );
  Result := Seek( 0, soFromEnd64     );
  Seek( OldPos, soFromBeginning64 );
end;

procedure TStream64.SetSize( const NewSize: Int64 );
begin
   raise EStream64Error.Create( 'TStream64.SetSize not supported!' );
end;

function TStream64.Seek( const Offset: Int64 ): Int64;
begin
   Result := Seek( Offset, soFromBeginning64 );
end;

procedure TStream64.Pad( const StartOffset, Length: Int64;
                         const ClearChar: Char = #0 );
const DataLen = 4096;
var   DataPtr: Pointer;
      LeftLen, CurrLen: Int64;
begin
   if GetSize < StartOffset+Length then SetSize( StartOffset+Length );

   GetMem( DataPtr, DataLen );
   try
      FillChar( DataPtr^, DataLen, ClearChar );

      Seek( StartOffset );
      LeftLen := Length;
      CurrLen := DataLen;

      while LeftLen > 0 do begin
         if LeftLen < CurrLen then CurrLen := LeftLen;
         Write( DataPtr^, CurrLen );
         dec( LeftLen, CurrLen );
      end;

   finally
      FreeMem( DataPtr );
   end;
end;

procedure TStream64.Flush;
begin
   //
end;

end.
