// ============================================================================
// 64 bit file streams with exception support.
// Copyright (c) 2002, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit cFileStream64;

interface

uses SysUtils, cStream64, Windows;

const
   // access modes
   fmOpenRead64       = $0000;
   fmOpenWrite64      = $0001;
   fmOpenReadWrite64  = $0002;

   // share modes
   fmShareCompat64    = $0000;
   fmShareExclusive64 = $0010;
   fmShareDenyWrite64 = $0020;
   fmShareDenyRead64  = $0030;
   fmShareDenyNone64  = $0040;

   // special modes
   fmCreateAlways64   = $F000; // -> use CREATE_ALWAYS
   fmOpenAlways64     = $0F00; // -> use OPEN_ALWAYS instead of OPEN_EXISTING

   // replacement for TFileStream's fmCreate ($FFFF):
   fmCreate64 = fmCreateAlways64 or fmOpenReadWrite64 or fmShareExclusive64;

   // some convenient combinations
   fmReadShared64         = fmOpenRead64   or fmShareDenyNone64;
   fmReadWriteShared64    = fmOpenAlways64 or fmOpenReadWrite64 or fmShareDenyNone64;
   fmReadWriteDenyWrite64 = fmOpenAlways64 or fmOpenReadWrite64 or fmShareDenyWrite64;
   fmReadWriteExclusive64 = fmOpenAlways64 or fmOpenReadWrite64 or fmShareExclusive64;

type
   EFileStream64Error = class( Exception );

   TFileStream64 = class( TStream64 )
      protected
         FFilename: String;
         FHandle  : LongWord;

         procedure SetSize( const NewSize: Int64 ); override;

      public
         class function FileSize( const AFilename: String ): Int64;

         property Handle: LongWord read FHandle;

         function Seek ( const Offset: Int64;
                         const Origin: LongWord ): Int64; override;

         function Read ( var   Buffer; const Count: LongWord ): LongWord; override;
         function Write( const Buffer; const Count: LongWord ): LongWord; override;

         function Lock  ( const Offset, Count: Int64 ): Boolean; overload;
         function Lock                                : Boolean; overload;
         function Unlock( const Offset, Count: Int64 ): Boolean; overload;
         function Unlock                              : Boolean; overload;

         procedure Flush; override;
         procedure SetEndOfFile;

         constructor Create( const AFileName: String;
                             const ALFMode  : Word );

         constructor CreateWinApi(
            const AFilename     : String;
            const AAccessMode   : LongWord = GENERIC_READ or GENERIC_WRITE;
            const AShareMode    : LongWord = FILE_SHARE_READ or FILE_SHARE_WRITE;
            const APSecurityAttr: PSecurityAttributes = nil;
            const ACreationMode : LongWord = OPEN_ALWAYS;
            const AFileAttribute: LongWord = FILE_ATTRIBUTE_NORMAL;
            const ATemplateFile : THandle = 0
         );

         destructor Destroy; override;
   end;


implementation

{$INCLUDE Compiler.inc}

{ TFileStream64 }

class function TFileStream64.FileSize( const AFilename: String ): Int64;
var  Handle: THandle;
     FindData: TWin32FindData;
begin
   Handle := Windows.FindFirstFile( PChar(AFileName), FindData );
   if Handle <> INVALID_HANDLE_VALUE then begin
      if (FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0 then begin
         Int64Rec(Result).Hi := FindData.nFileSizeHigh;
         Int64Rec(Result).Lo := FindData.nFileSizeLow;
      end;
      Windows.FindClose( Handle );
   end else begin
      RaiseLastWin32Error;
   end;
end;

function TFileStream64.Seek( const Offset: Int64; const Origin: LongWord ): Int64;
begin
   Result := Offset;
   Int64Rec(Result).Lo := SetFilePointer(
      FHandle, Int64Rec(Result).Lo, @Int64Rec(Result).Hi, Origin
   );
   if Int64Rec(Result).Lo = $FFFFFFFF then begin
      if GetLastError <> NO_ERROR then RaiseLastWin32Error;
   end;
end;

procedure TFileStream64.SetSize( const NewSize: Int64 );
begin
   Seek( NewSize, soFromBeginning64 );
   Win32Check( Windows.SetEndOfFile( FHandle ) );
end;

procedure TFileStream64.SetEndOfFile;
begin
   Win32Check( Windows.SetEndOfFile( FHandle ) );
end;

function TFileStream64.Read( var Buffer; const Count: LongWord ): LongWord;
begin
   Win32Check( ReadFile( FHandle, Buffer, Count, Result, nil ) );
end;

function TFileStream64.Write( const Buffer; const Count: LongWord ): LongWord;
begin
   Win32Check( WriteFile( FHandle, Buffer, Count, Result, nil) );
end;

function TFileStream64.Lock( const Offset, Count: Int64 ): Boolean;
begin
   Result := LockFile( FHandle,
                       Int64Rec(Offset).Lo, Int64Rec(Offset).Hi,
                       Int64Rec(Count).Lo,  Int64Rec(Count).Hi );
end;

function TFileStream64.Lock: Boolean;
begin
   Result := LockFile( FHandle, 0, 0, $FFFFFFFF, $FFFFFFFF );
end;

function TFileStream64.Unlock( const Offset, Count: Int64 ): Boolean;
begin
   Result := UnlockFile( FHandle,
                         Int64Rec(Offset).Lo, Int64Rec(Offset).Hi,
                         Int64Rec(Count).Lo,  Int64Rec(Count).Hi );
end;

function TFileStream64.Unlock: Boolean;
begin
   Result := UnlockFile( FHandle, 0, 0, $FFFFFFFF, $FFFFFFFF );
end;

procedure TFileStream64.Flush;
begin
   FlushFileBuffers( FHandle );
end;

constructor TFileStream64.CreateWinApi;
begin
   inherited Create;

   FFilename := AFilename;
   FHandle   := CreateFile( PChar( FFileName ), AAccessMode,
                            AShareMode, APSecurityAttr,
                            ACreationMode, AFileAttribute, 0 );
   if FHandle = INVALID_HANDLE_VALUE then RaiseLastWin32Error;
end;

constructor TFileStream64.Create( const AFileName: String;
                                  const ALFMode  : Word );
const AccessMode: array[0..2] of LongWord = (
                     GENERIC_READ, GENERIC_WRITE,
                     GENERIC_READ or GENERIC_WRITE );
      ShareMode : array[0..4] of LongWord = (
                     0, 0, FILE_SHARE_READ, FILE_SHARE_WRITE,
                     FILE_SHARE_READ or FILE_SHARE_WRITE );
var  ACreationMode: LongWord;
begin
   if ALFMode = $FFFF{=fmCreate} then begin
      // If TFileStream64 was used to replace a TFileStream, all fm-modes are
      // compatible except "fmCreate". It has to be replaced by "fmCreate64".
      raise EFileStream64Error.Create(
         'Invalid File Mode $FFFF for TFileStream64.Create -> use fmCreate64!'
      );
   end;

   if ( ALFMode and fmCreateAlways64 ) = fmCreateAlways64 then
      ACreationMode := CREATE_ALWAYS
   else if ( ALFMode and fmOpenAlways64 ) = fmOpenAlways64 then
      ACreationMode := OPEN_ALWAYS
   else
      ACreationMode := OPEN_EXISTING;

   CreateWinApi( AFilename, AccessMode[ ALFMode and 3 ],
                 ShareMode[ (ALFMode and $F0) shr 4 ], nil,
                 ACreationMode, FILE_ATTRIBUTE_NORMAL );
end;

destructor TFileStream64.Destroy;
begin
   if FHandle <> INVALID_HANDLE_VALUE then begin
      FileClose( FHandle );
      FHandle := INVALID_HANDLE_VALUE;
   end;

   inherited Destroy;
end;

end.
