// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit uHamTools;    

interface

{$INCLUDE Compiler.inc}

uses Classes, uType;

procedure CounterEnter;
procedure CounterLeave;
function  CounterInc( var CntVar: Int64; IncVal: Int64 = 1 ): Int64;
function  CounterDec( var CntVar: Int64; DecVal: Int64 = 1 ): Int64;
function  CounterSet( var CntVar: Int64; NewVal: Int64 ): Int64;
function  CounterGet( var CntVar: Int64 ): Int64;
procedure CounterChange;
function  CounterChanged( ResetChanged: Boolean = True ): Boolean;
function  CountersList: String;
procedure CountersReset;

function  NewUniqueID: Integer;

function  TasksStart: Integer;
procedure TasksChange;
function  TasksChanged( ResetChanged: Boolean = True ): Boolean;

function  ClientsStart: Integer;
procedure ClientsChange;
function  ClientsChanged( ResetChanged: Boolean = True ): Boolean;

procedure HamFileEnter;
procedure HamFileLeave;
procedure HamFileAppendList ( const Filename: String; const TS: TStringList );
procedure HamFileLoadList   ( const Filename: String; const TS: TStringList );
procedure HamFileAppendLine ( const Filename, LineText: String );
procedure HamFileRewriteLine( const Filename, LineText: String );

function GlobalListMarker( Action: TGlobalListMarker ): Boolean;
function GlobalGroupDesc( const GroupName: String ): String;
function POP3ServerNameToPath( const s: String ): String;
function IsLocalDomain( const Domain: String ): Boolean;
function GetUniqueMsgFilename( const DestPath, Which: String ): String;
function Logify( const s: String ): String;

implementation

uses SysUtils, IniFiles, uTools, uConst, uConstVar, uVar, SyncObjs,
     cLogFileHamster, cPCRE, cHamster;

var
   CS_COUNTER: TCriticalSection;
   CS_HAMFILE: TCriticalSection;

   CntChanged: Boolean = False;
   TskChanged: Boolean = False;
   CliChanged: Boolean = False;

   UniqueSeed: Integer = 0; // unique ID for task- and client-threads

//--------------------------------------------------------------- Counter -----

procedure CounterEnter;
begin
   CS_COUNTER.Enter;
end;

procedure CounterLeave;
begin
   CS_COUNTER.Leave;
end;

function CounterInc( var CntVar: Int64; IncVal: Int64 = 1 ): Int64;
begin
   CounterEnter;
   try
      inc( CntVar, IncVal );
      Result := CntVar;
      CntChanged := True;
   finally CounterLeave end;
end;

function CounterDec( var CntVar: Int64; DecVal: Int64 = 1 ): Int64;
begin
   CounterEnter;
   try
      dec( CntVar, DecVal );
      Result := CntVar;
      CntChanged := True;
   finally CounterLeave end;
end;

function CounterSet( var CntVar: Int64; NewVal: Int64 ): Int64;
begin
   CounterEnter;
   try
      CntVar := NewVal;
      Result := CntVar;
      CntChanged := True;
   finally CounterLeave end;
end;

function CounterGet( var CntVar: Int64 ): Int64;
begin
   CounterEnter;
   try Result := CntVar finally CounterLeave end;
end;

procedure CounterChange;
begin
   CounterEnter;
   try
      CntChanged := True;
   finally CounterLeave end;
end;

function CounterChanged( ResetChanged: Boolean = True ): Boolean;
begin
   CounterEnter;
   try
      Result := CntChanged;
      if ResetChanged then CntChanged := False;
   finally CounterLeave end;
end;

function CountersList: String;
begin
   Result :=
        'ta=' + inttostr(Hamster.ActiveThreads.CountActiveTasks) + CRLF
      + 'jo=' + inttostr(Hamster.NewsJobs.JobList.Count) + CRLF
      + 'no=' + inttostr(CounterOutboxN) + CRLF
      + 'mo=' + inttostr(CounterOutboxM) + CRLF
      + 'ni=' + inttostr(CounterArtNew) + CRLF
      + 'nl=' + inttostr(CounterArtLoad) + CRLF
      + 'nh=' + inttostr(CounterArtHist) + CRLF
      + 'nk=' + inttostr(CounterArtKill) + CRLF
      + 'mi=' + inttostr(CounterMailNew) + CRLF
      + 'bi=' + inttostr(CounterByteIn) + CRLF
      + 'bo=' + inttostr(CounterByteOut) + CRLF
      + 'by=' + inttostr(CounterByteIn+CounterByteOut) + CRLF;
end;

procedure CountersReset;
begin
   CounterEnter;
   try
      CounterArtNew  := 0;
      CounterArtLoad := 0;
      CounterArtHist := 0;
      CounterArtKill := 0;
      CounterMailNew := 0;
      CounterByteIn  := 0;
      CounterByteOut := 0;
      inc( CounterOutBoxChk );
   finally CounterLeave end;
end;

//--------------------------------------------------------- Tasks/Clients -----

function NewUniqueID: Integer;
begin
   CounterEnter;
   try
      inc( UniqueSeed );
      Result := UniqueSeed;
   finally CounterLeave end;
end;

function TasksStart: Integer;
begin
   CounterEnter;
   try
      Result := NewUniqueID;
      TskChanged := True;
   finally CounterLeave end;
end;

procedure TasksChange;
begin
   CounterEnter;
   try
      TskChanged := True;
   finally CounterLeave end;
end;

function TasksChanged( ResetChanged: Boolean = True ): Boolean;
begin
   CounterEnter;
   try
      Result := TskChanged;
      if ResetChanged then TskChanged := False;
   finally CounterLeave end;
end;

function ClientsStart: Integer;
begin
   CounterEnter;
   try
      Result := NewUniqueID;
      CliChanged := True;
   finally CounterLeave end;
end;

procedure ClientsChange;
begin
   CounterEnter;
   try
      CliChanged := True;
   finally CounterLeave end;
end;

function ClientsChanged( ResetChanged: Boolean = True ): Boolean;
begin
   CounterEnter;
   try
      Result := CliChanged;
      if ResetChanged then CliChanged := False;
   finally CounterLeave end;
end;

//--------------------------------------------------------------- HamFile -----

procedure HamFileEnter;
begin
   CS_HAMFILE.Enter;
end;

procedure HamFileLeave;
begin
   CS_HAMFILE.Leave;
end;

procedure HamFileLoadList( const Filename: String; const TS: TStringList );
begin
   HamFileEnter;
   try
      try
         TS.Clear;
         if FileExists( Filename ) then TS.LoadFromFile( Filename );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileLoadList '+ Filename + ': ' + E.Message );
      end;
   finally HamFileLeave end;
end;

procedure HamFileAppendList( const Filename: String; const TS: TStringList );
var  T : TextFile;
begin
   HamFileEnter;
   try
      try
         AssignFile( T, Filename );
         if FileExists( Filename ) then Append( T ) else Rewrite( T );
         write( T, TS.Text );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileAppendList '+ Filename + ': ' + E.Message );
      end;
   finally HamFileLeave end;
end;

procedure HamFileAppendLine( const Filename, LineText: String );
var  T : TextFile;
begin
   HamFileEnter;
   try
      try
         AssignFile( T, Filename );
         if FileExists( Filename ) then Append(T) else Rewrite(T);
         if IOResult<>0 then exit;
         writeln( T, LineText );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileAppendLine '+ Filename + ': ' + E.Message );
      end;
   finally HamFileLeave end;
end;

procedure HamFileRewriteLine( const Filename, LineText: String );
var  T : TextFile;
begin
   HamFileEnter;
   try
      try
         AssignFile( T, Filename );
         Rewrite( T );
         writeln( T, LineText );
         CloseFile( T );
      except
         on E: Exception do
            Log( LOGID_ERROR, 'FileWriteLine '+ Filename + ': ' + E.Message );
      end;
   finally HamFileLeave end;
end;

//--------------------------------------------------------------- Various -----

var mGlobalListMarker: Integer = -1;

function GlobalListMarker( Action: TGlobalListMarker ): Boolean;
begin
   Result := True;

   CS_MAINTENANCE.Enter;
   try

      if mGlobalListMarker < 0 then begin // init from .ini
         with TIniFile.Create( AppSettings.GetStr(asPathBase) + 'Hamster.ini' ) do try
            if ReadBool( 'Marker', 'GlobalList', True ) then mGlobalListMarker := 1
                                                        else mGlobalListMarker := 0;
         finally Free end;
      end;

      if Action = glTEST then begin
         Result := ( mGlobalListMarker = 1 );
         exit;
      end;

      with TIniFile.Create( AppSettings.GetStr(asPathBase) + 'Hamster.ini' ) do try
         case Action of
            glOPEN: begin
               mGlobalListMarker := 1;
               WriteBool( 'Marker', 'GlobalList', True  );
            end;
            glDONE: begin
               mGlobalListMarker := 0;
               WriteBool( 'Marker', 'GlobalList', False );
            end;
         end;
      finally Free end;
      
   finally CS_MAINTENANCE.Leave end;
end;

function GlobalGroupDesc( const GroupName: String ): String;
var  T   : TextFile;
     g, d: String;
     i   : Integer;
begin
     Result := '';
     HamFileEnter;
     try
        try
           if FileExists( AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS ) then begin
              AssignFile( T, AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS );
              Reset( T );
              while not EOF( T ) do begin
                 readln( T, g );
                 i := PosWhSpace( g );
                 d := '';
                 if i>0 then begin d:=copy(g,i+1,255); g:=copy(g,1,i-1); end;
                 if CompareText( g, GroupName )=0 then begin
                    Result := d;
                    break;
                 end;
              end;
              CloseFile( T );
           end;
        except
           Result := '';
        end;
     finally
        HamFileLeave;
     end;
end;

function POP3ServerNameToPath( const s: String ): String;
begin
   Result := StringReplace( s, '/', '-', [rfReplaceAll] );
end;

function IsLocalDomain( const Domain: String ): Boolean;
var  regex: TPCRE;
     re   : String;
begin
     // if not handled prior, force some domains to always be local-only
     Result := True;
     if Domain='' then exit; // no domain
     if Pos( '.', Domain )=0 then exit; // no valid domain
     if copy(Domain,length(Domain)-7,8)='.invalid' then exit; // .*\.invalid

     // otherwise, the domain is always assumed non-local
     // unless the given 'local domain'-regex matches
     Result := False;
     if Hamster.Config.Settings.GetStr(hsIsLocalDomain) = ''  then exit; // not set

     // finally, the given domain-regex decides
     regex := TPCRE.Create( False, PCRE_CASELESS );
     try
        re := '^(' + Hamster.Config.Settings.GetStr(hsIsLocalDomain) + ')$';
        if regex.Match( PChar(re), PChar(Domain) ) then Result:=True;
     except
        on E:Exception do Log( LOGID_ERROR, Format(
              'IsLocalDomain failed for %s: %s', [Re, E.Message] ) );
     end;
     regex.Free;
end;

function GetUniqueMsgFilename( const DestPath, Which: String ): String;
var  i, LfdNo: Integer;
     SR      : TSearchRec;
     Filename: String;
begin
   CounterEnter;
   try

      Filename := IncludeTrailingBackslash( AppSettings.GetStr(asPathBase) )
                + 'Hamster.ini';

      with TIniFile.Create( Filename ) do try

         LfdNo := ReadInteger( 'Setup', Which + '.number.lastused', 0 );
         inc( LfdNo );
         
         repeat
            Result := DestPath + inttostr(LfdNo) + '.msg';
            i := SysUtils.FindFirst( Result, faAnyFile, SR );
            SysUtils.FindClose( SR );
            if i<>0 then break;
            inc( LfdNo );
            if LfdNo>=99999999 then LfdNo:=1;
         until False;

         WriteInteger( 'Setup', Which + '.number.lastused', LfdNo );

      finally Free end;

   finally CounterLeave end;
end;

function Logify( const s: String ): String;
var  i: Integer;
begin
   Result := s;
   for i := 1 to length(Result) do begin
      if Result[i] < #32 then Result[i] := ' ';
   end;
end;

//-----------------------------------------------------------------------------


initialization
   CS_COUNTER := TCriticalSection.Create;
   CS_HAMFILE := TCriticalSection.Create;

finalization
   if Assigned( CS_COUNTER ) then CS_COUNTER.Free;
   if Assigned( CS_HAMFILE ) then CS_HAMFILE.Free;

end.
