// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cTraps;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, uTools, cArticle, cSyncObjects;

type
   // available match modes for mail traps
   TMailTrapMatchMode = ( mtmContainsWord, mtmContains, mtmEquals,
                          mtmMatchesRegex );

   // available actions for mail traps
   TMailTrapAction    = ( mtaAccept, mtaDelete, mtaScore );

   // single mail trap
   TMailTrapItem = class

      private
         FComment      : String;
         FHeader       : String;
         FMatchMode    : TMailTrapMatchMode;
         FMatchNot     : Boolean;
         FPattern      : String;
         FCaseSensitive: Boolean;
         FAction       : TMailTrapAction;
         FScoreValue   : Integer;
         FDisabled     : Boolean;

      public
         property Comment      : String read FComment write FComment;
         property Header       : String read FHeader  write FHeader;
         property MatchMode    : TMailTrapMatchMode read FMatchMode write FMatchMode;
         property MatchNot     : Boolean read FMatchNot write FMatchNot;
         property Pattern      : String read FPattern write FPattern;
         property CaseSensitive: Boolean read FCaseSensitive write FCaseSensitive;
         property Action       : TMailTrapAction read FAction write FAction;
         property ScoreValue   : Integer read FScoreValue write FScoreValue;
         property Disabled     : Boolean read FDisabled write FDisabled;

         function CheckAndFixTrap: Boolean;

         function AsDefLine: String;
         function AsInfo: String;

         constructor Create( const ADefLine: String );

   end;

   // manages list of mail traps (base class for actual mail trap in HService
   // and for mail trap tester in HControl)
   TMailTrapCustom = class

      protected
         FLock: TReaderWriterLock;
         FList: TList;
         FAcceptScore, FDeleteScore: Integer;

         procedure Clear; virtual;

         procedure NotifyHit( const HeaderValues: TStringList;
                              const MatchIndex: Integer;
                              const TrapItem: TMailTrapItem ); virtual;

      public
         property Lock: TReaderWriterLock read FLock;
         property List: TList read FList;
         
         function  IsActive: Boolean; virtual;
         function  Exists( const AsInfoStr: String ): Boolean; virtual;
         procedure Reload; virtual;
         function  Reject( const MailHeaders: TMess;
                           out Reason: String;
                           out DeleTrap: Boolean;
                           out TrapScore: Integer ): Boolean; virtual;

         constructor Create;
         destructor Destroy; override;
         
   end;


const
   // descriptions of mail trap match modes
   MailTrapMatchModes : array[ TMailTrapMatchMode ] of String =
                        ( 'contains-word', 'contains', 'equals',
                          'matches-regex' );

   // descriptions of mail trap actions
   MailTrapActions: array[ TMailTrapAction ] of String =
                    ( 'ACCEPT', 'DELETE', 'SCORE' );

   // placeholders for (groups of) special headers
   MTSpecialHeader_Contents        = '(contents)';
   MTSpecialHeader_RawContents     = '(raw contents)';
   MTSpecialHeader_AnyHeader       = '(any header)';
   MTSpecialHeader_AnySender       = '(any sender)';
   MTSpecialHeader_AnyRecipient    = '(any recipient)';
   MTSpecialHeader_MessageHeaders  = '(message headers)';
   MTSpecialHeader_MessageBody     = '(message body)';
   MTSpecialHeader_WholeMessage    = '(whole message)';
   MTSpecialHeader_BodySubject     = '(body+subject)';

   // headers covered by '(any sender)'
   MTAnySenderMax    = 8;
   MTAnySenderList   : array[ 0 .. MTAnySenderMax ] of String =
                       ( 'from', 'apparently-from', 'sender', 'reply-to',
                         'x-sender', 'envelope-from', 'x-envelope-from',
                         'return-path', 'received' );

   // headers covered by '(any recipient)'
   MTAnyRecipientMax = 9;
   MTAnyRecipientList: array[ 0 .. MTAnyRecipientMax ] of String =
                       ( 'to', 'apparently-to', 'cc', 'bcc',
                         'envelope-to', 'x-envelope-to',
                         'original-recipient', 'x-resent-for',
                          'x-resent-to', 'resent-to' );

   // separator for trap comments
   MailTrapAsInfoCommentSeparator = '   # ';

   // inline markers for decoded messages
   MailTrapContentsMarker   = ':::';
   MailTrapContentsNextPart = MailTrapContentsMarker + 'nextpart=';
   MailTrapContentsError    = MailTrapContentsMarker + 'error=';


procedure MailTrapGetHeaderValues( const MailHeaders: TMess;
                                   HeaderName: String;
                                   const HeaderValues: TStringList );

function MailTrapGetMatched( const HeaderValues: TStringList;
                             const MailTrapItem: TMailTrapItem ): Integer;


implementation

uses uEncoding, cPCRE, IdMessage, uSHA1;

// ------------------------------------------------------------ (helpers) -----

function MailTrapUnHtml( const html: String ): String;
// Removes all tags in given HTML text. Anchor and image link sources are
// retained for filtering purposes.

   function getPart( h, key: String ): String;
   var  j: Integer;
   begin
      Result := '';
      j := Pos( '>', h );
      if j > 0 then SetLength( h, j-1 );
      j := Pos( key, LowerCase(h) );
      if j > 0 then begin
         h := copy( h, j+5, MaxInt );
         if copy( h, 1, 1 ) = '"' then System.Delete( h, 1, 1 );
         j := Pos( '"', h ); if j > 0 then SetLength( h, j-1 );
         j := Pos( ' ', h ); if j > 0 then SetLength( h, j-1 );
         Result := h;
      end;
   end;

var  i: Integer;
     inTag: Boolean;
     h: String;
begin
   Result := '';
   inTag := False;

   for i := 1 to length(html) do begin

      if inTag then begin

         if html[i] = '>' then inTag := False;

      end else begin

         if html[i] = '<' then begin
            inTag := True;
            if LowerCase( copy( html, i, 3 ) ) = '<a ' then begin
               h := getPart( copy(html,i+3,1024), 'href=' );
               if h <> '' then Result := Result + ' [ ' + h + ' ] ';
            end;
            if LowerCase( copy( html, i, 5 ) ) = '<img ' then begin
               h := getPart( copy(html,i+5,1024), 'src=' );
               if h <> '' then Result := Result + ' [ ' + h + ' ] ';
            end;
         end else begin
            Result := Result + html[i];
         end;
         
      end;

   end;

   Result := StringReplace( Result, '&nbsp;', ' ', [rfReplaceAll] );
   Result := StringReplace( Result, #13#10#13#10#13#10, #13#10#13#10, [rfReplaceAll] );
   Result := StringReplace( Result, #13#10#13#10#13#10, #13#10#13#10, [rfReplaceAll] );
end;

function MailTrapExtractContents( const MailText: TMess; const UnHtml: Boolean ): String;
// Extract the contents text of given message, i. e. all text parts.
// If UnHtml is given as True, HMTL tags are removed from HTML messages.
var  msg: TIdMessage;
     str: TStringStream;
     i: Integer;
     txtPart: TIdText;
     attPart: TIdAttachment;
     unkPart: TIdMessagePart;
     Content: String;
     GotContent: Boolean;

     procedure Add( xType, xContentType, xText: String );
     var  xTemp: String;
     begin
        (*
        // make sure that no other lines start with our marker
        if Pos( MailTrapMarker, xText ) > 0 then begin
           if Pos( MailTrapMarker, xText ) = 1 then xText[1] := ' ';
           repeat
              j := Pos( #10 + MailTrapMarker, xText );
              if j = 0 then j := Pos( #13 + MailTrapMarker, xText );
              if j > 0 then xText[j+1] := ' ';
           until j = 0;
        end;
        *)

        // remove HTML tags
        if UnHtml and ( Pos('text/html', LowerCase(xContentType)) > 0 ) then begin
           xTemp := MailTrapUnHtml( xText );
           if length(xTemp) > 0 then xText := xTemp; // comment only HMTL spam
        end;

        // add introduction line followed by part's text
        Content := Content + #13#10
                 + MailTrapContentsNextPart + xType + ' ' + xContentType + #13#10
                 + #13#10
                 + xText;
        if length( xText ) > 0 then GotContent := True;
     end;

     function DecodingFailedResult( Reason: String ): String;
     begin
          Result := #13#10
                  + MailTrapContentsError + Reason + #13#10
                  + #13#10
                  + MailText.BodyText
     end;

     procedure tryToGetRidOfTheseGodDamnIndyTempFiles;
     var  i: Integer;
     begin
        for i := 0 to msg.MessageParts.Count - 1 do begin
            if msg.MessageParts[i] is TIdAttachment then begin
               attPart := TIdAttachment( msg.MessageParts[i] );
               attPart.DeleteTempFile := true;
            end;
         end;
     end;

     procedure loadMessage;
     begin
        try
           msg.LoadFromStream( str );
           tryToGetRidOfTheseGodDamnIndyTempFiles;
        except
           tryToGetRidOfTheseGodDamnIndyTempFiles;
           raise;
        end;
     end;

begin
   try

      msg := TIdMessage.Create( nil );
      try

         str := TStringStream.Create( StringReplace( MailText.FullText,
                                      #13#10'.'#13#10, #13#10'..'#13#10,
                                      [rfReplaceAll] )
              + #13#10 + '.' + #13#10 );

         try

            Content := '';
            GotContent := False;
            loadMessage;

            // add body part of message
            Add( 'text', msg.ContentType, msg.Body.Text );

            // add MIME parts of message
            for i := 0 to msg.MessageParts.Count - 1 do begin

               if msg.MessageParts[i] is TIdText then begin
                  // text (text/plain, text/html)
                  txtPart := TIdText( msg.MessageParts[i] );
                  Add( 'text', txtPart.ContentType, txtPart.Body.Text );

               end else if msg.MessageParts[i] is TIdAttachment then begin
                  // attachments
                  attPart := TIdAttachment( msg.MessageParts[i] );
                  attPart.DeleteTempFile := true;
                  Add( 'attachment', attPart.ContentType, attPart.FileName + #13#10 );

               end else begin
                  // other parts (presumably never happens)
                  unkPart := TIdMessagePart( msg.MessageParts[i] );
                  Add( 'unknown', unkPart.ContentType, '' );

               end;

            end;

            if GotContent then begin
               Result := Content;
            end else begin
               Result := DecodingFailedResult( 'Decoding failed, using raw body.' );
            end;

         finally str.Free end;

      finally msg.Free end;

   except
      // if anything goes wrong, use original, undecoded message text instead
      on E: Exception do begin
         Result := DecodingFailedResult(
            'Decoding error, using raw body. Error=' + E.Message );
      end;
   end;
end;

var LastGetHash: String = '';
    LastGetText: String = '';

procedure MailTrapGetHeaderValues( const MailHeaders: TMess;
                                   HeaderName: String;
                                   const HeaderValues: TStringList );
// Fills "HeaderValues" with all header values of message "MailHeaders"
// that belong to the given "HeaderName", which is either the name of a
// real header like "Subject" or any of the special ones "(...)".

   procedure AddSubjectValue;
   var  s: String;
   begin
      s := MailHeaders.HeaderValueByNameSL( HDR_NAME_SUBJECT );
      HeaderValues.Insert( 0, s );
      if Pos( '=?', s ) > 0 then begin
         HeaderValues.Insert( 1, DecodeHeaderValue( s ) );
      end;
   end;

var  Index, i: Integer;
     MsgName, MsgValue, Hash, s: String;
     Any, Ok: Boolean;
begin
   Hash := SHA1ofStr( HeaderName + #7 + MailHeaders.FullText );
   if LastGetHash = Hash then begin
      HeaderValues.Text := LastGetText;
      exit;
   end;

   try

      HeaderValues.Clear;

      // make sure, given header is lowercase and has no trailing colon
      HeaderName := LowerCase( HeaderName );
      if copy( HeaderName, length(HeaderName), 1 ) = ':' then begin
         System.Delete( HeaderName, length(HeaderName), 1 );
      end;

      // handle special headers
      if HeaderName = MTSpecialHeader_Contents then begin // "(contents)"
         HeaderValues.Text := MailTrapExtractContents( MailHeaders, True );
         AddSubjectValue;
         exit;
      end;
      if HeaderName = MTSpecialHeader_RawContents then begin // "(raw contents)"
         HeaderValues.Text := MailTrapExtractContents( MailHeaders, False );
         AddSubjectValue;
         exit;
      end;
      if HeaderName = MTSpecialHeader_MessageHeaders then begin // "(message headers)"
         HeaderValues.Text := MailHeaders.HeaderText; // header lines
         exit;
      end;
      if HeaderName = MTSpecialHeader_MessageBody then begin // "(message body)"
         HeaderValues.Text := MailHeaders.BodyText; // body lines
         exit;
      end;
      if HeaderName = MTSpecialHeader_WholeMessage then begin // "(whole message)"
         HeaderValues.Text := MailHeaders.FullText; // header+body lines
         exit;
      end;
      if HeaderName = MTSpecialHeader_BodySubject then begin // "(body+subject)"
         HeaderValues.Text := MailHeaders.BodyText; // body lines
         AddSubjectValue;
         exit;
      end;

      // loop through all headers of message
      Any := ( HeaderName = MTSpecialHeader_AnyHeader ); // optimize "(any header)"
      Index := 0;
      while MailHeaders.NextHeader( Index, MsgName, MsgValue ) do begin

         MsgName := LowerCase( MsgName );
         Ok := Any or ( MsgName = HeaderName ); // "(any header)" or given one?

         if not Ok then begin // check additional special headers

            if HeaderName = MTSpecialHeader_AnySender then begin
               // "(any sender)": check list of common sender headers
               for i := 0 to MTAnySenderMax do begin
                  if MsgName = MTAnySenderList[i] then begin Ok:=True; break; end;
               end;
            end;

            if HeaderName = MTSpecialHeader_AnyRecipient then begin
               // "(any recipient)": check list of common recipient headers
               for i := 0 to MTAnyRecipientMax do begin
                  if MsgName = MTAnyRecipientList[i] then begin Ok:=True; break; end;
               end;
            end;

         end;

         if Ok then begin

            HeaderValues.Add( MsgValue );

            if Pos( '=?', MsgValue ) > 0 then begin
               // if value is encoded, add decoded value as well
               HeaderValues.Add( DecodeHeaderValue( MsgValue ) );
            end;

         end;

      end;

   finally
      // convert 'no-break space' into normal space for filtering
      if Pos( #160, HeaderValues.Text ) > 0 then begin
         s := HeaderValues.Text;
         repeat
            i := Pos( #160, s );
            if i > 0 then s[i] := ' ';
         until i = 0;
         HeaderValues.Text := s;
      end;
      LastGetHash := Hash;
      LastGetText := HeaderValues.Text;
   end;
end;

function MailTrapGetMatched( const HeaderValues: TStringList;
                             const MailTrapItem: TMailTrapItem ): Integer;
// Returns "True" if any of the strings in "HeaderValues" matches the given
// "MailTrapItem".
var  i, ReOpts : Integer;
     Pattern   : String;
     Matched   : Boolean;
     RE: TPCRE;
begin
   // prepare pattern
   Pattern := MailTrapItem.Pattern;

   case MailTrapItem.MatchMode of

      mtmContainsWord:  // line contains pattern at word-boundaries
         Pattern := '\b' + EscRegex( Pattern ) + '\b';

      mtmContains:      // line contains pattern
         Pattern := EscRegex( Pattern );

      mtmEquals:        // pattern equals line (no wildcards supported)
         Pattern := '^' + EscRegex( Pattern ) + '$';

      mtmMatchesRegex:  // line matches regular expression
         {as given};

   end;

   // check given lines
   if MailTrapItem.CaseSensitive then ReOpts := 0
                                 else ReOpts := PCRE_CASELESS;
   RE := TPCRE.Create( True, REOpts );
   try

      RE.Compile( PChar(Pattern) );

      Matched := False;
      Result  := -1;

      for i := 0 to HeaderValues.Count - 1 do begin
         if RE.Exec( PChar(HeaderValues[i]), -1 ) then begin
            Matched := True;
            if Result < 0 then Result := i;
         end;
      end;

      if MailTrapItem.FMatchNot then Matched := not Matched;

      if Matched then begin
         // report first header for a multiheader "not" match
         if Result < 0 then Result := 0;
      end else begin
         Result := -1;
      end;

   finally RE.Free end;
end;



// -------------------------------------------------------- TMailTrapItem -----

function TMailTrapItem.CheckAndFixTrap: Boolean;
// Fix current definition; return "False" if needed values are missing.
begin
   // replace forbidden TAB characters (used as separators in file) by spaces
   FComment := StringReplace( FComment, #9, ' ', [rfReplaceAll] );
   FHeader  := StringReplace( FHeader,  #9, ' ', [rfReplaceAll] );
   FPattern := StringReplace( FPattern, #9, ' ', [rfReplaceAll] );

   // header and pattern have no (useful) defaults
   Result := ( FHeader <> '' ) and ( FPattern <> '' );
end;

function TMailTrapItem.AsDefLine: String;
// Returns the definition line as it is stored in file.
begin
   CheckAndFixTrap;
   Result := FComment
      + #9 + FHeader
      + #9 + inttostr( ord( FMatchMode ) )
      + #9 + FPattern
      + #9 + iif( FCaseSensitive, '1', '0' )
      + #9 + inttostr( ord( FAction ) )
      + #9 + inttostr( FScoreValue )
      + #9 + iif( FDisabled, '1', '0' )
      + #9 + iif( FMatchNot, '1', '0' )
      ;
end;

function TMailTrapItem.AsInfo: String;
// Returns the definition in a human readable format.
begin
   CheckAndFixTrap;
   Result := ''
           + iif( FDisabled, '(disabled) ' )
           + 'if  "' + FHeader + '" '
           + iif( FMatchNot, ' not' )
           + ' ' + MailTrapMatchModes[ FMatchMode ]
           + iif( FCaseSensitive, ' (case-sensitive)' )
           + '  "' + FPattern + '" '
           + ' then  ' + MailTrapActions[ FAction ]
           + iif( FAction=mtaScore,
                  ' ' + iif( FScoreValue>0, '+' ) + inttostr(FScoreValue) )
           + iif( FComment <>'', MailTrapAsInfoCommentSeparator + FComment )
           ;
end;

constructor TMailTrapItem.Create( const ADefLine: String );
// Create new item by given definition line (i. e. in file format).
var  TS: TStringList;
begin
   inherited Create;

   TS := TStringList.Create;
   try
      // split TAB separated line
      ArgsSplitChar( ADefLine, TS, #9, 10 );

      FComment       := TS[0];
      FHeader        := TS[1];
      FMatchMode     := TMailTrapMatchMode( strtointdef( TS[2], 0 ) );
      FPattern       := TS[3];
      FCaseSensitive := ( TS[4] = '1' );
      FAction        := TMailTrapAction( strtointdef( TS[5], 0 ) );
      FScoreValue    := strtointdef( TS[6], 0 );
      FDisabled      := ( TS[7] = '1' );
      FMatchNot      := ( TS[8] = '1' );

      CheckAndFixTrap;

   finally TS.Free end;
end;


// ------------------------------------------------------ TMailTrapCustom -----

constructor TMailTrapCustom.Create;
begin
   inherited Create;
   FLock := TReaderWriterLock.Create;
   FList := TList.Create;
   FAcceptScore := +10;
   FDeleteScore := -10;
   Reload;
end;

destructor TMailTrapCustom.Destroy;
begin
   FLock.BeginWrite;
   if Assigned( FList ) then begin
      Clear;
      FreeAndNil( FList );
   end;
   FLock.Free;
   inherited Destroy;
end;

procedure TMailTrapCustom.Clear;
// Clear and free all traps.
var  i: Integer;
begin
   for i := 0 to FList.Count - 1 do TMailTrapItem( FList[i] ).Free;
   FList.Clear;
end;

function TMailTrapCustom.IsActive: Boolean;
// Returns "True" if at least one item is available.
begin
   FLock.BeginRead;
   try
      Result := ( FList.Count > 0 );
   finally FLock.EndRead end;
end;

procedure TMailTrapCustom.Reload;
// Intended to be used to reload a modified file.
begin
   //
end;

function TMailTrapCustom.Exists( const AsInfoStr: String ): Boolean;
// Returns "True" if a trap with an .AsInfo of "AsInfoStr" still exists.
var  i: Integer;
begin
   FLock.BeginRead;
   try

      for i := 0 to FList.Count - 1 do begin
         if TMailTrapItem( FList[i] ).AsInfo = AsInfoStr then begin
            Result := True;
            exit;
         end;
      end;

      Result := False;

   finally FLock.EndRead end;
end;

procedure TMailTrapCustom.NotifyHit( const HeaderValues: TStringList;
                                     const MatchIndex: Integer;
                                     const TrapItem: TMailTrapItem );
// Called whenever a trap has matched. "TrapItem" is the trap that matched,
// "HeaderValues[MatchIndex]" is the matching line of the message.
begin
   //
end;

function TMailTrapCustom.Reject( const MailHeaders: TMess;
                                 out   Reason     : String;
                                 out   DeleTrap   : Boolean;
                                 out   TrapScore  : Integer ): Boolean;
// Tests all traps for the message given in "MailHeaders" and returns "True"
// if the final result is to reject (i. e. DELETE) it.
// Both "Reason" (list of matching traps) and "ScoreSum" (final score value)
// only contain the traps up to the point, where the final decision was made
// to either accept or delete the message.
// NOTE: TMailTrapItem.Disabled is ignored here. It is assumed, that unwanted
// traps were not added to the list (background: HControl's tester also tests
// disabled traps).
var  iTrap, iMatch: Integer;
     Item: TMailTrapItem;
     ScoreUsed: Boolean;
     HeaderValues: TStringList;
     LastHeader: String;
begin
   HeaderValues := TStringList.Create;
   FLock.BeginRead;
   try

      Result    := False;
      Reason    := '';
      DeleTrap  := False;
      TrapScore := 0;
      if FList.Count = 0 then exit;

      ScoreUsed  := False;
      LastHeader := #0;

      // loop through all traps until we have a final decision
      for iTrap := 0 to FList.Count - 1 do begin

         Item := TMailTrapItem( FList[iTrap] );

         // extract header values to test
         if Item.Header <> LastHeader then begin
            LastHeader := Item.Header;
            MailTrapGetHeaderValues( MailHeaders, Item.Header, HeaderValues );
         end;

         if HeaderValues.Count > 0 then begin

            // check trap
            iMatch := MailTrapGetMatched( HeaderValues, Item );
            if iMatch >= 0 then begin

               NotifyHit( HeaderValues, iMatch, Item );

               // we have a match, so take appropriate action
               case Item.Action of

                  mtaAccept: begin // "ACCEPT immediately"
                     Reason := Item.AsInfo;
                     Result := False; // accept
                     exit;
                  end;

                  mtaDelete: begin // "DELETE immediately"
                     Reason   := Item.AsInfo;
                     DeleTrap := True;
                     Result   := True; // delete
                     exit;
                  end;

                  mtaScore: begin // "change SCORE"
                     ScoreUsed := True;
                     inc( TrapScore, Item.ScoreValue );
                     Reason := iif( Reason <> '', Reason + #9 ) + Item.AsInfo;

                     // check, if current score is above given accept-limit
                     if TrapScore >= FAcceptScore then begin
                        Result := False; // accept
                        exit;
                     end;
                  end;

               end; // case Item.Action ...

            end; // if MailTrapGetMatched ...

         end; // if HeaderValues.Count > 0 ...

      end; // for iTrap ...

      // check, if final score is below given delete-limit
      if ScoreUsed and (TrapScore < 0) and (TrapScore < FDeleteScore) then begin
         Result := True; // delete
         exit;
      end;

   finally
      FLock.EndRead;
      HeaderValues.Free;
   end;
end;

end.
