// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cSettingsDef; // settings definition

interface

uses SysUtils, Classes;

type
   ESettingsDefError = class( Exception );

   TSettingDef = class
      private
         FID     : Integer;
         FSection: String;
         FKeyword: String;
         FDefault: String;

      public
         property ID     : Integer read FID;
         property Section: String  read FSection;
         property Keyword: String  read FKeyword;
         property Default: String  read FDefault;

         constructor Create( const AID: Integer;
                             const ASection, AKeyword, ADefault: String );
   end;

   TSettingsDef = class    
      private
         FList: TList;
         FMinID, FMaxID: Integer;

         function GetDef( ID: Integer ): TSettingDef;

      public
         property MinID: Integer read FMinID;
         property MaxID: Integer read FMaxID;
         property Def[ ID: Integer ]: TSettingDef read GetDef;

         procedure Add( const ID: Integer;
                        const Section, Keyword, Default: String );
         function IDExists( const ID: Integer ): Boolean;

         constructor Create;
         destructor Destroy; override;
   end;


implementation

{ TSettingDef }

constructor TSettingDef.Create( const AID: Integer;
                                const ASection, AKeyword, ADefault: String);
begin
   inherited Create;

   FID      := AID;
   FSection := ASection;
   FKeyword := AKeyword;
   FDefault := ADefault;
end;


{ TSettingsDef }

constructor TSettingsDef.Create;
begin
   inherited Create;
   FMinID := -1;
   FMaxID := -1;
   FList := TList.Create;
end;

destructor TSettingsDef.Destroy;
var  ID: Integer;
begin
   for ID := FMinID to FMaxID do begin
      if IDExists( ID ) then GetDef( ID ).Free;
   end;
   FList.Free;
   inherited Destroy;
end;

function TSettingsDef.IDExists( const ID: Integer ): Boolean;
begin
   Result := ( ID >= FMinID ) and ( ID <= FMaxID ) and Assigned( FList[ ID ] );
end;

procedure TSettingsDef.Add( const ID: Integer;
                            const Section, Keyword, Default: String );
begin
   while ID >= FList.Count do FList.Add( nil );

   if IDExists( ID ) then
      raise ESettingsDefError.CreateFmt( 'ID %d is not unique!', [ID] );

   FList[ ID ] := TSettingDef.Create( ID, Section, Keyword, Default );

   if ( FMinID = -1 ) or ( (FMinID>-1) and (ID<FMinID) ) then FMinID := ID;
   if ( FMaxID = -1 ) or ( (FMaxID>-1) and (ID>FMaxID) ) then FMaxID := ID;
end;

function TSettingsDef.GetDef( ID: Integer ): TSettingDef;
begin
   if not IDExists( ID ) then
      raise ESettingsDefError.CreateFmt( 'ID %d is undefined!', [ID] );
   Result := FList.Items[ ID ];
end;

end.
