// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cSchedulerCommon;

interface

uses SysUtils, Classes, uDateTime, SyncObjs;

type
   TSchedulerEntryCustom = class

      protected
         FActionKey: String;
         FFromTime : TDateTime;
         FTilTime  : TDateTime;
         FWeekDays : String;
         FMinutes  : Integer;

         FRunOnce  : Boolean;
         FRunCount : Integer;

         FLastDT   : TDateTime;
         FNextDT   : TDateTime;

         function IsWeekdaySelected ( const dt: TDateTime ): Boolean; virtual;
         function IsDateTimeSelected( const dt: TDateTime ): Boolean; virtual;
         function CalculateNextAfter( const dt: TDateTime ): TDateTime; virtual;

      public
         property ActionKey: String read FActionKey;
         property FromTime : TDateTime read FFromTime;
         property TilTime  : TDateTime read FTilTime;
         property Weekdays : String    read FWeekdays;
         property Minutes  : Integer   read FMinutes;
         property RunCount : Integer   read FRunCount;
         property LastDT   : TDateTime read FLastDT;
         property NextDT   : TDateTime read FNextDT;

         function ToString: String;

         procedure Restart; virtual;
         function Trigger: Boolean; virtual;

         constructor Create( const AActionKey: String;
                             const AFromTime, ATilTime: String;
                             const AWeekDays: String;
                             const AMinutes: Integer );

   end;

   TSchedulerCustom = class

      protected
         FList: TThreadList;

      public
         procedure Clear; virtual;
         procedure Add( Item: TSchedulerEntryCustom ); virtual;
         procedure Remove( Item: TSchedulerEntryCustom ); virtual;
         function  Check: TSchedulerEntryCustom; virtual;

         constructor Create;
         destructor Destroy; override;

   end;

   TGlobalSchedulerEntry = class( TSchedulerEntryCustom )

      public
         function IsUserTask: Boolean;
         function GetDefLine: String;

         constructor Create( const ADefLine: String );

   end;

   TGlobalSchedulerCustom = class( TSchedulerCustom )

      public
         function  GetDefText: String; virtual;
         procedure SetDefText( const NewDefText: String ); virtual;

   end;


implementation

uses uConst, uTools, cUserTasks;

// ------------------------------------------------ TSchedulerEntryCustom -----

function TSchedulerEntryCustom.IsWeekdaySelected( const dt: TDateTime ): Boolean;
var  w: Integer;
begin
   Result := False;

   w := DayOfWeek( dt ) - 1;
   if w = 0 then w := 7;
   if (w <= length(FWeekDays)) and (FWeekDays[w] = '0') then exit;

   Result := True;
end;

function TSchedulerEntryCustom.IsDateTimeSelected( const dt: TDateTime ): Boolean;
var  t: TDateTime;
begin
   Result := False;

   if not IsWeekdaySelected( dt ) then exit;
   t := TimeOfDateTime( dt );
   if (t < FFromTime) or (t > FTilTime) then exit;

   Result := True;
end;

function TSchedulerEntryCustom.CalculateNextAfter( const dt: TDateTime ): TDateTime;
var  dtDate, dtTime: TDateTime;
     n, minutes: Integer;
begin
   if (FRunOnce and (FRunCount>0)) or (FWeekDays = '0000000') then begin
      Result := EncodeDate( 2999, 12, 31 );
      exit;
   end;

   DateTimeSplit( dt, dtDate, dtTime );
   n := 0;

   if FMinutes = 0 then minutes := 1 else minutes := FMinutes;

   repeat

      dtTime := DateTimeAdd( dtTime, stepMinutes, minutes );

      if dtTime > FTilTime then begin
         if dtTime < 1.0 then begin
            dtDate := DateTimeAdd( dtDate, stepDays, 1 );
         end else begin
            dtDate := DateTimeAdd( dtDate, stepDays, Trunc(dtTime) );
         end;
         dtTime := FFromTime;
      end;

      inc( n );
      if n > 7 * 24 * 60 * 60 then begin
         Result := EncodeDate( 2999, 12, 31 );
         exit; // assuming endless loop by misconfiguration
      end;

      Result := DateTimeCombine( dtDate, dtTime );

   until (Result >= Now) and IsWeekdaySelected( Result );
end;

procedure TSchedulerEntryCustom.Restart;
var  dt: TDateTime;
     i: Integer;
begin
   FRunCount  := 0;
   FRunOnce   := ( FMinutes = 0 ); // true = exotic script option

   // make assumption for last execution (last selected weekday at start time)
   dt := DateOfDateTime( Now );
   for i := 1 to 7 do begin
      dt := DateTimeAdd( dt, stepDays, -1 );
      if IsWeekDaySelected( dt ) then break;
   end;
   FLastDT := DateTimeCombine( dt, FFromTime );

   // calculate first starting point based on assumption
   FNextDT := CalculateNextAfter( FLastDT );
end;

function TSchedulerEntryCustom.Trigger: Boolean;
begin
   if Now < FNextDT then begin
      Result := False;
   end else begin
      FLastDT  := FNextDT;
      FNextDT  := CalculateNextAfter( FLastDT );
      inc( FRunCount );
      Result   := True;
   end;
end;

function TSchedulerEntryCustom.ToString: String;
const wd = 'MonTueWedThuFriSatSun';
var  s, w: String;
     i: Integer;
     d: TDateTime;
begin
   w := FWeekdays;
   if      w = '1111111' then s := 'Mon-Sun'
   else if w = '1111110' then s := 'Mon-Sat'
   else if w = '1111100' then s := 'Mon-Fri'
   else if w = '0000011' then s := 'Sat-Sun'
   else if w = '0000000' then s := '(never)'
   else begin
      for i := 1 to 7 do begin
         if w[i] <> '0' then begin
            if length(s) > 0 then s := s + ', ';
            s := s + copy( wd, i*3-2, 3 );
         end;
      end;
   end;

   s := s + ', ' + TimeToTimeString( FFromTime, 5 );
   s := s + '-' + TimeToTimeString( FTilTime,  5 );

   s := s + ', ' + FActionKey;

   s := s + ', ' + inttostr(FMinutes) + ' min.';

   s := s + ', next at ' + TimeToTimeString( NextDT, 5 ) + ' (';
   d := DateOfDateTime( NextDT );
   if      d = Date   then s := s + 'today'
   else if d = Date+1 then s := s + 'tomorrow'
   else                    s := s + DateToStr( NextDT );
   s := s + ')';

   Result := s;
end;

constructor TSchedulerEntryCustom.Create( const AActionKey: String;
                                          const AFromTime : String;
                                          const ATilTime  : String;
                                          const AWeekDays : String;
                                          const AMinutes  : Integer );
begin
   inherited Create;

   FActionKey := AActionKey;
   FFromTime  := TimeStringToTime( AFromTime );
   FTilTime   := TimeStringToTime( ATilTime  );
   FWeekDays  := copy( AWeekDays, 1, 7 );
   while length( FWeekdays ) < 7 do FWeekdays := FWeekdays + '1';
   FMinutes   := AMinutes;
   FRunCount  := 0;
   FRunOnce   := ( FMinutes = 0 ); // true = exotic script option

   Restart;
end;


// ----------------------------------------------------- TSchedulerCustom -----

procedure TSchedulerCustom.Add( Item: TSchedulerEntryCustom );
begin
   FList.Add( Item );
end;

procedure TSchedulerCustom.Remove( Item: TSchedulerEntryCustom );
begin
   FList.Remove( Item );
end;

procedure TSchedulerCustom.Clear;
begin
   with FList.LockList do try
      while Count > 0 do begin
         TObject( Items[ Count-1 ] ).Free;
         Delete( Count-1 );
      end;
   finally FList.UnlockList end;
end;

function TSchedulerCustom.Check: TSchedulerEntryCustom;
var  i: Integer;
begin
   Result := nil;
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         if TSchedulerEntryCustom( Items[i] ).Trigger then begin
            Result := Items[i];
            break;
         end;
      end;
   finally FList.UnlockList end;
end;

constructor TSchedulerCustom.Create;
begin
   inherited Create;
   FList := TThreadList.Create;
end;

destructor TSchedulerCustom.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;


// ------------------------------------------------ TGlobalSchedulerEntry -----

function TGlobalSchedulerEntry.IsUserTask: Boolean;
var  s: String;
     i: Integer;
begin
   i := length( USERTASK_EXTENSION );
   s := TrimWhSpace( FActionKey );
   s := copy( s, length(s)-i+1, MaxInt );
   Result := ( CompareText( s, USERTASK_EXTENSION ) = 0 );
end;

function TGlobalSchedulerEntry.GetDefLine: String;
begin
   Result := DQuoteStr( FActionKey )
      + #9 + DQuoteStr( TimeToTimeString( FFromTime, 5 ) )
      + #9 + DQuoteStr( TimeToTimeString( FTilTime, 5 ) )
      + #9 + DQuoteStr( FWeekDays )
      + #9 + inttostr( FMinutes );
end;

constructor TGlobalSchedulerEntry.Create( const ADefLine: String );
var  Args: TStringList;
     AActionKey, AFromTime, ATilTime, AWeekDays: String;
     AMinutes: Integer;
begin
   Args := TStringList.Create;
   try

      ArgsWhSpaceDQuoted( ADefLine, Args, 9 );
      AActionKey := Args[0];
      AFromTime  := Args[1];
      ATilTime   := Args[2];
      AWeekDays  := Args[3];
      AMinutes   := strtointdef( Args[4], 60 );

      inherited Create( AActionKey, AFromTime, ATilTime, AWeekDays, AMinutes );

   finally Args.Free end;
end;


// ----------------------------------------------- TGlobalSchedulerCustom -----

function TGlobalSchedulerCustom.GetDefText: String;
var  i: Integer;
begin
   Result := '';
   with FList.LockList do try
      for i := 0 to Count - 1 do begin
         Result := Result + TGlobalSchedulerEntry( Items[i] ).GetDefLine + CRLF;
      end;
   finally FList.UnlockList end;
end;

procedure TGlobalSchedulerCustom.SetDefText( const NewDefText: String );
var  SL: TStringList;
     i: Integer;
begin
   with FList.LockList do try

      Clear;

      SL := TStringList.Create;
      try
         SL.Text := NewDefText;
         for i := 0 to SL.Count - 1 do begin
            if SL[i] <> '' then begin
               Add( TGlobalSchedulerEntry.Create( SL[i] ) );
            end;
         end;
      finally SL.Free end;

   finally FList.UnlockList end;
end;

end.
