// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cLiveQueue; 

interface

uses SysUtils, Classes, SyncObjs, cLiveMsg;

type
   TLiveQueue = class
      protected
         FList     : TThreadList;
         FQueueSize: Integer;
         FOverflow : Boolean;
         FEvent    : TEvent;

         function GetResetOverflow: Boolean; virtual;

      public
         property QueueOverflowed: Boolean read GetResetOverflow;

         function WaitFor( TimeoutMS: Cardinal ): Boolean; virtual;

         procedure Add( Item: TLiveMsg ); virtual;
         function  Get( var Item: TLiveMsg ): Boolean; virtual;

         constructor Create( const ASize: Integer );
         destructor Destroy; override;
   end;

implementation

// ----------------------------------------------------------- TLiveQueue -----

constructor TLiveQueue.Create( const ASize: Integer );
begin
   inherited Create;
   
   FList      := TThreadList.Create;
   FQueueSize := ASize;
   FOverflow  := False;
   FEvent     := TEvent.Create( nil, True, False, '' );
end;

destructor TLiveQueue.Destroy;
begin
   with FList.LockList do try
      while Count > 0 do begin
         TLiveMsg( Items[0] ).Free;
         Delete( 0 );
      end;
   finally
      FList.UnlockList;
   end;
   FreeAndNil( FList  );
   
   FreeAndNil( FEvent );

   inherited Destroy;
end;

function TLiveQueue.GetResetOverflow: Boolean;
begin
   Result := FOverflow;
   FOverflow := False;
end;

procedure TLiveQueue.Add( Item: TLiveMsg );
begin
   with FList.LockList do try
      if Count >= FQueueSize then begin
         // queue overflow, remove oldest item
         TLiveMsg( Items[0] ).Free;
         Delete( 0 );
         FOverflow := True;
      end;

      FList.Add( Item );
      FEvent.SetEvent;

   finally
      FList.UnlockList;
   end;
end;

function TLiveQueue.Get( var Item: TLiveMsg ): Boolean;
begin
   Result := False;

   with FList.LockList do try
      if Count > 0 then begin
         Item := Items[0];
         Delete( 0 ); // Note: caller has to free item
         if Count = 0 then FEvent.ResetEvent;
         Result := True;
      end;

   finally
      FList.UnlockList;
   end;
end;

function TLiveQueue.WaitFor( TimeoutMS: Cardinal ): Boolean;
begin
   Result := ( FEvent.WaitFor( TimeoutMS ) = wrSignaled );
end;

end.
