// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cLiveMsg; 

interface

type
   TLiveMsg = class
      protected
         FMsgType   : Word;
         FMsgData   : String;
         FSessionKey: String;

         function  GetTransferStr: String; virtual;
         procedure SetTransferStr( const s: String ); virtual;

      public
         class function LiveTransferEncode( const s: String ): String;
         class function LiveTransferDecode( const s: String;
                                            out Failed: Boolean ): String;

         property MsgType: Word   read FMsgType write FMsgType;
         property MsgData: String read FMsgData write FMsgData;
         property SessionKey: String read FSessionKey write FSessionKey;

         property TransferStr: String read GetTransferStr write SetTransferStr;

         constructor Create( const AMsgType: Word;
                             const AMsgData: String ); overload;
         constructor Create( const ATransferStr: String ); overload;
         constructor Create( const ATransferStr: String;
                             const ASessionKey: String ); overload;
   end;

implementation

uses SysUtils, uBlowfish, uConst;

const
   IVecSeed1 = $72a7429c;
   IVecSeed2 = $b923fe31;

class function TLiveMsg.LiveTransferEncode( const s: String ): String;
// Encodes characters NUL, TAB, LF and CR with a '^' followed by character's
// value plus 64. The quote character itself ('^') is doubled ('^^'), other
// characters are left unchanged.
var  i, o: Integer;
begin
   SetLength( Result, length(s) );

   o := 0;
   i := 0;

   while i < length(s) do begin

      inc( i );
      inc( o );

      case s[i] of
         #0{NUL}, #9{TAB}, #10{LF}, #13{CR}, '^':
            begin
               Result[o] := '^';
               inc( o );
               SetLength( Result, length(Result)+1 );
               if s[i] = '^' then Result[o] := '^'
                             else Result[o] := chr( ord(s[i]) + 64 );
            end;
         else
            Result[o] := s[i];
      end;

   end;
end;

class function TLiveMsg.LiveTransferDecode( const s: String; out Failed: Boolean ): String;
// Decodes a "LiveDataEncoded" string. Invalid encoding is replaced by '?'
// and returns "Failed" as True.
var  i, o: Integer;
begin
   SetLength( Result, length(s) );
   Failed := False;

   o := 0;
   i := 0;

   while i < length(s) do begin

      inc( i );
      inc( o );

      if s[i] = '^' then begin
         inc( i );
         if i > length(s) then begin
            Failed := True;
            Result[o] := '?'; // invalid Encoding ('=' at end of string)
         end else begin
            SetLength( Result, length(Result)-1 );
            if ord(s[i]) < 64 then begin
               Failed := True;
               Result[o] := '?'; // invalid Encoding (invalid char after '=')
            end else begin
               if s[i] = '^' then Result[o] := '^'
                             else Result[o] := chr( ord(s[i]) - 64 );
            end;
         end;
      end else begin
         Result[o] := s[i];
      end;

   end;
end;


{ TLiveMsg }

constructor TLiveMsg.Create( const AMsgType: Word;
                             const AMsgData: String );
begin
   inherited Create;
   FMsgType := AMsgType;
   FMsgData := AMsgData;
end;

constructor TLiveMsg.Create( const ATransferStr: String );
begin
   inherited Create;
   SetTransferStr( ATransferStr );
end;

constructor TLiveMsg.Create( const ATransferStr: String;
                             const ASessionKey: String );
begin
   inherited Create;
   FSessionKey := ASessionKey;
   SetTransferStr( ATransferStr );
end;

function TLiveMsg.GetTransferStr: String;
var  Data: String;
     i: Integer;
begin
   try
      Data := inttohex( FMsgType, 4 ) + ' ' + FMsgData;

      if length( FSessionKey ) > 0 then begin
         Data := StringOfChar( #0, bf_BLOCKSIZE ) + Data;
         for i:=1 to bf_BLOCKSIZE do Data[i] := chr( Random(256) );
         Data := HamBlowfishEncipherCBC( SessionKey, Data, IVecSeed1, IVecSeed2 );
      end;

      Result := LiveTransferEncode( Data );

   except
      Result := '';
   end;
end;

procedure TLiveMsg.SetTransferStr( const s: String );
var  Data  : String;
     Failed: Boolean;
begin
   try
      FMsgType := LMREQ_INVALID;
      FMsgData := '';

      Data := LiveTransferDecode( s, Failed );
      if Failed then exit;

      if length( FSessionKey ) > 0 then begin
         Data := HamBlowfishDecipherCBC( SessionKey, Data, IVecSeed1, IVecSeed2 );
         System.Delete( Data, 1, bf_BLOCKSIZE );
      end;

      if length( Data ) >= 4 then begin
         FMsgType := strtointdef( '$' + copy( Data, 1, 4 ), LMREQ_INVALID );
         FMsgData := copy( Data, 6, MaxInt );
      end;

   except
      FMsgType := LMREQ_INVALID;
      FMsgData := '';
   end;
end;

end.
