// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cArticle; // Class to handle a single message.

// ----------------------------------------------------------------------------
// Contains a class to handle a single message, which, for examples, allows
// direct access to specific header-lines.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses
  Classes;

const
   HDR_NAME_APPROVED                  = 'Approved';
   HDR_NAME_CONTENT_TRANSFER_ENCODING = 'Content-Transfer-Encoding';
   HDR_NAME_CONTENT_TYPE              = 'Content-Type';
   HDR_NAME_CONTROL                   = 'Control';
   HDR_NAME_DATE                      = 'Date';
   HDR_NAME_DISTRIBUTION              = 'Distribution';
   HDR_NAME_FOLLOWUP_TO               = 'Followup-To';
   HDR_NAME_FROM                      = 'From';
   HDR_NAME_IN_REPLY_TO               = 'In-Reply-To';
   HDR_NAME_LINES                     = 'Lines';
   HDR_NAME_MESSAGE_ID                = 'Message-ID';
   HDR_NAME_MIME_VERSION              = 'MIME-Version';
   HDR_NAME_NEWSGROUPS                = 'Newsgroups';
   HDR_NAME_PATH                      = 'Path';
   HDR_NAME_RECEIVED                  = 'Received';
   HDR_NAME_REFERENCES                = 'References';
   HDR_NAME_RETURN_PATH               = 'Return-Path';
   HDR_NAME_SUBJECT                   = 'Subject';
   HDR_NAME_TO                        = 'To';
   HDR_NAME_USER_AGENT                = 'User-Agent';
   HDR_NAME_XREF                      = 'Xref';
   HDR_NAME_X_HAMSTER_TO              = 'X-Hamster-To';
   HDR_NAME_X_HAMSTER_INFO            = 'X-Hamster-Info';
   HDR_NAME_X_HAMSTER_FEED            = 'X-Hamster-Feed';
   HDR_NAME_X_NEWSGROUPS              = 'X-Newsgroups';
   HDR_NAME_X_OLD_XREF                = 'X-Old-Xref';
   HDR_NAME_X_TRACE                   = 'X-Trace';

type
   TMess = class
      private
         fHeaderLines: TStringList;
         fBodyText   : String;

         function  GetFullText: String;
         procedure SetFullText( const sText: String );
         function  GetHeaderText: String;
         procedure SetHeaderText( const sNewHdr: String );
         function  GetExportText: String;
         function  GetHeaderCountSL: Integer;
         function  GetHeaderLineSL( Index: Integer ): String;
         procedure SetHeaderLineSL( Index: Integer; const sNewValue: String );
         function  GetDateTimeGMT: TDateTime;
         function  GetBodyLines: Integer;
         function  GetMailSummary: String;

      public
         class function HeaderValueFromText( const ArtText, HdrName: String ): String;
         class function HasHeaderEnd( const ArtText: String ): Boolean;

         function IsValid: Boolean;
         property FullText: String read GetFullText write SetFullText;
         property BodyText: String read fBodyText write fBodyText;
         property BodyLines: Integer read GetBodyLines;
         property HeaderText: String read GetHeaderText write SetHeaderText;
         property ExportText: String read GetExportText;
         property MailSummary: String read GetMailSummary;

         property DateTimeGMT: TDateTime read GetDateTimeGMT;

         property HeaderLineCountSL: Integer read GetHeaderCountSL;
         property HeaderLineSL[ Index: Integer ]: String read GetHeaderLineSL write SetHeaderLineSL;
         function NextHeader( var Index: Integer; out Name, Value: String ): Boolean;

         function  AddHeaderSL( const sHdrName, sHdrValue: String ): Integer;
         function  AddHeaderML( const sHdrName, sHdrValue: String ): Integer;
         procedure InsertHeaderSL( Index: Integer; const sHdrName, sHdrValue: String );

         function  SetHeaderSL( sHdrName: String; const sHdrValue: String; sHdrNameSaveOld: String ): Integer;
         procedure DeleteHeaderML( Index: Integer );
         procedure DelHeaderML( const sHdrName: String );
         function  IndexOfHeader( const sHdrName: String; iAfterPos: Integer = -1 ): Integer;
         function  HeaderExists( const sHdrName: String ): Boolean;
         function  HeaderValueByIndexSL( Index: Integer ): String;
         function  HeaderValueByNameSL( const sHdrName: String ): String;
         function  HeaderValueByIndexML( Index: Integer ): String;
         function  HeaderValueByNameML( const sHdrName: String ): String;

         procedure AddBodyLine( const line: String );
         procedure AddBodyText( const txt: String );

         function  GetOurXHeader( const XNam: String ) : String;
         function  GetReceivedDT: TDateTime;
         function  GetHeaderLine( const HdrNam: String; var LineNo: Integer ) : String;

         function  LoadFromFile( const Filename: String ): Integer;
         function  SaveToFile( const Filename: String ): Integer;

         procedure Clear;

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses SysUtils, uConst, uTools, uEncoding, uDateTime;

// ---------------------------------------------------------------- TMess -----

class function TMess.HeaderValueFromText( const ArtText, HdrName: String ): String;
var  TempMess: TMess;
     i: Integer;
begin
   TempMess := TMess.Create;
   try
      i := Pos( CRLF+CRLF, ArtText );
      if i > 0 then TempMess.HeaderText := copy( ArtText, 1, i+1 );
      Result := TempMess.HeaderValueByNameSL( HdrName );
   finally TempMess.Free end;
end;

class function TMess.HasHeaderEnd( const ArtText: String ): Boolean;
begin
   Result := ( Pos( CRLF+CRLF, ArtText ) > 0 );
end;

function TMess.IsValid: Boolean;
begin
   Result := False;
   if fHeaderLines.Count = 0 then exit;
   Result := True;
end;

function TMess.GetFullText: String;
var  i: Integer;
begin
     Result := '';
     for i:=0 to fHeaderLines.Count-1 do begin
        Result := Result + fHeaderLines[i] + CRLF;
     end;
     Result := Result + CRLF;
     Result := Result + fBodyText;
end;

procedure TMess.SetFullText( const sText: String );
var  i: Integer;
begin
     fHeaderLines.Clear;
     fBodyText := '';

     i := Pos( CRLF+CRLF, sText );
     if i=0 then begin
        fHeaderLines.Text := sText;
     end else begin
        fHeaderLines.Text := copy( sText, 1, i+1 );
        fBodyText := sText;
        System.Delete( fBodyText, 1, i+3 );
     end;
end;

function TMess.GetHeaderText: String;
var  i: Integer;
begin
     Result := '';
     for i:=0 to fHeaderLines.Count-1 do begin
        Result := Result + fHeaderLines[i] + CRLF;
     end;
end;

procedure TMess.SetHeaderText( const sNewHdr: String );
begin
     fHeaderLines.Clear;
     fHeaderLines.Text := sNewHdr;
end;

function TMess.GetHeaderCountSL: Integer;
begin
     Result := fHeaderLines.Count;
end;

function TMess.GetHeaderLineSL( Index: Integer ): String;
begin
     Result := fHeaderLines[ Index ];
end;

procedure TMess.SetHeaderLineSL( Index: Integer; const sNewValue: String );
begin
     fHeaderLines[ Index ] := sNewValue;
end;

function TMess.AddHeaderSL( const sHdrName, sHdrValue: String ): Integer;
begin
     if sHdrName[ length(sHdrName) ] = ':' then begin
        Result := fHeaderLines.Add( sHdrName + ' ' + sHdrValue );
     end else begin
        Result := fHeaderLines.Add( sHdrName + ': ' + sHdrValue );
     end;
end;

function TMess.AddHeaderML( const sHdrName, sHdrValue: String ): Integer;
var  TS: TStringList;
     i : Integer;
begin
     TS := TStringList.Create;
     if sHdrName[ length(sHdrName) ] = ':' then begin
        TS.Text := sHdrName + ' ' + sHdrValue;
     end else begin
        TS.Text := sHdrName + ': ' + sHdrValue;
     end;
     Result := fHeaderLines.Add( TS[0] );
     for i := 1 to TS.Count - 1 do fHeaderLines.Add( TS[i] );
     TS.Free;
end;

procedure TMess.InsertHeaderSL( Index: Integer; const sHdrName, sHdrValue: String );
begin
     if sHdrName[ length(sHdrName) ] = ':' then begin
        fHeaderLines.Insert( Index, sHdrName + ' ' + sHdrValue );
     end else begin
        fHeaderLines.Insert( Index, sHdrName + ': ' + sHdrValue );
     end;
end;

function TMess.SetHeaderSL( sHdrName: String; const sHdrValue: String; sHdrNameSaveOld: String ): Integer;
var  Index : Integer;
     sOldVal: String;
begin
     if sHdrName[length(sHdrName)]<>':' then sHdrName:=sHdrName+':';
     Index := IndexOfHeader( sHdrName, -1 );

     if Index>=0 then begin
        sOldVal := HeaderValueByIndexML( Index );
        DeleteHeaderML( Index );
        if sHdrNameSaveOld<>'' then begin
           if sHdrNameSaveOld[length(sHdrNameSaveOld)]<>':' then sHdrNameSaveOld:=sHdrNameSaveOld+':';
           AddHeaderML( sHdrNameSaveOld, sOldVal );
        end;
     end;

     Result := AddHeaderSL( sHdrName, sHdrValue );
end;

procedure TMess.DeleteHeaderML( Index: Integer );
begin
     fHeaderLines.Delete( Index );
     repeat
        if Index>=fHeaderLines.Count then break;
        if fHeaderLines[Index] = '' then break;
        if not( fHeaderLines[Index][1] in [#9,' '] ) then break;
        fHeaderLines.Delete( Index );
     until False;
end;

procedure TMess.DelHeaderML( const sHdrName: String );
var  Index : Integer;
begin
     Index := IndexOfHeader( sHdrName, -1 );
     if Index>=0 then DeleteHeaderML( Index );
end;

function TMess.IndexOfHeader( const sHdrName: String; iAfterPos: Integer = -1 ): Integer;
var  i, k: Integer;
     h, s: String;
begin
     Result := -1;

     h := LowerCase( TrimWhSpace( sHdrName ) );
     if h = '' then exit;
     if h[length(h)]=':' then System.Delete(h,length(h),1);

     i := iAfterPos + 1;
     while i<fHeaderLines.Count do begin
        s := fHeaderLines[i];
        if s='' then break;
        if not( s[1] in [#9,' '] ) then begin
           k := Pos( ':', s );
           if k>0 then begin
              s := LowerCase( copy( s, 1, k-1 ) );
              if s=h then begin Result:=i; break; end;
           end;
        end;
        inc( i );
     end;
end;

function TMess.HeaderExists( const sHdrName: String ): Boolean;
begin
   Result := ( IndexOfHeader(sHdrName) >= 0 );
end;

function TMess.NextHeader( var Index: Integer; out Name, Value: String ): Boolean;
var  i: Integer;
     s: String;
begin
   Result := False;
   if Index >= fHeaderLines.Count then exit;

   Result := True;
   s := fHeaderLines[Index];
   i := Pos( ':', s );
   Name := copy( s, 1, i-1 );
   System.Delete( s, 1, i );
   if copy( s, 1, 1 ) = ' ' then System.Delete( s, 1, 1 );
   Value := s;
   inc( Index );

   while Index < fHeaderLines.Count do begin
      if fHeaderLines[Index] = '' then break;
      if not( fHeaderLines[Index][1] in [#9,' '] ) then break;
      Value := Value + fHeaderLines[Index];
      inc( Index );
   end;
end;

function TMess.HeaderValueByIndexSL( Index: Integer ): String;
var  i: Integer;
begin
     Result := fHeaderLines[Index];
     i := Pos( ':', Result );
     System.Delete( Result, 1, i );
     if copy( Result, 1, 1 ) = ' ' then System.Delete( Result, 1, 1 );

     while Index+1<fHeaderLines.Count do begin
        inc( Index );
        if fHeaderLines[Index] = '' then break;
        if not( fHeaderLines[Index][1] in [#9,' '] ) then break;
        Result := Result + fHeaderLines[Index];
     end;
end;

function TMess.HeaderValueByNameSL( const sHdrName: String ): String;
var  Index: Integer;
begin
     Index := IndexOfHeader( sHdrName, -1 );
     if Index>=0 then Result := HeaderValueByIndexSL( Index )
                 else Result := '';
end;

function TMess.HeaderValueByIndexML( Index: Integer ): String;
var  i: Integer;
begin
     Result := fHeaderLines[Index];
     i := Pos( ':', Result );
     System.Delete( Result, 1, i );
     if copy( Result, 1, 1 ) = ' ' then System.Delete( Result, 1, 1 );

     while Index+1<fHeaderLines.Count do begin
        inc( Index );
        if fHeaderLines[Index] = '' then break;
        if not( fHeaderLines[Index][1] in [#9,' '] ) then break;
        Result := Result + #13#10 + fHeaderLines[Index];
     end;
end;

function TMess.HeaderValueByNameML( const sHdrName: String ): String;
var  Index: Integer;
begin
     Index := IndexOfHeader( sHdrName, -1 );
     if Index>=0 then Result := HeaderValueByIndexML( Index )
                 else Result := '';
end;

function TMess.GetBodyLines: Integer;
var  SL: TStringList;
begin
   SL := TStringList.Create;
   try
      SL.Text := BodyText;
      Result := SL.Count;
   finally
      SL.Free;
   end;
end;

procedure TMess.AddBodyLine( const line: String );
begin
   if length( fBodyText ) > 0 then begin
      if fBodyText[length(fBodyText)] <> #10 then fBodyText := fBodyText + #13#10;
   end;
   fBodyText := fBodyText + line + #13#10;
end;

procedure TMess.AddBodyText( const txt: String );
begin
   if length( fBodyText ) > 0 then begin
      if fBodyText[length(fBodyText)] <> #10 then fBodyText := fBodyText + #13#10;
   end;
   fBodyText := fBodyText + txt + #13#10;
end;

function TMess.GetDateTimeGMT: TDateTime;
begin
   Result := RfcDateTimeToDateTimeGMT( HeaderValueByNameSL( 'Date:' ) );
end;

function TMess.GetOurXHeader( const XNam: String ) : String;
var  h: String;
     i: Integer;
begin
     Result := '';

     h := HeaderValueByNameSL( HDR_NAME_X_HAMSTER_INFO );
     if h='' then exit;

     i := Pos( LowerCase(XNam), LowerCase(h) );
     if i=0 then exit;

     System.Delete( h, 1, i+length(XNam) );
     i := Pos(' ',h);
     if i>0 then h:=copy(h,1,i-1);

     Result := h;
end;

function TMess.GetReceivedDT: TDateTime;
var  s: String;
begin
     Result := 0;
     s := GetOurXHeader( 'Received' );
     if s<>''    then Result := TimeStampToDateTime(s);
     if Result=0 then Result := DateTimeGMTToLocal( GetDateTimeGMT );
end;

function TMess.GetHeaderLine( const HdrNam: String; var LineNo: Integer ) : String;
begin
   LineNo := IndexOfHeader( HdrNam );
   if LineNo < 0 then Result := ''
                 else Result := HeaderValueByIndexSL( LineNo );
end;

function TMess.GetExportText: String;
const DOW = 'SunMonTueWedThuFriSat';
      MOY = 'JanFebMayAprMayJunJulAugSepOctNovDec';

     function IntroductionLine: String;
     var  DT: TDateTime;
     begin
          // introduction line ("From ???@??? Fri Sep 8 13:21:43 1995")
          DT := DateTimeGMT;
          Result := 'From' + ' '
                  + ExtractMailAddr( HeaderValueByNameSL('From:') ) + ' '
                  + copy( DOW, DayOfWeek(DT)*3-2, 3 ) + ' '
                  + copy( MOY, strtoint(FormatDateTime('mm',DT))*3-2, 3 ) + ' '
                  + FormatDateTime( 'd hh:nn:ss yyyy', DT );
     end;

var  SL: TStringList;
     i : Integer;
begin
   SL := TStringList.Create;
   try
      SL.Text := FullText;
      if SL.Count > 0 then begin
         if copy( SL[0], 1, 5 ) <> 'From ' then begin

            SL.Insert( 0, IntroductionLine ); // add 'From ...' line

            for i:=1 to SL.Count-1 do begin // quote lines with leading "From "
               if ( copy( SL[i], 1, 5 ) = 'From ' ) (* and
                  ( Pos( '@', SL[i] ) > 0 ) *) then begin
                  SL[i] := '>' + SL[i];
               end;
            end;

         end;
      end;

      Result := SL.Text;

   finally
      SL.Free;
   end;
end;

function TMess.LoadFromFile( const Filename: String ): Integer;
var  Data: String;
     FS  : TFileStream;
begin
   Result := -1; // file not found
   Data   := '';

   try
      if FileExists( Filename ) then begin
         FS := TFileStream.Create( Filename, fmOpenRead or fmShareDenyNone );
         try
            SetLength( Data, FS.Size );
            FS.Read( Data[1], length(Data) );
            Result := 0; // ok
         finally FS.Free end;
      end;
   except
      Result := -2; // error
      Data   := '';
   end;

   FullText := Data;
end;

function TMess.SaveToFile( const Filename: String ): Integer;
var  Data: String;
     FS  : TFileStream;
begin
   Data   := FullText;
   Result := 0; // ok

   try
      FS := TFileStream.Create( Filename, fmCreate );
      try
         if length(Data) > 0 then FS.Write( Data[1], length(Data) );
      finally FS.Free end;
   except
      Result := -2; // error
      Data   := '';
   end;
end;

constructor TMess.Create;
begin
     inherited Create;
     fHeaderLines := TStringList.Create;
     fBodyText    := '';
end;

destructor TMess.Destroy;
begin
     if Assigned(fHeaderLines) then fHeaderLines.Free;
     inherited Destroy;
end;

function TMess.GetMailSummary: String;
var  h: String;
begin
   h := HeaderValueByNameSL( HDR_NAME_SUBJECT );
   h := StringReplace( h, #9, ' ', [rfReplaceAll] );
   if h = '' then h := '(none)';
   Result := '"' + h + '"';

   h := HeaderValueByNameSL( HDR_NAME_FROM );
   h := StringReplace( h, #9, ' ', [rfReplaceAll] );
   if h = '' then h := '(unknown)';
   Result := Result + ' from "' + h + '"';
end;

procedure TMess.Clear;
begin
   fHeaderLines.Clear;
   fBodyText := '';
end;

end.

