// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit xApplication;

interface

uses uType;

procedure AppStart( NewCheckPointFunc: TNewCheckPointFunc );
procedure AppStop ( NewCheckPointFunc: TNewCheckPointFunc );
procedure AppWaitFor;

procedure AppHide;
procedure AppShow;


implementation

uses SysUtils, Classes, Windows, Messages, uTools, cHamster, uConst,
     uConstVar, uVar, cLogFileHamster, xAppServiceManager;

procedure AppSettingChanged( const Qualifier: String;
                             const ID: Integer );
var  ErrMsg: String;
begin
   case ID of
      asServiceFile, asServiceName: begin
         AppSettings.SetChanged( ID, False ); // not intended to be changed
      end;

      asDisplayName, asDependencies, asInteractive: begin
         if AppServiceIsInstalled( ErrMsg ) then begin
            if AppServiceChangeConfig( ErrMsg ) then begin
               if Assigned(AppIsServiceMutex) and (ID<>asDisplayName) then begin
                  Log( LOGID_WARN, 'Note: Service changes will not take effect until service is restarted.' );
               end;
            end else begin
               Log( LOGID_ERROR, 'Error changing service config: ' + ErrMsg );
            end;
         end;
      end;

      asLogFileMask: LogFile.FileMask := AppSettings.GetHexInt(asLogFileMask);
      asLogViewMask: LogFile.ViewMask := AppSettings.GetHexInt(asLogViewMask);
      asLogTaskMask: LogFile.TaskMask := AppSettings.GetHexInt(asLogTaskMask);
      asLogFileMax:  LogFile.FileMax  := AppSettings.GetInt   (asLogFileMax );
      asLogViewMax:  LogFile.ViewMax  := AppSettings.GetInt   (asLogViewMax );
      asLogMSecs:    LogFile.LogMSecs := AppSettings.GetBoo   (asLogMSecs   );

      else SafeLog( LOGID_WARN, 'AppSettingChanged, unhandled ID ' + inttostr(ID) );
   end;
end;

procedure AppStart( NewCheckPointFunc: TNewCheckPointFunc );
begin
   LogConsole( LOGID_INFO, 'Starting application ...' );
   if Assigned( NewCheckPointFunc ) then NewCheckPointFunc( 5000 );

   Hamster := THamster.Create( NewCheckPointFunc );
   AppSettings.OnChanged := AppSettingChanged;

   LogConsole( LOGID_INFO, 'Application started.' );
end;

procedure AppStop( NewCheckPointFunc: TNewCheckPointFunc );
begin
   AppShow; // show console window if it was /hidden

   LogConsole( LOGID_INFO, 'Stopping application ...' );
   if Assigned( NewCheckPointFunc ) then NewCheckPointFunc( 5000 );

   AppTerminateEvent.SetEvent;
   Hamster.ActiveThreads.StopAll;
   AppSettings.OnChanged := nil;
   try if Assigned( Hamster ) then Hamster.Free except end;
   Hamster := nil;

   LogConsole( LOGID_INFO, 'Application stopped.' );
end;

procedure AppWaitFor;
// Waiting for app to be terminated.
var  ObjArray: array[0..0] of THandle;
     ObjCount, WaitRes: DWORD;
     Msg: TMsg;
begin
   (*
      Unfortunately, TThread of Classes-unit creates a hidden window for
      synchronization purposes. This causes strange effects (e. g. DDE
      applications will hang when starting, same with Office documents
      when double-clicked) if we do not provide a (dummy) message loop.

      As TThread cannot be avoided easily (Indy uses it also), line ...
         WaitForSingleObject( AppTerminateEvent, INFINITE );
      .. had to be replaced by the message loop code below.

      SetTimer() used by THamster also needs a message loop, even if it's
      solely used with a callback function. It may have been replaced by a
      multimedia timer, but as we need and have a message loop ...
   *)

   // create message queue
   while not PostThreadMessage( GetCurrentThreadId,WM_NULL,0,0 ) do Sleep(10);

   // prepare waitable objects array
   ObjArray[0] := AppTerminateEvent.Handle;
   ObjCount := 1;

   // message loop
   repeat
   
      // process queued messages
      while PeekMessage( Msg, 0, 0, 0, PM_REMOVE ) do begin
         if Msg.message = WM_QUIT then begin
            AppTerminateEvent.SetEvent;
            Hamster.ActiveThreads.StopAll;
            break;
         end;
         DispatchMessage( Msg );
      end;

      // wait for new messages or AppTerminateEvent
      WaitRes := MsgWaitForMultipleObjects( ObjCount, ObjArray, False,
                                            INFINITE, QS_ALLINPUT );

   until WaitRes = WAIT_OBJECT_0; // AppTerminateEvent was set
end;

//-----------------------------------------------------------------------------

var
   ConsoleHWnd: Cardinal = 0;

procedure AppHide;
var  SaveTitle, TempTitle: String;
     buf: array[0..256] of char;
begin
   if ConsoleHWnd = 0 then begin
      GetConsoleTitle( buf, 256 );
      SaveTitle := buf;

      TempTitle := 'Find Hamster ... ' + inttostr(GetCurrentProcessID);
      SetConsoleTitle( PChar(TempTitle) );
      Sleep( 100 );

      ConsoleHWnd := FindWindow( nil, PChar(TempTitle) );

      SetConsoleTitle( PChar(SaveTitle) );
   end;
   
   if ConsoleHWnd <> 0 then ShowWindow( ConsoleHWnd, SW_HIDE );
end;

procedure AppShow;
begin
   if ConsoleHWnd <> 0 then ShowWindow( ConsoleHWnd, SW_SHOW );
end;

end.
