// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit xAppStarter;

interface

uses Windows;

const
   EVTID_INFO       = 100;
   EVTID_INFO_START = EVTID_INFO + 10;
   EVTID_INFO_STOP  = EVTID_INFO + 90;

   EVTID_WARNING    = 800;

   EVTID_ERROR      = 900;
   EVTID_ERROR_DIED = EVTID_ERROR + 99;


procedure AppStartup( StartAsServiceEnabled: Boolean );
function EvtLogError( EventMsg: String; EventID: DWORD = 0 ): String;


implementation

uses SysUtils, Messages, uWinSvc, uTools, uConst, uConstVar, uVar,
     xAppServiceManager, xApplication, cLogFileHamster, cSyncObjects;

var
   AppServiceStatus       : TServiceStatus;             
   AppServiceStatusHandle : DWORD;
   AppServiceDispatchTable: array[0..1] of TServiceTableEntry;

//--------------------------------------------- Console's control handler -----

function ConsoleCtrlHandlerFunc( dwCtrlType: DWORD ): Boolean; stdcall;
var  ErrMsg: String;
begin
   Result := False;

   case dwCtrlType of
      CTRL_C_EVENT, CTRL_BREAK_EVENT, CTRL_CLOSE_EVENT, CTRL_SHUTDOWN_EVENT:
         try
            // catch all "close" events and stop in a safe manner
            LogConsole( LOGID_INFO, '<STOP> in Console Control Handler' );

            // stop service
            if Assigned( AppIsServiceMutex ) then begin
               if not AppServiceStop( ErrMsg ) then begin
                  LogConsole( LOGID_ERROR, ErrMsg );
               end;
            end;

            // set terminate event
            AppTerminateEvent.SetEvent;

            Result := True;

         except 
            on E: Exception do begin
               LogConsole( LOGID_ERROR, 'ConsoleCtrlHandlerFunc: '+E.Message );
            end;
         end;

      CTRL_LOGOFF_EVENT:
         begin
            // ignore, just mark as handled
            Result := True;
         end;
   end;
end;

//----------------------------------------------------- Service functions -----

function EvtLogEntry( EventMsg: String; EventID, EventType: DWORD ): String;
var  H: THandle;
     P: Pointer;
begin
   Result := EventMsg;
   
   H := RegisterEventSource( nil, PChar(AppSettings.GetStr(asServiceName)) );
   if H <> 0 then begin
      P := PChar( EventMsg );
      ReportEvent( H, EventType, 0, EventID, nil, 1, 0, @P, nil);
      DeregisterEventSource( H );
   end;
end;

function EvtLogError( EventMsg: String; EventID: DWORD = 0 ): String;
begin
   if EventId = 0 then EventId := EVTID_ERROR;
   Result := EvtLogEntry( EventMsg, EventID, EVENTLOG_ERROR_TYPE );
   LogConsole( LOGID_ERROR, Result );
end;

function EvtLogWarning( EventMsg: String; EventID: DWORD = 0 ): String;
begin
   if EventId = 0 then EventId := EVTID_WARNING;
   Result := EvtLogEntry( EventMsg, EventID, EVENTLOG_WARNING_TYPE );
   LogConsole( LOGID_WARN, Result );
end;

function EvtLogInfo( EventMsg: String; EventID: DWORD = 0 ): String;
begin
   if EventId = 0 then EventId := EVTID_INFO;
   Result := EvtLogEntry( EventMsg, EventID, EVENTLOG_INFORMATION_TYPE );
   LogConsole( LOGID_INFO, Result );
end;

function NewServiceStatus( dwState, dwCheckPoint, dwWaitHint: DWORD ): Boolean;
begin
   AppServiceStatus.dwCurrentState := dwState;
   AppServiceStatus.dwCheckPoint   := dwCheckPoint;
   AppServiceStatus.dwWaitHint     := dwWaitHint;

   Result := SetServiceStatus( AppServiceStatusHandle, AppServiceStatus );

   if not Result then begin
      EvtLogError( 'UpdateServiceStatus failed: ' + WinErrMsg );
   end;
end;

function NewCheckPoint( dwWaitHint: DWORD ): Boolean;
// Notifies service control manager, that we have safely reached a new
// checkpoint while starting/stopping and we expect the next checkpoint
// to be reached within next dwWaitHint milliseconds. 
begin
   inc( AppServiceStatus.dwCheckPoint );
   AppServiceStatus.dwWaitHint := dwWaitHint;

   Result := NewServiceStatus( AppServiceStatus.dwCurrentState,
                               AppServiceStatus.dwCheckPoint,
                               AppServiceStatus.dwWaitHint );
end;

procedure AppServiceCtrlHandler( Opcode: DWORD ); stdcall;
begin
   case Opcode of

      SERVICE_CONTROL_SHUTDOWN, SERVICE_CONTROL_STOP:
         try
            NewServiceStatus( SERVICE_STOP_PENDING, 0, 500 );
            AppServiceStatus.dwWin32ExitCode := 0;
            AppServiceStatus.dwServiceSpecificExitCode := 0;
            AppTerminateEvent.SetEvent;
            exit;
         except
            on E: Exception do begin
               EvtLogError( 'AppServiceCtrlHandler: ' + E.Message );
               exit;
            end;
         end;

      // SERVICE_CONTROL_PAUSE, SERVICE_CONTROL_CONTINUE: begin
      //    // not supported
      // end;

      SERVICE_CONTROL_INTERROGATE: begin
         // just fall through to send current status
      end;

      else begin
         EvtLogError( 'SvcCtrl: Unrecognized Opcode: ' + inttostr(Opcode) );
      end;
         
   end;

   // Send current status.
   if not SetServiceStatus( AppServiceStatusHandle, AppServiceStatus ) then begin
      EvtLogError( 'SvcCtrl.SetServiceStatus failed: ' + WinErrMsg );
   end;
end;

procedure AppServiceMain( argc: DWORD; argv: PLPSTR ); stdcall;
var  Status, SpecificError: DWORD;
begin
   with AppServiceStatus do begin
      dwServiceType             := SERVICE_WIN32;
      dwCurrentState            := SERVICE_START_PENDING;
      dwControlsAccepted        := SERVICE_ACCEPT_STOP
                                   // or SERVICE_ACCEPT_PAUSE_CONTINUE
                                   or SERVICE_ACCEPT_SHUTDOWN;
      dwWin32ExitCode           := 0;
      dwServiceSpecificExitCode := 0;
      dwCheckPoint              := 0;
      dwWaitHint                := 0;
   end;

   AppServiceStatusHandle := RegisterServiceCtrlHandler(
                                AppServiceDispatchTable[0].lpServiceName,
                                @AppServiceCtrlHandler );
   if AppServiceStatusHandle = 0 then begin
      EvtLogError( 'Run.RegisterServiceCtrlHandler failed: ' + WinErrMsg );
      exit;
   end;

   if not SetServiceStatus( AppServiceStatusHandle, AppServiceStatus ) then begin
      EvtLogError( 'Run.SetServiceStatus failed: ' + WinErrMsg );
      exit;
   end;

   // initialization
   Status := NO_ERROR;
   SpecificError := 0;

   // create and lock "running as service" mutex
   AppIsServiceMutex := TMutexObj.Create( lpSecurityAttributesEveryone, False,
                                          AppIsServiceMutexName );
   if not AppIsServiceMutex.Acquire( 0 ) then begin
      SpecificError := 20;
      EvtLogError( 'Run.CreateSvcMutex failed: ' + WinErrMsg );
   end;

   // start application
   if SpecificError = 0 then begin
      try
         AppStart( NewCheckPoint );
      except
         on E: Exception do begin
            SpecificError := 10;
            EvtLogError( 'Run.AppStart failed: ' + E.Message );
         end;
      end;
   end;

   // handle error condition
   if (Status = NO_ERROR) and (SpecificError <> 0) then begin
      Status := ERROR_SERVICE_SPECIFIC_ERROR;
   end;
   if Status <> NO_ERROR then begin
      with AppServiceStatus do begin
         dwCurrentState            := SERVICE_STOPPED;
         dwCheckPoint              := 0;
         dwWaitHint                := 0;
         dwWin32ExitCode           := Status;
         dwServiceSpecificExitCode := SpecificError;
      end;
      SetServiceStatus( AppServiceStatusHandle, AppServiceStatus );
      exit;
   end;

   // ok, the service is now running
   NewServiceStatus( SERVICE_RUNNING, 0, 0 );
   EvtLogInfo( 'Service is running.', EVTID_INFO_START );

   // wait until terminated
   try
      AppWaitFor;
   except
      on E: Exception do EvtLogError( 'AppServiceMain (wait): ' + E.Message );
   end;

   // stop service
   try
      NewServiceStatus( SERVICE_STOP_PENDING, 1, 500 );
      try
         AppStop( NewCheckPoint );
      except
         on E: Exception do EvtLogError( 'AppServiceMain (stop/2): ' + E.Message );
      end;
      AppIsServiceMutex.Release;
      NewServiceStatus( SERVICE_STOPPED, 0, 0 );
      EvtLogInfo( 'Service is stopped.', EVTID_INFO_STOP );

   except
      on E: Exception do EvtLogError( 'AppServiceMain (stop): ' + E.Message );
   end;
end;

//------------------------------- Startup (either service or application) -----

procedure AppStartup( StartAsServiceEnabled: Boolean );
var  ErrorCode: DWORD;
     ErrMsg: String;
begin
   // activate control handler for console window
   SetConsoleCtrlHandler( @ConsoleCtrlHandlerFunc, True );

   // check, if service is installed
   if StartAsServiceEnabled then begin
      if not AppServiceIsInstalled(ErrMsg) then begin
         StartAsServiceEnabled := False;
         if ErrMsg <> '' then LogConsole( LOGID_INFO, 'Service disabled: ' + ErrMsg );
      end;
   end;

   // start as service           changes.txt
   if StartAsServiceEnabled then begin

      LogConsole( LOGID_INFO, 'Trying to start as service ...' );

      FillChar( AppServiceDispatchTable, sizeof(AppServiceDispatchTable), 0 );
      with AppServiceDispatchTable[0] do begin
         lpServiceName := AppSettings.GetStrPtr(asServiceName);
         lpServiceProc := @AppServiceMain;
      end;

      // Note: StartServiceCtrlDispatcher will take up to 30 seconds here and
      //       return ERROR_FAILED_SERVICE_CONTROLLER_CONNECT if application
      //       was started directly and not by service control manager.
      if not StartServiceCtrlDispatcher( AppServiceDispatchTable[0] ) then begin
         ErrorCode := GetLastError;
         if ErrorCode = ERROR_FAILED_SERVICE_CONTROLLER_CONNECT then begin
            // not started as service, so fall through to start as application
            StartAsServiceEnabled := False;
         end else begin
            LogConsole( LOGID_WARN, 'Run.StartServiceCtrlDispatcher failed: '
                                    + WinErrMsg( ErrorCode ) );
         end;
      end;

   end;

   // start as application
   if not StartAsServiceEnabled then begin

      LogConsole( LOGID_INFO, 'Trying to start as application ...' );

      try
         AppStart( nil );
         try
            AppWaitFor;
         finally
            AppStop( nil );
         end;

      except
         on E: Exception do begin
            LogConsole( LOGID_ERROR,
                        'Exception running application: ' + E.Message );
         end;
      end;

   end;

   // deactivate control handler for console window
   SetConsoleCtrlHandler( @ConsoleCtrlHandlerFunc, False );
end;

end.
