// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit xAppServiceManager;

interface

uses Windows;

function AppServiceInstall     ( var ErrMsg: String ): Boolean;
function AppServiceChangeConfig( var ErrMsg: String ): Boolean;
function AppServiceUninstall   ( var ErrMsg: String ): Boolean;
function AppServiceIsInstalled ( var ErrMsg: String ): Boolean;

function AppServiceStart  ( var ErrMsg: String ): Boolean;
function AppServiceStop   ( var ErrMsg: String ): Boolean;
function AppServiceControl( const fdwControl: DWORD;
                            var ErrMsg: String ): Boolean;

implementation

uses SysUtils, uWinSvc, uConstVar, uTools, uConst, uVar;

//--------------------------------------------- Install/Uninstall Service -----

function AppServiceInstall( var ErrMsg: String ): Boolean;
var  schSCManager, schService: SC_HANDLE;
     dwServiceType, dwStartType: DWORD;
     lpServiceStartName, lpServiceStartPass: PChar;
     sDependencies0: String;
     i: Integer;
begin
   Result := False;
   ErrMsg := '';

   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( nil, nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Install.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   dwServiceType := SERVICE_WIN32_OWN_PROCESS;
   dwStartType   := SERVICE_AUTO_START; // vs. SERVICE_DEMAND_START
   lpServiceStartName := nil;
   lpServiceStartPass := nil;

   if AppSettings.GetBoo( asInteractive ) then begin
      dwServiceType := dwServiceType or SERVICE_INTERACTIVE_PROCESS;
   end else begin
      // opt.: lpServiceStartName := 'domainname\username'
      // opt.: lpServiceStartPass := 'userpass'
   end;

   sDependencies0 := AppSettings.GetStr( asDependencies );
   for i:=1 to length(sDependencies0) do begin
      if sDependencies0[i]=',' then sDependencies0[i]:=#0;
   end;
   sDependencies0 := sDependencies0 + #0#0;

   schService := CreateService(
      schSCManager,                   // SCManager database
      PChar(AppSettings.GetStr(asServiceName)), // service's name
      PChar(AppSettings.GetStr(asDisplayName)), // service's display name
      SERVICE_ALL_ACCESS,             // desired access
      dwServiceType,                  // service type
      dwStartType,                    // start type
      SERVICE_ERROR_NORMAL,           // error control type
      PChar(AppSettings.GetStr(asServiceFile)), // service's binary
      nil,                            // no load ordering group
      nil,                            // no tag identifier
      PChar(sDependencies0),          // dependencies
      lpServiceStartName,             // nil=LocalSystem, 'domain\name'=user
      lpServiceStartPass );           // password if lpServiceStartName<>nil

   if schService <> 0 then begin
      CloseServiceHandle( schService );
      Result := True;
   end else begin
      ErrMsg := 'Install.CreateService failed: ' + WinErrMsg;
   end;

   CloseServiceHandle( schSCManager );
end;

function AppServiceChangeConfig( var ErrMsg: String ): Boolean;
var  schSCManager, schService: SC_HANDLE;
     dwServiceType, dwStartType: DWORD;
     lpServiceStartName, lpServiceStartPass: PChar;
     sDependencies0: String;
     i: Integer;
begin
   Result := False;
   ErrMsg := '';

   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( nil, nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'ChangeConfig.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( AppSettings.GetStr(asServiceName) ), 
                              SERVICE_CHANGE_CONFIG );
   if schService = 0 then begin
      ErrMsg := 'ChangeConfig.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   dwServiceType := SERVICE_WIN32_OWN_PROCESS;
   dwStartType   := SERVICE_AUTO_START; // vs. SERVICE_DEMAND_START
   lpServiceStartName := nil;
   lpServiceStartPass := nil;

   if AppSettings.GetBoo( asInteractive ) then begin
      dwServiceType := dwServiceType or SERVICE_INTERACTIVE_PROCESS;
   end else begin
      // opt.: lpServiceStartName := 'domainname\username'
      // opt.: lpServiceStartPass := 'userpass'
   end;

   sDependencies0 := AppSettings.GetStr( asDependencies );
   for i:=1 to length(sDependencies0) do begin
      if sDependencies0[i]=',' then sDependencies0[i]:=#0;
   end;
   sDependencies0 := sDependencies0 + #0#0;

   if ChangeServiceConfig(
         schService,                     // service handle
         dwServiceType,                  // service type
         dwStartType,                    // start type
         SERVICE_ERROR_NORMAL,           // error control type
         PChar(AppSettings.GetStr(asServiceFile)), // service's binary
         nil,                            // no load ordering group
         nil,                            // no tag identifier
         PChar(sDependencies0),          // dependencies
         lpServiceStartName,             // nil=LocalSystem, 'domain\name'=user
         lpServiceStartPass,             // password if lpServiceStartName<>nil
         PChar(AppSettings.GetStr(asDisplayName)) // service's display name
      ) then begin

      Result := True;

   end else begin

      ErrMsg := 'ChangeConfig.ChangeServiceConfig failed: ' + WinErrMsg;

   end;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

function AppServiceUninstall( var ErrMsg: String ): Boolean;
var  schSCManager, schService: SC_HANDLE;
begin
   Result := False;
   ErrMsg := '';
   
   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( nil, nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Uninstall.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( AppSettings.GetStr(asServiceName) ),
                              _DELETE );
   if schService = 0 then begin
      ErrMsg := 'Uninstall.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   if DeleteService( schService ) then begin
      Result := True;
   end else begin
      ErrMsg := 'Uninstall.DeleteService failed: ' + WinErrMsg;
   end;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

function AppServiceIsInstalled( var ErrMsg: String ): Boolean;
var  schSCManager, schService: SC_HANDLE;
begin
   Result := False;
   ErrMsg := '';
   
   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( nil, nil, GENERIC_READ );
   if schSCManager=0 then begin
      ErrMsg := 'IsInstalled.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( AppSettings.GetStr(asServiceName) ),
                              SERVICE_QUERY_CONFIG );
   if schService = 0 then begin
      ErrMsg := 'IsInstalled.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   Result := True;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

//------------------------------------------------- Start/Control Service -----

function AppServiceStart( var ErrMsg: String ): Boolean;
var  ssStatus: TServiceStatus;
     schSCManager, schService: SC_HANDLE;
     dwServiceArgs, dwOldCheckPoint: DWORD;
     lpServiceArgs: PChar;
begin
   Result := False;
   ErrMsg := '';

   schSCManager := OpenSCManager( nil, nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Start.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   schService := OpenService( schSCManager,
                              PChar( AppSettings.GetStr(asServiceName) ),
                              SERVICE_ALL_ACCESS );
   if schService = 0 then begin
      ErrMsg := 'Start.OpenService failed: ' + WinErrMsg;
      CloseServiceHandle( schSCManager );
      exit;
   end;

   dwServiceArgs := 0;
   lpServiceArgs := nil;

   if StartService( schService, dwServiceArgs, lpServiceArgs ) then begin

      if QueryServiceStatus( schService, ssStatus) then begin

         while ssStatus.dwCurrentState = SERVICE_START_PENDING do begin
            dwOldCheckPoint := ssStatus.dwCheckPoint;
            Sleep( ssStatus.dwWaitHint );
            if not QueryServiceStatus( schService, ssStatus ) then break;
            if dwOldCheckPoint >= ssStatus.dwCheckPoint then break;
         end;

         if ssStatus.dwCurrentState = SERVICE_RUNNING then begin
            Result := True;
         end else begin
            ErrMsg := 'Start.WaitTilRunning failed: '
            + '  State='   + inttostr ( ssStatus.dwCurrentState )
            + '  SvcCode=' + inttostr ( ssStatus.dwServiceSpecificExitCode )
            + '  WinCode=' + WinErrMsg( ssStatus.dwWin32ExitCode );
         end;

      end else begin
         ErrMsg := 'Start.QueryServiceStatus failed ' + WinErrMsg;
      end;

   end else begin
      ErrMsg := 'Start.StartService failed: ' + WinErrMsg;
   end;

   CloseServiceHandle( schService );
   CloseServiceHandle( schSCManager );
end;

function AppServiceControl( const fdwControl: DWORD;
                            var   ErrMsg: String ): Boolean;
var  ssStatus: TServiceStatus;
     fdwAccess: DWORD;
     schSCManager, schService: SC_HANDLE;
begin
   Result := False;
   ErrMsg := '';
   
   if not IsWinSvcAvailable then begin
      ErrMsg := 'Services are not supported on this platform!';
      exit;
   end;

   schSCManager := OpenSCManager( nil, nil, SC_MANAGER_ALL_ACCESS );
   if schSCManager=0 then begin
      ErrMsg := 'Control.OpenSCManager failed: ' + WinErrMsg;
      exit;
   end;

   case fdwControl of
      SERVICE_CONTROL_STOP:        fdwAccess := SERVICE_STOP;
      SERVICE_CONTROL_PAUSE:       fdwAccess := SERVICE_PAUSE_CONTINUE;
      SERVICE_CONTROL_CONTINUE:    fdwAccess := SERVICE_PAUSE_CONTINUE;
      SERVICE_CONTROL_INTERROGATE: fdwAccess := SERVICE_INTERROGATE;
      else                         fdwAccess := SERVICE_INTERROGATE;
   end;

   schService := OpenService( schSCManager,
                              PChar( AppSettings.GetStr(asServiceName) ),
                              fdwAccess );
   if schService = 0 then begin

      ErrMsg := 'Control.OpenService failed: ' + WinErrMsg;

   end else begin

      if not ControlService( schService, fdwControl, ssStatus ) then begin
         ErrMsg := 'Control.ControlService failed: ' + WinErrMsg;
      end else begin
         Result := True;
      end;

      CloseServiceHandle( schService );
      
   end;

   CloseServiceHandle( schSCManager );
end;

function AppServiceStop( var ErrMsg: String ): Boolean;
begin
   Result := AppServiceControl( SERVICE_CONTROL_STOP, ErrMsg );
end;

//-----------------------------------------------------------------------------

end.
