// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tUserTasks; // thread to execute "user tasks"

interface

uses Windows, SysUtils, Classes, cUserTasks, tBase;

type
   TUserTaskThread = class( TBaseThread )

      private
         FUserTaskName: String;
         FUserTaskList: TThreadList;
         FNewsCache   : TStringList;
         FRasHangup   : Boolean;
         FSilent      : Boolean;

         procedure DoNewsCache ( const NntpServer: String );
         procedure DoNewsExecute;
         procedure DoFetchMails( const Pop3Server: String );
         procedure DoSendMails ( const SmtpServer: String );
         procedure DoSendMailsMX;
         procedure DoRasDial   ( const RasName: String );
         procedure DoRasHangup ( const WaitIdle: Boolean );
         procedure DoWait      ( const All: Boolean );
         procedure DoScript    ( const ScriptName: String;
                                 const WaitForEnd: Boolean );

         procedure StartTasks( const TaskList: TStringList );
         function ActiveUserTasks: Integer;

      protected
         procedure Execute; override;

      public
         procedure Terminate; override;

         class function  UserTasksFilename( const Nam: String ): String;
         class procedure UserTasksEnter;
         class procedure UserTasksLeave;

         class function UserTasks_Lst( const Lst: TStringList ): Boolean;
         class function UserTasks_Del( const Nam: String ): Boolean;
         class function UserTasks_Get( const Nam: String; const Lst: TStringList ): Boolean;
         class function UserTasks_Set( const Nam: String; const Lst: TStringList ): Boolean;

         constructor Create( const AUserTaskName: String; ASilent: Boolean );
         destructor Destroy; override;

   end;


implementation

uses cHamster, tTransfer, tScript, uRasDyn, cLogFileHamster, uConst, uType,
     uVar, uTools, uHamTools;

{ TUserTaskThread }

procedure TUserTaskThread.DoNewsCache( const NntpServer: String );
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Select NNTP transfer from/to ' + NntpServer );
   FNewsCache.Add( NntpServer );
end;

procedure TUserTaskThread.DoNewsExecute;
var  IdList: TList;
     i, id: Integer;
     NntpServers: String;
begin
   if FNewsCache.Count = 0 then exit;

   IdList := TList.Create();
   try

      TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Start NNTP transfers' );

      NntpServers := '';
      try
         for i := 0 to FNewsCache.Count - 1 do begin
            if length( NntpServers ) > 0 then NntpServers := NntpServers + ';';
            NntpServers := NntpServers + FNewsCache[i];
         end;
      finally FNewsCache.Clear end;

      if length( NntpServers ) > 0 then begin
         Hamster.NewsJobs.AddPostDef( NntpServers );
         Hamster.NewsJobs.AddPullDef( NntpServers );
         Hamster.NewsJobs.StartThreads( NntpServers, IdList );
      end;

      for i := 0 to IdList.Count - 1 do begin
         id := Integer( IdList[i] );
         FUserTaskList.Add( Pointer(id) );
      end;

   finally IdList.Free end;
end;

procedure TUserTaskThread.DoFetchMails( const Pop3Server: String );
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Fetch mails from POP3 server ' + Pop3Server );
   
   with TThreadPop3Fetch.Create( Pop3Server, '', '', '', '', '', '?' ) do begin
      FUserTaskList.Add( Pointer(UniqueID) );
      Resume;
   end;
end;

procedure TUserTaskThread.DoSendMails( const SmtpServer: String );
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Send mails to SMTP server ' + SmtpServer );

   with TThreadSmtpSend.Create( False, SmtpServer, '', '', '', '', '' ) do begin
      FUserTaskList.Add( Pointer(UniqueID) );
      Resume;
   end;
end;

procedure TUserTaskThread.DoSendMailsMX;
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Send mails by SMTP-MX' );
   
   with TThreadSmtpSend.Create( True, '', '', '', '', '', '' ) do begin
      FUserTaskList.Add( Pointer(UniqueID) );
      Resume;
   end;
end;

procedure TUserTaskThread.DoWait( const All: Boolean );
begin
   // Start all cached news transfers first
   DoNewsExecute;

   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Wait until ' + iif( All, 'all running', 'started' ) + ' tasks are finished' );

   // wait for all tasks started by this user task
   while not Terminated do begin
      if ActiveUserTasks = 0 then break;
      Sleep( 250 );
   end;

   // wait for all currently running tasks (except self)
   if All then begin
      while not Terminated do begin
         if Hamster.ActiveThreads.CountAllExcept( attUserTask ) = 0 then break;
         // if Hamster.ActiveThreads.IsIdle( Self.UniqueID ) then break;
         Sleep( 250 );
      end;
   end;
end;

procedure TUserTaskThread.DoRasDial( const RasName: String );
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ), 'Dial ' + RasName );

   if RasDynIsConnected then begin
      TLog( LOGID_INFO, 'Dial skipped, already conntected: ' + RasDynGetConnection );
   end else begin
      FRasHangup := ( Hamster.RasDialer.Dial( RasName, '', '' ) = 0 );
      if not FRasHangup then begin
         TLog( LOGID_WARN, 'RAS-Dial failed - stopping all tasks.' );
         Terminate;
      end;
   end;
end;

procedure TUserTaskThread.DoRasHangup( const WaitIdle: Boolean );
begin
   if WaitIdle then DoWait(False);
   
   if FRasHangup then begin
      TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ), 'Hangup' );
      Hamster.RasDialer.HangUp;
      FRasHangup := False;
   end;
end;

procedure TUserTaskThread.DoScript( const ScriptName: String; const WaitForEnd: Boolean );
var  t: TThreadExecuteScript;
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_INFO ),
         'Start script' + iif(WaitForEnd,' and wait') + ': ' + ScriptName );
   StartNewScript( ScriptName, '', WaitForEnd, t, tftFreeOnTerminate, False, False );
   if (not WaitForEnd) and (t <> nil) then begin
      FUserTaskList.Add( Pointer( t.UniqueID ) );
      // unsolved issue: tasks started by scripts are not added to list
      //                 -> WAIT does not work, WAIT_ALL required instead
      //                 -> Appropriate note added in helpfile
   end;
end;

function TUserTaskThread.ActiveUserTasks: Integer;
// Remove finished tasks from list and return number of remaining tasks
var  i, id: Integer;
begin
   with FUserTaskList.LockList do try

      for i := Count - 1 downto 0 do begin
         id := Integer( Items[i] );
         if id > 0 then begin
            if not Hamster.ActiveThreads.IsRunning( id ) then Delete( i );
         end;
      end;

      Result := Count;

   finally FUserTaskList.UnlockList end;
end;

procedure TUserTaskThread.Terminate;
// Make sure, that remaining user tasks are terminated as well.
var  i, id: Integer;
begin
   with FUserTaskList.LockList do try
      try
         for i := Count - 1 downto 0 do begin
            id := Integer( Items[i] );
            if i > 0 then Hamster.ActiveThreads.StopByID( id );
         end;

      except
         on E: Exception do begin
            TLog( LOGID_ERROR, 'ERROR terminating user tasks: ' + E.Message );
         end;
      end;

   finally
      FUserTaskList.UnlockList;
      inherited;
   end;
end;

procedure TUserTaskThread.StartTasks( const TaskList: TStringList );
// Start given tasks
var  i: Integer;
     Args: TStringList;
     Line: String;
begin
   Args := TStringList.Create;
   try

      for i := 0 to TaskList.Count - 1 do begin

         if Terminated then break;

         Line := TaskList[i];
         StateInfo := 'Starting tasks ... ' + line;
         TLog( LOGID_DEBUG, 'Line ' + inttostr(i+1) + ': ' + Line );
         ArgsWhSpaceDQuoted( Line, Args, 9 );

         if      Args[0]=USERTASKCMD_RAS_DIAL      then DoRasDial( Args[1] )
         else if Args[0]=USERTASKCMD_RAS_HANGUP    then DoRasHangup( True )
         else if Args[0]=USERTASKCMD_RAS_REQUIRED  then begin
            if not RasDynIsConnected then begin
               TLog( LOGID_DEBUG, 'Required RAS connection missing - stopping all tasks.' );
               Terminate;
               break;
            end;
         end
         else if Args[0]=USERTASKCMD_MAILS_SEND    then DoSendMails( Args[1] )
         else if Args[0]=USERTASKCMD_MAILS_SEND_MX then DoSendMailsMX
         else if Args[0]=USERTASKCMD_MAILS_FETCH   then DoFetchMails( Args[1] )
         else if Args[0]=USERTASKCMD_NEWS          then DoNewsCache( Args[1] )
         else if Args[0]=USERTASKCMD_WAIT          then DoWait(False)
         else if Args[0]=USERTASKCMD_WAIT_ALL      then DoWait(True)
         else if Args[0]=USERTASKCMD_SCRIPT        then DoScript( Args[1], False )
         else if Args[0]=USERTASKCMD_SCRIPT_WAIT   then DoScript( Args[1], True )

         else TLog( LOGID_WARN, 'Line ' + inttostr(i+1) + ' ignored: ' + Line );

      end;

   finally Args.Free end;
end;

procedure TUserTaskThread.Execute;
var  TaskList: TStringList;
     s: String;
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_SYSTEM ), 'Start' );

   TaskList := TStringList.Create;
   try

      try
         // load user task file (<taskname>.hut in script directory)
         StateInfo := 'Loading tasks ...';
         s := UserTasksFilename( FUserTaskName );
         if FileExists( s ) then begin
            UserTasksEnter;
            try
               TaskList.LoadFromFile( s );
            finally UserTasksLeave end;
         end else begin
            TLog( LOGID_WARN, 'User Task file not found: ' + s );
         end;

         // start all user tasks
         StateInfo := 'Starting tasks ...';
         StartTasks( TaskList );

         // wait until all started user tasks are finished
         StateInfo := 'Waiting for tasks to finish ...';
         DoWait( False );

      except
         on E: Exception do begin
            TLog( LOGID_ERROR, 'ERROR executing user tasks: ' + E.Message );
         end;
      end;

   finally
      try TaskList.Free; except end;
      try if FRasHangup then DoRasHangup( False ); except end;
      TLog( iif( FSilent, LOGID_DEBUG, LOGID_SYSTEM ), 'End' );
   end;
end;

constructor TUserTaskThread.Create( const AUserTaskName: String; ASilent: Boolean );
var  i: Integer;
begin
   FUserTaskName := AUserTaskName;
   FSilent := ASilent;

   i := Pos( '\', FUserTaskName );
   if i > 0 then System.Delete( FUserTaskName, 1, i );
   i := Pos( USERTASK_EXTENSION, LowerCase(FUserTaskName) );
   if i > 0 then SetLength( FUserTaskName, i-1 );

   inherited Create( attUserTask, '{usertask ' + FUserTaskName + '}', tftFreeOnTerminate );

   FUserTaskList := TThreadList.Create;
   FNewsCache    := TStringList.Create;
   FRasHangup    := False;
end;

destructor TUserTaskThread.Destroy;
begin
   if FUserTaskList <> nil then FUserTaskList.Free;
   if FNewsCache    <> nil then FNewsCache.Free;
   inherited Destroy;
end;


class function TUserTaskThread.UserTasksFilename( const Nam: String ): String;
begin
   Result := AppSettings.GetStr(asPathScripts) + Nam + USERTASK_EXTENSION;
end;

class procedure TUserTaskThread.UserTasksEnter;
begin
   HamFileEnter;
end;

class procedure TUserTaskThread.UserTasksLeave;
begin
   HamFileLeave;
end;

class function TUserTaskThread.UserTasks_Lst( const Lst: TStringList ): Boolean;
// list user tasks files
var  SR: TSearchRec;
     s : String;
     i : Integer;
begin
   UserTasksEnter;
   try
      try
         Result := False;
         Lst.Clear;

         if FindFirst( AppSettings.GetStr(asPathScripts)
                     + '*' + USERTASK_EXTENSION, faAnyFile, SR ) = 0 then try
            repeat
               if (SR.Attr and faDirectory) = 0 then begin
                  s := SR.Name;
                  i := LastDelimiter( '.', s );
                  if i > 0 then SetLength( s, i-1 );
                  Lst.Add( s );
               end;
            until FindNext( SR ) <> 0;

            Lst.Sort;
            Result := True;

         finally FindClose( SR ) end;

      except
         on E: Exception do begin
            Result := False;
            Log( LOGID_ERROR, 'UserTasks_Lst: ' + E.Message );
         end;
      end;

   finally UserTasksLeave end;
end;

class function TUserTaskThread.UserTasks_Del( const Nam: String ): Boolean;
// delete user tasks file
var  s: String;
begin
   UserTasksEnter;
   try
      try
         Result := False;
         s := UserTasksFilename( Nam );
         if FileExists( s ) then begin
            DeleteFile( s );
            Result := True;
         end;
      except
         on E: Exception do begin
            Result := False;
            Log( LOGID_ERROR, 'UserTasks_Del: ' + E.Message );
         end;
      end;
   finally UserTasksLeave end;
end;

class function TUserTaskThread.UserTasks_Get( const Nam: String;
                                              const Lst: TStringList ): Boolean;
// load user tasks file
var  s: String;
begin
   UserTasksEnter;
   try
      try
         Result := False;
         s := UserTasksFilename( Nam );
         if FileExists( s ) then begin
            Lst.LoadFromFile( s );
            Result := True;
         end;
      except
         on E: Exception do begin
            Result := False;
            Log( LOGID_ERROR, 'UserTasks_Get: ' + E.Message );
         end;
      end;
   finally UserTasksLeave end;
end;

class function TUserTaskThread.UserTasks_Set( const Nam: String;
                                              const Lst: TStringList): Boolean;
// save user tasks file
var  s: String;
begin
   UserTasksEnter;
   try
      try
         s := UserTasksFilename( Nam );
         Lst.SaveToFile( s );
         Result := True;
      except
         on E: Exception do begin
            Result := False;
            Log( LOGID_ERROR, 'UserTasks_Set: ' + E.Message );
         end;
      end;
   finally UserTasksLeave end;
end;

end.
