// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tTransfer; // Threads for news and mail transfers.

// ----------------------------------------------------------------------------
// Contains all threads, which handle the transfer of news and mails from/to
// remote servers.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, tBase, cClientNNTP, cClientPOP3, cClientSMTP,
     cWebChannel, uTools;

type
   TThreadPop3Fetch = class( TTaskThread )
      private
         FServerAlias: String;
         FServer, FPort, FUser, FPass,
         FDestUsername, FFilterSection, FLeaveServer : String;
         MailClient: TClientPOP3;
         procedure FetchMailFromServer( LfdServer: Integer );
      protected
         procedure Execute; override;
      public
         procedure Terminate; override;
         constructor Create( AServer, APort, AUser, APass, ADestUsername,
                             AFilterSection, ALeaveServer: String );
   end;

   TThreadSmtpSend = class( TTaskThread )
      private
         FMXMode: Boolean;
         FServerAlias, FServer, FPort, FUser, FPass: String;
         FFromSelect, FToSelect: String;
         ToSendSnapshot: TStringList;
         SendMailClient: TClientSMTP;
      protected
         procedure ConvertToMXFiles( FilePath, FileName: String );
         procedure BuildSendSnapshot;
         procedure SendMail;
         procedure SendMailMX;
         procedure Execute; override;
      public
         procedure Terminate; override;
         constructor Create( AMXMode: Boolean;
                             AServer, APort, AUser, APass: String;
                             AFromSelect, AToSelect: String );
   end;

   TThreadNewsJobs = class( TTaskThread )
      private
         FServerAlias: String;
         NewsClient: TClientNNTP;
         procedure NukePostedArticle( const PostFile: String;
                                      ArtText: String;
                                      const DestGroup, Reason, DestSrv: String );
         procedure PostArticle( const PostFile: String );
      protected
         procedure Execute; override;
      public
         procedure Terminate; override;
         constructor Create( AServerAlias: String );
   end;

   TThreadWebChannelReader = class( TTaskThread )
   
      private
         FFeed, FUrl, FGroup, FSubject, FFrom: String;
         FMode: Integer;
         FIsFile, FSaveXml: Boolean;
         FProxyServer, FProxyUser, FProxyPass: String;
         FProxyPort: Integer;

      protected
         procedure SetDefaults( var chSubject, chFrom: String );
         procedure SaveXml( reader: TWebChannelReader );
         procedure CreateArticleMode0 ( reader: TWebChannelReader; channel: TWebChannel );
         procedure CreateArticleMode13( reader: TWebChannelReader; channel: TWebChannel );
         procedure CreateArticleMode2 ( reader: TWebChannelReader; channel: TWebChannel );
         procedure Execute; override;
         
      public
         constructor Create( const AFeed, AGroup, ASubject, AFrom: String;
                             const AMode: Integer;
                             const AIsFile, ASaveXml: Boolean;
                             const AProxyServer: String;
                             const AProxyPort: Integer;
                             const AProxyUser, AProxyPass: String );
   end;

   
   TThreadLookupHostAddr = class( TBaseThread )

      private
         FHostName: String;
         FHostAddr: String;

      protected
         procedure Execute; override;

      public
         property HostAddr: String read FHostAddr;
         constructor Create( const AHostName: String );

   end;
   
   TThreadLookupHostAddrs = class( TBaseThread )

      private
         FHostNames: TStringList;
         FHostAddrs: TStringList;
         FDnsCache : TKeyValueCache;

      protected
         procedure Execute; override;

      public
         constructor Create( const AHostNames, AHostAddrs: TStringList;
                             const ADnsCache: TKeyValueCache = nil );

   end;


implementation

uses uType, uConst, uConstVar, uVar, uCRC32, cPCRE, cPasswords,
     cAccounts, cArticle, cArtFiles, cNewsJobs, cClientBase, cSmtpRouter,
     uDateTime, IniFiles, cLogFileHamster, cHamster, uHamTools, cHscAction,
     uMD5, cHscHelpers, uEncoding, uWinSock;

// ----------------------------------------------------- TThreadPop3Fetch -----

procedure TThreadPop3Fetch.FetchMailFromServer( LfdServer: Integer );
var  LeaveOnServer: Boolean;
begin
   // get default/overridden 'leave on server' setting
   if FLeaveServer = '1' then
      LeaveOnServer := True
   else if FLeaveServer = '0' then
      LeaveOnServer := False
   else begin
      LeaveOnServer := Hamster.Config.Settings.GetBoo(hsLeaveMailsOnServer);
      if not LeaveOnServer then begin
         with Hamster.Config.Pop3Servers do begin 
            LeaveOnServer := Settings(LfdServer).GetBoo(ssPop3LeaveOnServer);
         end;
      end;
   end;

   // connect to server
   StateInfo := 'Connecting ...';
   TLog( LOGID_INFO, StateInfo );
   MailClient := TClientPOP3.Create( self.ReportSubStateInfo,
                                     FServerAlias, FServer, FPort, FUser, FPass,
                                     FDestUserName, FFilterSection);
   try
      // connect to server
      MailClient.Connect;
      if Terminated then exit;
      if not MailClient.Connected then exit;

      // get mails from server
      StateInfo := 'get mails';
      MailClient.GetNewMails( LeaveOnServer, FDestUsername, FFilterSection );

      // disconnect from server
      StateInfo := 'Disconnecting ...';
      TLog( LOGID_INFO, StateInfo );
      MailClient.Disconnect;

   finally
      try FreeAndNil( MailClient ) except end;
   end;
end;

procedure TThreadPop3Fetch.Execute;
var  OK: Boolean;
     s : String;
     UID: Integer;
     LfdServer: Integer;
begin
   // identify server and get its defaults
   Hamster.Config.BeginRead;
   try
      with Hamster.Config.Pop3Servers do begin
         LfdServer := IndexOfAlias( FServerAlias );
         if LfdServer < 0 then begin
            TLog( LOGID_ERROR, 'Unknown POP3 server: ' + FServerAlias );
            exit;
         end;
         if IsDisabled[LfdServer] then begin
            TLog( LOGID_INFO, 'Skip disabled POP3 server: ' + FServerAlias );
            exit;
         end;

         FServer := SrvName[ LfdServer ];
         FPort   := SrvPort[ LfdServer ];
         if (FUser='') and (FPass='') then begin
            FUser := SrvUser[ LfdServer ];
            FPass := SrvPass[ LfdServer ];
         end;

         if FDestusername = ''  then
            FDestUsername  := Settings(LfdServer).GetStr(ssPop3LocalUser);
         if FFilterSection = '' then
            FFilterSection := Settings(LfdServer).GetStr(ssPop3FilterSection);
      end;
   finally Hamster.Config.EndRead end;

   // limit number of concurrent task
   WaitLimitTasks;

   TLog( LOGID_SYSTEM, 'Start' );
   OK := True;

   // check given user
   if FDestUsername='' then UID:=ACTID_ADMIN
                       else UID:=Hamster.Accounts.UserIDOf(FDestUsername);
   if UID=ACTID_INVALID then begin
      TLog( LOGID_ERROR, Format(
          'Unknown username "%s"!', [FDestUsername] ) );
      OK := False;
   end else begin
      if not Hamster.Accounts.HasMailbox( UID ) then begin
         TLog( LOGID_ERROR, Format(
            'User "%s" does not have a mailbox!', [FDestUsername] ) );
         OK := False;
      end;
   end;

   // resolve username and password
   if OK and ( copy(FUser,1,1) = '$' ) and ( FPass = '' ) then begin
      s := FUser;  // script-password ($0..$99)
      if not Hamster.Passwords.UsePassword( s, FUser, FPass ) then begin
         TLog( LOGID_ERROR, Format(
            'Missing username/password for "%s"!', [s] ) );
         OK := False;
      end;
   end;

   // get mails from selected server
   if OK then FetchMailFromServer( LfdServer );

   TLog( LOGID_SYSTEM, 'End');
end;

procedure TThreadPop3Fetch.Terminate;
begin
   if Assigned( MailClient ) then try MailClient.Terminate except end;
   inherited;
end;

constructor TThreadPop3Fetch.Create( AServer, APort, AUser, APass,
                                     ADestUsername, AFilterSection, ALeaveServer: String );
var  s: String;
begin
   s := AServer;
   if ( copy(AUser,1,1) = '$' ) and ( APass = '' ) then s := s + ',' + AUser;
   inherited Create( attTransfer, '{fetchmail ' + s + '}', tftFreeOnTerminate );

   FServerAlias   := AServer;
   FServer        := AServer;
   FPort          := APort;
   FUser          := AUser;
   FPass          := APass;
   FDestUsername  := ADestUsername;
   FFilterSection := AFilterSection;
   FLeaveServer   := ALeaveServer;

   MailClient := nil;
end;

// ------------------------------------------------------ TThreadSmtpSend -----

function CompareByObjectAsInt( List: TStringList;
                               Index1, Index2: Integer ): Integer;
begin
   Result := Integer( List.Objects[Index1] )
           - Integer( List.Objects[Index2] );
end;

procedure TThreadSmtpSend.ConvertToMXFiles( FilePath, FileName: String );
// split given MailOut message into separate files for each recipient domain
var  MailList, MailTemp, RcptList: TStringList;
     i, FirstRcpt, DomainHash: Integer;
     RcptLine, Domain: String;
begin
   if LowerCase( copy( FileName, 1, 2 ) ) = 'mx' then exit; // already done

   MailList := TStringList.Create;
   MailTemp := TStringList.Create;
   RcptList := TStringList.Create;
   try

      try
         // load message
         MailList.LoadFromFile( FilePath + FileName );

         // get recipients and remove them from message
         i := 0;
         FirstRcpt := -1;
         while i < MailList.Count do begin
            RcptLine := MailList[i];
            if copy( RcptLine, 1, 1 ) <> '!' then break;

            if UpperCase( copy(RcptLine,1,9) ) = '!RCPT TO:' then begin
               if FirstRcpt < 0 then FirstRcpt := i;
               DomainHash := StrToCRC32( ExtractMailDomain( RcptLine ) );
               RcptList.AddObject( RcptLine, Pointer(DomainHash) );
               MailList.Delete( i );
            end else begin
               inc( i );
            end;
         end;

         if RcptList.Count > 0 then begin

            // sort by domain hash to have same domains in sequence
            RcptList.CustomSort( CompareByObjectAsInt );

            // prepare one message for each recipient domain
            MailTemp.Text := MailList.Text; // init template
            i := RcptList.Count - 1;
            while i >= 0 do begin

               // add recipient to template and get its domain
               MailTemp.Insert( FirstRcpt, RcptList[i] );
               Domain := ExtractMailDomain( RcptList[i] );

               // save domain mail unless a recipient for same domain follows
               if ( i = 0 ) or
                  ( Domain <> ExtractMailDomain( RcptList[i-1] ) ) then begin

                  MailTemp.SaveToFile( // save template
                     FilePath + 'mx' + inttostr(i+1) + '_' + FileName
                  );
                  MailTemp.Text := MailList.Text; // reset template

               end;

               dec( i );
            end;

            // delete old message
            DeleteFile( FilePath + FileName );

            CounterInc( CounterOutboxM, RcptList.Count - 1 );
            CounterInc( CounterOutboxChk );

         end;

      except
         on E: Exception do begin
            TLog( LOGID_ERROR, 'ERROR in ConvertToMXFiles: ' + E.Message );
         end;
      end;

   finally RcptList.Free; MailTemp.Free; MailList.Free end;
end;

procedure TThreadSmtpSend.BuildSendSnapshot;
var  SR: TSearchRec;
     selFrom, selTo, okFrom, okTo: Boolean;
     rexFrom, rexTo: TPCRE;
     T: TextFile;
     s: String;
     LfdMail: Integer;
begin
   // prevent local smtp-server from adding mails while building snapshot
   CS_LOCK_MAILOUT_ADD.Enter;
   try

      // in MX mode, convert all files to single recipient files
      if FMXMode then begin
         if FindFirst( AppSettings.GetStr(asPathMailOut) + '*.msg',
                       faAnyFile-faDirectory, SR ) = 0 then try
            repeat
               if LowerCase( copy(SR.Name,1,2) ) <> 'mx' then begin
                  // convert to single recipient files
                  ConvertToMXFiles( AppSettings.GetStr(asPathMailOut), SR.Name );
               end;
            until FindNext( SR ) <> 0;
         finally FindClose( SR ) end;
      end;
      
      // build a snapshot of all mails to send
      ToSendSnapshot := TStringList.Create;
      if FindFirst( AppSettings.GetStr(asPathMailOut) + '*.msg',
                    faAnyFile-faDirectory, SR ) = 0 then try
         repeat
            if FMXMode then begin
               if LowerCase( copy(SR.Name,1,2) ) = 'mx' then begin
                  ToSendSnapshot.Add( AppSettings.GetStr(asPathMailOut) + SR.Name );
               end;
            end else begin
               ToSendSnapshot.Add( AppSettings.GetStr(asPathMailOut) + SR.Name );
            end;
         until FindNext( SR ) <> 0;
      finally FindClose( SR ) end;

   finally
      // now we have a reliable list of mailfiles to send, so local
      // smtp-server may continue to add new mails in mail.out
      CS_LOCK_MAILOUT_ADD.Leave;
   end;

   // filter out unselected mails
   selFrom := (FFromSelect <> '') and (FFromSelect <> '.*');
   selTo   := (FToSelect   <> '') and (FToSelect   <> '.*');

   if (ToSendSnapshot.Count > 0) and (selFrom or selTo) then begin

      rexFrom := TPCRE.Create( True, PCRE_CASELESS );
      rexTo   := TPCRE.Create( True, PCRE_CASELESS );

      try
         try
            if selFrom then rexFrom.Compile( PChar(FFromSelect) );
            if selTo   then rexTo  .Compile( PChar(FToSelect  ) );

            LfdMail := 0;
            while LfdMail < ToSendSnapShot.Count do begin

               okFrom := (FFromSelect = '') or (FFromSelect = '.*');
               okTo   := (FToSelect   = '') or (FToSelect   = '.*');

               Assign( T, ToSendSnapShot[LfdMail] );
               Reset ( T );

               try
                  while not eof( T ) do begin

                     ReadLn( T, s );

                     if lowercase(copy(s,1,11))='!mail from:' then begin

                        if selFrom then begin
                           System.Delete( s, 1, 11 );
                           s := TrimWhSpace( s );
                           if rexFrom.Exec( PChar(s), 0 ) then begin
                              TLog( LOGID_DEBUG, Format(
                                 'From-Match(%s,%s) OK', [FFromSelect, s] ));
                              okFrom := True;
                              if okFrom and okTo then break;
                           end else begin
                              TLog( LOGID_DEBUG, Format(
                                 'From-Match(%s,%s) failed', [FFromSelect, s] ));
                           end;
                        end;

                     end else if lowercase(copy(s,1,9))='!rcpt to:' then begin

                        if selTo then begin
                           System.Delete( s, 1, 9 );
                           s := TrimWhSpace( s );
                           if rexTo.Exec( PChar(s), 0 ) then begin
                              TLog( LOGID_DEBUG, Format(
                                 'To-Match(%s,%s) OK', [FToSelect, s] ));
                              okTo := True;
                              if okFrom and okTo then break;
                           end else begin
                              TLog( LOGID_DEBUG, Format(
                                 'To-Match(%s,%s) failed', [FToSelect, s] ));
                           end;
                        end;

                     end else begin
                        break;
                     end;

                  end;

               finally Close( T ) end;

               if okFrom and okTo then inc( LfdMail )
                                  else ToSendSnapShot.Delete( LfdMail );

            end; // while LfdMail < ToSendSnapShot.Count do begin

         except
            on E: Exception do begin
               ToSendSnapshot.Clear; // don't send anything on errors
               TLog( LOGID_ERROR, 'Filter mails error: ' + E.Message );
            end;
         end;
      finally
         rexFrom.Free;
         rexTo.Free;
      end;
   end;
end;

procedure TThreadSmtpSend.SendMail;
var  MailFile, s: String;
begin
   // resolve username and password
   if ( copy(FUser,1,1) = '$' ) and ( FPass = '' ) then begin
      s := FUser;  // script-password ($0..$99)
      if not Hamster.Passwords.UsePassword( s, FUser, FPass ) then begin
         TLog( LOGID_DEBUG, Format(
            'Missing username/password for "%s"!', [s] ) );
      end;
   end;

   // connect to server
   StateInfo := 'Connecting ...';
   TLog( LOGID_INFO, StateInfo );
   SendMailClient := TClientSMTP.Create( self.ReportSubStateInfo,
                                         False, FServerAlias,
                                         FServer, FPort, FUser, FPass );

   try
      SendMailClient.Connect;

      // send mails
      while ToSendSnapshot.Count > 0 do begin
         if Terminated then break;
         if not SendMailClient.Connected then break;

         MailFile := ToSendSnapshot[0];
         ToSendSnapshot.Delete( 0 );

         StateInfo := 'send mail ' + ExtractFilename( MailFile );
         TLog( LOGID_INFO, StateInfo );
         if not SendMailClient.SendMailfile( MailFile ) then break;
         CounterDec( CounterOutboxM );
      end;

      // disconnect from server
      StateInfo := 'Disconnecting ...';
      TLog( LOGID_INFO, StateInfo );
      try SendMailClient.Disconnect; except end;

   finally
      try FreeAndNil( SendMailClient ) except end;
      CounterInc( CounterOutboxChk );
   end;
end;

procedure TThreadSmtpSend.SendMailMX;
var  MailFile: String;
begin
   // prepare delivery client
   SendMailClient := TClientSMTP.Create( self.ReportSubStateInfo,
                                         True, '', '', '', '', '' );

   try
      // send mails
      while ToSendSnapshot.Count > 0 do begin
         if Terminated then break;

         MailFile := ToSendSnapshot[0];
         ToSendSnapshot.Delete( 0 );

         StateInfo := 'Send mail ' + ExtractFilename( MailFile ) + ' by MX';
         TLog( LOGID_INFO, StateInfo );

         if SendMailClient.SendMailfileMX( MailFile ) then begin
            CounterDec( CounterOutboxM );
         end;
      end;

   finally
      try FreeAndNil( SendMailClient ) except end;
      CounterInc( CounterOutboxChk );
   end;
end;

procedure TThreadSmtpSend.Execute;
var  LfdServer: Integer;
begin
   // identify the given server and get its defaults
   if not FMXMode then begin
      Hamster.Config.BeginRead;
      try
         with Hamster.Config.SmtpServers do begin
            LfdServer := IndexOfAlias( FServerAlias );
            if LfdServer < 0 then begin
               TLog( LOGID_ERROR, 'Unknown SMTP server: ' + FServerAlias );
               exit;
            end;
            if IsDisabled[LfdServer] then begin
               TLog( LOGID_INFO, 'Skip disabled SMTP server: ' + FServerAlias );
               exit;
            end;

            FServer := SrvName[ LfdServer ];
            FPort   := SrvPort[ LfdServer ];
            if (FUser='') and (FPass='') then begin
               FUser := SrvUser[ LfdServer ];
               FPass := SrvPass[ LfdServer ];
            end;
         end;
      finally Hamster.Config.EndRead end;
   end;

   // limit number of concurrent task
   WaitLimitTasks;

   TLog( LOGID_SYSTEM, 'Start' );

   // block other smtpsend-threads
   CS_LOCK_MAILOUT_USE.Enter;
   try
   
      // build a snapshot of all mails to send
      StateInfo := 'checking mails to send';
      BuildSendSnapshot;

      // send mails
      if ToSendSnapshot.Count > 0 then begin
         if FMXMode then SendMailMX  // send directly to recipient's server
                    else SendMail;   // send to given server
      end else begin
         TLog( LOGID_INFO, 'No mails to send.' );
      end;

   finally
      // release other smtpsend-threads
      CS_LOCK_MAILOUT_USE.Leave;
   end;

   ToSendSnapshot.Free;
   TLog( LOGID_SYSTEM, 'End' );
end;

procedure TThreadSmtpSend.Terminate;
begin
   if Assigned( SendMailClient ) then try SendMailClient.Terminate except end;
   inherited;
end;

constructor TThreadSmtpSend.Create( AMXMode: Boolean;
                                    AServer, APort, AUser, APass: String;
                                    AFromSelect, AToSelect: String );
begin
   if AMXMode then
      inherited Create( attTransfer, '{sendmail MX}', tftFreeOnTerminate )
   else
      inherited Create( attTransfer, '{sendmail ' + AServer + '}', tftFreeOnTerminate );

   FMXMode := AMXMode;

   FServerAlias := AServer;
   FServer      := AServer;
   FPort        := APort;
   FUser        := AUser;
   FPass        := APass;
   FFromSelect  := AFromSelect;
   FToSelect    := AToSelect;

   SendMailClient := nil;
end;


// ------------------------------------------------------ TThreadNewsJobs -----

procedure TThreadNewsJobs.NukePostedArticle( const PostFile: String;
                                             ArtText: String;
                                             const DestGroup, Reason, DestSrv: String );
var  Newsgroups, MessageID, s: String;
     DestHdl, ArtNo, i       : Integer;
     Art                     : TMess;
begin
   // delete posted article, even if post failed
   DeleteFile( PostFile );

   // prepare article to post a copy of it to internal.misc
   Art := TMess.Create;
   Art.FullText := ArtText;

   Newsgroups := Art.HeaderValueByNameSL( 'Newsgroups:' );
   Art.SetHeaderSL( 'Newsgroups:', DestGroup, 'X-Post-Newsgroups:' );

   MessageID := Art.HeaderValueByNameSL( 'Message-ID:' );
   if MessageID='' then MessageID := MidGenerator( 'posted.without.mid.invalid' );
   MessageID := copy(MessageID,1,1) + 'archive.' + copy(MessageID,2,MaxInt);
   Art.SetHeaderSL( 'Message-ID:', MessageID, 'X-Post-Message-ID:' );

   Art.GetHeaderLine( 'Date:', i );
   if i<0 then Art.InsertHeaderSL( 0, HDR_NAME_DATE, DateTimeGMTToRfcDateTime(NowGMT,NowRfcTimezone) );

   s := '[Hamster: archive-copy of posted article (' + Newsgroups + ')]' + CRLF
      + 'Result: ' + Reason + CRLF
      + 'Server: ' + FServerAlias + CRLF
      + CRLF;
   Art.BodyText := s + Art.BodyText;

   Art.DelHeaderML( 'Lines:' );
   Art.InsertHeaderSL( 0, HDR_NAME_LINES, inttostr( Art.BodyLines ) );

   Art.SetHeaderSL( HDR_NAME_DISTRIBUTION, 'local', '' );

   ArtText := Art.FullText;
   Art.Free;

   // add timestamp and post-result
   s := 'Received=' + DateTimeToTimeStamp(Now)
      + 'Server='   + DestSrv + ' '
      + 'Result='   + ReplaceChar( Reason, ' ', '_' ) + ' '
      + 'UID=' + IntToStr( GetUID(0) );
   ArtText := HDR_NAME_X_HAMSTER_INFO + ': ' + s + #13#10 + ArtText;

   // save in DestGroup
   DestHdl := Hamster.ArticleBase.Open( DestGroup );
   if DestHdl>=0 then begin
      ArtNo := Hamster.ArticleBase.WriteArticle( DestHdl, 0, ArtText );
      Hamster.ArticleBase.Close( DestHdl );
      Hamster.NewsHistory.AddEntryDupes( MessageID, StrToCRC32(DestGroup), ArtNo, 0 );
   end else begin
      TLog( LOGID_ERROR, Format(
            'Could not open %s to archive posted article', [DestGroup] ));
   end;
end;

procedure TThreadNewsJobs.PostArticle( const PostFile: String );
var  Art       : TMess;
     SrvResult : String;
     DestGroup : String;
begin
   if not FileExists( PostFile ) then exit;

   Art := TMess.Create;
   try
      Art.LoadFromFile( PostFile );

      CS_LOCK_NEWSOUT.Enter;
      try
         if NewsClient.PostArticle( PostFile, Art.FullText, SrvResult ) then begin
            // POST successful
            DestGroup := Hamster.Config.Newsgroups.HamGroupName(hsHamGroupPostOk);
            TLog( LOGID_INFO, Format(
               'Posting OK: %s (%s)', [SrvResult,ExtractFileName(PostFile)]));
         end else begin
            // POST failed
            DestGroup := Hamster.Config.Newsgroups.HamGroupName(hsHamGroupPostErrors);
            with TFileStream.Create( AppSettings.GetStr(asPathNewsErr)
                                   + ExtractFileName(Postfile), fmCreate ) do try
               Write( Art.FullText[1], Length(Art.FullText) )
            finally free end;
            TLog( LOGID_WARN, Format(
               'Posting failed: %s (%s)', [SrvResult,ExtractFileName(PostFile)]));
         end;

         // save article in DestGroup and delete it from Post-group/-list
         NukePostedArticle( PostFile, Art.FullText, DestGroup, SrvResult, FServerAlias );
         CounterDec( CounterOutboxN   );
         CounterInc( CounterOutboxChk );
         
      finally CS_LOCK_NEWSOUT.Leave end;

   finally Art.Free end;
end;

procedure TThreadNewsJobs.Execute;
var  FServer, FPort, FUser, FPass: String;
     GroupName, JobPar, LastGroup: String;
     LfdServer, JobType, GroupHandle, DelayCount: Integer;
     IsConnected, FeedMode, OK: Boolean;
begin
   // identify server and get its defaults
   Hamster.Config.BeginRead;
   try
      with Hamster.Config.NntpServers do begin
         LfdServer := IndexOfAlias( FServerAlias );
         if LfdServer < 0 then begin
            TLog( LOGID_ERROR, 'Unknown NNTP server: ' + FServerAlias );
            exit;
         end;
         if IsDisabled[LfdServer] then begin
            TLog( LOGID_INFO, 'Skip disabled NNTP server: ' + FServerAlias );
            exit;
         end;

         FServer := SrvName[ LfdServer ];
         FPort   := SrvPort[ LfdServer ];
         if (FUser='') and (FPass='') then begin
            FUser := SrvUser[ LfdServer ];
            FPass := SrvPass[ LfdServer ];
         end;
      end;
   finally Hamster.Config.EndRead end;

   if not Hamster.Config.NntpServers.UseInc( LfdServer ) then begin
      // the jobs for this server are already executed by other threads
      Terminate;
      exit;
   end;

   try
      // limit number of concurrent task
      WaitLimitTasks;

      NewsClient  := nil;
      IsConnected := False;

      try
         TLog( LOGID_SYSTEM, 'Start' );

         LastGroup   := '';
         DelayCount  := 0;

         repeat
            // stop download?
            if Terminated then break;
            if Assigned(NewsClient) and NewsClient.Terminated then break;

            // get (next) job
            if not Hamster.NewsJobs.JobList.JobGet( FServerAlias, JobType, JobPar ) then break;

            // connect to server
            if not IsConnected then begin
               StateInfo := 'Connecting ...';
               TLog( LOGID_INFO, StateInfo );
               FeedMode := (JobType = JOBTYPE_NEWSFEED);
               NewsClient := TClientNNTP.Create( self.ReportSubStateInfo,
                                                 FServerAlias,
                                                 FServer, FPort, FUser, FPass,
                                                 FeedMode );
               NewsClient.Connect;
               if not NewsClient.Connected then begin
                  if JobType=JOBTYPE_NEWSPULL then begin
                     Hamster.NewsJobs.JobList.JobDelay( FServerAlias, JobType, JobPar, 3 );
                  end;
                  break;
               end;
               StateInfo := 'Connected';
               IsConnected := True;
            end;

            // execute job
            case JobType of

               JOBTYPE_NEWSFEED: begin
                  // feed articles of group to remote server
                  GroupName := JobPar;
                  StateInfo := 'feed ' + GroupName;
                  TLog( LOGID_INFO, StateInfo );
                  GroupHandle := Hamster.ArticleBase.Open( GroupName );
                  if GroupHandle >= 0 then try
                     NewsClient.FeedNewArticlesOfGroup( GroupName, GroupHandle );
                  finally
                     Hamster.ArticleBase.Close( GroupHandle );
                  end;
               end;

               JOBTYPE_SRVINFOS: begin
                  // get infos for new servers and new groups on old servers
                  StateInfo := 'get server-info';
                  TLog( LOGID_INFO, StateInfo );
                  NewsClient.GetServerInfos;
               end;

               JOBTYPE_GETBYMID: begin
                  // get articles by list of Message-IDs
                  StateInfo := 'get list of MIDs';
                  TLog( LOGID_INFO, StateInfo );
                  NewsClient.GetListOfMIDs;
               end;

               JOBTYPE_NEWSPOST: begin
                  StateInfo := 'post ' + ExtractFilename(JobPar);
                  TLog( LOGID_INFO, StateInfo );
                  PostArticle( JobPar ); // post article
               end;

               JOBTYPE_NEWSPULL: begin
                  // load new articles for group
                  GroupName := JobPar;
                  StateInfo := 'pull ' + GroupName;
                  TLog( LOGID_INFO, StateInfo );

                  OK := True;
                  if Hamster.ArticleBase.UseCount( GroupName ) > 0 then begin
                     // Avoid ineffective downloads of groups, that are already "in
                     // use" by other (pull-) threads. Due to synchronized access on
                     // group files, they would block each other often and therefore
                     // delay download.
                     if (GroupName=LastGroup) or (DelayCount>5) then begin
                        // no alternatives left/found
                        OK := True;
                     end else begin
                        // Put job back into job-list as (up to) 4th job for server.
                        OK := False;
                        inc( DelayCount );
                        TLog( LOGID_DEBUG, Format(
                           'Group %s already in use, trying later.', [GroupName] ));
                        Hamster.NewsJobs.JobList.JobDelay( FServerAlias, JobType, JobPar, 3 );
                     end;
                  end;

                  if OK then begin
                     GroupHandle := Hamster.ArticleBase.Open( GroupName );
                     if GroupHandle >= 0 then try
                        DelayCount := 0;
                        NewsClient.GetNewArticlesForGroup( GroupName, GroupHandle );
                     finally
                        Hamster.ArticleBase.Close( GroupHandle );
                        if not NewsClient.Connected then begin
                           Hamster.NewsJobs.JobList.JobDelay( FServerAlias, JobType, JobPar, 3 );
                        end;
                     end;
                  end;

                  LastGroup := GroupName;
               end;

               else TLog( LOGID_WARN,
                  'Unknown job-type' + ': ' + inttostr(JobType) );
            end;

         until not NewsClient.Connected;

      finally
         if Assigned( NewsClient ) then begin
            // disconnect from server
            StateInfo := 'Disconnecting ...';
            TLog( LOGID_INFO, StateInfo );
            if IsConnected then try NewsClient.Disconnect; except end;
            try FreeAndNil( NewsClient ) except end; 
         end else begin
            TLog( LOGID_INFO, Format(
               'News-jobs for %s abandoned (no jobs)', [FServerAlias] ) );
         end;

         Hamster.NewsHistory.SaveToFile;
         TLog( LOGID_SYSTEM, 'End' );
      end;

   finally
      Hamster.Config.NntpServers.UseDec( LfdServer );
   end;
end;

procedure TThreadNewsJobs.Terminate;
begin
   if Assigned( NewsClient ) then try NewsClient.Terminate except end;
   inherited;
end;

constructor TThreadNewsJobs.Create( AServerAlias: String );
begin
   inherited Create( attTransfer, '{newsjobs ' + AServerAlias + '}', tftFreeOnTerminate );

   FServerAlias := AServerAlias;
   NewsClient   := nil;
end;

// ---------------------------------------------- TThreadWebChannelReader -----

procedure TThreadWebChannelReader.SetDefaults( var chSubject, chFrom: String );
begin
   if FSubject <> '' then begin
      if FSubject[ length(FSubject) ] = '*' then begin
         chSubject := copy( FSubject, 1, length(FSubject)-1 ) + chSubject;
      end else begin
         chSubject := FSubject;
      end;
   end;
   if FFrom <> '' then begin
      if FFrom[ length(FFrom) ] = '*' then begin
         chFrom := copy( FFrom, 1, length(FFrom)-1 ) + chFrom;
      end else begin
         chFrom := FFrom;
      end;
   end;
end;

procedure TThreadWebChannelReader.SaveXml( reader: TWebChannelReader );
var  sl: TStringList;
     filename, filepath, s: String;
     i: Integer;
begin
   try
      if length( reader.WebLink ) = 0 then exit;
      if length( reader.WebText ) = 0 then exit;

      filename := '';
      s := reader.WebLink;
      if copy( s, 1, 7 ) = 'http://' then System.Delete( s, 1, 7 );
      if copy( s, length(s), 1 ) = '/' then System.Delete( s, length(s), 1 );
      for i := 1 to length( s ) do begin
         if s[i] = '.' then begin
            filename := filename + '.';
         end else if s[i] = '/' then begin
            filename := filename + '!';
         end else if ( s[i] = '%' ) or ( s[i] = '!' ) or ( not IsFilename( s[i] ) ) then begin
            filename := filename + '%' + IntToHex( ord(s[i]), 2 );
         end else begin
            filename := filename + s[i];
         end;
      end;
      if length( filename ) = 0 then exit;
      filename := filename + '.txt';

      filepath := AppSettings.GetStr( asPathLogs ) + 'rss\';
      ForceDirectories( filepath );

      sl := TStringList.Create;
      try
         sl.Text := reader.WebText;
         sl.SaveToFile( filepath + filename );
      finally sl.Free end;
   except
      on ex: Exception do TLog( LOGID_WARN, 'Error saving feed file: ' + ex.Message );
   end;
end;

procedure TThreadWebChannelReader.CreateArticleMode0(reader: TWebChannelReader; channel: TWebChannel);
// mode 0: all items in one article (if any item has changed)
//         also used for channel errors and normal html files
var  chSubject, chFrom, chText, chMid: String;
     chTime: TDateTime;
     msg: TMess;
begin
   try
      msg := TMess.Create;
      try

         // report error
         if channel = nil then begin
            TLog( LOGID_WARN, reader.Error );
            exit;
         end;

         // convert complete channel into article
         try
            reader.GetAllItemsAsText( channel, chSubject, chFrom, chText, chTime );
         except
            on ex: Exception do begin
               chText := chText + #13#10#13#10 + '-- ' + #13#10
                       + 'Error converting channel (0):' + #13#10 + ex.Message;
            end;
         end;
         SetDefaults( chSubject, chFrom );

         // prepare a content based Message-ID so duplicates will not be stored
         chMid := '<' + MD5toHex( Md5OfStr( chText ) ) + '@webreader>';

         // create and store article
         msg.BodyText := chText;
         msg.AddHeaderSL( HDR_NAME_NEWSGROUPS, FGroup );
         msg.AddHeaderSL( HDR_NAME_SUBJECT, chSubject );
         msg.AddHeaderSL( HDR_NAME_FROM, chFrom );
         msg.AddHeaderSL( HDR_NAME_DATE, uDateTime.DateTimeLocalToRfcDateTime( chTime, NowRfcTimezone ) );
         msg.AddHeaderSL( HDR_NAME_LINES, IntToStr( msg.BodyLines ) );
         msg.AddHeaderSL( HDR_NAME_MESSAGE_ID, chMid );
         msg.AddHeaderSL( HDR_NAME_DISTRIBUTION, 'local' );
         if Contains8BitChars( chText ) then begin
            msg.AddHeaderSL( HDR_NAME_MIME_VERSION, '1.0' );
            msg.AddHeaderSL( HDR_NAME_CONTENT_TYPE, 'text/plain; charset="utf-8"' );
            msg.AddHeaderSL( HDR_NAME_CONTENT_TRANSFER_ENCODING, '8bit' );
         end;
         if channel = nil then begin
            msg.AddHeaderSL( HDR_NAME_X_HAMSTER_FEED, 'unknown' + ' ' + reader.WebLink );
         end else begin
            msg.AddHeaderSL( HDR_NAME_X_HAMSTER_FEED, channel.WebChannelVersionStr + ' ' + reader.WebLink );
         end;

         if SaveArticle( msg, '', -1, 'none-rss', false, '', actIgnore ) then begin
            TLog( LOGID_INFO, 'New message.' );
         end else begin
            TLog( LOGID_INFO, 'Not modified.' );
         end;

      finally
         msg.Free;
      end;
   except
      on ex: Exception do begin
         TLog( LOGID_WARN, 'Error creating article (mode 0): ' + ex.Message );
         raise;
      end;
   end;
end;

procedure TThreadWebChannelReader.CreateArticleMode13(reader: TWebChannelReader; channel: TWebChannel);
// mode 1: one item per article (only new or modified ones)
// mode 3: like 1, but additionally attach linked HTML file
var  chSubject, chFrom, chText, chMid, chHtml, chType, chSep, s: String;
     chTime: TDateTime;
     msg: TMess;
     index, counter, i, idxMax: Integer;
begin
   try
      // always use mode 0 to report errors or converted html files
      if channel = nil then begin
         CreateArticleMode0( reader, channel );
         exit;
      end;
      if (channel.WebChannelVersion = html) and (FMode <> 3) then begin
         CreateArticleMode0( reader, channel );
         exit;
      end;

      msg := TMess.Create;
      try
         counter := 0;

         if FMode = 3 then idxMax := 0 // dummy item (=HTML text)
                      else idxMax := channel.Count - 1;

         for index := 0 to idxMax do begin

            // convert channel item into article
            try
               if FMode = 1 then begin
                  reader.GetSingleItemAsText( channel, index,
                                              chSubject, chFrom, chText, chTime,
                                              False, chHtml, chType );
                  chHtml := '';
               end else begin // FMode = 3
                  reader.GetSingleItemAsText( channel, index,
                                              chSubject, chFrom, chText, chTime,
                                              True, chHtml, chType );
               end;
            except
               on ex: Exception do begin
                  chText := chText + #13#10#13#10 + '-- ' + #13#10
                          + 'Error converting channel (1):' + #13#10 + ex.Message;
               end;
            end;
            SetDefaults( chSubject, chFrom );

            // create a content based Message-ID so duplicates will not be stored
            s := MD5toHex( Md5OfStr( chText ) );
            chMid := '<' + s + '@webreader' + inttostr(FMode) + '>';
            chSep := '_' + s;

            // create and store article
            msg.FullText := '';
            msg.AddHeaderSL( HDR_NAME_NEWSGROUPS, FGroup );
            msg.AddHeaderSL( HDR_NAME_SUBJECT, chSubject );
            msg.AddHeaderSL( HDR_NAME_FROM, chFrom );
            msg.AddHeaderSL( HDR_NAME_DATE, uDateTime.DateTimeLocalToRfcDateTime( chTime, NowRfcTimezone ) );
            msg.AddHeaderSL( HDR_NAME_MESSAGE_ID, chMid );
            msg.AddHeaderSL( HDR_NAME_DISTRIBUTION, 'local' );
            msg.AddHeaderSL( HDR_NAME_X_HAMSTER_FEED, channel.WebChannelVersionStr + ' ' + reader.WebLink );

            if (FMode = 1) or (chHtml = '') then begin

               if Contains8BitChars( chText ) then begin
                  msg.AddHeaderSL( HDR_NAME_MIME_VERSION, '1.0' );
                  msg.AddHeaderSL( HDR_NAME_CONTENT_TYPE, 'text/plain; charset="utf-8"' );
                  msg.AddHeaderSL( HDR_NAME_CONTENT_TRANSFER_ENCODING, '8bit' );
               end;

               msg.BodyText := chText;
               
            end else begin

               msg.AddHeaderSL( HDR_NAME_MIME_VERSION, '1.0' );
               msg.AddHeaderSL( HDR_NAME_CONTENT_TYPE, 'multipart/alternative; boundary="=' + chSep + '"' );

               msg.AddBodyLine( '--=' + chSep );

               if Contains8BitChars( chText ) then begin
                  msg.AddBodyLine( HDR_NAME_CONTENT_TYPE + ': ' + 'text/plain; charset="utf-8"' );
                  msg.AddBodyLine( HDR_NAME_CONTENT_TRANSFER_ENCODING + ': ' + '8bit' );
               end else begin
                  msg.AddBodyLine( HDR_NAME_CONTENT_TYPE + ': ' + 'text/plain; charset="us-ascii"' );
                  msg.AddBodyLine( HDR_NAME_CONTENT_TRANSFER_ENCODING + ': ' + '7bit' );
               end;
               msg.AddBodyLine( '' );

               msg.AddBodyText( chText );

               msg.AddBodyLine( '--=' + chSep );

               i := Pos( 'charset=', chType );
               s := '';
               if i > 0 then begin
                  s := copy( chType, i+8, MaxInt );
                  if copy( s, 1, 1 ) = '"' then System.Delete( s, 1, 1 );
                  i := Pos( '"', s );
                  if i > 0 then s := copy( s, 1, i-1 );
                  i := Pos( ';', s );
                  if i > 0 then s := copy( s, 1, i-1 );
                  s := Trim( s );
               end;
               if s = '' then s := 'ISO-8859-1';

               msg.AddBodyLine( HDR_NAME_CONTENT_TYPE + ': ' + 'text/html; charset="' + s + '"' );
               msg.AddBodyLine( HDR_NAME_CONTENT_TRANSFER_ENCODING + ': ' + '8bit' );
               msg.AddBodyLine( '' );

               msg.AddBodyText( chHtml );

               msg.AddBodyLine( '--=' + chSep + '--' );
               
            end;

            msg.AddHeaderSL( HDR_NAME_LINES, IntToStr( msg.BodyLines ) );
            
            if SaveArticle( msg, '', -1, 'none-rss', false, '', actIgnore ) then begin
               Inc( Counter );
            end;

         end;

         if counter > 0 then TLog( LOGID_INFO, 'New messages: ' + IntToStr( counter ) )
                        else TLog( LOGID_INFO, 'Not modified.' );

      finally
         msg.Free;
      end;

   except
      on ex: Exception do begin
         TLog( LOGID_WARN, 'Error creating article (mode 1): ' + ex.Message );
         raise;
      end;
   end;
end;

procedure TThreadWebChannelReader.CreateArticleMode2(reader: TWebChannelReader; channel: TWebChannel);
// mode 2: only modified items in one article
var  chSubject, chFrom, chText, chMid, grpName, chHtml, chType: String;
     chTime: TDateTime;
     msg: TMess;
     index, grpHash, artNo: Integer;
     indexes: TList;
     mids: TStringList;
     newMsg: Boolean;
begin
   try
      // always use mode 0 to report errors or html files
      if (channel = nil) or (channel.WebChannelVersion = html) then begin
         CreateArticleMode0( reader, channel );
         exit;
      end;

      msg := TMess.Create;
      indexes := TList.Create;
      mids := TStringList.Create;
      try
         grpHash := StrToCRC32( FGroup );
         newMsg  := False;

         for index := 0 to channel.Count - 1 do begin

            // convert channel item into article
            try
               reader.GetSingleItemAsText( channel, index,
                                           chSubject, chFrom, chText, chTime,
                                           False, chHtml, chType );
            except
               on ex: Exception do begin
                  chText := chText + #13#10#13#10 + '-- ' + #13#10
                          + 'Error converting channel (2):' + #13#10 + ex.Message;
               end;
            end;
            SetDefaults( chSubject, chFrom );

            // create a content based Message-ID to recognize duplicates
            chMid := '<' + MD5toHex( Md5OfStr( chText ) ) + '@webreader2item>';

            // do we have this item already?
            if not Hamster.NewsHistory.ContainsMID( chMid ) then begin
               indexes.Add( Pointer(index) );
               mids.Add( chMid );
            end;

         end;

         if indexes.Count > 0 then begin

            // convert new or modified channel items into article
            reader.GetSelectedItemsAsText( channel, indexes, chSubject, chFrom, chText, chTime );
            SetDefaults( chSubject, chFrom );

            // prepare a content based Message-ID so duplicates will not be stored
            chMid := '<' + MD5toHex( Md5OfStr( chText ) ) + '@webreader2>';

            // create and store article
            msg.BodyText := chText;
            msg.AddHeaderSL( HDR_NAME_NEWSGROUPS, FGroup );
            msg.AddHeaderSL( HDR_NAME_SUBJECT, chSubject );
            msg.AddHeaderSL( HDR_NAME_FROM, chFrom );
            msg.AddHeaderSL( HDR_NAME_DATE, uDateTime.DateTimeLocalToRfcDateTime( chTime, NowRfcTimezone ) );
            msg.AddHeaderSL( HDR_NAME_LINES, IntToStr( msg.BodyLines ) );
            msg.AddHeaderSL( HDR_NAME_MESSAGE_ID, chMid );
            msg.AddHeaderSL( HDR_NAME_DISTRIBUTION, 'local' );
            if Contains8BitChars( chText ) then begin
               msg.AddHeaderSL( HDR_NAME_MIME_VERSION, '1.0' );
               msg.AddHeaderSL( HDR_NAME_CONTENT_TYPE, 'text/plain; charset="utf-8"' );
               msg.AddHeaderSL( HDR_NAME_CONTENT_TRANSFER_ENCODING, '8bit' );
            end;
            msg.AddHeaderSL( HDR_NAME_X_HAMSTER_FEED, channel.WebChannelVersionStr + ' ' + reader.WebLink );

            if SaveArticle( msg, '', -1, 'none-rss', false, '', actIgnore ) then begin

               // create additional history entries for contained channel items
               Hamster.NewsHistory.LocateMID( chMid, grpName, artNo );
               for index := 0 to mids.Count - 1 do begin
                  Hamster.NewsHistory.AddEntryFirst( mids[index], grpHash, artNo, 0 );
               end;

               newMsg := True;
               TLog( LOGID_INFO, 'New messages: ' + IntToStr( indexes.Count ) );

            end;

         end;

         if not newMsg then TLog( LOGID_INFO, 'Not modified.' );

      finally
         mids.Free;
         indexes.Free;
         msg.Free;
      end;

   except
      on ex: Exception do begin
         TLog( LOGID_WARN, 'Error creating article (mode 2): ' + ex.Message );
         raise;
      end;
   end;
end;

procedure TThreadWebChannelReader.Execute;
var  reader: TWebChannelReader;
     lockKey: String;
     i: Integer;
begin
   // limit number of concurrent tasks
   WaitLimitTasks;

   TLog( LOGID_SYSTEM, 'Start' );
   lockKey := 'system.' + FUrl;
   HscGlobal.Enter( lockKey, nil, True );
   try

      // check feed URL
      if FIsFile then begin
         if not FileExists( FFeed ) then begin
            TLog( LOGID_ERROR, Format( 'Feed file not found: "%s"', [FFeed] ) );
            exit;
         end;
      end else begin
         if ( copy( FFeed, 1, 7 ) <> 'http://' ) and
            ( copy( FFeed, 1, 8 ) <> 'https://' ) then begin
            if Pos( ')http://', FFeed ) = 0 then begin
               TLog( LOGID_ERROR, Format( '"http://" URL expected: "%s"', [FFeed] ) );
               exit;
            end;
         end;
      end;

      // check group
      Hamster.Config.BeginRead;
      try
         if FGroup = '' then begin
            FGroup := Hamster.Config.Newsgroups.HamGroupName( hsHamGroupUnknownGroup );
         end;
         if Hamster.Config.Newsgroups.IndexOf( FGroup ) < 0 then begin
            TLog( LOGID_ERROR, Format( 'Unknown group: "%s"', [FGroup] ) );
            exit;
         end;
      finally Hamster.Config.EndRead end;

      // load and process feed
      try
         if FIsFile then begin
            reader := TWebChannelReader.Create( wcitFile, FFeed, 'mode ' + IntToStr(FMode),
                                                '', 0, '', '' );
         end else begin
            reader := TWebChannelReader.Create( wcitUrl,  FFeed, 'mode ' + IntToStr(FMode),
                                                FProxyServer, FProxyPort, FProxyUser, FProxyPass );
         end;

         try
            // create article(s) from feed
            if reader.Modified then begin

               try
                  // save loaded source text
                  if FSaveXml then SaveXml( reader );

                  if reader.Channels.Count = 0 then begin
                     CreateArticleMode0( reader, nil );
                  end else begin
                     for i := 0 to reader.Channels.Count - 1 do begin
                        case FMode of
                            1 : CreateArticleMode13( reader, reader.Channels.Items[i] );
                            2 : CreateArticleMode2 ( reader, reader.Channels.Items[i] );
                            3 : CreateArticleMode13( reader, reader.Channels.Items[i] );
                           else CreateArticleMode0 ( reader, reader.Channels.Items[i] );
                        end;
                     end;
                  end;

               except
                  on ex: Exception do begin
                     TLog( LOGID_ERROR, 'Error converting feed: ' + ex.Message );
                  end;
               end;

            end else begin
               TLog( LOGID_INFO, 'Not modified (304).' );
            end;

         finally reader.Free end;

      except
         on ex: Exception do begin
            TLog( LOGID_ERROR, 'Error reading/processing feed: ' + ex.Message );
         end;
      end;

   finally
      HscGlobal.Leave( lockKey, True );
      TLog( LOGID_SYSTEM, 'End' );
   end;
end;

constructor TThreadWebChannelReader.Create( const AFeed, AGroup, ASubject, AFrom: String;
                                            const AMode: Integer;
                                            const AIsFile, ASaveXml: Boolean;
                                            const AProxyServer: String;
                                            const AProxyPort: Integer;
                                            const AProxyUser, AProxyPass: String );
var  url, u, p: String;
begin
   SplitUrl( AFeed, url, u, p );
   inherited Create( attTransfer,
                     '{webreader' + inttostr(AMode) + ' ' + url + '}',
                     tftFreeOnTerminate );

   FFeed    := AFeed;
   FUrl     := url;
   FGroup   := AGroup;
   FSubject := ASubject;
   FFrom    := AFrom;
   FMode    := AMode and $7;
   FIsFile  := AIsFile;
   FSaveXml := ASaveXml;

   FProxyServer := AProxyServer;
   FProxyPort   := AProxyPort;
   FProxyUser   := AProxyUser;
   FProxyPass   := AProxyPass;
end;


// ------------------------------------------------ TThreadLookupHostAddr -----

procedure TThreadLookupHostAddr.Execute;
begin
   try
      TLog( LOGID_DETAIL, 'Start' );
      FHostAddr := nAddrToStr( LookupHostAddr( FHostName ) );
      TLog( LOGID_DETAIL, 'End, Result=' + FHostAddr );
      Terminate;
   except
      on ex: Exception do TLog( LOGID_ERROR, 'Error: ' + ex.Message );
   end;
end;

constructor TThreadLookupHostAddr.Create( const AHostName: String );
begin
   inherited Create( attTransfer, '{dns lookup ' + AHostName + '}', tftFreeByCode );
   FHostName := AHostName;
   FHostAddr := '';
end;


// ----------------------------------------------- TThreadLookupHostAddrs -----

procedure TThreadLookupHostAddrs.Execute;
var  i: Integer;
     s: String;
     SubThread: TThreadLookupHostAddr;
     SubThreads: TList;
     SubThreadsLeft: Boolean;
begin
   TLog( LOGID_DETAIL, 'Start' );
   try

      try
         // prepare empty results
         FHostAddrs.Clear;
         for i := 0 to FHostNames.Count - 1 do FHostAddrs.Add( '' );

         SubThreads := TList.Create;
         try

            // start threads
            SetStateInfo( 'Start lookup threads' );
            for i := 0 to FHostNames.Count - 1 do begin

               if Assigned( FDnsCache ) then s := FDnsCache.Get( FHostNames[i] )
                                        else s := '';

               if s = '' then begin
                  TLog( LOGID_DEBUG, 'Start lookup: ' + FHostNames[i] );
                  SubThread := TThreadLookupHostAddr.Create( FHostNames[i] );
                  SubThreads.Add( SubThread );
                  SubThread.Resume;
               end else begin
                  TLog( LOGID_DEBUG, 'Result from cache: ' + FHostNames[i] + ' -> ' + s );
                  FHostAddrs[i] := s;
                  SubThreads.Add( nil );
               end;
               
            end;

            // wait for all threads to finish
            SetStateInfo( 'Wait for results' );
            repeat

               Sleep( 50 );
            
               SubThreadsLeft := False;

               for i := 0 to FHostNames.Count - 1 do begin
                  if Assigned( SubThreads[i] ) then begin

                     SubThread := TThreadLookupHostAddr( SubThreads[i] );
                     if SubThread.Terminated then begin

                        // get result, then free
                        FHostAddrs[i] := SubThread.HostAddr;
                        SubThreads[i] := nil;
                        try SubThread.Free except end;

                        // cache result
                        if Assigned( FDnsCache ) and (FHostAddrs[i] <> '') then begin
                           FDnsCache.Add( FHostNames[i], FHostAddrs[i] );
                        end;

                        TLog( LOGID_DEBUG, 'Result: ' + FHostNames[i] + ' -> ' + FHostAddrs[i] );

                     end else begin
                        SubThreadsLeft := True;
                     end;

                  end;
               end;

               if not SubThreadsLeft then Terminate;

            until Terminated or not SubThreadsLeft;

            SetStateInfo( 'Done' );

         finally
            // clean up any orphaned threads if terminated
            for i := 0 to FHostNames.Count - 1 do begin
               if Assigned( SubThreads[i] ) then begin
                  try TThreadLookupHostAddr( SubThreads[i] ).Terminate except end;
                  try TThreadLookupHostAddr( SubThreads[i] ).Free except end;
               end;
            end;
            SubThreads.Free;
         end;

      except
         on ex: Exception do begin
            TLog( LOGID_ERROR, 'Error: ' + ex.Message );
         end;
      end;

   finally
      TLog( LOGID_DETAIL, 'End' );
   end;
end;

constructor TThreadLookupHostAddrs.Create( const AHostNames, AHostAddrs: TStringList;
                                           const ADnsCache: TKeyValueCache = nil );
begin
   inherited Create( attTransfer,
                     '{dns lookup ' + inttostr(AHostNames.Count) + ' hosts}',
                     tftFreeByCode );

   FHostNames := AHostNames;
   FHostAddrs := AHostAddrs;
   FDnsCache  := ADnsCache;
end;

end.
