// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit tScript; // Threads which runs scripts

// ----------------------------------------------------------------------------
// Contains the thread which executes scripts and the Hamster-related
// enhancements of the script-engine.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses Windows, SysUtils, Classes, uType, tBase, SyncObjs, cSyncObjects;

type
   TThreadExecuteScript = class( TBaseThread )
      private
         FScriptFile: String;
         FParamsText: String;
         FStopEvent : TEvent;
         FSilent    : Boolean;
         FOutputBuffer: TThreadStringList;

      protected
         procedure Execute; override;

      public
         property OutputBuffer: TThreadStringList read FOutputBuffer;

         procedure Terminate; override;

         constructor Create( const AScriptFile, AParamsText: String;
                             const FreeType: TThreadFreeTypes;
                             const ASilent: Boolean;
                             const DoBufferOutput: Boolean );
         destructor Destroy; override;
   end;

function StartNewScript( const ScriptFile, ParamsText: String;
                         const WaitForEnd: Boolean;
                         out   ScriptThread: TThreadExecuteScript;
                         const FreeType: TThreadFreeTypes;
                         const Silent: Boolean;
                         const DoBufferOutput: Boolean ): Integer; overload;
function StartNewScript( const ScriptFile, ParamsText: String;
                         const WaitForEnd: Boolean;
                         const Silent: Boolean ): Integer; overload;

implementation

uses uConst, uConstVar, uVar, cLogFileHamster, cHamster,
     cHscHelpers, cHscEngine, cHscHamster, uTools
     {$IFDEF H_NEED_VARIANTS} , Variants {$ENDIF} ;

// ------------------------------------------------- TThreadExecuteScript -----

procedure TThreadExecuteScript.Terminate;
begin
   inherited;
   FStopEvent.SetEvent;
end;

procedure TThreadExecuteScript.Execute;
var  MyScriptList: TStringList;
     MyScriptName: String;
     Result      : THscVariant;
     OurRasConn  : Boolean;
begin
   TLog( iif( FSilent, LOGID_DEBUG, LOGID_SYSTEM ), 'Start' );

   StateInfo := 'prepare script';
   MyScriptList := TStringList.Create;

   try
      MyScriptName := ExtractFilename( FScriptFile );

      try
         MyScriptList.LoadFromFile( FScriptFile );
      except
         TLog( LOGID_ERROR, Format( 'Error opening %s', [FScriptFile]) );
         exit;
      end;

      Result := THscVariant.Create( 0, False );
      try
         StateInfo := 'run script';
         with THscEngineHamster.Create(
                 FStopEvent, AppSettings.GetStr(asPathModules), FOutputBuffer
              ) do try

            ExecuteFromList( MyScriptName, MyScriptList, FParamsText, Result );
            OurRasConn := RasOurConnection;
            
         finally
            StateInfo := 'end script';
            Free;
         end;
      finally
         Result.Free;
      end;

      Hamster.NewsHistory.SaveToFile;

      // if script was stopped with an open RAS connection -> hangup
      // Note: this only applies to HamRasDial, not RasDial
      if OurRasConn then begin // script left with an open HamRasDial
         if Terminated then begin // script was stopped
            if Hamster.RasDialer.IsConnected then Hamster.RasDialer.HangUp;
         end;
      end;

   finally
      MyScriptList.Free;
      TLog( iif( FSilent, LOGID_DEBUG, LOGID_SYSTEM ), 'End' );
   end;
end;

constructor TThreadExecuteScript.Create( const AScriptFile, AParamsText: String;
                                         const FreeType: TThreadFreeTypes;
                                         const ASilent: Boolean;
                                         const DoBufferOutput: Boolean );
begin
   inherited Create( attScript,
                     '{script ' + ExtractFilename(AScriptFile)+ '}', FreeType );

   FScriptFile := AScriptFile;
   FParamsText := AParamsText;
   FStopEvent  := TEvent.Create( lpSecurityAttributesEveryone,
                                 True, False, '' );
   FSilent := ASilent;

   if DoBufferOutput then
      FOutputBuffer := TThreadStringList.Create( False, dupAccept )
   else
      FOutputBuffer := nil;
end;

destructor TThreadExecuteScript.Destroy;
begin
   if Assigned( FStopEvent ) then FStopEvent.Free;
   if Assigned( FOutputBuffer ) then FOutputBuffer.Free;
   inherited Destroy;
end;


// ----------------------------------------------------------------------------

function StartNewScript( const ScriptFile, ParamsText: String;
                         const WaitForEnd : Boolean;
                         out   ScriptThread: TThreadExecuteScript;
                         const FreeType: TThreadFreeTypes;
                         const Silent: Boolean;
                         const DoBufferOutput: Boolean ): Integer;
var  ScriptPath, Params: String;
begin
   try
      if not Silent then begin
         Log( LOGID_SYSTEM, 'Starting script' + ': ' + ScriptFile
                          + ', Wait=' + inttostr(ord(WaitForEnd)) );
         Log( LOGID_DEBUG,  'Starting script' + ': Parameters = "'
                          + copy( ParamsText, pos(LF,ParamsText)+1, 255 ) + '"' );
      end;

      ScriptThread := nil;

      ScriptPath := '';
      if Pos( '\', ScriptFile ) = 0 then begin
         ScriptPath := AppSettings.GetStr( asPathScripts ) + ScriptFile
      end else begin
         if FileExists( ScriptFile ) then begin
            ScriptPath := ExpandFilename( ScriptFile );
         end else begin
            ScriptPath := AppSettings.GetStr( asPathScripts ) + ScriptFile
         end;
      end;

      if not FileExists( ScriptPath ) then ScriptPath := '';

      if ScriptPath = '' then begin

         Result := -2;
         Log( LOGID_ERROR, Format( 'Script "%s" not found!', [ScriptFile] ) );

      end else begin

         Result := 0;
         Params := ScriptPath;
         if ParamsText <> '' then Params := Params + CRLF + ParamsText;

         if WaitForEnd then begin
            ScriptThread := TThreadExecuteScript.Create(
               ScriptPath, Params, tftFreeByCode, Silent, DoBufferOutput
            );
            ScriptThread.Resume;
            ScriptThread.WaitFor;
            ScriptThread.Free;
         end else begin
            ScriptThread := TThreadExecuteScript.Create(
               ScriptPath, Params, FreeType, Silent, DoBufferOutput
            );
            ScriptThread.Resume;
         end;

      end;

   except
      on E: Exception do begin
         Result := -1;
         ScriptThread := nil;
         Log( LOGID_ERROR, 'Error starting script: ' + E.Message );
      end;
   end;
end;

function StartNewScript( const ScriptFile, ParamsText: String;
                         const WaitForEnd : Boolean;
                         const Silent: Boolean ): Integer;
var  ScriptThread: TThreadExecuteScript;
begin
    Result := StartNewScript( ScriptFile, ParamsText, WaitForEnd,
                              ScriptThread, tftFreeOnTerminate, Silent, False );
end;


end.

