// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cSmtpRouter;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, cSyncObjects, cDnsResolver;

type
   TSmtpRouter = class
      protected
         FDestDomain    : String;
         FOverrideServer: String;
         FFallbackServer: String;
         FDnsAuto       : TDnsAutoDetect;
         FDnsServers    : String;
         FDnsTimeout    : Integer;
         FLimitDNS      : Integer;
         FLimitMX       : Integer;
         FMXServers     : TStringList;

         procedure DoPrepare;

      public
         property DestDomain    : String      read FDestDomain;
         property OverrideServer: String      read FOverrideServer;
         property FallbackServer: String      read FFallbackServer;
         property MXServers     : TStringList read FMXServers;
         
         constructor Create;
         destructor Destroy; override;
   end;

   TSmtpRouterFactory = class
      private
         FFilename: String;
         FRawList : TThreadStringList;

         function DoCreateListFor(
                     const RasName: String;
                     const RasIgnore: Boolean = False ): TStringList;

         function Do_SET( const R: TSmtpRouter; var Pars: String ): Boolean;
         function Do_USE( const R: TSmtpRouter; var Pars: String ): Boolean;

      public
         procedure Refresh;

         function CreateSmtpRouter( const ADestDomain: String ): TSmtpRouter;

         constructor Create( const AFilename: String );
         destructor Destroy; override;
   end;

implementation

uses uTools, cPCRE, IdAssignedNumbers, uConst, cLogFileHamster, cFiltersBase,
     uWinsock, uRasDyn, cHamster;

// ---------------------------------------------------------- TSmtpRouter -----

constructor TSmtpRouter.Create;
begin
   inherited Create;
   FMXServers := TStringList.Create;
end;

destructor TSmtpRouter.Destroy;
begin
   FMXServers.Free;
   inherited Destroy;
end;

procedure TSmtpRouter.DoPrepare;
var  DnsMX: TDnsMxResolver;
begin
   FMXServers.Clear;

   // was MX delivery disabled by setting a specific server to use?
   if FOverrideServer <> '' then exit; // yes

   // find MX servers for given domain
   DnsMX := TDnsMxResolver.Create( FDnsAuto, FDnsServers, FDnsTimeout,
                                   FLimitDNS, FLimitMX );
   try
      try
         if DnsMX.QueryMXRecords( FDestDomain ) then begin
            FMXServers.Assign( DnsMX.MXRecords );
         end;
      except
         on E: Exception do begin
            Log( LOGID_WARN, 'QueryMXRecords-Error: ' + E.Message );
         end;
      end;
   finally DnsMX.Free end;
end;

// --------------------------------------------------- TSmtpRouterFactory -----

constructor TSmtpRouterFactory.Create( const AFilename: String );
begin
   inherited Create;

   FFilename := AFilename;
   FRawList  := TThreadStringList.Create( False, dupAccept );

   Refresh;
end;

destructor TSmtpRouterFactory.Destroy;
begin
   FRawList.Free;
   inherited Destroy;
end;

procedure TSmtpRouterFactory.Refresh;
// (re-) load configuration file
begin
   with FRawList.LockList do try

      FRawList.Clear;
      
      try
         if FileExists( FFilename ) then LoadFromFile( FFilename );
      except
         on E: Exception do begin
            Log( LOGID_WARN, Format( 'Error loading file "%s": %s',
                                     [ FFilename, E.Message ] ) );
         end;
      end;

   finally FRawList.UnlockList end;
end;

function TSmtpRouterFactory.DoCreateListFor(
                               const RasName: String;
                               const RasIgnore: Boolean = False
                            ): TStringList;
// returns a new TStringList with all lines that are valid for given RasName
var  RasOK: Boolean;
     i, j: Integer;
     s, Patterns: String;
     RegEx: TPCRE;
begin
   Result := TStringList.Create;

   RegEx := TPCRE.Create( False, PCRE_CASELESS );
   try

      with FRawList.LockList do try

         RasOK := False;

         for i := 0 to Count - 1 do begin

            s := TrimWhSpace( Strings[i] );
            if copy( s, 1, 1 ) = '#' then s := ''; // comment
            if copy( s, 1, 1 ) = ';' then s := ''; // comment

            if copy( s, 1, 1 ) = '[' then begin // new section

               // extract section patterns
               Patterns := copy( s, 2, MaxInt );
               j := Pos( ']', Patterns );
               if j = 0 then begin
                  Log( LOGID_WARN, Format( 'Invalid []-section in file %s: %s',
                                           [ FFilename, Strings[i] ] ) );
                  continue;
               end;
               SetLength( Patterns, j - 1 );

               // check if patterns match the given RAS name
               if RasIgnore           then RasOK := True
               else if Patterns = '*' then RasOK := True
               else if Patterns = ''  then RasOK := ( RasName = '' )
               else RasOK := MatchPatterns( RasName, Patterns, RegEx );

            end else begin

               // add meaningful lines that belong to the given RAS name
               if RasOK and ( length(s) > 0 ) then begin
                  Result.AddObject( s, Pointer(i) ); // line, line-no.
               end;

            end;

         end;

      finally FRawList.UnlockList end;

   finally RegEx.Free end;
end;

function TSmtpRouterFactory.Do_SET( const R: TSmtpRouter; var Pars: String ): Boolean;
// "set" command:  keyword 1*WSP value *( 1*WSP value )
var  OrgPars, Keyword, Value: String;
     i: Integer;
begin
   Result  := False;
   OrgPars := Pars;
   Keyword := LowerCase( NextWhSpacePart( Pars ) );
   Value   := NextWhSpacePart( Pars );

   if Keyword = 'dnsauto' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      i := strtointdef( Value, ord(R.FDnsAuto) );
      R.FDnsAuto := TDnsAutoDetect( i );

   end else if Keyword = 'dnsserver' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      R.FDnsServers := Value;
      repeat
         Value := NextWhSpacePart( Pars );
         if Value = '' then break;
         R.FDnsServers := R.FDnsServers + ' ' + Value;
      until False;

   end else if Keyword = 'dnstimeout' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      R.FDnsTimeout := strtointdef( Value, R.FDnsTimeout );

   end else if Keyword = 'limitdns' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      R.FLimitDNS := strtointdef( Value, R.FLimitDNS );

   end else if Keyword = 'limitmx' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      R.FLimitMX := strtointdef( Value, R.FLimitMX );

   end else if Keyword = 'fallbackserver' then begin
      Log( LOGID_DEBUG, 'SmtpRouter entry matched: set ' + OrgPars );
      Result  := True;
      i := Hamster.Config.SmtpServers.IndexOfAlias( Value );
      if i >= 0 then begin
         R.FFallbackServer := Value;
      end else begin
         Log( LOGID_WARN, Format( 'Unknown SMTP server in file %s: set %s',
                                  [ FFilename, OrgPars ] ) );
      end;

   end else begin
      Log( LOGID_WARN, Format( 'Unknown set-keyword in file %s: set %s',
                               [ FFilename, OrgPars ] ) );

   end;
end;

function TSmtpRouterFactory.Do_USE( const R: TSmtpRouter; var Pars: String ): Boolean;
// "use" command:  server 1*WSP pattern *( 1*WSP pattern )
var  OrgPars, Server, Patterns: String;
     RegEx: TPCRE;
     i: Integer;
begin
   Result   := False;
   OrgPars  := Pars;
   Server   := NextWhSpacePart( Pars );
   Patterns := TrimWhSpace( Pars );
   Pars     := '';

   RegEx := TPCRE.Create( False, PCRE_CASELESS );
   try
      if MatchPatterns( R.FDestDomain, Patterns, RegEx ) then begin
         Log( LOGID_DEBUG, 'SmtpRouter entry matched: use ' + OrgPars );
         Result := True;

         i := Hamster.Config.SmtpServers.IndexOfAlias( Server );
         if i >= 0 then begin
            R.FOverrideServer := Server;
         end else begin
            Log( LOGID_WARN, Format( 'Unknown SMTP server in file %s: use %s',
                                     [ FFilename, OrgPars ] ) );
         end;
      end;
   finally RegEx.Free end;
end;

function TSmtpRouterFactory.CreateSmtpRouter( const ADestDomain: String ): TSmtpRouter;
// create a new SmtpRouter for the given destination domain based on the
// configuration settings for the currently active RAS connection
var  SL: TStringList;
     i: Integer;
     s, Cmd, RasName: String;
     Terminator, Matched: Boolean;
begin
   // prepare default/inactive SmtpRouter
   Result := TSmtpRouter.Create;
   with Result do begin
      FDestDomain     := ADestDomain;
      FOverrideServer := '';
      FFallbackServer := '';
      FDnsAuto        := dnsDetectPreferred;
      FDnsServers     := '';
      FDnsTimeout     := 10000;
      FLimitDNS       := 3;
      FLimitMX        := 3;
   end;

   // create list for current RAS connection
   RasName := RasDynGetConnection;
   Log( LOGID_DEBUG, Format( 'Processing SmtpRouter.hst for RAS name "%s" and '
                           + 'domain "%s"', [ RasName, ADestDomain ] ) );
   SL := DoCreateListFor( RasName );

   try

      // check and execute all specific list entries
      for i := 0 to SL.Count - 1 do begin

         s := SL[i];
         Cmd := LowerCase( NextWhSpacePart( s ) );

         Terminator := False;
         if copy( Cmd, 1, 1 ) = '=' then begin
            Terminator := True;
            System.Delete( Cmd, 1, 1 );
         end;

         Matched := False;

         if      Cmd = 'set'  then Matched := Do_SET( Result, s )
         else if Cmd = 'use'  then Matched := Do_USE( Result, s )
         
         else begin
            Log( LOGID_WARN, Format(
               'Unknown command in line %d of file %s: %s',
               [ Integer( SL.Objects[i] ), FFilename, SL[i] ] ) );
            s := '';
         end;

         if TrimWhSpace( s ) <> '' then begin
            Log( LOGID_WARN, Format(
               'Too many parameters in line %d of file %s: %s',
               [ Integer( SL.Objects[i] ), FFilename, SL[i] ] ) );
         end;

         if Matched and Terminator then break;
         
      end;

   finally SL.Free end;

   // finally prepare SmtpRouter with settings of config file
   Result.DoPrepare;
end;

end.
