// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cServerSMTP;

interface

{$INCLUDE Compiler.inc}

uses Classes, cServerBase, cAccounts, cMailItem;

type
  TServerSMTP = class( TServerBase );

type
  TSmtpStage = (
     ExpectHeloOrMail,
     ExpectMail,
     ExpectRcpt,
     ExpectRcptOrData,
     ExpectData
  );

const
  SmtpStageText: array[ TSmtpStage ] of String = (
     'HELO/EHLO or MAIL',
     'MAIL',
     'RCPT',
     'RCPT or DATA',
     'DATA'
  );

type
  TServerClientSMTP = class( TServerClientBase )
    private
      AuthReq, AuthDone, ApplyMailTraps: Boolean;
      MailItem: TMailItem;
      SmtpFqdn: String;
      SmtpStage: TSmtpStage;

    protected
      function FormatErrorReply( const ErrType: TClientErrorType;
                                 const ErrMsg: String ): String; override;
      procedure SendGreeting( var KeepConnection: Boolean;
                              var Reason: String ); override;

      function Cmd_MAIL( const Cmd, Par: String ): Boolean;
      function Cmd_RCPT( const Cmd, Par: String ): Boolean;
      function Cmd_DATA( const Cmd, Par: String ): Boolean;
      function Cmd_AUTH( const Cmd, Par: String ): Boolean;
      function Cmd_NOOP( const Cmd, Par: String ): Boolean;
      function Cmd_RSET( const Cmd, Par: String ): Boolean;
      function Cmd_QUIT( const Cmd, Par: String ): Boolean;
      function Cmd_HELO( const Cmd, Par: String ): Boolean;
      function Cmd_HELP( const Cmd, Par: String ): Boolean;

    public
      procedure HandleCommand( const CmdLine: String ); override;

      constructor Create( ACreateSuspended: Boolean ); override;
      destructor Destroy; override;
  end;

implementation

uses uConst, uConstVar, uVar, SysUtils, Windows, uTools, uWinSock, cIPAccess,
     cArticle, uDateTime, uEncoding, uSASL, cLogFileHamster, uHamTools,
     cHamster, cMailUsers, cMailDispatcher, cHscAction;

// ---------------------------------------------------- TServerClientSMTP -----

procedure TServerClientSMTP.HandleCommand( const CmdLine: String );
var  Cmd, Par: String;
     j: Integer;
begin
     try
        if not HConnected then exit;

        Log( LOGID_INFO, '> ' + CmdLine );
        if CmdLine='' then exit;

        // Extract command
        j := PosWhSpace( CmdLine );
        if j=0 then begin
           Cmd := UpperCase( CmdLine );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( CmdLine, 1, j-1 ) );
           Par := TrimWhSpace( copy( CmdLine, j+1, 512 ) );
        end;

        // commands (no authentication required)
        if Cmd='QUIT' then if Cmd_QUIT( Cmd, Par ) then exit;
        if Cmd='HELP' then if Cmd_HELP( Cmd, Par ) then exit;
        if Cmd='NOOP' then if Cmd_NOOP( Cmd, Par ) then exit;
        if Cmd='RSET' then if Cmd_RSET( Cmd, Par ) then exit;
        if Cmd='HELO' then if Cmd_HELO( Cmd, Par ) then exit;
        if Cmd='EHLO' then if Cmd_HELO( Cmd, Par ) then exit;
        if Cmd='AUTH' then if Cmd_AUTH( Cmd, Par ) then exit;

        // check authentication
        if AuthReq and (not AuthDone) then begin
           if not Hamster.Config.Settings.GetBoo(hsLocalSmtpNoAuthForLocal) then begin
              HWriteLn( '500 Permission denied' );
              exit;
           end;
        end;

        // commands (might require authentication)
        if Cmd='MAIL' then if Cmd_MAIL( Cmd, Par ) then exit;
        if Cmd='RCPT' then if Cmd_RCPT( Cmd, Par ) then exit;
        if Cmd='DATA' then if Cmd_DATA( Cmd, Par ) then exit;

        HWriteLn( '502 Command not implemented.' );
        Log( LOGID_INFO, 'Unsupported SMTP-command: ' + CmdLine );

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + CmdLine );
           try HDisconnect except end;
        end;
     end;
end;

function TServerClientSMTP.Cmd_MAIL( const Cmd, Par: String ): Boolean;
var  s: String;
begin
   Result := True;
   
   if (SmtpStage <> ExpectHeloOrMail) and (SmtpStage <> ExpectMail) then begin
      HWriteLn( '503 Bad sequence of commands ('
                + SmtpStageText[SmtpStage] + ' expected)' );
      exit;
   end;

   if UpperCase( copy(Par,1,5) ) <> 'FROM:' then begin
      HWriteLn( '504 Command parameter not implemented' );
      exit;
   end;

   MailItem.Sender.EnvelopeAddr := ExtractEnvelopeAddr( copy(Par,6,MaxInt) );
   MailItem.Recipients.Clear;

   if MailItem.Sender.MailUserType <> muMissing then begin
      HWriteLn( '250 OK' );
      SmtpStage := ExpectRcpt;
   end else begin
      s := MailItem.Sender.OrgEnvAddr;
      MailItem.Sender.Clear;
      HWriteLn( '550 Invalid address in MAIL FROM - ' + s );
   end;
end;

function TServerClientSMTP.Cmd_RCPT( const Cmd, Par: String ): Boolean;
var  Reply, s: String;
     j: Integer;
begin
   Result := True;

   if (SmtpStage <> ExpectRcpt) and (SmtpStage <> ExpectRcptOrData) then begin
      HWriteLn( '503 Bad sequence of commands ('
                + SmtpStageText[SmtpStage] + ' expected)' );
      exit;
   end;

   if UpperCase( copy(Par,1,3) ) <> 'TO:' then begin
      HWriteLn( '504 Command parameter not implemented' );
      exit;
   end;

   j := MailItem.Recipients.Add( ExtractEnvelopeAddr( copy(Par,4,MaxInt) ) );
   s := MailItem.Recipients.Items[j].OrgEnvAddr;
   
   case MailItem.Recipients.Items[j].MailUserType of

      muLocal:        begin
                         Reply := '250 OK';
                         SmtpStage := ExpectRcptOrData;
                      end;

      muRemote:       if (not AuthReq) or AuthDone then begin
                         Reply := '250 OK';
                         SmtpStage := ExpectRcptOrData;
                      end else begin
                         Reply := '550 Not a local recipient - ' + s;
                         MailItem.Recipients.Delete( j );
                      end;

      muLocalUnknown: begin
                         Reply := '550 Unknown local address - ' + s;
                         MailItem.Recipients.Delete( j );
                      end;

      muLocalInvalid: begin
                         Reply := '550 Invalid (local) address - ' + s;
                         MailItem.Recipients.Delete( j );
                      end;

      else            begin // muMissing
                         Reply := '550 Unknown or invalid address - ' + s;
                         MailItem.Recipients.Delete( j );
                      end;

   end;

   HWriteLn( Reply );
end;

function TServerClientSMTP.Cmd_DATA( const Cmd, Par: String ): Boolean;
var  mdr: TMailDeliveryResults;
     LogNow: TDateTime;
     Reply, Envelope, MailErr, TrapReason, ModifiedHeaders, s: String;
     DeleTrapScore, i: Integer;
     DeleTrap, MailOk: Boolean;
begin
   Result := True;

   if (SmtpStage <> ExpectRcptOrData) and (SmtpStage <> ExpectData) then begin
      HWriteLn( '503 Bad sequence of commands ('
                + SmtpStageText[SmtpStage] + ' expected)' );
      exit;
   end;

   SmtpStage := ExpectData;

   try

      // check if recipients were given
      if MailItem.Recipients.Count = 0 then begin
         HWriteLn( '554 No valid recipients' );
         exit;
      end;

      // add sender identification
      MailItem.Sender.IPAddr := FClientAD.S_addr;
      if AuthDone then MailItem.Sender.AuthUserID := CurrentUserID;

      // get mail text
      MailItem.MailText.FullText := HRequestText( '354 Start mail input; end with <CRLF>.<CRLF>' );
      if not HConnected then exit;
      MailOk  := True;
      MailErr := '';
      Reply   := '';

      // Add received header if required for filtering
      if ApplyMailTraps or
         Hamster.HscActions.IsAssigned( actMailInHeader ) or
         Hamster.HscActions.IsAssigned( actMailInHeaderFile ) then begin

         if Hamster.Config.Settings.GetBoo(hsMailAddReceived) then begin
            MailItem.AddHeaderReceived;
         end;

      end;

      // Process "MailHeaderIn" actions to check headers of mail.
      if ProcessAction_MailInHeader( MailItem.Origin, MailItem.MailText, ModifiedHeaders ) then begin
         if ModifiedHeaders <> '' then MailItem.MailText.HeaderText := ModifiedHeaders;
      end else begin
         if ModifiedHeaders <> '' then MailItem.MailText.HeaderText := ModifiedHeaders;
         MailOk  := False;
         MailErr := 'Action "MailInHeader[File]" has invalidated mail.';
         Log( LOGID_WARN, 'Mail rejected: ' + MailErr );
         Reply  := '554 Transaction failed: Mail violates server policy (A)';
      end;

      // apply "Mail Traps" if configured to do so
      if MailOk and ApplyMailTraps then begin

         Log( LOGID_DEBUG, 'Check Mail Traps ...' );

         if Hamster.MailTrap.Reject( MailItem.MailText, TrapReason, DeleTrap, DeleTrapScore ) then begin

            MailOk  := False;

            MailErr := 'Mail Trap ';
            if DeleTrap then begin
               MailErr := MailErr + 'Delete';
               if DeleTrapScore <> 0 then begin
                  MailErr := MailErr + ', Score ' + inttostr(DeleTrapScore);
               end;
            end else begin
               MailErr := MailErr + 'Score ' + inttostr(DeleTrapScore);
            end;
            Log( LOGID_WARN, 'Mail rejected: ' + MailErr );
            Log( LOGID_WARN, 'Mail rejected: ' + TrapReason );

            s := ' (' + iif( DeleTrap, 'D', '' ) + 'S' + inttostr(DeleTrapScore) + ')';
            Reply  := '554 Transaction failed: Mail violates server policy' + s;

         end else begin
            if DeleTrapScore <> 0 then begin
               Log( LOGID_DEBUG, 'MailTrap score: ' + inttostr(DeleTrapScore) );
            end;
         end;

      end;

      // process mail
      if MailOk then begin
         mdr := Hamster.MailDispatcher.Process( MailItem );
         if mdr = mdrSuccess then begin
            Reply := '250 OK';
         end else begin
            MailOk  := False;
            MailErr := MailDeliveryResultTexts[ mdr ];
            Reply   := '554 Transaction failed: ' + MailDeliveryResultTexts[ mdr ];
         end;
      end;

      // send result
      HWriteLn( Reply );

      // save result in SmtpServer.log
      LogNow := Now;

      Envelope := 'From: ' + MailItem.Sender.OrgEnvAddr;
      for i := 0 to MailItem.Recipients.Count - 1 do begin
         Envelope := Envelope + ', To: ' + MailItem.Recipients[i].OrgEnvAddr;
      end;

      s := DateTimeToLogTime( LogNow )
         + #9 + 'User='     + CurrentUserName
         + #9 + 'IP='       + nAddrToStr( FClientAD.S_addr )
         + #9 + 'Action='   + 'DATA'
         + #9 + 'Result='   + Reply
         + #9 + 'Envelope=' + Envelope
         + #9 + 'From='     + Logify( MailItem.MailText.HeaderValueByNameSL(HDR_NAME_FROM) )
         + #9 + 'To='       + Logify( MailItem.MailText.HeaderValueByNameSL(HDR_NAME_TO) )
         + #9 + 'Subject='  + Logify( MailItem.MailText.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
         + #9 + 'Date='     + Logify( MailItem.MailText.HeaderValueByNameSL(HDR_NAME_DATE) )
         ;
      if not MailOk then s := s + #9 + 'Reason=' + MailErr;

      HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                         + LOGFILE_SMTPSERVER
                         + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                         + LOGFILE_EXTENSION,
                         s );

      MailItem.Clear;

   finally
      SmtpStage := ExpectHeloOrMail;
   end;
end;

function TServerClientSMTP.Cmd_AUTH( const Cmd, Par: String ): Boolean;
var  SASL_Name, SASL_Par: String;
     j: Integer;
begin
   Result := True;

   if MailItem.Origin <> moESMTP then begin
      HWriteLn( '503 Bad sequence of commands (AUTH without EHLO)' );
      exit;
   end;

   j := Pos( ' ', Par );
   if j=0 then begin
      SASL_Name := Uppercase( Par );
      SASL_Par  := '';
   end else begin
      SASL_Name := Uppercase( copy( Par, 1, j-1 ) );
      SASL_Par  := copy( Par, j+1, MaxInt );
   end;

   if Pos( ' ' + SASL_Name + ' ',
           ' ' + Hamster.Config.Settings.GetStr(hsLocalSmtpSASL) + ' ' )=0 then begin
      HWriteLn( '504 Unrecognized authentication type ' + SASL_Name );
      exit;
   end;

   if Local_SASL_Login(
         Self, '334 %s',
         MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) ),
         SASL_NAME, SASL_Par, ClientIPn, CurrentUserID, CurrentUserName
      ) then begin
      HWriteLn( '235 Authentication successful.' );
      ClientsChange;
      AuthDone := True;
   end else begin
      HWriteLn( '535 Authentication failed.' );
   end;
end;

function TServerClientSMTP.Cmd_NOOP( const Cmd, Par: String ): Boolean;
begin
   Result := True;
   HWriteLn( '250 OK' );
end;

function TServerClientSMTP.Cmd_RSET( const Cmd, Par: String ): Boolean;
begin
   Result := True;
   MailItem.Clear;
   SmtpStage := ExpectHeloOrMail;
   HWriteLn( '250 OK' );
end;

function TServerClientSMTP.Cmd_QUIT( const Cmd, Par: String ): Boolean;
begin
   Result := True;

   if HConnected then HWriteLn( '221 ' + SmtpFqdn + ' closing connection - goodbye!' );
   Sleep( Hamster.Config.Settings.GetInt(hsLocalTimeoutQuitDelay) );

   try
      HDisconnect;
   except
      on E:Exception do Log(LOGID_DEBUG, 'Exception on .Close: ' + E.Message );
   end;

   Terminate;
end;

function TServerClientSMTP.Cmd_HELO( const Cmd, Par: String ): Boolean;
begin
   Result := True;

   MailItem.Clear;
   MailItem.Sender.HELOStr := TrimWhSpace( Par );

   if MailItem.Sender.HELOStr = '' then begin

      HWriteLn( '501 Syntax error (missing domain name)' );

   end else begin

      if Cmd='HELO' then begin

         MailItem.Origin := moSMTP;
         HWriteLn( '250 ' + SmtpFqdn );
         SmtpStage := ExpectMail;


      end else if Cmd='EHLO' then begin

         MailItem.Origin := moESMTP;
         HWriteLn(
            '250-' + SmtpFqdn + #13#10 +
            '250-AUTH=' + Hamster.Config.Settings.GetStr(hsLocalSmtpSASL) + #13#10 + {Mozilla}
            '250-AUTH ' + Hamster.Config.Settings.GetStr(hsLocalSmtpSASL) + #13#10 + {RFC}
            '250 HELP'
         );
         SmtpStage := ExpectMail;

      end else Result := False;

   end;
end;

function TServerClientSMTP.Cmd_HELP( const Cmd, Par: String ): Boolean;
begin
   Result := True;
   HWriteLn( '214-Implemented commands follow:' );
   HWriteLn( '214-data' );
   HWriteLn( '214-helo' );
   HWriteLn( '214-ehlo' );
   HWriteLn( '214-mail' );
   HWriteLn( '214-noop' );
   HWriteLn( '214-quit' );
   HWriteLn( '214-rcpt' );
   HWriteLn( '214 rset' );
end;

function TServerClientSMTP.FormatErrorReply( const ErrType: TClientErrorType;
                                             const ErrMsg: String ): String;
begin
   case ErrType of
      cetRefusedDontRetry, cetRefusedTemporary:
         Result := '500 ' + ErrMsg;
      cetLineTooLong, cetTextTooLarge:
         Result := '554 ' + ErrMsg;
      else
         Result := '500 ' + ErrMsg;
   end;
end;

procedure TServerClientSMTP.SendGreeting( var KeepConnection: Boolean;
                                          var Reason: String );
var  OK: Boolean;
     Msg: String;
begin
     KeepConnection := False;

     OK  := True;
     Msg := 'Permission denied - closing connection.';

     if AuthReq and not( AuthDone ) then begin
        if Hamster.Config.Settings.GetBoo(hsLocalSmtpAuthReq) then begin
           // SMTP-after-POP3
           if Hamster.Accounts.ChkAuthenticatedLogin( FClientAD.S_addr ) then begin
              AuthDone := True;
           end else begin
              // OK  := False;
              // Msg := 'Permission denied (not authorized by POP3)';
           end;
        end;
     end;

     if OK and ((IPAccess and IPACC_ACCESS_WO)=IPACC_ACCESS_WO) then begin

        HWriteLn( '220 ' + SmtpFqdn + ' Hamster-SMTP, '
                    + GetMyStringFileInfo('ProductName','Hamster') + ' '
                    + GetMyVersionInfo );
        SmtpStage := ExpectHeloOrMail;
        KeepConnection := True;

     end else begin

        Reason := Msg;
        HWriteLn( FormatErrorReply( cetRefusedDontRetry, Reason ) );

     end;
end;

constructor TServerClientSMTP.Create( ACreateSuspended: Boolean );
var  i: Integer;
begin
     inherited Create( True );

     FLimitLineLen  := Hamster.Config.Settings.GetInt(hsLocalSmtpLimitLineLen);
     FLimitTextSize := Hamster.Config.Settings.GetInt(hsLocalSmtpLimitTextSize);

     i := Hamster.Config.Settings.GetInt( hsLocalSmtpInactivitySec );
     if i > 0 then FInactivitySecs := i;

     AuthReq  := Hamster.Config.Settings.GetBoo(hsLocalSmtpAuthReq )
              or Hamster.Config.Settings.GetBoo(hsLocalSmtpEAuthReq);
     AuthDone := False;

     ApplyMailTraps := Hamster.Config.Settings.GetBoo(hsLocalSmtpApplyTraps);

     SmtpFqdn := Hamster.Config.Settings.GetStr( hsFQDNforMID );
     if SmtpFqdn = '' then SmtpFqdn := LookupLocalHostName;
     if SmtpFqdn = '' then SmtpFqdn := 'localhost';

     SmtpStage := ExpectHeloOrMail;

     CurrentUserID   := ACTID_INVALID;
     CurrentUserName := '';

     MailItem := TMailItem.Create( moSMTP );
     MailItem.Sender.IPAddr := FClientAD.S_addr;

     if not ACreateSuspended then Resume;
end;

destructor TServerClientSMTP.Destroy;
begin
     MailItem.Free;
     inherited;
end;

// ----------------------------------------------------------------------------

end.

