// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cServerRECO; // Remote Control Server

interface

{$INCLUDE Compiler.inc}

uses Classes, cServerBase, cLiveServer, uConst, cLiveMsg;

type
   TServerRECO = class( TServerBase );

const
   ReCoCmdCount = 10;

type
   TReCoCmdInfo = record
      CmdStr   : String; // command
      MinPar   : Byte;   // min. no. of parameters
      MaxPar   : Byte;   // max. no. of parameters
      AuthReq  : Byte;   // 1=authentication required to use command
   end;

   TReCoCmdArray = array[ 0 .. ReCoCmdCount-1 ] of TReCoCmdInfo;

type
   TServerClientRECO = class;

   TLiveClientQueueSenderThread = class( TLiveClientQueueHandlerThread )
      protected
         FServerClient: TServerClientRECO;
         procedure HandleMessage( var LiveMsg: TLiveMsg ); override;
      public
         constructor Create( ACreateSuspended: Boolean;
                             ALiveClient     : TLiveClient;
                             AServerClient   : TServerClientRECO );
   end;

   TReCoProfile = class
      private
         FDescription   : String;
         FGrantScriptPut: Boolean;
         FGrantLiveOn   : Boolean;
         FCmdLineRE     : array[ 0 .. ReCoCmdCount-1 ] of String;

         function  GetRE( const Command: String ): String;
         procedure SetRE( const Command: String; const NewRE: String );

      public
         property Description   : String  read FDescription;
         property GrantScriptPut: Boolean read FGrantScriptPut;
         property GrantLiveOn   : Boolean read FGrantLiveOn;
         property CmdLineRE[ const Command: String ]: String read GetRE write SetRE;

         constructor Create;
   end;

   TReCoProfiles = class
      private
         FProfiles: array[ RC_PROFILE_FIRST .. RC_PROFILE_LAST ] of TReCoProfile;

         function GetP( const ProfileNo: Integer ): TReCoProfile;

      public
         property Profile[ const ProfileNo: Integer ]: TReCoProfile read GetP;

         function  GetList: String;
         procedure LoadProfiles;
         procedure SaveProfiles;

         constructor Create;
         destructor Destroy; override;
   end;

   TServerClientRECO = class( TServerClientBase )
     private
        CurrentUserProfile: Integer;
        FReCoProfiles     : TReCoProfiles;
        FLiveClient       : TLiveClient;
        FLiveSenderThread : TLiveClientQueueSenderThread;

        function IsCommandAllowed( CmdIndex: Integer;
                                   const CmdLine: String ): Boolean;

     protected
        function FormatErrorReply( const ErrType: TClientErrorType;
                                   const ErrMsg: String ): String; override;
        procedure SendGreeting( var KeepConnection: Boolean;
                                var Reason: String ); override;

        procedure TerminateLiveMode;
        procedure TerminateConnection;

        procedure HandleLiveCommand( const CmdLine: String );

        function Cmd_AUTH  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_USER  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_HELP  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_LOG   ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_NEWS  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_QUIT  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_SCRIPT( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_SERVER( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_TASK  ( AC: Integer; AV: TStringList ): Boolean;
        function Cmd_LIVE  ( AC: Integer; AV: TStringList ): Boolean;

     public
        procedure HandleCommand( const CmdLine: String ); override;

        constructor Create( ACreateSuspended: Boolean ); override;
        destructor Destroy; override;
   end;


implementation

uses SysUtils, Windows, uTools, cIPAccess, cArticle, cArtFiles,
     cAccounts, uCRC32, cPCRE, uDateTime, cLogFileHamster, tScript, uEncoding,
     uSASL, cLiveQueue, uType, uConstVar, uVar, cHamster, cHscFiles,
     uHamTools, IniFiles, uMD5;

const
   ReCoCmdArray: TReCoCmdArray = (
      ( CmdStr:'AUTH';   MinPar:2; MaxPar:4; AuthReq:0 ),
      ( CmdStr:'HELP';   MinPar:0; MaxPar:1; AuthReq:0 ),
      ( CmdStr:'LIVE';   MinPar:1; MaxPar:2; AuthReq:1 ),
      ( CmdStr:'LOG';    MinPar:1; MaxPar:5; AuthReq:1 ),
      ( CmdStr:'NEWS';   MinPar:2; MaxPar:4; AuthReq:1 ),
      ( CmdStr:'QUIT';   MinPar:0; MaxPar:0; AuthReq:0 ),
      ( CmdStr:'SCRIPT'; MinPar:1; MaxPar:9; AuthReq:1 ),
      ( CmdStr:'SERVER'; MinPar:1; MaxPar:2; AuthReq:1 ),
      ( CmdStr:'TASK';   MinPar:1; MaxPar:3; AuthReq:1 ),
      ( CmdStr:'USER';   MinPar:3; MaxPar:3; AuthReq:1 )
   );

const
   R_INFO     = 100;

   R_OK0      = 200;
   R_OK1      = 201;
   R_AUTHOK   = 280;

   R_SENDDATA = 300;
   R_AUTHCHLG = 380;

   R_FAILED0  = 400;
   R_FAILED1  = 401;
   R_AUTHREQ  = 480;

   R_SYNTAX   = 500;
   R_UNKNOWN  = 501;
   R_NOPERM   = 502;
   R_SYSERR   = 503;
   R_AUTHFAIL = 580;

function Res( ResCode: Integer; ResText: String ): String;
begin
   Result := Format( '%3d %s', [ ResCode, ResText ] );
end;

function OptBoo( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=1 to AC do begin
      if CompareText( Opt, AV[i] ) = 0 then begin
         Result := True;
         AV.Delete( i );
         dec( AC );
         break;
      end;
   end;
end;

function OptStr( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String;
                 const Default: String ): String;
var  i: Integer;
begin
   Result := Default;
   for i:=1 to AC-1 do begin
      if CompareText( Opt, AV[i] ) = 0 then begin
         Result := AV[ i+1 ];
         AV.Delete( i+1 );
         AV.Delete( i   );
         dec( AC, 2 );
         break;
      end;
   end;
end;

function OptInt( var AC: Integer;
                 var AV: TStringList;
                 const Opt: String;
                 const Default: Integer ): Integer;
begin
   Result := strtointdef( OptStr( AC, AV, Opt, '' ), Default );
end;

function CmdIndexOf( const Cmd: String ): Integer;
var  i: Integer;
begin
   Result := -1;
   for i:=0 to ReCoCmdCount-1 do begin
      if AnsiCompareText( ReCoCmdArray[i].CmdStr, Cmd ) = 0 then begin
         Result := i;
         break;
      end;
   end;
end;


// --------------------------------------------------------- TReCoProfile -----

constructor TReCoProfile.Create;
var  i: Integer;
begin
   inherited Create;

   FDescription    := '';
   FGrantScriptPut := False;
   FGrantLiveOn    := False;
   for i := 0 to ReCoCmdCount-1 do FCmdLineRE[ i ] := '';
end;

function TReCoProfile.GetRE( const Command: String ): String;
var  i: Integer;
begin
   i := CmdIndexOf( Command );
   if (i<0) or (i>=ReCoCmdCount) then Result := ''
                                 else Result := FCmdLineRE[ i ];
end;

procedure TReCoProfile.SetRE( const Command, NewRE: String );
var  i: Integer;
begin
   i := CmdIndexOf( Command );
   if (i>=0) and (i<ReCoCmdCount) then FCmdLineRE[ i ] := NewRE;
end;


// -------------------------------------------------------- TReCoProfiles -----

constructor TReCoProfiles.Create;
var  i: Integer;
begin
   inherited Create;

   for i := RC_PROFILE_FIRST to RC_PROFILE_LAST do begin
      FProfiles[ i ] := TReCoProfile.Create;
   end;

   LoadProfiles;
end;

destructor TReCoProfiles.Destroy;
var  i: Integer;
begin
   for i := RC_PROFILE_FIRST to RC_PROFILE_LAST do begin
      FProfiles[ i ].Free;
   end;

   inherited Destroy;
end;

function TReCoProfiles.GetList: String;
var  i: Integer;
begin
   Result := '';
   for i := RC_PROFILE_FIRST to RC_PROFILE_LAST do begin
      Result := Result + inttostr( i )
              + TAB + DQuoteStr( Profile[i].Description )
              + CRLF;
   end;
end;

procedure TReCoProfiles.LoadProfiles;
var  p, i: Integer;
     ProfileSection, s: String;
     Ini: TIniFile;
begin
   // read profiles
   Ini := TIniFile.Create( AppSettings.GetStr(asPathBase) + CFGFILE_RCPROFILES );
   try
      for p := RC_PROFILE_FIRST to RC_PROFILE_LAST do with FProfiles[ p ] do begin

         ProfileSection := 'RC profile ' + inttostr( p );
         FDescription   := Ini.ReadString( ProfileSection, 'Description', '' );

         // init "allowed command line" patterns
         for i:=0 to ReCoCmdCount-1 do begin
            with ReCoCmdArray[i] do begin

               // set pattern for "allowed command line"
               FCmdLineRE[i] := Ini.ReadString( ProfileSection, 'Command ' + ReCoCmdArray[i].CmdStr, '' );
               if FCmdLineRE[i] <> '' then begin
                  // make sure, pattern covers complete line: '^' + <pattern> + '$'
                  if FCmdLineRE[i][ 1 ] <> '^' then
                     FCmdLineRE[i] := '^' + FCmdLineRE[i];
                  if FCmdLineRE[i][ length(FCmdLineRE[i]) ] <> '$' then
                     FCmdLineRE[i] := FCmdLineRE[i] + '$';
               end;

            end;
         end;

         // "security hole" SCRIPT PUT needs extra confirmation to become active
         FGrantScriptPut := Ini.ReadBool( ProfileSection, 'Confirm SCRIPT PUT', False );

         // "security hole" LIVE ON needs extra confirmation to become active
         FGrantLiveOn := Ini.ReadBool( ProfileSection, 'Confirm LIVE ON', False );

      end;
   finally Ini.Free end;

   // Force predefined profile 0: No access
   with FProfiles[ RC_PROFILE_NOACCESS ] do begin
      if FDescription = '' then FDescription := 'No access';
      FGrantScriptPut := False;
      FGrantLiveOn    := False;
      for i := 0 to ReCoCmdCount-1 do FCmdLineRE[ i ] := '';
   end;

   // Force predefined profile 1: Change own password only
   with FProfiles[ RC_PROFILE_CHANGEPASSWORD ] do begin
      if FDescription = '' then FDescription := 'Change own password only';
      FGrantScriptPut := False;
      FGrantLiveOn    := False;
      for i := 0 to ReCoCmdCount-1 do FCmdLineRE[ i ] := '';
      CmdLineRE['USER'  ] := '^USER\s+(PASSWORD)\s+.+$';
   end;

   // Force predefined profile 2: No actions, info only.
   with FProfiles[ RC_PROFILE_INFORMATIONAL ] do begin
      if FDescription = '' then FDescription := 'Informational access and change password';
      FGrantScriptPut := False;
      FGrantLiveOn    := False;
      for i := 0 to ReCoCmdCount-1 do FCmdLineRE[ i ] := '';
      CmdLineRE['LOG'   ] := '^LOG\s+(LIST).*$';
      CmdLineRE['NEWS'  ] := '^NEWS\s+(LIST).*$';
      CmdLineRE['SCRIPT'] := '^SCRIPT\s+(LIST|DIR).*$';
      CmdLineRE['SERVER'] := '^SERVER\s+(LIST|STATE).*$';
      CmdLineRE['TASK'  ] := '^TASK\s+(LIST).*$';
      CmdLineRE['USER'  ] := '^USER\s+(PASSWORD)\s+.+$';
   end;

   // Force predefined profile 9: Full access for administration purposes.
   with FProfiles[ RC_PROFILE_FULLACCESS ] do begin
      if FDescription = '' then FDescription := 'Full access (administration)';
      FGrantScriptPut := True;
      FGrantLiveOn    := True;
      for i:=0 to ReCoCmdCount-1 do begin
         with ReCoCmdArray[i] do begin
            s := ReCoCmdArray[i].CmdStr;
            FCmdLineRE[i] := '^' + s + '\s.*$';
         end;
      end;
   end;
end;

procedure TReCoProfiles.SaveProfiles;
var  p, i: Integer;
     ProfileSection: String;
     Ini: TIniFile;
begin
   Ini := TIniFile.Create( AppSettings.GetStr(asPathBase) + CFGFILE_RCPROFILES );
   try
      for p := RC_PROFILE_FIRST to RC_PROFILE_LAST do with FProfiles[ p ] do begin

         ProfileSection := 'RC profile ' + inttostr( p );
         Ini.WriteString( ProfileSection, 'Description', FDescription );

         for i:=0 to ReCoCmdCount-1 do begin
            with ReCoCmdArray[i] do begin
               if AuthReq <> 0 then begin
                  Ini.WriteString(
                     ProfileSection,
                     'Command ' + ReCoCmdArray[i].CmdStr,
                     FCmdLineRE[i]
                  );
               end;
            end;
         end;

         Ini.WriteBool( ProfileSection, 'Confirm SCRIPT PUT', FGrantScriptPut );
         Ini.WriteBool( ProfileSection, 'Confirm LIVE ON', FGrantLiveOn );
         
      end;
   finally Ini.Free end;
end;

function TReCoProfiles.GetP( const ProfileNo: Integer ): TReCoProfile;
begin
   if (ProfileNo>=RC_PROFILE_FIRST) and (ProfileNo<=RC_PROFILE_LAST) then
      Result := FProfiles[ ProfileNo ]
   else
      Result := FProfiles[ 0 ];
end;


// ----------------------------------------- TLiveClientQueueSenderThread -----

procedure TLiveClientQueueSenderThread.HandleMessage( var LiveMsg: TLiveMsg );
begin
   // Note: Using .Connection directly here to avoid new Log()
   //       entries created by FServerClient's .HWriteLn.
   with FServerClient.Connection do begin
      if Connected then WriteLn( LiveMsg.TransferStr );
   end;
end;

constructor TLiveClientQueueSenderThread.Create( ACreateSuspended: Boolean;
                                                 ALiveClient     : TLiveClient;
                                                 AServerClient   : TServerClientRECO );
begin
   inherited Create( True, ALiveClient );
   FServerClient := AServerClient;
   if not ACreateSuspended then Resume;
end;


// ---------------------------------------------------- TServerClientRECO -----

procedure TServerClientRECO.HandleCommand( const CmdLine: String );
var  AC, CmdIndex: Integer;
     AV: TStringList;
     LogCmdLine: String;
begin
   try
      if not HConnected then exit;

      // handle requests in "Live Mode"
      if Assigned( FLiveClient ) then begin
         HandleLiveCommand( CmdLine );
         exit;
      end;

      AV := TStringList.Create;
      try
         // parse command and parameters
         // AC: number of arguments (i.e. not including AV[0])
         // AV[0]: command; AV[1]: 1st arg.; ...; AV[AC]: last arg.
         AC := ArgsWhSpaceDQuoted( CmdLine, AV, 2 ) - 1;
         AV[0] := UpperCase( AV[0] ); // command
         AV[1] := UpperCase( AV[1] ); // [action, i.e. 1st param]

         if AV[0] = 'AUTH' then begin
            LogCmdLine := 'AUTH [...]';
         end else begin
            LogCmdLine := CmdLine;
         end;

         Log( LOGID_INFO, '> ' + LogCmdLine );
         if CmdLine='' then exit;

         // check: supported command?
         CmdIndex := CmdIndexOf( AV[0] );
         if CmdIndex < 0 then begin
            HWriteLn( Res( R_UNKNOWN, 'Unsupported command' ) );
            exit;
         end;

         // check: authentication required and given?
         if ( ReCoCmdArray[CmdIndex].AuthReq <> 0 ) and
            ( CurrentUserID = ACTID_INVALID ) then begin
            HWriteLn( Res( R_AUTHREQ, 'Authentication required' ) );
            exit;
         end;

         // check: min./max. param count
         if ( AC < ReCoCmdArray[CmdIndex].MinPar ) or
            ( AC > ReCoCmdArray[CmdIndex].MaxPar ) then begin
            HWriteLn( Res( R_SYNTAX, 'Invalid parameter count' ) );
            exit;
         end;

         // check: sub-command allowed by user's profile?
         if not IsCommandAllowed( CmdIndex, CmdLine ) then begin
            HWriteLn( Res( R_UNKNOWN, 'Unsupported command' ) );
            exit;
         end;

         // execute command
         if AV[0]='AUTH'   then if Cmd_AUTH  ( AC, AV ) then exit;
         if AV[0]='HELP'   then if Cmd_HELP  ( AC, AV ) then exit;
         if AV[0]='LIVE'   then if Cmd_LIVE  ( AC, AV ) then exit;
         if AV[0]='LOG'    then if Cmd_LOG   ( AC, AV ) then exit;
         if AV[0]='NEWS'   then if Cmd_NEWS  ( AC, AV ) then exit;
         if AV[0]='QUIT'   then if Cmd_QUIT  ( AC, AV ) then exit;
         if AV[0]='SCRIPT' then if Cmd_SCRIPT( AC, AV ) then exit;
         if AV[0]='SERVER' then if Cmd_SERVER( AC, AV ) then exit;
         if AV[0]='TASK'   then if Cmd_TASK  ( AC, AV ) then exit;
         if AV[0]='USER'   then if Cmd_USER  ( AC, AV ) then exit;

         // unknown (sub-) command [should not happen]
         HWriteLn( Res( R_UNKNOWN, 'Unsupported command' ) );
         exit;

      finally
         AV.Free;
      end;

   except
      on E: Exception do begin
         Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
         Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
         TerminateConnection;
      end;
   end;
end;

procedure TServerClientRECO.TerminateLiveMode;
begin
   if Assigned( FLiveSenderThread ) then try
      FLiveSenderThread.Terminate;
      FLiveClient.LiveQueue.Add( nil ); // wake up
      FLiveSenderThread.WaitFor;
      FreeAndNil( FLiveSenderThread );
      FLimitLineLen := Hamster.Config.Settings.GetInt(hsLocalRecoLimitLineLen);
   except
      on E:Exception do begin
         Log( LOGID_DEBUG, Self.Classname
                         + '.TerminateLiveMode.A Exception: ' + E.Message );
      end;
   end;

   if Assigned( FLiveClient ) then try
      FreeAndNil( FLiveClient );
   except
      on E:Exception do begin
         Log( LOGID_DEBUG, Self.Classname
                         + '.TerminateLiveMode.B Exception: ' + E.Message );
      end;
   end;
end;

procedure TServerClientRECO.TerminateConnection;
begin
   TerminateLiveMode;

   try
      Sleep( Hamster.Config.Settings.GetInt(hsLocalTimeoutQuitDelay) );
      HDisconnect;
   except
      on E:Exception do begin
         Log( LOGID_DEBUG, Self.Classname
                         + '.TerminateConnection Exception: ' + E.Message );
      end;
   end;

   Terminate;
end;

procedure TServerClientRECO.HandleLiveCommand( const CmdLine: String );
var  LiveMsg: TLiveMsg;
     Data, s1: String;
     i1, i2: Integer;
begin
   LiveMsg := TLiveMsg.Create( CmdLine, FLiveClient.SessionKey );

   try
      // Sepecial: Request for returning to "Telnet Mode"?
      if LiveMsg.MsgType = LMXXX_RC_LIVE_OFF {=$8888} then begin

         TerminateLiveMode;
         HWriteLn( Res( R_OK0, 'LIVE mode stopped' ) );

         exit;

      end;

      // Sepecial: Request for stopping/restarting RC server?
      if LiveMsg.MsgType = LMREQ_LOCALSERVER_CONTROL then begin

         Data := LiveMsg.MsgData; // ("*"/servertype) CRLF servercontrol
         s1 := NextTopLine( Data );
         i1 := strtointdef( s1, -1 );
         i2 := strtointdef( NextTopLine( Data ), -1 );

         if ( s1 = '*' ) or ( i1 = ord(stRECO) ) then begin
            if ( i2 = ord(scSTOP) ) or ( i2 = ord(scRESTART) ) then begin

               // Yes, RC server will be stopped by this request.
               try
                  // send final confirmation
                  FLiveClient.ReceiveMsg( TLiveMsg.Create( LMREP_OK, '' ) );
                  Sleep( 1000 );

                  // terminate "Live Mode" and connection
                  TerminateConnection;

               finally
                  // cause delayed execution of request
                  Hamster.LiveServer.DelayedRequest( LiveMsg );
                  LiveMsg := nil;
               end;

               exit;

            end;
         end;
      end;

      // All other requests can be handled by LiveServer.
      try
         Hamster.LiveServer.ClientRequest( FLiveClient, LiveMsg );
      finally
         LiveMsg := nil;
      end;

   finally
      if Assigned( LiveMsg ) then LiveMsg.Free;
   end;
end;

function TServerClientRECO.Cmd_HELP( AC: Integer; AV: TStringList ): Boolean;

   function eq( const TestCmd: String;
                const TestSubCmdPar: String = '' ): Boolean;
   var  i: Integer;
   begin
      Result := False;

      if ( AC <> 1 ) or ( AV[1] <> TestCmd ) then exit;

      i := CmdIndexOf( TestCmd );
      if i < 0 then exit;

      with FReCoProfiles.Profile[CurrentUserProfile], ReCoCmdArray[i] do begin
         if ( AuthReq = 0 ) or ( CurrentUserID <> ACTID_INVALID ) then begin
            if ( AuthReq = 0 ) or ( CmdLineRE[TestCmd] <> '' ) then begin
               if TestSubCmdPar = '' then begin
                  Result := True;
               end else begin
                  Result := IsCommandAllowed( i,
                                              TestCmd + ' ' + TestSubCmdPar );
               end;
            end;
         end;
      end;
   end;

var  i: Integer;
     s: String;
     SL: TStringList;
begin
   Result := True;

   SL := TStringList.Create;
   try
                             
      // HELP [Command]
      if eq('AUTH') then begin
         SL.Add( 'AUTH [-SIMPLE] Username Password' );
         SL.Add( '   Login with given username and password.' );
         SL.Add( 'AUTH -SASL Mechanism [InitialParameters]' );
         SL.Add( '   Login with given SASL mechanism.' );
         SL.Add( '   Supported mechanisms: ' + Hamster.Config.Settings.GetStr(hsLocalRecoSASL) );
      end else if eq('LOG') then begin
         SL.Add( 'LOG LIST [-S Size] [-P Pattern]' );
         SL.Add( '   Show end of current logfile.' );
         SL.Add( '   -S: Size to read in KB (default 8)' );
         SL.Add( '   -P: Only lines which match given regex pattern' );
         SL.Add( 'LOG ROTATE' );
         SL.Add( '   Start a new log file.' );
      end else if eq('NEWS') then begin
         SL.Add( 'NEWS (ADD|DEL) GROUP Groupname' );
         SL.Add( '   Add/delete the given newsgroup.' );
         SL.Add( 'NEWS (ADD|DEL) PULL Groupname Servername' );
         SL.Add( '   Add/delete the given news-pull.' );
         SL.Add( 'NEWS LIST (GROUP|PULL) [-P Pattern]' );
         SL.Add( '   Return list of active groups/pulls.' );
         SL.Add( '   -P: Only lines which match given regex pattern' );
      end else if eq('QUIT') then begin
         SL.Add( 'QUIT' );
         SL.Add( '   Logout and terminate connection.' );
      end else if eq('SCRIPT') then begin
         SL.Add( 'SCRIPT LIST [-P Pattern]' );
         SL.Add( '   List available script files.' );
         SL.Add( '   -P: Only lines which match given regex pattern' );
         SL.Add( 'SCRIPT DIR [-P Pattern]' );
         SL.Add( '   Like LIST, but with extended information:' );
         SL.Add( '   Filetime TAB Filesize TAB [Path\]Filename' );
         SL.Add( 'SCRIPT START [-W] Scriptname [Scriptparameters]' );
         SL.Add( '   Start script with given parameters.' );
         SL.Add( '   -W: Wait until finished; also shows script output' );
         SL.Add( 'SCRIPT STOP' );
         SL.Add( '   Stop all running scripts.' );
         SL.Add( 'SCRIPT GET Scriptname' );
         SL.Add( '   Sends the content of the given server script.' );
         if FReCoProfiles.Profile[CurrentUserProfile].GrantScriptPut then begin
            SL.Add( 'SCRIPT PUT Scriptname' );
            SL.Add( '   Stores the text following this command under the' );
            SL.Add( '   given filename on the server.' );
         end;
         SL.Add( 'SCRIPT DEL Scriptname' );
         SL.Add( '   Deletes the given script on the server.' );
      end else if eq('SERVER') then begin
         SL.Add( 'SERVER LIST' );
         SL.Add( '   List servers with their current states.' );
         SL.Add( 'SERVER (STATE|START|STOP|RESTART) (NNTP|POP3|SMTP|RECO)' );
         SL.Add( '   Get or change state of the given local server.' );
      end else if eq('TASK') then begin
         SL.Add( 'TASK LIST [-P Pattern]' );
         SL.Add( '   Show current task list.' );
         SL.Add( '   -P: Only lines which match given regex pattern' );
      end else if eq('USER') then begin
         SL.Add( 'USER PASSWORD newpass1 newpass2' );
         SL.Add( '   Change password.' );
      end else if eq('LIVE') then begin
         if FReCoProfiles.Profile[CurrentUserProfile].GrantLiveOn then begin
            SL.Add( 'LIVE ON [ENCRYPT]' );
            SL.Add( '   Enters "Live Mode" with optional encryption.' );
            SL.Add( 'LIVE REQUEST Requestcode' );
            SL.Add( '   Execute Live request in "Telnet Mode".' );
         end;

      end else begin
         s := '';
         with FReCoProfiles.Profile[CurrentUserProfile] do begin
            for i := 0 to ReCoCmdCount - 1 do begin
               with ReCoCmdArray[i] do begin
                  if ( AuthReq = 0 ) or ( CurrentUserID <> ACTID_INVALID ) then begin
                     if ( AuthReq = 0 ) or ( CmdLineRE[CmdStr] <> '' ) then begin
                        if length(s) > 0 then s := s + ', ';
                        s := s + CmdStr;
                     end;
                  end;
               end;
            end;
         end;
         SL.Add( 'HELP [Command]' );
         SL.Add( '   Show description for the given command.' );
         SL.Add( '   Commands: ' + s );
      end;

      
      HWriteLn( Res( R_INFO, 'Help text follows' ) );
      for i:=0 to SL.Count-1 do HWriteLnQ( SL[i] );
      HWriteLn( '.' );

   finally
      SL.Free;
   end;
end;

function TServerClientRECO.Cmd_NEWS( AC: Integer; AV: TStringList ): Boolean;
var  RE, s: String;
     TS: TStringList;
     ok: Boolean;
     i: Integer;
begin
   Result := True;

   // NEWS (ADD|DEL) GROUP Groupname
   // NEWS (ADD|DEL) PULL Groupname Servername
   if (AV[1]='ADD') or (AV[1]='DEL') then begin
      s := UpperCase( AV[2] );

      if ( AC = 3 ) and ( s = 'GROUP' ) then begin
         if AV[1]='ADD' then ok := Hamster.Config.Newsgroups.Add( AV[3] )
                        else ok := Hamster.Config.Newsgroups.Del( AV[3] );
         if ok then HWriteLn( Res( R_OK0, AV[1] + ' GROUP successful.' ) )
               else HWriteLn( Res( R_FAILED0, AV[1] + ' GROUP failed.' ) );
         exit;

      end else if ( AC = 4 ) and ( s = 'PULL' ) then begin
         if Hamster.Config.NntpServers.IndexOfAlias( AV[4] ) < 0 then begin
            HWriteLn( Res( R_FAILED1,
                             AV[1] + ' PULL failed: Unknown server.' ) );
            exit;
         end;

         if AV[1]='ADD' then ok := Hamster.Config.NewsPulls.Add( AV[4], AV[3] )
                        else ok := Hamster.Config.NewsPulls.Del( AV[4], AV[3] );
         if ok then HWriteLn( Res( R_OK0, AV[1] + ' PULL successful.' ) )
               else HWriteLn( Res( R_FAILED0, AV[1] + ' PULL failed.' ) );
         exit;
      end;
   end;

   // NEWS LIST (GROUP|PULL) [-P Pattern]
   if AV[1]='LIST' then begin
      s := UpperCase( AV[2] );
      RE := OptStr( AC, AV, '-P', '.*' );

      if ( AC = 2 ) and ( (s='GROUP') or (s='PULL') ) then begin
         TS := TStringList.Create;
         try
            Hamster.Config.BeginRead;
            try
               if s='GROUP' then begin
                  for i:=0 to Hamster.Config.Newsgroups.Count-1 do
                     TS.Add( Hamster.Config.Newsgroups.Name[i] );
               end else begin
                  for i:=0 to Hamster.Config.NewsPulls.Count-1 do
                     TS.Add(      Hamster.Config.NewsPulls.Group[i]
                           + #9 + Hamster.Config.NewsPulls.Server[i] );
               end;
            finally
               Hamster.Config.EndRead;
            end;

            HWriteLn( Res( R_OK0, s + '-list follows' ) );
            for i:=0 to TS.Count-1 do begin
               if RE_Match( TS[i], RE, PCRE_CASELESS ) then HWriteLnQ( TS[i] );
            end;
            HWriteLn( '.' );
         finally
            TS.Free;
         end;
         exit;

      end;
   end;

   HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action/params)' ) );
end;

function TServerClientRECO.Cmd_SCRIPT( AC: Integer; AV: TStringList ): Boolean;
var  ScriptThread: TThreadExecuteScript;

   procedure SendScriptOutput;
   var  s: String;
   begin
      if not Assigned( ScriptThread ) then exit;
      if not Assigned( ScriptThread.OutputBuffer ) then exit;

      with ScriptThread do begin
         with OutputBuffer.LockList do try
            while Count > 0 do begin
               s := Strings[ 0 ];
               Delete( 0 );
               if HConnected then HWriteLnQ( s );
            end;
         finally OutputBuffer.UnlockList end;
      end;
   end;

var  i: Integer;
     s: String;
     RE, Filename, PathSub, ScriptPars, ScriptText: String;
     WaitForEnd, FullMode, Success: Boolean;
     FreeType: TThreadFreeTypes;
     SL: TStringList;
     HscFile: THscFile;
begin
   Result := True;

   // SCRIPT START Scriptname [Scriptparameters]
   if ( AC >= 2 ) and ( AV[1] = 'START' ) then begin

      WaitForEnd := OptBoo( AC, AV, '-W' );

      if not HscFileCheckSplitName( AV[2], PathSub, Filename ) then begin
         HWriteLn( Res( R_SYNTAX, 'Invalid script filename: ' + AV[2] ) );
         exit;
      end;

      HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsScripts );
      if not Assigned( HscFile ) then begin
         HWriteLn( Res( R_SYNTAX, 'Unknown script file: ' + AV[2] ) );
         exit;
      end;

      Scriptpars := '';
      if AC > 2 then ScriptPars := AV[3];
      for i:=4 to AC do Scriptpars := ' ' + Scriptpars + AV[i];

      if WaitForEnd then FreeType := tftFreeByCode
                    else FreeType := tftFreeOnTerminate;
      i := Hamster.HscFiles.Start( HscFile, ScriptPars, False,
                                   ScriptThread, FreeType, WaitForEnd );

      if i = 0 then begin

         if WaitForEnd then begin

            HWriteLn( Res( R_OK1, 'Script started - waiting' ) );

            try
               while WaitForSingleObject( ScriptThread.Handle, 1000 )
                     = WAIT_TIMEOUT do begin

                  if not HConnected then begin
                     ScriptThread.Terminate;
                     break;
                  end;
                  SendScriptOutput;
                  
               end;
            except
               ScriptThread.Terminate;
            end;

            try if HConnected then SendScriptOutput except end;
            try FreeAndNil( ScriptThread ) except end;

            HWriteLn( '.' )

         end else begin
            HWriteLn( Res( R_OK0, 'Script started' ) )
         end;

      end else begin
         if i=-2 then
            HWriteLn( Res( R_FAILED0, 'Script not found' ) )
         else
            HWriteLn( Res( R_FAILED1, 'Script start failed' ) );
      end;
      
   // SCRIPT LIST [-P Pattern]
   // SCRIPT DIR [-P Pattern]
   end else if ( AC >= 1 ) and
               ( (AV[1] = 'LIST') or (AV[1] = 'DIR') ) then begin

      RE := OptStr( AC, AV, '-P', '.*' );
      FullMode := ( AV[1] = 'DIR' );

      HWriteLn( Res( R_OK0, 'List of available script files follows' ) );
      Hamster.HscFiles.Refresh;
      SL := TStringList.Create;
      try
         if FullMode then
            Hamster.HscFiles.List( hfsAnyFile, hlftFileInfo, SL )
         else
            Hamster.HscFiles.List( hfsAnyFile, hlftPathFile, SL );
         for i:=0 to SL.Count-1 do begin
            s := SL[i];
            if RE_Match( s, RE, PCRE_CASELESS ) then HWriteLnQ( s );
         end;
      finally
         SL.Free;
      end;

      HWriteLn( '.' );

   // SCRIPT GET Scriptname
   end else if ( AC = 2 ) and ( AV[1] = 'GET' ) then begin
      if not HscFileCheckSplitName( AV[2], PathSub, Filename ) then begin
         HWriteLn( Res( R_SYNTAX, 'Invalid script filename: ' + AV[2] ) );
         exit;
      end;

      HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsAnyFile );
      if not Assigned( HscFile ) then begin
         HWriteLn( Res( R_SYNTAX, 'Unknown script file: ' + AV[2] ) );
         exit;
      end;

      SL := TStringList.Create;
      try
         Success := False;
         try
            Hamster.HscFiles.Load( HscFile, SL );
            Success := ( SL.Count > 0 );
         except on E: Exception do Log( LOGID_WARN, E.Message ) end;

         if Success then begin
            HWriteLn( Res( R_OK0, 'Script text follows' ) );
            for i:=0 to SL.Count-1 do HWriteLnQ( SL[i] );
            HWriteLn( '.' );
         end else begin
            HWriteLn( Res( R_FAILED0, 'Script file could not be loaded' ) );
         end;
      finally
         SL.Free;
      end;

   // SCRIPT PUT Scriptname                            
   end else if FReCoProfiles.Profile[CurrentUserProfile].GrantScriptPut
               and ( AC = 2 ) and ( AV[1] = 'PUT' ) then begin

      if not HscFileCheckSplitName( AV[2], PathSub, Filename ) then begin
         HWriteLn( Res( R_SYNTAX, 'Invalid script filename: ' + AV[2] ) );
         exit;
      end;

      try
         ScriptText := HRequestText( Res(R_SENDDATA,'Ok, send script text') );
      except
         on E: Exception do begin
            Log( LOGID_WARN, E.Message );
            ScriptText := '';
         end;
      end;
      if length( ScriptText ) = 0 then begin
         HWriteLn( Res( R_FAILED0, 'No script text received' ) );
         exit;
      end;

      Success := False;
      SL := TStringList.Create;
      try
         SL.Text := ScriptText;

         try
            Hamster.HscFiles.Save( PathSub, Filename, SL );
            Success := True;
         except on E: Exception do Log( LOGID_WARN, E.Message ) end;
      finally
         SL.Free;
      end;

      if Success then begin
         s := 'Note: Script "' + PathSub + Filename + '" uploaded by '
            + 'user ' + inttostr(CurrentUserID) + ' (' + CurrentUserName
            + ') via RC server from ' + ClientID + '!';
         Log( LOGID_WARN, s );
         HWriteLn( Res( R_OK0, 'Ok, script file saved' ) );
      end else begin
         HWriteLn( Res( R_FAILED0, 'Script file could not be saved' ) );
      end;

   // SCRIPT DELETE Scriptname
   end else if ( AC = 2 ) and ( AV[1] = 'DEL' ) then begin
      if not HscFileCheckSplitName( AV[2], PathSub, Filename ) then begin
         HWriteLn( Res( R_SYNTAX, 'Invalid script filename: ' + AV[2] ) );
         exit;
      end;

      HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsAnyFile );
      if not Assigned( HscFile ) then begin
         HWriteLn( Res( R_SYNTAX, 'Unknown script file: ' + AV[2] ) );
         exit;
      end;

      if Hamster.HscFiles.Delete( HscFile ) then begin
         HWriteLn( Res( R_OK0, 'Script file deleted' ) );
      end else begin
         HWriteLn( Res( R_FAILED0, 'Script file could not be deleted' ) );
      end;

   // SCRIPT STOP
   end else if ( AC = 1 ) and ( AV[1] = 'STOP' ) then begin
      Hamster.ActiveThreads.StopAllByType( attScript );
      HWriteLn( Res( R_OK0, 'Scripts stopped' ) )

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;

function TServerClientRECO.Cmd_AUTH( AC: Integer; AV: TStringList ): Boolean;
var  SASL_Name, Par, TempName: String;
     TempID, TempProfile, i: Integer;
begin
   Result := True;

   CurrentUserID      := ACTID_INVALID;
   CurrentUserName    := '';
   CurrentUserProfile := RC_PROFILE_NOACCESS;

   // AUTH [-SIMPLE] Username Password
   // AUTH -SASL Mechanism [InitialParameters]

   if OptBoo( AC, AV, '-SIMPLE' ) then
      SASL_Name := ''
   else
      SASL_Name := UpperCase( OptStr( AC, AV, '-SASL', '' ) );

   // identify user
   TempProfile := RC_PROFILE_NOACCESS;
   if SASL_Name = '' then begin // SIMPLE

      TempName := AV[1];
      TempID   := Hamster.Accounts.LoginID( TempName, AV[2], ClientIPn );

   end else begin // SASL

      if Pos( ' ' + SASL_Name + ' ',
              ' ' + Hamster.Config.Settings.GetStr(hsLocalRecoSASL) + ' ' )=0 then begin
         HWriteLn( Res( R_AUTHFAIL, 'Unrecognized authentication type' ) );
         exit;
      end;

      Par := '';
      if AC > 0 then Par := AV[1];
      for i:=2 to AC do Par := ' ' + Par + AV[i];

      Local_SASL_Login(
         Self, Res( R_AUTHCHLG, '%s' ),
         MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) ),
         SASL_NAME, Par, ClientIPn, TempID, TempName
      );

   end;

   // check user's rc-permission
   if TempID <> ACTID_INVALID then begin
      TempProfile := strtointdef(
                        Hamster.Accounts.Value[ TempID, apRemoteControl ],
                        RC_PROFILE_NOACCESS
                     );
      if ( TempProfile = RC_PROFILE_NOACCESS ) or
         ( TempProfile < RC_PROFILE_FIRST    ) or
         ( TempProfile > RC_PROFILE_LAST     ) then begin
         HWriteLn( Res( R_AUTHFAIL,
                        'No permission for remote control server.' ) );
         exit;
      end;
   end;

   // login user
   if TempID <> ACTID_INVALID then begin
      CurrentUserProfile := TempProfile;
      CurrentUserName    := TempName;
      CurrentUserID      := TempID;
      HWriteLn( Res( R_AUTHOK,   'Authentication successful.' ) );
      ClientsChange;
   end else begin
      HWriteLn( Res( R_AUTHFAIL, 'Authentication failed.'     ) );
   end;
end;

function TServerClientRECO.Cmd_USER( AC: Integer; AV: TStringList ): Boolean;
begin
   Result := True;

   // USER PASSWORD newpass1 newpass2
   if ( AC = 3 ) and ( AV[1] = 'PASSWORD' ) then begin

      if AV[2] = AV[3] then begin
         Hamster.Accounts.Value[ CurrentUserID, apPassword ] := AV[2];
         HWriteLn( Res( R_OK0, 'Password changed' ) );
      end else begin
         HWriteLn( Res( R_FAILED0, 'Failed (passwords different)' ) );
      end;

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;

function TServerClientRECO.Cmd_QUIT( AC: Integer; AV: TStringList ): Boolean;
begin
   Result := True;

   // QUIT
   if HConnected then HWriteLn( Res( R_OK0, 'Closing connection.' ) );
   TerminateConnection;
end;

function TServerClientRECO.Cmd_LOG( AC: Integer; AV: TStringList ): Boolean;
var  i, KB: Integer;
     RE: String;
     TS: TStringList;
begin
   Result := True;

   // LOG LIST [-S Size] [-P Pattern]
   if ( AC >= 1 ) and ( AV[1] = 'LIST' ) then begin

      KB := OptInt( AC, AV, '-S', 8 );
      RE := OptStr( AC, AV, '-P', '.*' );

      HWriteLn( Res( R_OK0, 'Logfile follows' ) );
      TS := TStringList.Create;
      try
         TS.Text := LogFile.LastLines( KB );
         for i:=0 to TS.Count-1 do try
            if RE_Match( TS[i], RE, PCRE_CASELESS ) then HWriteLnQ( TS[i] );
         except break end;
      finally
         TS.Free;
      end;
      HWriteLn( '.' );

   // LOG ROTATE
   end else if ( AC = 1 ) and ( AV[1] = 'ROTATE' ) then begin
      LogFile.RotateLog;
      HWriteLn( Res( R_OK0, 'Started new logfile.' ) );

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;

function TServerClientRECO.Cmd_TASK( AC: Integer; AV: TStringList ): Boolean;
var  i: Integer;
     RE: String;
     TS: TStringList;
begin
   Result := True;

   // TASK LIST [-P Pattern]
   if ( AC >= 1 ) and ( AV[1] = 'LIST' ) then begin

      RE := OptStr( AC, AV, '-P', '.*' );

      HWriteLn( Res( R_OK0, 'Current task list follows' ) );
      TS := TStringList.Create;
      try
         TS.Text := Hamster.ActiveThreads.ListAll + Hamster.ServerClients;

         for i:=0 to TS.Count-1 do try
            if RE_Match( TS[i], RE, PCRE_CASELESS ) then HWriteLnQ( TS[i] );
         except break end;
      finally
         TS.Free;
      end;
      HWriteLn( '.' );

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;

function LocalServerStrToType( const ServerStr: String;
                               out   ServerTyp: TServerTypes ) : Boolean;
var  s: String;
begin
   Result := True;
   s := UpperCase( ServerStr );
   if      s='NNTP' then ServerTyp := stNNTP
   else if s='POP3' then ServerTyp := stPOP3
   else if s='SMTP' then ServerTyp := stSMTP
   else if s='RECO' then ServerTyp := stRECO
   else Result := False;
end;

function TServerClientRECO.Cmd_SERVER( AC: Integer; AV: TStringList ): Boolean;
// SERVER (START|STOP|RESTART|STATE) (NNTP|POP3|SMTP|RECO)
var  st: TServerTypes;
begin
   Result := True;

   if ( AC = 1 ) and ( AV[1] = 'LIST' ) then begin

      HWriteLn( Res( R_INFO, 'Server list follows' ) );
      if Hamster.ServerControl( stNNTP, scIsActive )=1
         then HWriteLnQ( Res( R_OK0, 'NNTP started' ) )
         else HWriteLnQ( Res( R_OK1, 'NNTP stopped' ) );
      if Hamster.ServerControl( stPOP3, scIsActive )=1
         then HWriteLnQ( Res( R_OK0, 'POP3 started' ) )
         else HWriteLnQ( Res( R_OK1, 'POP3 stopped' ) );
      if Hamster.ServerControl( stSMTP, scIsActive )=1
         then HWriteLnQ( Res( R_OK0, 'SMTP started' ) )
         else HWriteLnQ( Res( R_OK1, 'SMTP stopped' ) );
      if Hamster.ServerControl( stRECO, scIsActive )=1
         then HWriteLnQ( Res( R_OK0, 'RECO started' ) )
         else HWriteLnQ( Res( R_OK1, 'RECO stopped' ) );
      HWriteLn( '.' );
      exit;
   end;

   if not LocalServerStrToType( AV[2], st ) then begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (unknown server type)' ) );
      exit;
   end;

   if ( AC = 2 ) and ( AV[1] = 'START' ) then begin
      if Hamster.ServerControl( st, scSTART )=1 then
         HWriteLn( Res( R_OK0, 'Server is started.' ) )
      else
         HWriteLn( Res( R_FAILED0, 'Server start failed.' ) );

   end else if ( AC = 2 ) and ( AV[1] = 'STOP' ) then begin
      if st=stRECO then begin
         HWriteLn( Res( R_OK0, 'Server will be stopped.' ) );
         Hamster.LiveServer.DelayedRequest( TLiveMsg.Create(
            LMREQ_LOCALSERVER_CONTROL,
            inttostr(ord(stRECO)) + CRLF + inttostr(ord(scSTOP))
         ) );
         TerminateConnection;
      end else begin
         if Hamster.ServerControl( st, scSTOP )=1 then
            HWriteLn( Res( R_OK0, 'Server is stopped.' ) )
         else
            HWriteLn( Res( R_FAILED0, 'Server stop failed.' ) );
      end;

   end else if ( AC = 2 ) and ( AV[1] = 'RESTART' ) then begin
      if st=stRECO then begin
         HWriteLn( Res( R_OK0, 'Server will be restarted.' ) );
         Hamster.LiveServer.DelayedRequest( TLiveMsg.Create(
            LMREQ_LOCALSERVER_CONTROL,
            inttostr(ord(stRECO)) + CRLF + inttostr(ord(scRESTART))
         ) );
         TerminateConnection;
      end else begin
         if Hamster.ServerControl( st, scRESTART )=1 then
            HWriteLn( Res( R_OK0, 'Server restarted.' ) )
         else
            HWriteLn( Res( R_FAILED0, 'Server restart failed.' ) );
      end;

   end else if ( AC = 2 ) and ( AV[1] = 'STATE' ) then begin
      if Hamster.ServerControl(st, scIsActive )=1 then
         HWriteLn( Res( R_OK0, 'Server is started.' ) )
      else
         HWriteLn( Res( R_OK1, 'Server is stopped.' ) );

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;

function TServerClientRECO.Cmd_LIVE( AC: Integer; AV: TStringList ): Boolean;
var  UKey, SKey, SessionKey: String;
     i: Integer;
     Reply: TLiveMsg;
     Msg: Word;
     Par: String;
     SL: TStringList;
begin
   Result := True;

   // LIVE ON [ENCRYPT]
   if FReCoProfiles.Profile[CurrentUserProfile].GrantLiveOn
      and ( AC >= 1 ) and ( AV[1] = 'ON' ) then begin

      SessionKey := '';

      if AC >= 2 then begin
         if UpperCase( AV[2] ) = 'ENCRYPT' then begin   
            UKey := MD5ofStr( Hamster.Accounts.Value[CurrentUserID,apPassword] );
            SKey := '';
            for i := 1 to 16 do SKey := SKey + inttohex( Random(256), 2 );
            HMAC_MD5( SKey, UKey, SessionKey );
         end else begin
            HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid parameter)' ) );
            exit;
         end;
      end;

      if SessionKey = '' then begin
         HWriteLn( Res( R_OK0, 'LIVE mode started ("8888" to stop)' ) )
      end else begin
         HWriteLn( Res( R_OK0, '<' + SKey + '>' ) );
      end;

      LimitLineLen := LimitTextSize;
      FLiveClient := TLiveClient.Create( Hamster.LiveServer, 1024, SessionKey );
      FLiveSenderThread := TLiveClientQueueSenderThread.Create( False, FLiveClient, Self );
      CounterChange; // trigger counter update
      TasksChange;   // trigger task list update
      ClientsChange; // trigger client list update

   end else if FReCoProfiles.Profile[CurrentUserProfile].GrantLiveOn
               and ( AC = 2 ) and ( AV[1] = 'REQUEST' ) then begin
      if length( AV[2] ) <> 4 then begin
         HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid 2nd parameter)' ) );
         exit;
      end;

      Msg := strtointdef( '$' + AV[2], 0 );
      if ( Msg and LMREQ ) <> LMREQ then begin
         HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid 2nd parameter)' ) );
         exit;
      end;

      Par := HRequestText( Res( R_SENDDATA, 'Ok, send parameters' ) );

      Reply := Hamster.LiveServer.ReplyFor( TLiveMsg.Create( Msg, Par ) );
      if Assigned( Reply ) then try
         if Reply.MsgType = LMREP_OK then begin
            HWriteLn( Res( R_OK0, inttohex(Reply.MsgType,4) + ' Ok, data follows' ) );
            SL := TStringList.Create;
            try
               SL.Text := Reply.MsgData;
               for i := 0 to SL.Count - 1 do HWriteLnQ( SL[i] );
               HWriteLn( '.' );
            finally SL.Free end;
         end else begin
            HWriteLn( Res( R_FAILED0, inttohex(Reply.MsgType,4) + ' Failed!' ) );
         end;
      finally Reply.Free end;

   end else begin
      HWriteLn( Res( R_SYNTAX, 'Syntax error (invalid action)' ) );
   end;
end;


function TServerClientRECO.FormatErrorReply( const ErrType: TClientErrorType;
                                             const ErrMsg: String ): String;
begin
   case ErrType of
      cetRefusedDontRetry, cetRefusedTemporary:
         Result := Res( R_NOPERM, ErrMsg );
      else
         Result := Res( R_SYSERR, ErrMsg );
   end;
end;

procedure TServerClientRECO.SendGreeting( var KeepConnection: Boolean;
                                          var Reason: String );
begin
   KeepConnection := False;

   if (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW then begin

      HWriteLn( Res( R_OK0, 'Hamster Remote Control, '
                + GetMyStringFileInfo( 'ProductName', 'Hamster' ) + ' '
                + GetMyVersionInfo ) );
      KeepConnection := True;

   end else begin

      Reason := 'Permission denied - closing connection.';
      HWriteLn( FormatErrorReply( cetRefusedDontRetry, Reason ) );

   end;
end;

function TServerClientRECO.IsCommandAllowed( CmdIndex: Integer;  
                                             const CmdLine: String ): Boolean;
begin
   Result := False;

   with ReCoCmdArray[ CmdIndex ] do begin
      if AuthReq = 0 then begin
         // no authentication required -> always ok
         Result := True;
      end else begin
         // check "allowed command-line" pattern
         with FReCoProfiles.Profile[CurrentUserProfile] do begin
            if ( CmdLineRE[CmdStr] <> '' ) and
               RE_Match( CmdLine, CmdLineRE[CmdStr], PCRE_CASELESS ) then Result := True;
         end;
      end;
   end;
end;

constructor TServerClientRECO.Create( ACreateSuspended: Boolean );
var  i: Integer;
begin
   inherited Create( True );

   FLimitLineLen  := Hamster.Config.Settings.GetInt(hsLocalRecoLimitLineLen);
   FLimitTextSize := Hamster.Config.Settings.GetInt(hsLocalRecoLimitTextSize);

   i := Hamster.Config.Settings.GetInt( hsLocalRecoInactivitySec );
   if i > 0 then FInactivitySecs := i;

   FReCoProfiles      := TReCoProfiles.Create;

   CurrentUserID      := ACTID_INVALID;
   CurrentUserName    := '';
   CurrentUserProfile := RC_PROFILE_NOACCESS;
   FLiveClient        := nil;
   FLiveSenderThread  := nil;

   if not ACreateSuspended then Resume;
end;

destructor TServerClientRECO.Destroy;
begin
   TerminateLiveMode;

   if Assigned( FReCoProfiles ) then FreeAndNil( FReCoProfiles );

   inherited;
end;

// ----------------------------------------------------------------------------

initialization
   Randomize;

end.
