// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cServerPOP3;

interface

{$INCLUDE Compiler.inc}

uses Classes, cArticle, cServerBase;

type
  TServerPOP3 = class( TServerBase );

type
  TServerClientPOP3 = class( TServerClientBase )
    private
      APOP_Stamp: String;
      lstRETR   : TStringList;
      lstDELE   : TStringList;

      function LoginUser( const CheckPassword: Boolean;
                          const Password: String ): String;
      function LoadMailFile( MailSL: TStringList; const MailFilename: String ): Boolean; overload;
      function LoadMailFile( MailSL: TMess;       const MailFilename: String ): Boolean; overload;

    protected
      function FormatErrorReply( const ErrType: TClientErrorType;
                                 const ErrMsg: String ): String; override;
      procedure SendGreeting( var KeepConnection: Boolean;
                              var Reason: String ); override;

      function Cmd_QUIT( const Par: String ): Boolean;
      function Cmd_USER( const Par: String ): Boolean;
      function Cmd_PASS( const Par: String ): Boolean;
      function Cmd_APOP( const Par: String ): Boolean;
      function Cmd_AUTH( const Par: String ): Boolean;
      function Cmd_RSET( const Par: String ): Boolean;
      function Cmd_STAT( const Par: String ): Boolean;
      function Cmd_LIST( const Par: String ): Boolean;
      function Cmd_RETR( const Par: String ): Boolean;
      function Cmd_TOP ( const Par: String ): Boolean;
      function Cmd_UIDL( const Par: String ): Boolean;
      function Cmd_DELE( const Par: String ): Boolean;
      function Cmd_NOOP( const Par: String ): Boolean;
      function Cmd_HELP( const Par: String ): Boolean;
      function Cmd_CAPA( const Par: String ): Boolean;

    public
      procedure HandleCommand( const CmdLine: String ); override;

      constructor Create( ACreateSuspended: Boolean ); override;
      destructor Destroy; override;
  end;

implementation

uses uConst, uVar, SysUtils, Windows, uTools, cIPAccess, uHamTools,
     cArtFiles, cAccounts, uMD5, uCRC32, cLogFileHamster, uConstVar, cHamster;

// ---------------------------------------------------- TServerClientPOP3 -----

function TServerClientPOP3.LoginUser( const CheckPassword: Boolean;
                                      const Password: String ): String;
var  No, res    : Integer;
     SR         : TSearchRec;
     MailBoxPath: String;
begin
     Result := '-ERR System-error, check logfile. [0]';

     try
        if Assigned(lstRETR) then lstRETR.Clear;
        if Assigned(lstDELE) then lstDELE.Clear;

        Result := '-ERR Authentication rejected';
        if CurrentUserName='' then exit;

        if CheckPassword then begin
           CurrentUserID := Hamster.Accounts.LoginID(
                               CurrentUserName, Password, ClientIPn );
        end;
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           Result := '-ERR No permission';
           exit;
        end;

        // log authentication for SMTP-after-POP3
        if Hamster.Accounts.Value[ CurrentUserID, apMailSend ] = '1' then begin
           try
              Hamster.Accounts.LogAuthenticatedLogin( FClientAD.S_addr );
           except
              on E:Exception do Log( LOGID_ERROR, 'Pop3.LogAuthLogin-Exception: ' + E.Message );
           end;
        end;

        if not Hamster.Accounts.HasMailbox( CurrentUserID ) then begin
           CurrentUserID := ACTID_INVALID;
           CurrentUserName := '';
           Result := '-ERR No mailbox';
           exit;
        end;

        // authentication ok, try to lock mailbox
        if not Hamster.Accounts.MailboxLock( CurrentUserID, True ) then begin
           Result := '-ERR unable to lock mailbox';
           CurrentUserName := '';
           CurrentUserID := ACTID_INVALID;
           exit;
        end;

        // read list of available mails
        CS_LOCK_MAILBOX_ALL.Enter;
        try
           try
              No := 0;
              MailBoxPath := Hamster.Accounts.MailboxPath( CurrentUserID );

              res := SysUtils.FindFirst( MailBoxPath + '*.msg', faAnyFile, SR );
              if res=0 then begin
                 while res=0 do begin
                    inc( No ); // 1..n
                    lstRETR.AddObject( MailBoxPath + SR.Name, Pointer(No) );
                    res := SysUtils.FindNext( SR );
                 end;
                 SysUtils.FindClose( SR );
              end;
              Result := '+OK mailbox locked, ' + inttostr(lstRETR.Count) + ' messages';
           except
              on E:Exception do begin
                 Log( LOGID_ERROR, 'Pop3.LoginUser.ReadMailbox-Exception: ' + E.Message );
              end;
           end;
        finally CS_LOCK_MAILBOX_ALL.Leave end;

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Pop3.LoginUser-Exception: ' + E.Message );
           Result := '-ERR System-error, check logfile.';
        end;
     end;
end;

function TServerClientPOP3.LoadMailFile( MailSL: TStringList;
                                         const MailFilename: String ): Boolean;
begin
     Result := True;

     try
        MailSL.LoadFromFile( MailFilename );
     except
        on E: Exception do begin
           Log( LOGID_ERROR, Format(
              'Couldn''t load mail: %s', [MailFilename] ) );
           Log( LOGID_ERROR, 'Error: ' + E.Message );
           Result := False;
        end;
     end;
end;

function TServerClientPOP3.LoadMailFile( MailSL: TMess;
                                         const MailFilename: String ): Boolean;
begin
     Result := True;

     try
        MailSL.LoadFromFile( MailFilename );
     except
        on E: Exception do begin
           Log( LOGID_ERROR, Format(
              'Couldn''t load mail: %s', [MailFilename] ) );
           Log( LOGID_ERROR, 'Error: ' + E.Message );
           Result := False;
        end;
     end;
end;

procedure TServerClientPOP3.HandleCommand( const CmdLine: String );
var  LogCmdLine, Cmd, Par, s: String;
     j: Integer;
begin
     try
        if not HConnected then exit;

        // Extract command
        j := PosWhSpace( CmdLine );
        if j=0 then begin
           Cmd := UpperCase( CmdLine );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( CmdLine, 1, j-1 ) );
           Par := TrimWhSpace( copy( CmdLine, j+1, 512 ) );
        end;

        if (Cmd='PASS') then begin
           LogCmdLine := 'PASS [...]';
        end else begin
           if (Cmd='APOP') or (Cmd='AUTH') then begin
              s := Par;
              j := PosWhSpace( s );
              if j>0 then s:=copy(s,1,j-1);
              LogCmdLine := Cmd + ' ' + s + ' [...]';
           end else begin
              LogCmdLine := CmdLine;
           end;
        end;

        Log( LOGID_INFO, '> ' + LogCmdLine );
        if CmdLine='' then exit;

        // commands (no authentication required)
        if Cmd='APOP'     then if Cmd_APOP( Par ) then exit;
        if Cmd='AUTH'     then if Cmd_AUTH( Par ) then exit;
        if Cmd='CAPA'     then if Cmd_CAPA( Par ) then exit;
        if Cmd='HELP'     then if Cmd_HELP( Par ) then exit;
        if Cmd='PASS'     then if Cmd_PASS( Par ) then exit;
        if Cmd='QUIT'     then if Cmd_QUIT( Par ) then exit;
        if Cmd='USER'     then if Cmd_USER( Par ) then exit;

        // check authentication
        if CurrentUserID=ACTID_INVALID then begin
           if Pos( '|'+Cmd+'|',
                   '|DELE|LIST|RETR|RSET|STAT|TOP|UIDL|NOOP|'
                 )>0 then begin
              HWriteLn( '-ERR Authentication required' );
              exit;
           end;
        end;

        // commands (authentication required)
        if Cmd='DELE'     then if Cmd_DELE( Par ) then exit;
        if Cmd='LIST'     then if Cmd_LIST( Par ) then exit;
        if Cmd='RETR'     then if Cmd_RETR( Par ) then exit;
        if Cmd='RSET'     then if Cmd_RSET( Par ) then exit;
        if Cmd='STAT'     then if Cmd_STAT( Par ) then exit;
        if Cmd='TOP'      then if Cmd_TOP ( Par ) then exit;
        if Cmd='UIDL'     then if Cmd_UIDL( Par ) then exit;
        if Cmd='NOOP'     then if Cmd_NOOP( Par ) then exit;

        // unknown (sub-) command
        HWriteLn( '-ERR Command not implemented.' );
        Log( LOGID_INFO, 'Unsupported POP3-command: ' + CmdLine );

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
           try HDisconnect except end;
        end;
     end;
end;

function TServerClientPOP3.Cmd_QUIT( const Par: String ): Boolean;
var  j: Integer;
begin
     Result := True;

     if lstDELE.Count>0 then begin
        for j:=0 to lstDELE.Count-1 do begin
           SysUtils.DeleteFile( lstDELE[j] );
        end;
        lstDELE.Clear;
     end;

     if CurrentUserID >= 0 then begin
        Hamster.Accounts.MailboxLock( CurrentUserID, False );
        CurrentUserID := ACTID_INVALID;
     end;

     if HConnected then HWriteLn( '+OK closing connection - goodbye!' );

     Sleep( Hamster.Config.Settings.GetInt(hsLocalTimeoutQuitDelay) );
     try
        HDisconnect;
     except
        on E:Exception do Log(LOGID_DEBUG, 'Exception on Socket.Close: ' + E.Message );
     end;

     Terminate;
end;

function TServerClientPOP3.Cmd_USER( const Par: String ): Boolean;
begin
     Result := True;

     if CurrentUserID >= 0 then begin
        HWriteLn( '-ERR Command not allowed in transaction state' );
        exit;
     end;

     CurrentUserName := TrimWhSpace( Par );
     CurrentUserID   := ACTID_INVALID;
     HWriteLn( '+OK More authentication information required' );
end;

function TServerClientPOP3.Cmd_PASS( const Par: String ): Boolean;
var  s: String;
begin
     Result := True;

     if CurrentUserID >= 0 then begin
        HWriteLn( '-ERR Command not allowed in transaction state' );
        exit;
     end;

     try
        s := LoginUser( True, TrimWhSpace( Par ) );
        HWriteLn( s );
        ClientsChange;
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Pop3.Cmd_PASS-Exception: ' + E.Message );
        end;
     end;
end;

function TServerClientPOP3.Cmd_APOP( const Par: String ): Boolean;
var  md5val, s, p, cmp: String;
     j: Integer;
begin
     Result := True;

     if CurrentUserID >= 0 then begin
        HWriteLn( '-ERR Command not allowed in transaction state' );
        exit;
     end;

     CurrentUserID := ACTID_INVALID;

     j := PosWhSpace( Par );
     if j=0 then begin
        HWriteLn( '-ERR Invalid params' );
        exit;
     end;
     CurrentUserName := copy( Par, 1, j-1 );
     md5val := TrimWhSpace( copy( Par, j+1, 255 ) );

     CurrentUserID := Hamster.Accounts.UserIDOf( CurrentUserName );
     if CurrentUserID=ACTID_INVALID then begin
        HWriteLn( '-ERR Authentication failed.' );
        exit;
     end;

     p := Hamster.Accounts.Value[ CurrentUserID, apPassword ];
     s := MD5ofStr( APOP_Stamp + p );
     cmp := '';
     for j:=1 to length(s) do cmp := cmp + lowercase( inttohex( ord(s[j]), 2 ) );

     if cmp=md5val then begin
        s := LoginUser( True, p );
        HWriteLn( s );
        ClientsChange;
     end else begin
        CurrentUserID := ACTID_INVALID;
        HWriteLn( '-ERR Authentication failed.' );
     end;
end;

function TServerClientPOP3.Cmd_AUTH( const Par: String ): Boolean;
// "AUTH Mechanism" (RFC 1734, 2222)
var  SASL_NAME, SASL_PAR, s, h: String;
     j: Integer;
begin
     Result := True;

     if CurrentUserID >= 0 then begin
        HWriteLn( '-ERR Command not allowed in transaction state' );
        exit;
     end;

     CurrentUserID := ACTID_INVALID;
     CurrentUserName := '';

     j := PosWhSpace( Par );
     if j = 0 then begin
        SASL_NAME := UpperCase( Par );
        SASL_PAR  := '';
     end else begin
        SASL_NAME := UpperCase( copy( Par, 1, j-1 ) );
        SASL_PAR  := copy( Par, j+1, 255 );
     end;

     // if no SASL mechanism is given, return list
     if SASL_NAME = '' then begin
        HWriteLn( '+OK List of supported SASL mechanisms follows' );

        s := Hamster.Config.Settings.GetStr( hsLocalPop3SASL );
        repeat
           h := NextWhSpacePart( s );
           if h = '' then break;
           HWriteLnQ( h );
        until s = '';

        HWriteLn( '.' );
        exit;
     end;

     // check, if requested mechanism is supported/enabled
     if Pos( ' ' + SASL_Name + ' ',
             ' ' + Hamster.Config.Settings.GetStr(hsLocalPop3SASL) + ' ' )=0 then begin
        HWriteLn( '-ERR Unrecognized authentication mechanism: '+ SASL_NAME );
        exit;
     end;

     // try SASL authentication with requested mechanism
     if not Local_SASL_Login(
           Self, '+ %s',
           MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) ),
           SASL_NAME, SASL_PAR, ClientIPn,
           CurrentUserID, CurrentUserName
        ) then begin
        HWriteLn( '-ERR Authentication failed.' );
        exit;
     end;

     // authentication successful, proceed with login
     HWriteLn( LoginUser( False, '' ) );
     ClientsChange;
end;

function TServerClientPOP3.Cmd_RSET( const Par: String ): Boolean;
var  i, k: Integer;
begin
     Result := True;

     while lstDELE.Count>0 do begin
        lstRETR.AddObject( lstDELE[0], lstDELE.Objects[0] );
        lstDELE.Delete( 0 );
     end;

     // resort by numbers
     for i:=0 to lstRETR.Count-2 do begin
        for k:=i+1 to lstRETR.Count-1 do begin
            if LongInt(lstRETR.Objects[i])>LongInt(lstRETR.Objects[k]) then begin
               lstRETR.Exchange( i, k );
            end;
        end;
     end;

     HWriteLn( '+OK mailbox has ' + inttostr(lstRETR.Count) + ' messages' );
end;

function TServerClientPOP3.Cmd_STAT( const Par: String ): Boolean;
var  cnt, byt, j: Integer;
begin
     Result := True;

     cnt := 0;
     byt := 0;
     for j:=0 to lstRETR.Count-1 do begin
        inc( cnt );
        byt := byt + GetFileSize( lstRETR[j] );
     end;

     HWriteLn( '+OK' + ' ' + inttostr(cnt) + ' ' + inttostr(byt) );
end;

function TServerClientPOP3.Cmd_LIST( const Par: String ): Boolean;
var  No, j, byt: Integer;
     h: String;
begin
     Result := True;

     if Par='' then begin
        HWriteLn( '+OK' + ' ' + inttostr(lstRETR.Count) + ' messages' );
        for j:=0 to lstRETR.Count-1 do begin
           byt := GetFileSize( lstRETR[j] );
           HWriteLnQ( inttostr(LongInt(lstRETR.Objects[j])) + ' ' + inttostr(byt) );
        end;
        HWriteLn( '.' );
        exit;
     end;

     No := strtoint( Par );
     h := '-ERR no such message';
     for j:=0 to lstRETR.Count-1 do begin
        if LongInt(lstRETR.Objects[j])=No then begin
           byt := GetFileSize( lstRETR[j] );
           h := '+OK' + ' ' + inttostr( LongInt(lstRETR.Objects[j]) ) + ' ' + inttostr(byt);
           break;
        end;
     end;
     HWriteLn( h );
end;

function TServerClientPOP3.Cmd_RETR( const Par: String ): Boolean;
var  OK: Boolean;
     MailText: TMess;
     MailNo, j, byt, snd: Integer;
     buf: String;
begin
     Result := True;
     OK := False;

     if Par <> '' then begin
        MailNo := strtoint( Par );
        MailText := TMess.Create;
        try
           for j := 0 to lstRETR.Count - 1 do begin
              if LongInt(lstRETR.Objects[j]) = MailNo then begin

                 Log( LOGID_DEBUG, 'RETR ' + inttostr(MailNo)
                                 + ' returns file ' + lstRETR[j] );
                 if not LoadMailFile( MailText, lstRETR[j] ) then break;

                 byt := GetFileSize( lstRETR[j] );
                 HWriteLn( '+OK' + ' ' + inttostr(byt) + ' octets' );

                 // HWrite  ( TextToRfcWireFormat( MailText.FullText ) );
                 buf := TextToRfcWireFormat( MailText.FullText );
                 snd := 1;
                 while snd <= length(buf) do begin
                    HWrite( copy( buf, snd, 1024 ) );
                    inc( snd, 1024 );
                 end;

                 HWriteLn( '.' );

                 OK := True;
                 break;

              end;
           end;
        finally MailText.Free end;
     end;

     if not OK then HWriteLn( '-ERR no such message' );
end;

function TServerClientPOP3.Cmd_TOP( const Par: String ): Boolean;
var  j, k, No, Cnt, byt: Integer;
     OK, Body: Boolean;
     Art: TStringList;
     SendBuf, s: String;
begin
     Result := True;

     j := Pos( ' ', Par );
     if j=0 then begin
        HWriteLn( '-ERR invalid format for TOP' );
        exit;
     end;

     No  := strtoint( copy(Par,1,j-1) );
     Cnt := strtoint( copy(Par,j+1,255) );
     OK  := False;
     Art := TStringList.Create;

     for j:=0 to lstRETR.Count-1 do begin
        if LongInt(lstRETR.Objects[j])=No then begin
           //Art.LoadFromFile( lstRETR[j] );
           if not LoadMailFile( Art, lstRETR[j] ) then break;

           byt := GetFileSize( lstRETR[j] );
           HWriteLn( '+OK' + ' ' + inttostr(byt) + ' octets' );

           Body := False;
           SetLength( SendBuf, 0 );
           
           for k := 0 to Art.Count-1 do begin

              // HWriteLnQ( Art[k] );
              s := Art[k];
              if copy(s,1,1) = '.' then SendBuf := SendBuf + '.';
              SendBuf := SendBuf + s + CRLF;

              if Body then begin
                 dec( Cnt );
                 if Cnt < 0 then break;
              end else begin
                 if Art[k]='' then Body := True;
              end;
           end;

           if length(SendBuf) > 0 then HWrite( SendBuf );
           HWriteLn( '.' );
           OK := True;
           break;
        end;
     end;

     Art.Free;
     if not OK then HWriteLn( '-ERR no such message' );
end;

function TServerClientPOP3.Cmd_UIDL( const Par: String ): Boolean;
var  Art: TMess;
     OK : Boolean;
     No, j: Integer;
     h: String;
begin
     Result := True;

     if Par<>'' then begin
        No := strtoint( Par );
        OK := False;
        Art := TMess.Create;
        try
           for j:=0 to lstRETR.Count-1 do begin
              if LongInt(lstRETR.Objects[j])=No then begin
                 // Art.LoadFromFile( lstRETR[j] );
                 if not LoadMailFile( Art, lstRETR[j] ) then break;

                 h := Art.GetOurXHeader( 'UIDL' );
                 if h = '' then h := MD5ToHex( MD5OfStr( Art.FullText ) );

                 HWriteLn( '+OK' + ' ' + inttostr(no) + ' ' + h );
                 ok := True;
                 break;
              end;
           end;
        finally Art.Free end;
        if not OK then HWriteLn( '-ERR no such message' );
        exit;
     end;


     HWriteLn( '+OK' );
     Art := TMess.Create;
     try
        for j:=0 to lstRETR.Count-1 do begin
           // Art.LoadFromFile( lstRETR[j] );
           if LoadMailFile( Art, lstRETR[j] ) then begin
              h := Art.GetOurXHeader( 'UIDL' );
              if h='' then h:=inttohex( StrToCRC32(Art.FullText), 8 );
              HWriteLnQ( inttostr(LongInt(lstRETR.Objects[j])) + ' ' + h );
           end;
        end;

        HWriteLn( '.' );
     finally Art.Free end;
end;


function TServerClientPOP3.Cmd_DELE( const Par: String ): Boolean;
var  ok: Boolean;
     No, j: Integer;
begin
     Result := True;
     ok := False;

     if Par<>'' then begin
        No := strtoint( Par );
        for j:=0 to lstRETR.Count-1 do begin
           if LongInt(lstRETR.Objects[j])=No then begin
              lstDELE.AddObject( lstRETR[j], lstRETR.Objects[j] );
              lstRETR.Delete( j );
              HWriteLn( '+OK' + ' ' + 'message ' + inttostr(no)
                              + ' marked for deletion' );
              ok := True;
              break;
           end;
        end;
     end;

     if not OK then HWriteLn( '-ERR no such message' );
end;

function TServerClientPOP3.Cmd_NOOP( const Par: String ): Boolean;
begin
     Result := True;
     HWriteLn( '+OK localhost' );
end;

function TServerClientPOP3.Cmd_HELP( const Par: String ): Boolean;
begin
     Result := True;

     HWriteLn( '+OK Implemented commands follow' );

     HWriteLnQ( '    apop' );
     HWriteLnQ( '    auth [mechanism]' );
     HWriteLnQ( '    capa' );
     HWriteLnQ( '    dele Number' );
     HWriteLnQ( '    help' );
     HWriteLnQ( '    list [Number]' );
     HWriteLnQ( '    noop' );
     HWriteLnQ( '    pass Password' );
     HWriteLnQ( '    quit' );
     HWriteLnQ( '    retr Number' );
     HWriteLnQ( '    rset' );
     HWriteLnQ( '    stat' );
     HWriteLnQ( '    top Number Lines' );
     HWriteLnQ( '    uidl [Number]' );
     HWriteLnQ( '    user Username' );

     HWriteLn( '.' );
end;

function TServerClientPOP3.Cmd_CAPA( const Par: String ): Boolean;
// CAPA (RFC 2449)
begin
     Result := True;

     HWriteLn( '+OK Capability list follows' );

     HWriteLnQ( 'TOP' );
     HWriteLnQ( 'USER' );
     if Hamster.Config.Settings.GetStr(hsLocalPop3SASL)<>'' then begin
        HWriteLnQ( 'SASL ' + Hamster.Config.Settings.GetStr(hsLocalPop3SASL) );
     end;
     HWriteLnQ( 'EXPIRE NEVER' );
     HWriteLnQ( 'UIDL' );
     if CurrentUserID<>ACTID_INVALID then
        HWriteLnQ( 'IMPLEMENTATION ' + OUR_VERINFO );

     HWriteLn( '.' );
end;

function TServerClientPOP3.FormatErrorReply( const ErrType: TClientErrorType;
                                             const ErrMsg: String ): String;
begin
   Result := '-ERR ' + ErrMsg;
end;

procedure TServerClientPOP3.SendGreeting( var KeepConnection: Boolean;
                                          var Reason: String );
begin
   KeepConnection := False;

   if (IPAccess and IPACC_ACCESS_RO)=IPACC_ACCESS_RO then begin

      HWriteLn( '+OK Hamster-POP3, '
                   + GetMyStringFileInfo('ProductName','Hamster') + ' '
                   + GetMyVersionInfo + ' ' + APOP_Stamp );
      KeepConnection := True;

   end else begin

      Reason := 'Permission denied - closing connection.';
      HWriteLn( FormatErrorReply( cetRefusedDontRetry, Reason ) );

   end;
end;

constructor TServerClientPOP3.Create( ACreateSuspended: Boolean );
var  i: Integer;
begin
     inherited Create( True );

     FLimitLineLen  := Hamster.Config.Settings.GetInt(hsLocalPop3LimitLineLen);
     FLimitTextSize := Hamster.Config.Settings.GetInt(hsLocalPop3LimitTextSize);

     i := Hamster.Config.Settings.GetInt( hsLocalPop3InactivitySec );
     if i > 0 then FInactivitySecs := i;

     APOP_Stamp := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
     CurrentUserID   := ACTID_INVALID;
     CurrentUserName := '';
     lstRETR := TStringList.Create;
     lstDELE := TStringList.Create;

     if not ACreateSuspended then Resume;
end;

destructor TServerClientPOP3.Destroy;
begin
     if CurrentUserID>=0 then Hamster.Accounts.MailboxLock( CurrentUserID, False );
     if Assigned(lstRETR) then begin lstRETR.Free; lstRETR:=nil; end;
     if Assigned(lstDELE) then begin lstDELE.Free; lstDELE:=nil; end;
     inherited;
end;

// ----------------------------------------------------------------------------

end.

