// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cServerNNTP;

interface

{$INCLUDE Compiler.inc}

uses Classes, cServerBase, cArticle, cFiltersNews;

type
  TServerNNTP = class( TServerBase );

  TNntpMode = ( nmDefault, nmIHAVE, nmTAKETHIS );

  TServerClientNNTP = class( TServerClientBase )
    private
      CurrentGroup   : LongInt;
      CurrentArtNo   : Integer;
      UserMayPost    : Boolean;
      UserMayFeed    : Boolean;
      UserMayNewNews : Boolean;
      UserMayXHSearch: Boolean;
      UserAutoSubMode: Integer;
      AutoGroupName  : String;
      AutoGroupPends : Boolean;
      GroupList      : TStringList;
      RecommendedMID : String;
      ScoreFileFeed  : TFiltersNews;
      ScoreFilePost  : TFiltersNews;
      NntpMode       : TNntpMode;
      PermRead, PermPost: String;

      function ReloadGroupList: Boolean;
      function LoginUser( const Password: String ): String;
      procedure SetCurrentGroup( const Groupname: String );
      function Report_XHSearchResults( const Groupname: String;
                                       const ArticleNo: LongWord;
                                       const Article  : TMess ): Boolean;
      function MailPostedArticle( const ArtText, SendTo: String;
                                  const IsModerated    : Boolean ): String;
      function PostPermission_KnownGroup( const GroupName: String ): Boolean;
      function PostPermission_UnknownGroup( const GroupName: String ): Boolean;
      function HandlePostedArticle( const ArtText: String ): String;
      function HandleFeededArticle( const MessageId, ArtText: String;
                                    const isIHAVE: Boolean ): String;

    protected
      function FormatErrorReply( const ErrType: TClientErrorType;
                                 const ErrMsg: String ): String; override;
      procedure SendGreeting( var KeepConnection: Boolean;
                              var Reason: String ); override;

      procedure AutoGroupAddPulls( const Groupname: String );
      procedure AutoGroupPrepare( const Groupname: String );
      procedure AutoGroupCommit;
      procedure AutoGroupRollback;

      function Cmd_ARTINFOS ( const Cmd, Par: String ): Boolean;
      function Cmd_LASTNEXT ( const Cmd, Par: String ): Boolean;
      function Cmd_QUIT     ( const Par: String ): Boolean;
      function Cmd_AUTHINFO ( const Par: String ): Boolean;
      function Cmd_POST     ( const Par: String ): Boolean;
      function Cmd_SLAVE    ( const Par: String ): Boolean;
      function Cmd_GROUP    ( const Par: String ): Boolean;
      function Cmd_NEWGROUPS( const Par: String ): Boolean;
      function Cmd_LIST     ( const Par: String ): Boolean;
      function Cmd_LISTGROUP( const Par: String ): Boolean;
      function Cmd_MODE     ( const Par: String ): Boolean;
      function Cmd_DATE     ( const Par: String ): Boolean;
      function Cmd_XOVER    ( const Par: String ): Boolean;
      function Cmd_XPAT     ( const Par: String ): Boolean;
      function Cmd_XHDR     ( const Par: String ): Boolean;
      function Cmd_NEWNEWS  ( const Par: String ): Boolean;
      function Cmd_XHSEARCH ( const Par: String ): Boolean;
      function Cmd_XHHELP   ( const Par: String ): Boolean;
      function Cmd_HELP     ( const Par: String ): Boolean;
      function Cmd_IHAVE    ( const Par: String ): Boolean;
      function Cmd_CHECK    ( const Par: String ): Boolean;
      function Cmd_TAKETHIS ( const Par: String ): Boolean;

    public
      procedure HandleCommand( const CmdLine: String ); override;

      constructor Create( ACreateSuspended: Boolean ); override;
      destructor Destroy; override;
  end;

const
  PERM_NOTH = 0;
  PERM_READ = 1;
  PERM_POST = 2;

function GetPermissionForGroup( Group, PermPost, PermRead: String ): LongInt;

implementation

uses uType, uConst, uConstVar, uVar, SysUtils, Windows, uTools, cIPAccess,
     cArtFiles, cAccounts, cPCRE, uCRC32, uDateTime, cLogFileHamster,
     cNewsSearcher, uHamTools, cHamster, cMailItem, cMailUsers,
     cMailDispatcher, uWinSock, cHscAction;

function NntpDateTimeToDateTime( DateStr, TimeStr: String ): TDateTime;
// Convert a NEWGROUPS/NEWNEWS-timepoint ('YYMMDD','HHNNSS') into TDateTime.
begin
   Result := 0;

   case length(DateStr) of
      5: begin // YMMDD, Y=[0-9] -> YYYYMMDD
            if DateStr[1] in ['0'..'9'] then DateStr:='200'+DateStr else exit;
         end;
      6: begin // YYMMDD -> YYYYMMDD
            if strtointdef(copy(DateStr,1,2),0)>=50 then DateStr:='19'+DateStr
                                                    else DateStr:='20'+DateStr;
         end;
      7: begin // 1YYMMDD -> YYYYMMDD
            System.Delete( DateStr, 1, 1 );
            if strtointdef(copy(DateStr,1,2),0)>=50 then DateStr:='19'+DateStr
                                                    else DateStr:='20'+DateStr;
         end;
   end;

   if length(DateStr)<>8 then exit;
   if length(TimeStr)<>6 then exit;

   Result := TimeStampToDateTime( DateStr + TimeStr );
end;

function GroupMatchesPatterns( const GroupName  : String;
                               const WildMatPats: TStringList ): Boolean;
// Check, if GroupName is matched by the wildmat-patterns given in WildMatPats.
var  i: Integer;
     w: String;
     g: PChar;
begin
   // optimize: all or none are selected
   if WildMatPats.Count = 1 then begin
      if WildMatPats[0] = '*'  then begin Result := True;  exit end;
      if WildMatPats[0] = '!*' then begin Result := False; exit end;
   end;

   // test patterns - last matching pattern wins
   Result := False;
   g := PChar( GroupName );
   for i := WildMatPats.Count-1 downto 0 do begin
       w := WildMatPats[i];
       if copy(w,1,1) = '!' then begin // '!' as 1st char means 'not'
          if WildMat( g, PChar( copy(w,2,999) ) ) then break;
       end else begin
          if WildMat( g, PChar(w) ) then begin Result := True; break end;
       end;
   end;
end;

function SplitXrefGroups( const XrefValue : String;
                          const XrefGroups: TStringList;
                          WantHost: Boolean = False ): Integer;
// Splits the value of a Xref:-header ('host WSP grp:art WSP grp:art ...').
// XrefGroups.Strings will be filled with the group-names, .Objects with the
// corresponding article-numbers. If WantHost is True, the leading host-name
// is returned in XrefGroups[0] (if .Sorted=False!) with a number of -1.
var  j, n: Integer;
     Xref, s: String;
begin
   Result := 0;
   XrefGroups.Clear;
   Xref := XrefValue;
   
   j := PosWhSpace( Xref );
   if j=0 then exit;
   if WantHost then
      XrefGroups.AddObject( copy( Xref, 1, j ), Pointer(-1) ); // host, -1
   System.Delete( Xref, 1, j );
   Xref := TrimWhSpace( Xref );

   while Xref <> '' do begin
      j := PosWhSpace( Xref );
      if j=0 then begin
         s := Xref;
         Xref := '';
      end else begin
         s := copy( Xref, 1, j-1 );
         System.Delete( Xref, 1, j );
         Xref := TrimWhSpace( Xref );
      end;

      j := Pos( ':', s );
      if j>0 then begin
         n := strtointdef( copy( s, j+1, 255 ), 0 );
         s := copy( s, 1, j-1 );
         XrefGroups.AddObject( s, Pointer(n) ); // groupname, article-number
      end;
   end;

   Result := XrefGroups.Count;
end;

// ---------------------------------------------------------- TSrvNNTPCli -----

function GetPermissionForGroup( Group, PermPost, PermRead: String ): LongInt;
var  rex: TPCRE;
     prs: TParser;

     function HasPermFor( Perm: String ): Boolean;
     var  i   : Integer;
          s, p: String;
     begin
          Result := False;

          prs.Parse( Perm, ' ' );
          i := 0;
          repeat
             s := prs.sPart( i, '' );
             if s<>'' then begin
                try
                   if s[1]='!' then begin
                      p := copy( s, 2, 255 );
                      if rex.Match( PChar(p), PChar(Group) ) then break;
                   end else begin
                      if rex.Match( PChar(s), PChar(Group) ) then begin
                         Result := True;
                         break;
                      end;
                   end;
                except
                   on E:Exception do begin
                      Log( LOGID_ERROR, 'Invalid group-permission: ' + Perm
                                      + ' Msg=' + E.Message );
                      break;
                   end;
                end;

                inc( i );
             end;
          until s='';
     end;

begin
     Result := PERM_NOTH;

     try
        // optimize for some common settings
        if PermRead='' then exit; //PERM_NOTH
        if PermRead='.*' then begin
           if PermPost=''   then begin Result:=PERM_READ; exit; end;
           if PermPost='.*' then begin Result:=PERM_POST; exit; end;
        end;

        prs := TParser.Create;
        rex := TPCRE.Create( False, PCRE_CASELESS );
        if HasPermFor(PermRead) then begin
           Result := PERM_READ;
           if HasPermFor(PermPost) then Result:=PERM_POST;
        end;

        rex.Free;
        prs.Free;
     except
        on E: Exception do begin
           Log( LOGID_ERROR, 'GetPermissionForGroup: ' + E.Message );
        end;
     end;
end;

function TServerClientNNTP.LoginUser( const Password: String ): String;
begin
     Result := '503 System-error, check logfile. [0]';

     try
        SetCurrentGroup( '' );
        if Assigned(GroupList) then GroupList.Clear;
        UserMayPost     := False;
        UserMayFeed     := False;
        UserMayNewNews  := False;
        UserMayXHSearch := False;
        UserAutoSubMode := NEWSAUTOSUB_NONE;
        RecommendedMID  := '';
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #1: ' + E.Message );
           Result := '503 System-error, check logfile. [1]';
           exit;
        end;
     end;

     try
        Result := '482 Authentication rejected';
        if CurrentUserName='' then exit;

        CurrentUserID := Hamster.Accounts.LoginID(
                            CurrentUserName, Password, ClientIPn );
        if CurrentUserID=ACTID_INVALID then begin
           CurrentUserName := '';
           Result := '502 No permission';
           exit;
        end;
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #2: ' + E.Message );
           Result := '503 System-error, check logfile. [2]';
           exit;
        end;
     end;

     try
        PermPost := Hamster.Accounts.Value[ CurrentUserID, apNewsPost ];
        PermRead := Hamster.Accounts.Value[ CurrentUserID, apNewsRead ];
        
        UserMayNewNews  := ( Hamster.Accounts.Value[ CurrentUserID, apNewsNewNews ] = '1' );
        UserMayXHSearch := ( Hamster.Accounts.Value[ CurrentUserID, apNewsXHSearch] = '1' );

        if Hamster.Accounts.Value[ CurrentUserID, apNewsPeer ] = '1' then begin
           if Hamster.Config.Settings.GetStr(hsFQDNforMID) <> '' then begin
              UserMayFeed := True;
           end else begin
              Log( LOGID_WARN, 'FQDN not set, feed permission ignored!' );
           end;
        end;

        UserAutoSubMode := strtointdef(
           Hamster.Accounts.Value[ CurrentUserID, apNewsAutoSub ],
           NEWSAUTOSUB_NONE );

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.LoginUser-Exception #3: ' + E.Message );
           Result := '503 System-error, check logfile. [3]';
           exit;
        end;
     end;

     // authentication ok, build user-specific list of newsgroups
     if ReloadGroupList then Result := '281 Authentication accepted'
                        else Result := '503 System-error, check logfile. [4]';
end;

function TServerClientNNTP.ReloadGroupList: Boolean;
var  i: Integer;
     g: String;
     p: LongInt;
begin
   Result := False;

   try
      if not Assigned( GroupList ) then exit;
      UserMayPost := False;
      GroupList.Clear;
      for i := 0 to Hamster.Config.Newsgroups.Count - 1 do begin
         g := Hamster.Config.Newsgroups.Name[i];
         p := GetPermissionForGroup( g, PermPost, PermRead );
         if p <> PERM_NOTH then GroupList.AddObject( g, Pointer( p ) );
         if p =  PERM_POST then UserMayPost:=True;
      end;
      Result := True;

   except
      on E:Exception do begin
         Log( LOGID_ERROR, Self.ClassName + '.ReloadGroupList: ' + E.Message );
         exit;
      end;
   end;
end;

procedure TServerClientNNTP.SetCurrentGroup( const Groupname: String );
begin
   with Hamster.ArticleBase do try
      if CurrentGroup>=0 then begin
         Close( CurrentGroup );
         CurrentGroup := -1;
      end;

      if length(Groupname) = 0 then exit;

      CurrentGroup := Open( Groupname );
      CurrentArtNo := GetInt( CurrentGroup, gsLocalMin );

   except
      on E:Exception do
         Log( LOGID_ERROR,
              'TSrvNNTPCli.SetCurrentGroup-Exception: ' + E.Message );
   end;
end;


procedure TServerClientNNTP.AutoGroupAddPulls( const Groupname: String );
// Automatically add pulls for the given group.
var  j: Integer;
     s: String;
     SrvList: TStringList;
begin
   // Only add, if:
   // - current user is allowed to do so
   // - group doesn't have any pulls yet
   // - group is not an internal one
   if UserAutoSubMode = NEWSAUTOSUB_NONE then exit;
   if Hamster.Config.NewsPulls.ExistPullServer( Groupname ) then exit;
   if Hamster.Config.Newsgroups.GroupClass( Groupname, True ) = aclInternal then exit;

   SrvList := TStringList.Create;
   with Hamster.Config do try
   
      // Create temporary list of known remote servers
      BeginRead;
      try
         for j:=0 to NntpServers.Count-1 do SrvList.Add( NntpServers.AliasName[j] );
      finally EndRead end;

      // Add group/pulls if group is available on wanted servers
      if UserAutoSubMode = NEWSAUTOSUB_PREF then begin

         // Add pull for preferred or first server only.
         if NntpServers.ServerHasGroup( Settings.GetStr(hsPostServer),
                                        Groupname ) then begin
            s := Settings.GetStr(hsPostServer);
         end else begin
            s := '';
            for j := SrvList.Count-1 downto 0 do begin
               if NntpServers.ServerHasGroup( SrvList[j], Groupname ) then begin
                  s := SrvList[j];
                  break;
               end;
            end;
         end;

         if ( s<>'' ) and ( not NewsPulls.ExistPull( s, Groupname ) ) then begin
            NewsPulls.Add( s, Groupname );
            Log( LOGID_WARN, 'Note: Auto-Pull added: ' + s + '/' + Groupname
                           + ' by ' + CurrentUserName );
         end;

      end else begin

         // Add pulls for all servers, that carry this group.
         for j := SrvList.Count-1 downto 0 do begin
            if NntpServers.ServerHasGroup( SrvList[j], Groupname ) then begin
               if not NewsPulls.ExistPull( SrvList[j], Groupname ) then begin
                  NewsPulls.Add( SrvList[j], Groupname );
                  Log( LOGID_WARN, 'Note: Auto-Pull added: '
                                 + SrvList[j] + '/' + Groupname
                                 + ' by ' + CurrentUserName );
               end;
            end;
         end;
         
      end;

   finally SrvList.Free end;
end;

procedure TServerClientNNTP.AutoGroupPrepare( const Groupname: String );
// Prepare adding of the given group, which is assumed to be "virtual" until
// it is either committed (=create group) or rolled back (=forget group).
begin
   if AutoGroupPends then AutoGroupRollback;

   AutoGroupPends := True;
   AutoGroupName  := Groupname;
   SetCurrentGroup( '' ); // unselect, no group files yet
end;

procedure TServerClientNNTP.AutoGroupCommit;
// Creates the current "virtual" group, i. e. creates group files, adds it
// to active and places the placeholder article in it.

   procedure CreatePlaceholder( Groupname: String );
   var  Art: TMess;
   begin
      Art := TMess.Create;
      try
         Art.BodyText :=
           'Newsgroup: ' + Groupname                                  + #13#10
                                                                      + #13#10
         + 'This group was created automatically. It will be filled ' + #13#10
         + 'with articles the next time when groups are pulled from ' + #13#10
         + 'remote servers again.'                                    + #13#10;

         Art.AddHeaderSL( HDR_NAME_SUBJECT,      'Hamster placeholder for ' + Groupname );
         Art.AddHeaderSL( HDR_NAME_DATE,         DateTimeGMTToRfcDateTime(
                                                 NowGMT, NowRfcTimezone ) );
         Art.AddHeaderSL( HDR_NAME_FROM,         'Hamster' );
         Art.AddHeaderSL( HDR_NAME_NEWSGROUPS,   Groupname );
         Art.AddHeaderSL( HDR_NAME_MESSAGE_ID,   MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) ) );
         Art.AddHeaderSL( HDR_NAME_DISTRIBUTION, 'local' );
         Art.AddHeaderSL( HDR_NAME_PATH,         'not-for-mail' );
         Art.AddHeaderSL( HDR_NAME_LINES,        inttostr(Art.BodyLines) );

         SaveArticle( Art, '', -1, '(none-placeholder)' + ' ' + 'NoArchive=1',
                      False, '', actIgnore );
      finally
         Art.Free;
      end;
   end;

begin
   if not AutoGroupPends then exit;

   // create group and add it to active
   Hamster.Config.Newsgroups.Add( AutoGroupName );
   Log( LOGID_WARN, 'Note: Auto-Group added: ' + AutoGroupName
                  + ' by ' + CurrentUserName );
   CreatePlaceholder( AutoGroupName ); // create placeholder in new group
   ReloadGroupList;                    // refresh user's group list

   // add pulls for new group
   AutoGroupAddPulls( AutoGroupName );

   // select group and initialize its LastClientRead marker
   SetCurrentGroup( AutoGroupName );
   Hamster.ArticleBase.SetDT( CurrentGroup, gsLastClientRead, Now );

   SetLength( AutoGroupName, 0 );
   AutoGroupPends := False;
end;

procedure TServerClientNNTP.AutoGroupRollback;
// A "virtual" group was not used, so forget it.
begin
   if not AutoGroupPends then exit;

   SetLength( AutoGroupName, 0 );
   AutoGroupPends := False;
end;

function TServerClientNNTP.MailPostedArticle( const ArtText, SendTo : String;
                                              const IsModerated: Boolean ): String;
var  MailItem: TMailItem;
     mdr: TMailDeliveryResults;
begin
   Result := '441 posting failed';

   MailItem := TMailItem.Create( moNewsToMail );
   try
      // init mail to send
      MailItem.MailText.FullText := ArtText;

      // set recipient
      MailItem.Recipients.Add( SendTo );
      MailItem.MailText.DelHeaderML( 'To:' );
      MailItem.MailText.InsertHeaderSL( 0, HDR_NAME_TO, SendTo );

      // gateway: remove news-specific headers
      if not IsModerated then begin
         MailItem.MailText.DelHeaderML( 'Newsgroups:' );
         MailItem.MailText.DelHeaderML( 'Path:' );
      end;

      // set sender
      MailItem.Sender.EnvelopeAddr := MailItem.MailText.HeaderValueByNameSL( 'From:' );
      MailItem.Sender.IPAddr := FClientAD.S_addr;

      mdr := Hamster.MailDispatcher.Process( MailItem );
      if mdr = mdrSuccess then begin
         Result := '240 article posted ok (mailed)';
      end else begin
         Result := '441 posting failed (' + MailDeliveryResultTexts[ mdr ] + ')';
      end;

   finally MailItem.Free end;
end;

function TServerClientNNTP.PostPermission_KnownGroup( const GroupName: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   i := GroupList.IndexOf( GroupName );
   if i >= 0 then Result := LongInt( GroupList.Objects[i] ) = PERM_POST;
end;

function TServerClientNNTP.PostPermission_UnknownGroup( const GroupName: String ): Boolean;
begin
   Result := GetPermissionForGroup(GroupName,PermPost,PermRead) = PERM_POST;
end;

function TServerClientNNTP.HandlePostedArticle( const ArtText: String ): String;

   function TraceContent: String;
   var  DT: TDateTime;
   begin
      if Hamster.Config.Settings.GetStr(hsFQDNforMID) = '' then
         Result := 'localhost'
      else
         Result := Hamster.Config.Settings.GetStr(hsFQDNforMID);
      DT := NowGMT;
      Result := Result + ' ' + inttostr( DateTimeToUnixTime( DT ) )
                       + ' ' + Format( '%u', [GetCurrentThreadID] )
                       + ' ' + FClientIP
                       + ' ('
                         + inttostr( CurrentUserID ) + ' '
                         + lowercase( inttohex(GetCurrentThreadID,1) ) + ' '
                         + FormatDateTime( 'yyyy"."mm"."dd hh":"nn":"ss', DT )
                       + ')';
   end;

type TMsgHeaders = ( hNewsgroups, hMessageID, hDate, hLines, hControl, hFrom,
                     hSubject, hPath, hApproved, hXref );
var  Msg: TMess;
     Hdr: array[ TMsgHeaders ] of String;
     DataBuf, GrpNam: String;
     CancelMID, XrefHostname, SendByMailTo, s, Moderator: String;
     DestLines, LfdGrp, GrpHdl, i: Integer;
     boDONE, HasLocal, HasRemote, HasFeedOnly, IsModerated, IsGateway, LocalInject, OK: Boolean;
     Parser: TParser;
     Xrefs: TList;
     Xref: TXrefArtNo;
     GroupType: Char;
     TS: TStringList;
     newsInType: THscNewsInTypes;
begin
   Result := '441 posting failed (unknown reason, see logfile)';

   Msg := TMess.Create;
   Parser := TParser.Create;

   try

     try
        if Length(ArtText) <= 0 then begin
           Result := '441 posting failed (missing ... all)';
           exit;
        end;

        DataBuf := ArtText;

        Msg.FullText := DataBuf;

        if not TMess.HasHeaderEnd( DataBuf ) then begin
           Result := '441 posting failed (missing header/body-separator)';
           exit;
        end;

        if Hamster.Config.Settings.GetBoo(hsNewsAddXHamster) then begin
           // add/expand User-Agent header
           i := Msg.IndexOfHeader( HDR_NAME_USER_AGENT );
           if i < 0 then begin // add
              Msg.AddHeaderSL( HDR_NAME_USER_AGENT, OUR_VERINFO );
           end else begin // expand
              s := Msg.HeaderLineSL[i];
              if Pos(OUR_VERINFO,s)=0 then Msg.HeaderLineSL[i]:=s+' '+OUR_VERINFO;
           end;
        end;

        if Hamster.Config.Settings.GetBoo(hsNewsAddXTrace) then begin
           // add Trace-Header
           Msg.SetHeaderSL( HDR_NAME_X_TRACE, TraceContent, 'X-Old-Trace:' );
        end;

        Hdr[hNewsgroups] := Msg.HeaderValueByNameSL( HDR_NAME_NEWSGROUPS );
        Hdr[hMessageID]  := Msg.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID );
        Hdr[hLines]      := Msg.HeaderValueByNameSL( HDR_NAME_LINES );
        Hdr[hDate]       := Msg.HeaderValueByNameSL( HDR_NAME_DATE );
        Hdr[hControl]    := Msg.HeaderValueByNameSL( HDR_NAME_CONTROL );
        Hdr[hFrom]       := Msg.HeaderValueByNameSL( HDR_NAME_FROM );
        Hdr[hSubject]    := Msg.HeaderValueByNameSL( HDR_NAME_SUBJECT );
        Hdr[hPath]       := Msg.HeaderValueByNameSL( HDR_NAME_PATH );
        Hdr[hApproved]   := Msg.HeaderValueByNameSL( HDR_NAME_APPROVED );
        DestLines        := Msg.BodyLines;

        Hdr[hXref] := Msg.GetHeaderLine( HDR_NAME_XREF, i );
        if (Hdr[hXref]<>'') and (i>=0) then Msg.DeleteHeaderML( i ); // remove existing Xref-header

        DataBuf := Msg.FullText;

        // check some headers
        if (Hdr[hFrom]='') or (Hdr[hSubject]='') or (Hdr[hNewsgroups]='') then begin
           Result := '441 posting failed (missing From|Subject|Newsgroups)';
           exit;
        end;

        i := Pos( '@', Hdr[hFrom] );
        if i = 0 then begin
           Result := '441 posting failed (invalid From)';
           exit;
        end;

        if Hdr[hDate] <> '' then begin
           if RfcDateTimeToDateTimeGMT(Hdr[hDate])<=EncodeDate( 1980, 1, 1 ) then begin
              Result := '441 posting failed (invalid Date)';
              exit;
           end;
        end;

        // add recommended MID if appropriate
        if (Hdr[hMessageID] = '') and (RecommendedMID <> '') then begin
           Hdr[hMessageID] := RecommendedMID;
           RecommendedMID := '';
           DataBuf := 'Message-ID: ' + Hdr[hMessageID] + #13#10 + DataBuf;
        end;

        // generate MID if not set already
        if Hdr[hMessageID] = '' then begin
           if Hamster.Config.Settings.GetBoo(hsGenerateNewsMID) then begin

              if Hamster.Config.Settings.GetStr(hsFQDNforMID) = '' then begin
                 Log( LOGID_WARN, 'Configuration error: No FQDN to generate Message-ID!' );
                 Result := '441 posting failed (server config error: MID-gen w/o FQDN)';
                 exit;
              end;
              
              Hdr[hMessageID] := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
              DataBuf := 'Message-ID: ' + Hdr[hMessageID] + #13#10 + DataBuf;

           end;
        end;

        // check Message-ID
        if Hdr[hMessageID] <> '' then begin
           if Hamster.NewsHistory.ContainsMID( Hdr[hMessageID] ) then begin
              Result := '441 posting failed (Message-ID already in history)';
              exit;
           end;
        end;

        // apply score file section [$POST$]
        if ScoreFilePost = nil then begin // init on first use
           ScoreFilePost := TFiltersNews.Create( AppSettings.GetStr(asPathBase) + CFGFILE_SCORES );
           ScoreFilePost.SelectSections( SCOREGROUP_POST );
        end;
        if ScoreFilePost.LinesCount > 0 then begin
           i := ScoreFilePost.ScoreArticle( Msg );
           if i < 0 then begin
              Result := '441 posting failed (filtered, score ' + inttostr(i) + ')';
              exit;
           end;
        end;

        // Check given newsgroups
        boDONE         := False;
        HasLocal       := False;
        HasRemote      := False;
        HasFeedOnly    := False;
        IsModerated    := False;
        IsGateway      := False;
        SendByMailTo   := '';
        Parser.Parse( Hdr[hNewsgroups], ',' );

        LfdGrp := 0;
        repeat
           GrpNam := TrimWhSpace( Parser.sPart( LfdGrp, '' ) );

           if GrpNam='' then begin
              if LfdGrp=0 then begin
                 Result := '441 posting failed (no newsgroups)';
                 boDONE := True;
              end;
              break;
           end;

           if Hamster.Config.Newsgroups.IndexOf(GrpNam) < 0 then begin

              // unknown group

              if not IsNewsgroup( GrpNam ) then begin
                 // posting to invalid group
                 Result := '441 posting failed (invalid newsgroup "'+GrpNam+'")';
                 boDONE := True;
              end else if not PostPermission_UnknownGroup( GrpNam ) then begin
                 // posting not allowed
                 Result := '440 posting failed (no permission to post to '+GrpNam+')';
                 boDONE := True;
              end else begin
                 // posting to unknown group
                 Log( LOGID_INFO, '(Cross-) Posting to unknown group: ' + GrpNam );
              end;

           end else begin

              // known group

              if Hamster.Config.NewsPulls.GetPostServer(GrpNam)='' then HasLocal :=True
                                                                   else HasRemote:=True;
              if not PostPermission_KnownGroup( GrpNam ) then begin
                 Result := '440 posting failed (no permission to post to '+GrpNam+')';
                 boDONE := True;
              end;

              // get newsgroup type
              GroupType := 'n';
              GrpHdl := Hamster.ArticleBase.Open( GrpNam );
              if GrpHdl >= 0 then try
                 GroupType := (Hamster.ArticleBase.GetStr( GrpHdl, gsGroupType )+'y')[1];
                 Moderator := Hamster.ArticleBase.GetStr( GrpHdl, gsModerator );
                 if Hamster.ArticleBase.GetBoo( GrpHdl, gsFeedOnly ) then HasFeedOnly := True;
              finally Hamster.ArticleBase.Close( GrpHdl ) end;

              // check if posting is allowed or should be sent by mail
              case GroupType of
                 'n': begin
                         Result := '440 posting failed (no postings allowed in '+GrpNam+')';
                         boDONE := True;
                      end;
                 'm': if (Moderator <> '') and (Hdr[hApproved] = '') then begin
                         IsModerated  := True;
                         SendByMailTo := Moderator;
                      end;
                 'g': if Moderator = '' then begin
                         Result := '440 posting failed (no postings allowed in '+GrpNam+')';
                         boDONE := True;
                      end else begin
                         IsGateway    := True;
                         SendByMailTo := Moderator;
                      end;
              end;

           end;

           inc( LfdGrp );
        until boDONE;

        if (not boDONE) and (LfdGrp>1) and (IsModerated or IsGateway) then begin
           Result := '441 posting failed (crossposted to moderated/gateway group)';
           boDONE := True;
        end;

        if (not boDONE) and (not (HasLocal or HasRemote)) then begin
           Result := '441 posting failed (no known groups)';
           boDONE := True;
        end;

        if boDONE then exit;

        // prepare and check local injection
        LocalInject := Hamster.Config.Settings.GetBoo(hsLocalNntpLocalInjection);
        if LocalInject then begin

           if Hdr[hMessageID] = '' then begin
              Log( LOGID_WARN, 'Configuration error: '
                 + 'Local injection requires Message-ID generation!' );
              Result := '441 posting failed (server config error: local-inject w/o MID-gen)';
              exit;
           end;

           HasLocal    := True;
           LocalInject := True;

        end;

        // check requirements for and handle feed-only groups
        if HasFeedOnly and not LocalInject then begin
           Log( LOGID_WARN, 'Configuration error: '
              + 'Marking groups as "feed-only" requires "local injection"!' );
           Result := '441 posting failed (server config error: feed-only w/o local-inject)';
           exit;
        end;

        if HasFeedOnly then begin
           HasLocal  := True;
           HasRemote := False;
        end;

        // execute 'NewsInLocal' action
        if Hamster.HscActions.IsAssigned(actNewsInLocal) then begin

           Msg.FullText := DataBuf;

           if IsModerated                 then newsInType := nitModerated
           else if IsGateway              then newsInType := nitGateway
           else if HasRemote and HasLocal then newsInType := nitRemoteAndLocal
           else if HasRemote              then newsInType := nitRemote
           else if HasLocal               then newsInType := nitLocal
           else                                newsInType := nitUnknown;

           if Hamster.HscActions.Execute(
              actNewsInLocal, inttostr(ord(newsInType)) + CRLF + inttostr( Integer(Msg) )
           ) then begin
              if Msg.IsValid then begin
                 DataBuf := Msg.FullText;
              end else begin
                 Log( LOGID_INFO, 'Note: Action "NewsInLocal" invalidated message.' );
                 Result := '441 posting failed (message violates server policy)';
                 exit;
              end;
           end;

        end;

        // Send article by mail (moderated or gateway group)
        if IsModerated or IsGateway then begin 
           Result := MailPostedArticle( DataBuf, SendByMailTo, IsModerated );
           exit;
        end;

        // Save in News.Out
        LfdGrp := 0;
        while HasRemote and not HasFeedOnly do begin
           GrpNam := Parser.sPart( LfdGrp, '' );
           if GrpNam='' then break;

           if Hamster.Config.NewsPulls.GetPostServer(GrpNam)<>'' then begin

              if not boDONE then begin
                 if SaveUniqueNewsMsg(
                       AppSettings.GetStr(asPathNewsOut), DataBuf,
                       Hamster.Config.Settings.GetBoo(hsGenerateNewsMID)
                    ) then begin
                    CounterInc( CounterOutboxN   );
                    CounterInc( CounterOutboxChk );
                    if Hdr[hMessageID]='' then Result:='240 article posted ok'
                                          else Result:='240 article posted ok '+Hdr[hMessageID];
                    boDONE := True;
                 end else begin
                    Result := '441 posting failed (couldn''t save in News.Out)';
                    boDONE := True;
                 end;

                 break; // only once!
              end;

           end;

           inc( LfdGrp );
        end;

        // Add headers for local groups
        if HasLocal then begin

           DataBuf := HDR_NAME_X_HAMSTER_INFO + ': '
                    + 'Received=' + DateTimeToTimeStamp(Now) + ' '
                    + 'UID=' + IntToStr( GetUID(0) )
                    + #13#10 + DataBuf;
           if Hdr[hLines] = '' then begin
              DataBuf := 'Lines: ' + inttostr(DestLines) + #13#10 + DataBuf;
           end;

           if Hdr[hDate] = '' then begin
              Hdr[hDate] := DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone );
              DataBuf := 'Date: ' + Hdr[hDate] + #13#10 + DataBuf;
           end;

           if Hdr[hMessageID] = '' then begin
              if Hamster.Config.Settings.GetStr(hsFQDNforMID) <> '' then
                 Hdr[hMessageID] := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) )
              else
                 Hdr[hMessageID] := MidGenerator( 'hamster.local.invalid' );
              DataBuf := 'Message-ID: ' + Hdr[hMessageID] + #13#10 + DataBuf;
           end;

           if Hdr[hPath] = '' then begin
              Hdr[hPath] := 'not-for-mail';
              if Hamster.Config.Settings.GetStr(hsFQDNforMID) <> '' then
                 Hdr[hPath] := Hamster.Config.Settings.GetStr(hsFQDNforMID)+'!'+Hdr[hPath];
              DataBuf := 'Path: ' + Hdr[hPath] + #13#10 + DataBuf;
           end else begin
              if Hamster.Config.Settings.GetStr(hsFQDNforMID) <> '' then begin
                 TS := TStringList.Create;
                 try
                    RE_Split( Hdr[hPath], '!', 0, -1, TS );
                    for i:=0 to TS.Count-1 do begin
                       if AnsiCompareText( TrimWhSpace(TS[i]),
                                           Hamster.Config.Settings.GetStr(hsFQDNforMID) )=0 then begin
                          Result := '441 posting failed (invalid "Path:")';
                          boDONE := True;
                          break;
                       end;
                    end;
                    if not boDONE then begin
                       Hdr[hPath] := Hamster.Config.Settings.GetStr(hsFQDNforMID) + '!' + Hdr[hPath];
                       Msg.FullText := DataBuf;
                       Msg.SetHeaderSL( HDR_NAME_PATH, Hdr[hPath], '' );
                       DataBuf := Msg.FullText;
                    end;
                 finally
                    TS.Free;
                 end;
              end;
           end;

        end;

        // cancel in local group?
        if HasLocal and ( not(boDONE) or LocalInject ) then begin
           if Pos( 'cancel', Lowercase(Hdr[hControl]) )=1 then begin

              OK := True;
              CancelMID := TrimWhSpace( copy(Hdr[hControl],8,255) ); // MID
              Msg.FullText := Hamster.ArticleBase.ReadArticleByMID( CancelMID );

              if Msg.HeaderLineCountSL=0 then begin
                 OK := False;
                 Result := '441 local-cancel failed (MID not found)';
              end else begin
                 if ExtractMailAddr( Msg.HeaderValueByNameSL(HDR_NAME_FROM) )
                 <> ExtractMailAddr( Hdr[hFrom] ) then begin
                    OK := False;
                    Result := '441 local-cancel failed (From-mismatch)';
                    Log( LOGID_DEBUG, 'From-mismatch on cancel: Article-From: ['
                                      + Msg.HeaderValueByNameSL( HDR_NAME_FROM )
                                      + '] Cancel-From: [' + Hdr[hFrom] + ']' );
                 end;
              end;

              if OK then begin
                 if Hamster.ArticleBase.DeleteArticleByMID( CancelMID ) then begin
                    Result := '240 local article cancelled';
                    s := 'Note: Message ' + CancelMID + ' was cancelled.' + #13#10;
                    s := s + #13#10 + 'Headers of cancelled-message:' + #13#10#13#10;
                    for i:=0 to Msg.HeaderLineCountSL-1 do begin
                       s := s + '> ' + Msg.HeaderLineSL[i] + #13#10;
                    end;
                    s := s + #13#10 + 'Cancel-message:' + #13#10#13#10 + DataBuf;
                    SaveInInternalGroup( hsHamGroupCancelNotify,
                                         '[Hamster] Cancel', s );
                 end else begin
                    Result := '441 local-cancel failed (DelByMid)';
                 end;
              end;

              exit;

           end;
        end;

        // Create Xref-header by opening groups and reserving article-numbers
        Hdr[hXref] := '';
        Xrefs := TList.Create;
        LfdGrp := 0;
        while HasLocal do begin
           GrpNam := Parser.sPart( LfdGrp, '' );
           if GrpNam='' then break;

           if LocalInject then begin
              OK := True;
           end else begin
              OK := ( Hamster.Config.NewsPulls.GetPostServer(GrpNam) = '' );
           end;
           if Hamster.Config.Newsgroups.IndexOf(GrpNam) < 0 then OK := False;

           if OK then begin
              GrpHdl := Hamster.ArticleBase.Open( GrpNam );
              if GrpHdl>=0 then begin
                 Xref := TXrefArtNo.Create;
                 Xref.GrpNam := GrpNam;
                 Xref.GrpHdl := GrpHdl;
                 Xref.ArtNo  := Hamster.ArticleBase.ReserveArtNo( Xref.GrpHdl );
                 Xrefs.Add( Xref );
                 Hdr[hXref] := Hdr[hXref] + ' ' + GrpNam + ':' + inttostr( Xref.ArtNo );
              end;
           end;

           inc( LfdGrp );
        end;
        // Note: already existing Xref would have been removed above
        XrefHostname := Hamster.Config.Settings.GetStr(hsFQDNforMID);
        if XrefHostname = '' then XrefHostname := 'localhost';
        DataBuf := 'Xref: ' + XrefHostname + Hdr[hXref] + #13#10 + DataBuf;

        // save article in local groups
        for LfdGrp:=0 to Xrefs.Count-1 do begin
           Xref := TXrefArtNo( Xrefs[LfdGrp] );

           OK := False;
           try
              Hamster.ArticleBase.WriteArticle( Xref.GrpHdl, Xref.ArtNo, DataBuf );
              Hamster.NewsHistory.AddEntryDupes( Hdr[hMessageID], StrToCRC32(Xref.GrpNam), Xref.ArtNo, 0 );
              OK := True;
           except
              on E: Exception do begin
                 Log( LOGID_ERROR, 'Exception saving article: ' + E.Message );
              end;
           end;

           if OK then begin
              if not boDONE then Result := '240 article posted local';
              boDONE := True;
           end else begin
              if not boDONE then Result := '441 local-posting failed (couldn''t save)';
              boDONE := True;
           end;
        end;

        // close groups and free Xref-entries and -list
        for LfdGrp:=Xrefs.Count-1 downto 0 do begin
           Xref := TXrefArtNo( Xrefs[LfdGrp] );
           Hamster.ArticleBase.Close( Xref.GrpHdl );
           Xrefs.Delete( LfdGrp );
           Xref.Free;
        end;
        Xrefs.Free;

        if not boDONE then Result := '441 posting failed';

        exit;

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.HandlePostedArticle error: ' + E.Message );
           try HDisconnect except end;
        end;
     end;
   finally
      Msg.Free;
      Parser.Free;
   end;
end;

function TServerClientNNTP.HandleFeededArticle( const MessageId, ArtText: String;
                                                const isIHAVE: Boolean ): String;
type TMsgHeaders = ( hNewsgroups, hMessageID, hDate, hLines, hControl, hFrom,
                     hSubject, hPath, hApproved, hXref );
var  Msg: TMess;
     Hdr: array[ TMsgHeaders ] of String;
     OkResult, ErrResult, FQDN, GrpNam, CancelMID, s: String;
     LfdGrp, GrpHdl, i: Integer;
     HasLocal: Boolean;
     Parser: TParser;
     Xrefs: TList;
     Xref: TXrefArtNo;
     TS: TStringList;
     dt: TDateTime;
begin
   if isIHAVE then begin // IHAVE
      OkResult  := '235' + ' article transferred ok ' + MessageId;
      ErrResult := '437' + ' article rejected - do not try again ' + MessageId;
   end else begin        // TAKETHIS
      OkResult  := '239' + ' article transferred ok ' + MessageId;
      ErrResult := '439' + ' article rejected - do not try again ' + MessageId;
   end;

   Result := ErrResult;
   FQDN   := Hamster.Config.Settings.GetStr( hsFQDNforMID );
   Msg    := TMess.Create;
   Parser := TParser.Create;
   Xrefs  := TList.Create;

   try
     try

        // prepare article and some headers
        Msg.FullText := ArtText;

        // check mandatory headers
        if not( Msg.HeaderExists( HDR_NAME_FROM       ) and
                Msg.HeaderExists( HDR_NAME_SUBJECT    ) and
                Msg.HeaderExists( HDR_NAME_NEWSGROUPS ) and
                Msg.HeaderExists( HDR_NAME_DATE       ) and
                Msg.HeaderExists( HDR_NAME_MESSAGE_ID ) and
                Msg.HeaderExists( HDR_NAME_PATH       )
           ) then begin
           Log( LOGID_DETAIL, 'Feed: mandatory header missing ' + MessageId );
           exit;
        end;

        Hdr[hNewsgroups] := Msg.HeaderValueByNameSL( HDR_NAME_NEWSGROUPS );
        Hdr[hMessageID]  := Msg.HeaderValueByNameSL( HDR_NAME_MESSAGE_ID );
        Hdr[hLines]      := Msg.HeaderValueByNameSL( HDR_NAME_LINES );
        Hdr[hDate]       := Msg.HeaderValueByNameSL( HDR_NAME_DATE );
        Hdr[hControl]    := Msg.HeaderValueByNameSL( HDR_NAME_CONTROL );
        Hdr[hFrom]       := Msg.HeaderValueByNameSL( HDR_NAME_FROM );
        Hdr[hSubject]    := Msg.HeaderValueByNameSL( HDR_NAME_SUBJECT );
        Hdr[hPath]       := Msg.HeaderValueByNameSL( HDR_NAME_PATH );
        Hdr[hApproved]   := Msg.HeaderValueByNameSL( HDR_NAME_APPROVED );

        // remove existing Xref-header
        Hdr[hXref] := Msg.GetHeaderLine( HDR_NAME_XREF, i );
        if i >= 0 then Msg.DeleteHeaderML( i );

        // check mandatory header values
        if (Hdr[hNewsgroups] = '') or (Hdr[hDate] = '') or
           (Hdr[hMessageID]  = '') or (Hdr[hPath] = '') then begin
           Log( LOGID_DETAIL, 'Feed: mandatory header value missing ' + MessageId );
           exit;
        end;

        // check Message-ID header
        if Hdr[hMessageID] <> MessageId then begin
           Log( LOGID_DETAIL, 'Feed: Message-ID mismatch ' + MessageId );
           exit;
        end;

        // check Path header
        TS := TStringList.Create;
        try
           RE_Split( Hdr[hPath], '!', 0, -1, TS );
           for i := 0 to TS.Count - 2 do begin
              if AnsiCompareText( TrimWhSpace( TS[i] ), FQDN ) = 0 then begin
                 Log( LOGID_DETAIL, 'Feed: FQDN already in Path: header ' + MessageId );
                 exit;
              end;
           end;
        finally
           TS.Free;
        end;

        // check Date header
        dt := RfcDateTimeToDateTimeGMT( Hdr[hDate] );
        if dt <= EncodeDate( 1980, 1, 1 ) then begin
           Log( LOGID_DETAIL, 'Feed: invalid Date ' + MessageId );
           exit;
        end;

        // check Newsgroups header
        HasLocal := False;
        Parser.Parse( Hdr[hNewsgroups], ',' );
        LfdGrp := 0;

        repeat

           GrpNam := TrimWhSpace( Parser.sPart( LfdGrp, '' ) );
           if GrpNam = '' then break;
           
           if Hamster.Config.Newsgroups.IndexOf( GrpNam ) < 0 then begin

              // unknown group

              if not IsNewsgroup( GrpNam ) then begin // valid group name?
                 Log( LOGID_DETAIL, 'Feed: invalid newsgroup ' + GrpNam
                                  + ' ' + MessageId );
                 exit;
              end;

              if not PostPermission_UnknownGroup( GrpNam ) then begin // permission?
                 Log( LOGID_DETAIL, 'Feed: no permission to post to ' + GrpNam
                                  + ' ' + MessageId );
                 exit;
              end;

           end else begin

              // known group

              if not PostPermission_KnownGroup( GrpNam ) then begin // permission?
                 Log( LOGID_DETAIL, 'Feed: no permission to post to ' + GrpNam
                                  + ' ' + MessageId );
                 exit;
              end;

              // only accept group types 'y' (posting allowed) and 'm' (moderated)
              s := 'n';
              GrpHdl := Hamster.ArticleBase.Open( GrpNam );
              if GrpHdl >= 0 then try
                 s := (Hamster.ArticleBase.GetStr( GrpHdl, gsGroupType ) + 'y')[1];
              finally Hamster.ArticleBase.Close( GrpHdl ) end;
              if (s <> 'y') and (s <> 'm') then begin
                 Log( LOGID_DETAIL, 'Feed: no postings allowed in ' + GrpNam
                                  + ' ' + MessageId );
                 exit;
              end;
              if (s = 'm') and (Hdr[hApproved] = '') then begin
                 Log( LOGID_DETAIL, 'Feed: not approved ' + MessageId );
                 exit;
              end;

              HasLocal := True;

           end;

           inc( LfdGrp );
           
        until False;

        // at least one known group found?
        if not HasLocal then begin
           Log( LOGID_DETAIL, 'Feed: no known groups ' + MessageId );
           exit;
        end;

        // apply score file section [$FEED$]
        if ScoreFileFeed = nil then begin // init on first use
           ScoreFileFeed := TFiltersNews.Create( AppSettings.GetStr(asPathBase) + CFGFILE_SCORES );
           ScoreFileFeed.SelectSections( SCOREGROUP_FEED );
        end;
        if ScoreFileFeed.LinesCount > 0 then begin
           i := ScoreFileFeed.ScoreArticle( Msg );
           if i < 0 then begin
              Log( LOGID_DETAIL, 'Feed: filtered (score=' + inttostr(i)
                               + ') ' + MessageId );
              exit;
           end;
        end;
        
        // execute 'NewsIn' action
        if Hamster.HscActions.IsAssigned(actNewsIn) then begin

           if Hamster.HscActions.Execute(
              actNewsIn, inttostr(ord(nitFeed)) + CRLF + inttostr( Integer(Msg) )
           ) then begin
              if not Msg.IsValid then begin
                 Log( LOGID_DETAIL, 'Feed: Action "NewsIn" invalidated message '
                                  + MessageId );
                 exit;
              end;
           end;

        end;

        // add Hamster specific header (=purge base time)
        Msg.SetHeaderSL( HDR_NAME_X_HAMSTER_INFO,
                         'Received=' + DateTimeToTimeStamp(Now) + ' '
                         + 'UID=' + IntToStr( GetUID(0) ), '' );

        // add FQDN to Path: header
        Hdr[hPath] := FQDN + '!' + Hdr[hPath];
        Msg.SetHeaderSL( HDR_NAME_PATH, Hdr[hPath], '' );

        CS_LOCK_NEWSFEED.Enter;
        try
           // reject known Message-ID
           // (should only happen if injected by concurrent connection)
           if Hamster.NewsHistory.ContainsMID( Hdr[hMessageID] ) then begin
              Log( LOGID_DETAIL, 'Feed: Message-ID already in history ' + MessageId );
              exit;
           end;

           // cancel in local group?
           if Pos( 'cancel', Lowercase(Hdr[hControl]) ) = 1 then begin

              CancelMID := TrimWhSpace( copy(Hdr[hControl],8,255) );
              s := Hamster.ArticleBase.ReadArticleByMID( CancelMID );

              if ExtractMailAddr( TMess.HeaderValueFromText( s, HDR_NAME_FROM ) )
               = ExtractMailAddr( Hdr[hFrom] ) then begin
                 if Hamster.ArticleBase.DeleteArticleByMID( CancelMID ) then begin
                    Log( LOGID_DETAIL, 'Feed: article ' + CancelMID
                                     + ' cancelled by ' + MessageId );
                    Result := OkResult;
                 end;
              end;

           end;

           // create Xref-header by opening groups and reserving article-numbers
           Xrefs.Clear;
           try
              Hdr[hXref] := '';
              LfdGrp := 0;

              while True do begin

                 GrpNam := Parser.sPart( LfdGrp, '' );
                 if GrpNam = '' then break;

                 if Hamster.Config.Newsgroups.IndexOf(GrpNam) >= 0 then begin
                    GrpHdl := Hamster.ArticleBase.Open( GrpNam );
                    if GrpHdl >= 0 then begin
                       Xref := TXrefArtNo.Create;
                       Xref.GrpNam := GrpNam;
                       Xref.GrpHdl := GrpHdl;
                       Xref.ArtNo  := Hamster.ArticleBase.ReserveArtNo( Xref.GrpHdl );
                       Xrefs.Add( Xref );
                       Hdr[hXref] := Hdr[hXref] + ' ' + GrpNam + ':' + inttostr( Xref.ArtNo );
                    end;
                 end;

                 inc( LfdGrp );

              end;

              // add Xref header
              Msg.SetHeaderSL( HDR_NAME_XREF, FQDN + Hdr[hXref], '' );

              // save article in local groups
              for LfdGrp := 0 to Xrefs.Count - 1 do begin

                 Xref := TXrefArtNo( Xrefs[LfdGrp] );

                 try
                    Hamster.ArticleBase.WriteArticle(
                       Xref.GrpHdl, Xref.ArtNo, Msg.FullText );

                    Hamster.NewsHistory.AddEntryDupes(
                       Hdr[hMessageID], StrToCRC32(Xref.GrpNam), Xref.ArtNo, 0 );

                    Result := OkResult;

                 except
                    on E: Exception do begin
                       Log( LOGID_ERROR, 'Feed: error saving article: '
                                       + E.Message + ' ' + MessageId );
                    end;
                 end;

              end;

           finally

              // close groups and free Xref-entries
              for LfdGrp := Xrefs.Count - 1 downto 0 do begin
                 Xref := TXrefArtNo( Xrefs[LfdGrp] );
                 Hamster.ArticleBase.Close( Xref.GrpHdl );
                 Xrefs.Delete( LfdGrp );
                 Xref.Free;
              end;

           end;
           
        finally
           CS_LOCK_NEWSFEED.Leave;
        end;

     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'NNTP.HandleFeededArticle error: ' + E.Message
                           + ' ' + MessageId );
           try HDisconnect except end;
        end;
     end;

   finally
      Xrefs.Free;
      Msg.Free;
      Parser.Free;
   end;
end;

procedure TServerClientNNTP.HandleCommand( const CmdLine: String );
var  LogCmdLine, Cmd, Par: String;
     j: Integer;
begin
     try
        if not HConnected then exit;

        // Extract command
        j := PosWhSpace( CmdLine );
        if j=0 then begin
           Cmd := UpperCase( CmdLine );
           Par := '';
        end else begin
           Cmd := UpperCase  ( copy( CmdLine, 1, j-1 ) );
           Par := TrimWhSpace( copy( CmdLine, j+1, 512 ) );
        end;

        if (Cmd='AUTHINFO') and (UpperCase(copy(Par,1,4))='PASS') then begin
           LogCmdLine := 'AUTHINFO PASS [...]';
        end else begin
           LogCmdLine := CmdLine;
        end;

        Log( LOGID_INFO, '> ' + LogCmdLine );
        if CmdLine='' then exit;

        // commands (no authentication required)
        if Cmd='AUTHINFO' then if Cmd_AUTHINFO( Par ) then exit;
        if Cmd='DATE'     then if Cmd_DATE    ( Par ) then exit;
        if Cmd='HELP'     then if Cmd_HELP    ( Par ) then exit;
        if Cmd='MODE'     then if Cmd_MODE    ( Par ) then exit;
        if Cmd='QUIT'     then if Cmd_QUIT    ( Par ) then exit;
        if Cmd='SLAVE'    then if Cmd_SLAVE   ( Par ) then exit;

        // check authentication
        if CurrentUserID = ACTID_INVALID then begin
           if Pos( '|'+Cmd+'|', 
                   '|ARTICLE|BODY|GROUP|HEAD|LAST|LIST|LISTGROUP|NEWGROUPS|'
                   + 'NEXT|POST|STAT|XHDR|XOVER|XPAT|NEWNEWS|XHSEARCH|'
                   + 'XHHELP|IHAVE|CHECK|TAKETHIS|') > 0 then begin
              HWriteLn( '480 Authentication required' );
              exit;
           end;
        end;

        // commands (authentication required)
        if Cmd='ARTICLE'   then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='BODY'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='GROUP'     then if Cmd_GROUP    (      Par ) then exit;
        if Cmd='HEAD'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='LAST'      then if Cmd_LASTNEXT ( Cmd, Par ) then exit;
        if Cmd='LIST'      then if Cmd_LIST     (      Par ) then exit;
        if Cmd='LISTGROUP' then if Cmd_LISTGROUP(      Par ) then exit;
        if Cmd='NEWGROUPS' then if Cmd_NEWGROUPS(      Par ) then exit;
        if Cmd='NEXT'      then if Cmd_LASTNEXT ( Cmd, Par ) then exit;
        if Cmd='POST'      then if Cmd_POST     (      Par ) then exit;
        if Cmd='STAT'      then if Cmd_ARTINFOS ( Cmd, Par ) then exit;
        if Cmd='XHDR'      then if Cmd_XHDR     (      Par ) then exit;
        if Cmd='XOVER'     then if Cmd_XOVER    (      Par ) then exit;
        if Cmd='XPAT'      then if Cmd_XPAT     (      Par ) then exit;
        if Cmd='NEWNEWS'   then if Cmd_NEWNEWS  (      Par ) then exit;
        if Cmd='XHSEARCH'  then if Cmd_XHSEARCH (      Par ) then exit;
        if Cmd='XHHELP'    then if Cmd_XHHELP   (      Par ) then exit;
        if Cmd='CHECK'     then if Cmd_CHECK    (      Par ) then exit;

        if Cmd='IHAVE'     then try
           NntpMode := nmIHAVE;
           if Cmd_IHAVE( Par ) then exit;
        finally NntpMode := nmDefault end;

        if Cmd='TAKETHIS'  then try
           NntpMode := nmTAKETHIS;
           if Cmd_TAKETHIS( Par ) then exit;
        finally NntpMode := nmDefault end;

        // unknown (sub-) command
        HWriteLn( '500 Command not implemented.' );
        Log( LOGID_INFO, 'Unsupported NNTP-command: ' + CmdLine );

     except
        on E: Exception do begin
           Log( LOGID_ERROR, SockDesc('.HandleCommand.Exception') + E.Message );
           Log( LOGID_ERROR, SockDesc('.HandleCommand.ErrorCommand') + LogCmdLine );
           try HDisconnect except end;
        end;
     end;
end;

function TServerClientNNTP.Cmd_ARTINFOS( const Cmd, Par: String ): Boolean;

   procedure SearchArticleNextTime( const MID: String;
                                    const IgnoreHist: Boolean );
   var  j: Integer;
        s: String;
   begin
      HWriteLn( '430 Missing article will be searched on next pull');
      for j := 0 to Hamster.Config.NntpServers.Count-1 do begin
         s := Hamster.Config.NntpServers.Path[j] + SRVFILE_GETMIDLIST;
         HamFileEnter;
         with TStringlist.Create do try
            if FileExists( s ) then LoadFromFile( s );
            if ( IndexOf(MID) < 0 ) and ( IndexOf('!'+MID) < 0 ) then begin
               if IgnoreHist then Add( '!' + MID ) else Add( MID );
               SaveToFile( s )
            end
         finally
            Free;
            HamFileLeave;
         end;
      end;
   end;

   procedure SendArticlePart( const ArtTxt: String;
                              ArtNo       : Integer;
                              ArtMid      : String );
   var  i: Integer;
        SendBuf, SendTxt, GrpLst, s: String;
        Ok: Boolean;
   begin
      if ArtMid = '' then begin
         // get Message-ID
         ArtMid := TMess.HeaderValueFromText( ArtTxt, 'Message-ID:' );
         if ArtMid = '' then ArtMid := '<0>';
      end else begin
         // check read permission for group if requested by MID
         Ok := False;
         GrpLst := TMess.HeaderValueFromText( ArtTxt, 'Newsgroups:' );
         while GrpLst <> '' do begin
            s := TrimWhSpace( NextSepPart( GrpLst, ',' ) );
            if s = '' then break;
            if GroupList.IndexOf( s ) >= 0 then begin
               Ok := True;
               break;
            end;
         end;
         if not Ok then begin
            HWriteLn( '430 no such article found (permission)' );
            exit;
         end;
      end;

      SendTxt := '';
      i := Pos( #13#10#13#10, ArtTxt );
      if Cmd='ARTICLE' then begin
         SendBuf := '220';
         SendTxt := ArtTxt;
      end else if Cmd='HEAD' then begin
         SendBuf := '221';
         SendTxt := copy( ArtTxt, 1, i-1 );
      end else if Cmd='BODY' then begin
         SendBuf := '222';
         SendTxt := copy( ArtTxt, i+4, MaxInt );
      end else if Cmd='STAT' then begin
         SendBuf := '223';
      end;

      SendBuf := SendBuf
               + ' ' + inttostr(ArtNo) + ' ' + ArtMid + ' ' + LowerCase(Cmd);
      Log( LOGID_INFO, '< ' + SendBuf );
      SendBuf := SendBuf + CRLF;

      if Cmd <> 'STAT' then begin
         SendBuf := SendBuf
                  + TextToRfcWireFormat( SendTxt )
                  + '.' + CRLF;
      end;

      HWrite( SendBuf );
   end;

var  ArtMid, GrpNam, ArtTxt: String;
     ArtNo, TempGrpHdl: Integer;
begin
     Result := True;

     if copy( Par, 1, 1 ) = '<' then begin
        // ARTICLE (selection by message-id)
        ArtMid := Par;
        if ArtMid[ length(ArtMid) ] <> '>' then ArtMid := ArtMid + '>';
        if not Hamster.NewsHistory.LocateMID( ArtMid, GrpNam, ArtNo ) then begin
           if Hamster.Config.Settings.GetBoo(hsLocalNntpGetUnknownMIDs) then
              SearchArticleNextTime( ArtMid, False )
           else
              HWriteLn( '430 no such article found (history)');
           exit;
        end;
        Log( LOGID_DEBUG, ArtMid + ' -> ' + GrpNam + ' #' + inttostr(ArtNo) );

        ArtTxt := '';
        TempGrpHdl := Hamster.ArticleBase.Open( GrpNam );
        if TempGrpHdl>=0 then begin
           try
              if ArtNo < Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMin) then GrpNam:='';
              if ArtNo > Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMax) then GrpNam:='';
              if GrpNam<>'' then ArtTxt := Hamster.ArticleBase.ReadArticle( TempGrpHdl, ArtNo );
           finally
              Hamster.ArticleBase.Close( TempGrpHdl );
           end;
        end else begin
           GrpNam := '';
        end;

        if (GrpNam = '') or (ArtTxt = '') then begin
           if Hamster.Config.Settings.GetBoo(hsLocalNntpGetUnknownMIDs) then
              SearchArticleNextTime( ArtMid, True )
           else
              HWriteLn( '430 no such article found (group)' );
           exit;
        end;

        SendArticlePart( ArtTxt, ArtNo, ArtMid );
        exit;
     end;

     if copy( Par, 1, 1 ) <> '<' then begin
        // ARTICLE (selection by number)
        if AutoGroupPends then AutoGroupCommit;
        if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
           HWriteLn( '412 no newsgroup has been selected' );
           exit;
        end;
        if CurrentArtNo=0 then begin
           HWriteLn( '420 no current article has been selected' );
           exit;
        end;

        if Par<>'' then ArtNo := strtoint( Par )
                   else ArtNo := CurrentArtNo;
        ArtTxt := '';
        if (ArtNo>0) and
           (ArtNo>=Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin)) and
           (ArtNo<=Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax)) then begin
           ArtTxt := Hamster.ArticleBase.ReadArticle( CurrentGroup, ArtNo );
           if ArtTxt <> '' then CurrentArtNo := ArtNo;
        end;

        if ArtTxt = '' then begin
           HWriteLn( '423 no such article number in this group' );
           exit;
        end;

        SendArticlePart( ArtTxt, ArtNo, '' );
        exit;
     end;
end;

function TServerClientNNTP.Cmd_LASTNEXT( const Cmd, Par: String ): Boolean;
var  h, s: String;
     j: Integer;
     Art: TMess;
begin
     Result := True;

     if AutoGroupPends then AutoGroupCommit;
     if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
        HWriteLn( '412 no newsgroup selected' );
        exit;
     end;
     if CurrentArtNo=0 then begin
        HWriteLn( '420 no current article has been selected' );
        exit;
     end;

     h := '';
     j := CurrentArtNo;
     repeat
        if Cmd='LAST' then begin
           dec( j );
           if (j<1) or (j<Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin)) then break;
        end else begin
           inc( j );
           if j>Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax) then break;
        end;
        Art := TMess.Create;
        try
           Art.FullText := Hamster.ArticleBase.ReadArticle( CurrentGroup, j );
           h := Art.HeaderValueByNameSL( 'Message-ID:' );
        finally Art.Free end;
     until (h<>'');

     if h<>'' then begin
        CurrentArtNo := j;
        s := '223' + ' '
           + inttostr(CurrentArtNo) + ' '
           + h + ' '
           + 'article selected';
        HWriteLn( s );
     end else begin
        if Cmd='LAST' then HWriteLn( '422 no previous article in this group' )
                      else HWriteLn( '421 no next article in this group' );
     end;
end;

function TServerClientNNTP.Cmd_QUIT( const Par: String ): Boolean;
begin
     Result := True;

     if HConnected then HWriteLn( '205 Closing connection.' );
     Sleep( Hamster.Config.Settings.GetInt(hsLocalTimeoutQuitDelay) );
     try
        HDisconnect;
     except
        on E:Exception do Log(LOGID_DEBUG, 'Exception on .Close: ' + E.Message );
     end;
     Terminate;
end;

function TServerClientNNTP.Cmd_AUTHINFO( const Par: String ): Boolean;
var  s: String;
begin
     Result := True;

     try
        if UpperCase(copy(Par,1,4))='USER' then begin
           CurrentUserName := TrimWhSpace( copy( Par,6,255 ) );
           CurrentUserID   := ACTID_INVALID;
           HWriteLn( '381 More authentication information required' );
           exit;
        end;

        if UpperCase(copy(Par,1,4))='PASS' then begin
           s := LoginUser( TrimWhSpace( copy( Par,6,255 ) ) );
           HWriteLn( s );
           ClientsChange;
           exit;
        end;

        s := Par;
        if UpperCase( NextWhSpacePart(s) ) = 'SIMPLE' then begin
           if s = '' then s := HReadLn;
           CurrentUserName := NextWhSpacePart(s);
           CurrentUserID   := ACTID_INVALID;
           s := LoginUser( NextWhSpacePart(s) );
           HWriteLn( s );
           ClientsChange;
           exit;
        end;

        HWriteLn( '501 Command not supported' );
     except
        on E:Exception do begin
           Log( LOGID_ERROR, 'Nntp.Cmd_AUTHINFO-Exception: ' + E.Message );
        end;
     end;
end;

function TServerClientNNTP.Cmd_POST( const Par: String ): Boolean;
var  Reply, ArtText, s: String;
     LogNow: TDateTime;
     Msg: TMess;
begin
   Result := True;

   if not( UserMayPost and
           ( (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW )
   ) then begin
      HWriteLn( '440 posting not allowed' );
      exit;
   end;

   if ( Hamster.Config.Settings.GetBoo(hsGenerateNewsMID)  ) and
      ( Hamster.Config.Settings.GetStr(hsFQDNforMID) <> '' ) then begin
      RecommendedMID := MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) );
      Reply := '340 OK, recommended ID ' + RecommendedMID;
   end else begin
      RecommendedMID := '';
      Reply := '340 send article to be posted. End with <CR-LF>.<CR-LF>';
   end;

   ArtText := HRequestText( Reply );
   if not HConnected then exit;

   Reply := HandlePostedArticle( ArtText );
   HWriteLn( Reply );

   // Save result in NntpServer.log
   LogNow := Now;
   Msg := TMess.Create;
   try
      Msg.FullText := ArtText;
      s := DateTimeToLogTime( LogNow )
         + #9 + 'User='       + CurrentUserName
         + #9 + 'IP='         + nAddrToStr( FClientAD.S_addr )
         + #9 + 'Action='     + 'POST'
         + #9 + 'Result='     + Reply
         + #9 + 'MID='        + Logify( Msg.HeaderValueByNameSL(HDR_NAME_MESSAGE_ID) )
         + #9 + 'From='       + Logify( Msg.HeaderValueByNameSL(HDR_NAME_FROM) )
         + #9 + 'Newsgroups=' + Logify( Msg.HeaderValueByNameSL(HDR_NAME_NEWSGROUPS) )
         + #9 + 'Subject='    + Logify( Msg.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
         ;
   finally Msg.Free end;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_NNTPSERVER
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );
end;

function TServerClientNNTP.Cmd_IHAVE( const Par: String ): Boolean;
var  MessageId, Reply, ArtText, s: String;
     LogNow: TDateTime;
     Msg: TMess;
begin
   Result := True;

   MessageId := TrimWhSpace( Par );
   if not IsMessageId( MessageId ) then begin
      HWriteLn( '435 article not wanted - do not send it ' + MessageId );
      exit;
   end;

   if not( UserMayFeed and
           ( (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW )
   ) then begin
      HWriteLn( '480 Transfer permission denied ' + MessageId );
      exit;
   end;

   if Hamster.NewsHistory.ContainsMID( MessageID ) then begin
      HWriteLn( '435 article not wanted - do not send it ' + MessageId );
      exit;
   end;

   Reply := '335 send article to be transferred. ' + MessageId;

   ArtText := HRequestText( Reply );
   if not HConnected then exit;

   Reply := HandleFeededArticle( MessageId, ArtText, True {IHAVE} );
   HWriteLn( Reply );

   // Save result in NntpServer.log
   LogNow := Now;
   Msg := TMess.Create;
   try
      Msg.FullText := ArtText;
      s := DateTimeToLogTime( LogNow )
         + #9 + 'User='       + CurrentUserName
         + #9 + 'IP='         + nAddrToStr( FClientAD.S_addr )
         + #9 + 'Action='     + 'IHAVE'
         + #9 + 'Result='     + Reply
         + #9 + 'MID='        + Logify( Msg.HeaderValueByNameSL(HDR_NAME_MESSAGE_ID) )
         + #9 + 'From='       + Logify( Msg.HeaderValueByNameSL(HDR_NAME_FROM) )
         + #9 + 'Newsgroups=' + Logify( Msg.HeaderValueByNameSL(HDR_NAME_NEWSGROUPS) )
         + #9 + 'Subject='    + Logify( Msg.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
         ;
   finally Msg.Free end;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_NNTPSERVER
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );
end;

function TServerClientNNTP.Cmd_CHECK( const Par: String ): Boolean;
var  MessageId: String;
begin
   Result := True;

   MessageId := TrimWhSpace( Par );
   if not IsMessageId( MessageId ) then begin
      HWriteLn( '438 already have it, please don''t send it ' + MessageId );
      exit;
   end;

   if not( UserMayFeed and
           ( (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW )
   ) then begin
      HWriteLn( '480 Transfer permission denied ' + MessageId );
      exit;
   end;

   if Hamster.NewsHistory.ContainsMID( MessageID ) then begin
      HWriteLn( '438 already have it, please don''t send it ' + MessageId );
   end else begin
      HWriteLn( '238 no such article found, please send it ' + MessageId );
   end;
end;

function TServerClientNNTP.Cmd_TAKETHIS( const Par: String ): Boolean;
var  MessageId, Reply, ArtText, s: String;
     LogNow: TDateTime;
     Msg: TMess;
begin
   Result := True;

   MessageId := TrimWhSpace( Par );

   ArtText := HReadLnText;
   if not HConnected then exit;

   if not IsMessageId( MessageId ) then begin
      HWriteLn( '439 article transfer failed ' + MessageId );
      exit;
   end;

   if not( UserMayFeed and
           ( (IPAccess and IPACC_ACCESS_RW) = IPACC_ACCESS_RW )
   ) then begin
      HWriteLn( '480 Transfer permission denied ' + MessageId );
      exit;
   end;

   if Hamster.NewsHistory.ContainsMID( MessageID ) then begin
      HWriteLn( '239 article transferred ok but already known ' + MessageId );
      exit;
   end;

   Reply := HandleFeededArticle( MessageId, ArtText, False {TAKETHIS} );
   HWriteLn( Reply );

   // Save result in NntpServer.log
   LogNow := Now;
   Msg := TMess.Create;
   try
      Msg.FullText := ArtText;
      s := DateTimeToLogTime( LogNow )
         + #9 + 'User='       + CurrentUserName
         + #9 + 'IP='         + nAddrToStr( FClientAD.S_addr )
         + #9 + 'Action='     + 'TAKETHIS'
         + #9 + 'Result='     + Reply
         + #9 + 'MID='        + Logify( Msg.HeaderValueByNameSL(HDR_NAME_MESSAGE_ID) )
         + #9 + 'From='       + Logify( Msg.HeaderValueByNameSL(HDR_NAME_FROM) )
         + #9 + 'Newsgroups=' + Logify( Msg.HeaderValueByNameSL(HDR_NAME_NEWSGROUPS) )
         + #9 + 'Subject='    + Logify( Msg.HeaderValueByNameSL(HDR_NAME_SUBJECT) )
         ;
   finally Msg.Free end;
   HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                      + LOGFILE_NNTPSERVER
                      + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                      + LOGFILE_EXTENSION,
                      s );
end;

function TServerClientNNTP.Cmd_SLAVE( const Par: String ): Boolean;
begin
     Result := True;
     HWriteLn( '202 slave status noted' );
end;

function TServerClientNNTP.Cmd_GROUP( const Par: String ): Boolean;
var  GroupOK, AutoAddPulls: Boolean;
     Groupname, s: String;
begin
   Result := True;

   if AutoGroupPends then AutoGroupRollback;
   Groupname := Par;

   // check if group is in user's group-list
   GroupOK := ( GroupList.IndexOf( Groupname ) >= 0 );

   // add missing group/pulls automatically if enabled for current user
   AutoAddPulls := False;
   if UserAutoSubMode <> NEWSAUTOSUB_NONE then begin

      if GroupOK then begin

         // Already existing group, just check/add the pulls for it.
         // Note: Pulls are added independent from groups, as a group might
         // just have lost its pulls in the first step of auto-unsubscribe.
         AutoAddPulls := True;

      end else begin

         // Unkown group, check if it can be created automatically.
         // a) Does user have permission for given groupname?
         // b) Is group available on any server?
         if GetPermissionForGroup( Groupname, PermPost, PermRead ) <> PERM_NOTH then begin
            if Hamster.Config.NntpServers.AnyServerHasGroup( Groupname ) then begin
               GroupOK := True;
               AutoGroupPrepare( Groupname );
            end;
         end;

      end;

   end;

   // send reply
   if GroupOK then begin

      // 211 101 100 200 group.name
      // 0:errno 1:count 2:first 3:last 4:name
      
      if AutoGroupPends then begin

         // return values for the "virtual" placeholder article
         HWriteLn( '211 1 1 1 ' + AutoGroupName );

      end else begin

         // select group and return its values
         SetCurrentGroup( Groupname );
         with Hamster.ArticleBase do begin
            if AutoAddPulls and GetBoo(CurrentGroup,gsAutoAddPulls) then begin
               AutoGroupAddPulls( Groupname );
            end;

            // return group's values
            s := '211' + SP + inttostr( Count[CurrentGroup] )
                       + SP + inttostr( GetInt(CurrentGroup,gsLocalMin) )
                       + SP + inttostr( GetInt(CurrentGroup,gsLocalMax) )
                       + SP + Name[CurrentGroup];
            HWriteLn( s );

            // update LastClientRead marker
            SetDT( CurrentGroup, gsLastClientRead, Now );
         end;

      end;

   end else begin

      HWriteLn( '411 no such news group (' + Groupname + ')' );

   end;
end;

function TServerClientNNTP.Cmd_NEWGROUPS( const Par: String ): Boolean;
var  j, TempGrpHdl: Integer;
     s, g, m: String;
     DT: TDateTime;
     TS: TStringList;
begin
     Result := True;

     // NEWGROUPS YYMMDD HHMMSS [GMT]
     TS := TStringList.Create;
     try
        ArgsWhSpace( Par, TS, 3 );
        DT := NntpDateTimeToDateTime( TS[0], TS[1] );
        if DT=0 then begin
           HWriteLn( '431 Unexpected date-format (' + Par + ')' );
           exit;
        end;
        s := UpperCase( TS[2] );
        if (s<>'GMT') and (s<>'UTC') then DT:=DateTimeLocalToGMT(DT);

     finally
        TS.Free;
     end;

     if ReloadGroupList then begin
        // note: no full list support here, required information not available
        HWriteLn( '231 list of new newsgroups follows' );
        for j:=0 to GroupList.Count-1 do begin
           TempGrpHdl := Hamster.ArticleBase.Open( GroupList[j] );
           if TempGrpHdl>=0 then begin
              if Hamster.ArticleBase.DTCreated(TempGrpHdl)>=DT then begin
                 // group last first type
                 g := Hamster.ArticleBase.GetStr( TempGrpHdl, gsGroupType );
                 m := Hamster.ArticleBase.GetStr( TempGrpHdl, gsModerator );
                 if g = 'g' then begin // report gateway as y or n
                    if m = '' then g := 'n' else g := 'y';
                 end;
                 if (g<>'y') and (g<>'n') and (g<>'m') then g := 'y';
                 s := Hamster.ArticleBase.Name[TempGrpHdl] + ' '
                    + inttostr( Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMax) ) + ' '
                    + inttostr( Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMin) ) + ' '
                    + g;
                 HWriteLnQ( s );
              end;
              Hamster.ArticleBase.Close( TempGrpHdl );
           end;
        end;
        HWriteLn( '.' );
     end else begin
        HWriteLn( '503 System-error, check logfile. [ng]' );
     end;
end;

function TServerClientNNTP.Cmd_LIST( const Par: String ): Boolean;

   procedure DescSplit( const Line: String; out GrpName, GrpDesc: String );
   var  i: Integer;
   begin
      i := PosWhSpace( Line );
      if i > 0 then begin
         GrpName := copy( Line, 1, i-1 );
         GrpDesc := copy( Line, i+1, 255 );
      end else begin
         GrpName := Line;
         GrpDesc := '';
      end;
   end;

   function SendActive( var SendBuf: String;
                        const GrpName, Pattern: String ): Boolean;
   var  TempGrpHdl: Integer;
   begin
      Result := True;
      try
         if GrpName = '' then exit;

         if Pattern<>'' then begin
            if not WildMat( PChar(GrpName), PChar(Pattern) ) then exit;
         end;

         if Hamster.Config.Newsgroups.IndexOf(GrpName) >= 0 then begin
            TempGrpHdl := Hamster.ArticleBase.Open( GrpName );
            if TempGrpHdl >= 0 then try
               // group last first p
               SendBuf := SendBuf
                  + Hamster.ArticleBase.Name[TempGrpHdl] + ' '
                  + inttostr( Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMax) ) + ' '
                  + inttostr( Hamster.ArticleBase.GetInt(TempGrpHdl,gsLocalMin) ) + ' '
                  + Hamster.ArticleBase.GetStr( TempGrpHdl, gsGroupType )
                  + CRLF;
            finally
               Hamster.ArticleBase.Close( TempGrpHdl );
            end;
         end else begin
            // group last first p
            // '1' = dummy article of auto subscribe
            SendBuf := SendBuf
                     + GrpName + ' ' + '1' + ' ' + '1' + ' ' + 'y'
                     + CRLF;
         end;

         Result := HConnected;
         
      except
         Result := False;
      end;
   end;

   function SendDesc( var SendBuf: String;
                      const GrpName, GrpDesc, Pattern: String;
                      IsLocalGroup: Boolean ): Boolean;
   var  TempGrpHdl: Integer;
   begin
      Result := True;
      try
         if GrpName = '' then exit;

         if Pattern<>'' then begin
            if not WildMat( PChar(GrpName), PChar(Pattern) ) then exit;
         end;

         if IsLocalGroup then begin
            TempGrpHdl := Hamster.ArticleBase.Open( GrpName );
            if TempGrpHdl>=0 then try
               SendBuf := SendBuf
                     + Hamster.ArticleBase.Name[TempGrpHdl] + #9
                     + Hamster.ArticleBase.GetStr( TempGrpHdl, gsDescription )
                     + CRLF;
            finally
               Hamster.ArticleBase.Close( TempGrpHdl );
            end;
         end else begin
            if GetPermissionForGroup( GrpName, PermPost, PermRead )
                 <> PERM_NOTH then begin
               SendBuf := SendBuf
                        + GrpName + #9 + GrpDesc
                        + CRLF;
            end;
         end;
         
         Result := HConnected;
         
      except
         Result := False;
      end;
   end;

var  j, TempGrpHdl: Integer;
     s, Pattern, GrpName, GrpDesc, SendBuf: String;
     TempList: TStringList;
begin
     Result := True;

     if ( Par = '' ) or ( UpperCase(Copy(Par,1,6)) = 'ACTIVE' ) then begin
        if Par = '' then begin
           Pattern := '';
        end else begin
           j := PosWhSpace( Par );
           if j=0 then Pattern := ''
                  else Pattern := TrimWhSpace( copy(Par,j+1,255) );
        end;

        HWriteLn( '215 list of newsgroups follows' );
        SetLength( SendBuf, 0 );

        if UserAutoSubMode = NEWSAUTOSUB_NONE then begin

           // send user's list of groups
           for j:=0 to GroupList.Count-1 do begin
              if not SendActive( SendBuf, GroupList[j], Pattern ) then break;
              if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
           end;

        end else begin

           // a) send groups available on any remote server
           TempList := TStringList.Create;
           try
              HamFileLoadList( AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS, TempList );
              for j := 0 to TempList.Count-1 do begin
                 DescSplit( TempList[j], GrpName, GrpDesc );
                 if GetPermissionForGroup( GrpName, PermPost, PermRead )
                    <> PERM_NOTH then begin
                    if not SendActive( SendBuf, GrpName, Pattern ) then break;
                    if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
                 end;
              end;
           finally
              TempList.Free;
           end;

           // b) append (user's) internal and local groups
           for j := 0 to GroupList.Count-1 do begin
              GrpName := GroupList[j];
              if Hamster.Config.Newsgroups.GroupClass(GrpName) in
                 [ aclInternal, aclLocal, aclWasPulled ] then begin
                 if not SendActive( SendBuf, GrpName, Pattern ) then break;
                 if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
              end;
           end;

        end;

        if length(SendBuf) > 0 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
        HWriteLn( '.' );

        exit;
     end;

     if UpperCase(copy(Par,1,10))='NEWSGROUPS' then begin
        j := PosWhSpace( Par );
        if j=0 then Pattern:=''
               else Pattern:=TrimWhSpace( copy(Par,j+1,255) );

        HWriteLn( '215 information follows' );
        SetLength( SendBuf, 0 );

        if UserAutoSubMode = NEWSAUTOSUB_NONE then begin

           // send user's list of groups
           for j:=0 to GroupList.Count-1 do begin
              if not SendDesc( SendBuf, GroupList[j], '', Pattern, True ) then break;
              if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
           end;

        end else begin

           // a) send groups available on any remote server
           TempList := TStringList.Create;
           try
              HamFileLoadList( AppSettings.GetStr(asPathServer) + SRVFILE_ALLDESCS, TempList );
              for j:=0 to TempList.Count-1 do begin
                 DescSplit( TempList[j], GrpName, GrpDesc );
                 if length(GrpDesc) > 1 then begin
                    if not SendDesc( SendBuf, GrpName, GrpDesc, Pattern, False ) then break;
                    if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
                 end;
              end;
           finally
              TempList.Free;
           end;

           // b) append (user's) internal and local groups
           for j:=0 to GroupList.Count-1 do begin
              GrpName := GroupList[j];
              if Hamster.Config.Newsgroups.GroupClass(GrpName) in [aclInternal, aclLocal] then begin
                 if not SendDesc( SendBuf, GroupList[j], '', Pattern, True ) then break;
                 if length(SendBuf) >= 4096 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
              end;
           end;

        end;

        if length(SendBuf) > 0 then begin HWrite(SendBuf); SetLength(SendBuf,0) end;
        HWriteLn( '.' );
        
        exit;
     end;

     if UpperCase(Par)='ACTIVE.TIMES' then begin
        // note: an optional full list makes not much sense here, as there is
        //       no meaningful creation date amongst different servers, so we
        //       just return the information of our own groups.
        HWriteLn( '215 list of newsgroups follows' );
        for j:=0 to GroupList.Count-1 do begin
           TempGrpHdl := Hamster.ArticleBase.Open( GroupList[j] );
           if TempGrpHdl>=0 then begin
              s := Hamster.ArticleBase.Name[TempGrpHdl] + ' '
                 + inttostr( DateTimeToUnixTime(Hamster.ArticleBase.DTCreated(TempGrpHdl)) ) + ' '
                 + 'local';
              HWriteLnQ( s );
              Hamster.ArticleBase.Close( TempGrpHdl );
           end;
        end;
        HWriteLn( '.' );
        exit;
     end;

     if UpperCase(Par)='OVERVIEW.FMT' then begin
        HWriteLn( '215 information follows' );
        HWriteLnQ( 'Subject:' );
        HWriteLnQ( 'From:' );
        HWriteLnQ( 'Date:' );
        HWriteLnQ( 'Message-ID:' );
        HWriteLnQ( 'References:' );
        HWriteLnQ( 'Bytes:' );
        HWriteLnQ( 'Lines:' );
        HWriteLnQ( 'Xref:full' );
        HWriteLn( '.' );
        exit;
     end;

     if UpperCase(Par)='EXTENSIONS' then begin
        // HWriteLn( '501 Bad command use' );
        HWriteLn( '202 Extensions supported:' );
        HWriteLnQ( 'AUTHINFO' );
        HWriteLnQ( 'LISTGROUP' );
        HWriteLn( '.' );
        exit;
     end;

     Result := False;
end;

function TServerClientNNTP.Cmd_LISTGROUP( const Par: String ): Boolean;
var  OK: Boolean;
     j, vv, bb: Integer;
     s: String;
begin
     Result := True;

     if AutoGroupPends then AutoGroupCommit;

     if Par<>'' then begin
        OK := False;
        for j:=0 to GroupList.Count-1 do begin
           if GroupList[j]=Par then begin OK:=True; break; end;
        end;
        if not OK then begin
           HWriteLn( '411 no such news group' );
           exit;
        end;
        SetCurrentGroup(Par);
     end;

     if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
        HWriteLn( '412 not currently in newsgroup' );
        exit;
     end;

     vv := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin);
     bb := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);

     HWriteLn( '211 list of article numbers follows' );
     s := '';
     for j:=vv to bb do begin
        // if ClientData.CurrentGroup.ReadSize(j)>0 then begin
        if Hamster.ArticleBase.GetKeyIndex(CurrentGroup,j)>=0 then begin
           if s<>'' then s:=s+#13#10;
           s := s + inttostr(j); // last CRLF is added by HWriteLnQ() below
        end;
     end;
     if s<>'' then HWriteLnQ( s );
     HWriteLn( '.' );
end;

function TServerClientNNTP.Cmd_MODE( const Par: String ): Boolean;
begin
     Result := True;

     if UpperCase(Par)='READER' then begin
        if (IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW then begin
           HWriteLn( '200 ignored' );
        end else begin
           HWriteLn( '201 ignored' );
        end;
        exit;
     end;

     if UpperCase(Par)='STREAM' then begin
        HWriteLn( '203 Streaming is OK' );
        exit;
     end;

     Result := False;
end;

function TServerClientNNTP.Cmd_DATE( const Par: String ): Boolean;
begin
     Result := True;
     HWriteLn( '111 ' + DateTimeToTimeStamp(NowGMT) );
end;

function TServerClientNNTP.Cmd_XOVER( const Par: String ): Boolean;
var  s, SendBuf: String;
     j, vv, bb: Integer;
     Art: TMess;
begin
     Result := True;

     if AutoGroupPends then AutoGroupCommit;
     if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
        HWriteLn( '412 No news group selected' );
        exit;
     end;
     if (Par='') and (CurrentArtNo=0) then begin
        HWriteLn( '420 No article(s) selected' );
        exit;
     end;

     s := Par;
     if s='' then s:=inttostr( CurrentArtNo );

     j := Pos('-',s);
     if j>0 then begin
        if j=1 then vv := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin)
               else vv := strtoint( copy(s,1,j-1) );
        s  := copy( s, j+1, 255 );
        if s<>'' then bb := strtoint( s )
                 else bb := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);
     end else begin
        vv := strtoint( s );
        bb := vv;
     end;

     HWriteLn( '224 Overview information follows' );

     Art := TMess.Create;
     try
        SendBuf := '';
        for j := vv to bb do begin
           s := Hamster.ArticleBase.ReadOverview( CurrentGroup, j, Art );
           if length(s) > 0 then begin
              SendBuf := SendBuf + s + CRLF;
              if length(SendBuf)>4096 then begin HWrite(SendBuf); SendBuf:='' end;
           end;
        end;
     finally Art.Free end;

     if length(SendBuf) > 0 then HWrite( SendBuf );
     HWriteLn( '.' );
end;

function TServerClientNNTP.Cmd_XPAT( const Par: String ): Boolean;
var  Parser: TParser;
     HdrNam, HdrVal, Pattern, ArtText, s: String;
     j, vv, bb, k, ArtSize: Integer;
     Art: TMess;
     ok: Boolean;
begin
     Result := True;

     if AutoGroupPends then AutoGroupCommit;
     if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
        HWriteLn( '412 No news group current selected' );
        exit;
     end;
     if (Par='') and (CurrentArtNo=0) then begin
        HWriteLn( '420 No article(s) selected' );
        exit;
     end;

     Parser := TParser.Create;
     Parser.Parse( Par, ' ' );

     HdrNam := Parser.sPart( 0, '' );
     s := Parser.sPart( 1, '' ); // range|mid
     if (HdrNam='') or (s='') or (pos('@',s)>0) then begin
        Parser.Free;
        HWriteLn( '501 Unknown/unsupported xpat-syntax' );
        exit;
     end;

     j := Pos('-',s);
     if j>0 then begin
        if s='-' then begin
           vv := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin);
           bb := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);
        end else begin
           vv := strtoint( copy(s,1,j-1) );
           if vv<Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin) then
              vv:=Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin);

           s  := copy( s, j+1, 255 );
           if s<>'' then begin
              bb := strtoint( s );
              if bb>Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax) then
                 bb:=Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);
           end else begin
              bb := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);
           end;
        end;
     end else begin
        vv := strtoint( s );
        bb := vv;
     end;

     HWriteLn( '221 Header follows' );

     Art := TMess.Create;
     for j:=vv to bb do begin
        ArtSize := 4096;
        ArtText := Hamster.ArticleBase.ReadArticleSized( CurrentGroup, j, ArtSize );
        if TMess.HasHeaderEnd( ArtText ) then begin
           Art.FullText := ArtText;
        end else begin
           Art.FullText := Hamster.ArticleBase.ReadArticle( CurrentGroup, j );
        end;

        HdrVal := Art.HeaderValueByNameSL( HdrNam + ':' );

        ok := False;
        if HdrVal<>'' then begin
           k := 2;
           repeat
              Pattern := Parser.sPart( k, '' );
              if (Pattern='') and (k=2) then begin
                 ok := True; // impliziter "*"
                 break;
              end;
              if Pattern='' then break;

              if WildMat( PChar(HdrVal), PChar(Pattern) ) then begin
                 ok := True;
                 break;
              end;
              inc(k);
           until False;
        end;

        if ok then begin
           s := inttostr(j) + ' ' + HdrVal;
           HWriteLnQ( s );
        end;
     end;
     Art.Free;
     Parser.Free;

     HWriteLn( '.' );
end;

function TServerClientNNTP.Cmd_XHDR( const Par: String ): Boolean;
var  j, vv, bb, ArtSize: Integer;
     HdrNam, Range, ArtText, s, h: String;
     Art: TMess;
begin
     Result := True;

     j := Pos( ' ', Par );
     if j=0 then begin
        HdrNam := Trim(Par);
        Range  := '';
     end else begin
        HdrNam := Trim( copy( Par, 1, j-1 ) );
        Range  := Trim( copy( Par, j+1, 255 ) );
     end;

     if copy(Range,1,1)='<' then begin
        HWriteLn( '500 XHDR with Message-ID not implemented.' );
        Log( LOGID_INFO, 'Unsupported NNTP-server-command: XHDR <Message-ID>' );
        exit;
     end;

     if AutoGroupPends then AutoGroupCommit;
     if (CurrentGroup<0) or (Hamster.ArticleBase.Name[CurrentGroup]='') then begin
        HWriteLn( '412 No news group selected' );
        exit;
     end;

     if (Range='') and (CurrentArtNo=0) then begin
        HWriteLn( '420 No article(s) selected' );
        exit;
     end;

     s := Range;
     if s='' then s:=inttostr( CurrentArtNo );

     j := Pos('-',s);
     if j>0 then begin
        if j=1 then vv := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMin)
               else vv := strtoint( copy(s,1,j-1) );
        s  := copy( s, j+1, 255 );
        if s<>'' then bb := strtoint( s )
                 else bb := Hamster.ArticleBase.GetInt(CurrentGroup,gsLocalMax);
     end else begin
        vv := strtoint( s );
        bb := vv;
     end;

     HWriteLn( '221 Header follow' );

     Art := TMess.Create;
     for j:=vv to bb do begin
        ArtSize := 4096;
        ArtText := Hamster.ArticleBase.ReadArticleSized( CurrentGroup, j, ArtSize );
        if TMess.HasHeaderEnd( ArtText ) then begin
           Art.FullText := ArtText;
        end else begin
           Art.FullText := Hamster.ArticleBase.ReadArticle( CurrentGroup, j );
        end;

        if length(Art.HeaderText) > 0 then begin
           h := Art.HeaderValueByNameSL( HdrNam + ':' );
           if h='' then h:='(none)';
           if h<>'' then begin
              s := inttostr(j) + ' ' + h;
              HWriteLnQ( s );
           end;
        end;
     end;
     Art.Free;

     HWriteLn( '.' );
end;

function TServerClientNNTP.Cmd_NEWNEWS( const Par: String ): Boolean;
// NEWNEWS newsgroups date time ["GMT"|"UTC"] [<distributions>]
// Note: if <distributions> is given, it is ignored
var  GrpNo, ArtNo, RefNo, GrpHdl, ArtSize, ArtMin, ArtMax: Integer;
     sGroups, sDate, sTime, sGMT, MessageID, ArtText: String;
     GroupPats, TS: TStringList;
     dtStart: TDateTime;
     Art: TMess;
begin
   Result := True;

   if not UserMayNewNews then begin
      HWriteLn( '502 NEWNEWS-permission denied.' );
      exit;
   end;

   if AutoGroupPends then AutoGroupCommit;

   TS := TStringList.Create;
   GroupPats := TStringList.Create;
   Art := TMess.Create;

   try
      // parse arguments
      ArgsWhSpace( Par, TS, 5 );
      sGroups := TS[0];
      sDate   := TS[1];
      sTime   := TS[2];
      if copy(TS[3],1,1)<>'<' then sGMT:=UpperCase(TS[3]) else sGMT:='';

      // parse newsgroups-arg.
      if ArgsWhSpace( ReplaceChar( sGroups, ',', ' ' ), GroupPats ) = 0 then begin
         HWriteLn( '501 Bad newsgroup specifier: ' + sGroups );
         exit;
      end;

      // parse date-arg.
      dtStart := NntpDateTimeToDateTime( sDate, sTime );
      if dtStart = 0 then begin
         HWriteLn( '501 Bad date: ' + sDate + ' ' + sTime + ' ' + sGMT );
         exit;
      end;
      if (sGMT='GMT') or (sGMT='UTC') then dtStart:=DateTimeGMTToLocal(dtStart);

      // loop through user's newsgroup-list and check if group is selected;
      // if selected, loop through all articles of the group and report its
      // new articles not reported so far

      HWriteLn( '230 New news follow' );

      for GrpNo:=0 to GroupList.Count-1 do begin

         if GroupMatchesPatterns( GroupList[GrpNo], GroupPats ) then begin

            GrpHdl := Hamster.ArticleBase.Open( GroupList[GrpNo] );
            if GrpHdl>=0 then try

               ArtMin := Hamster.ArticleBase.GetInt(GrpHdl,gsLocalMin);
               ArtMax := Hamster.ArticleBase.GetInt(GrpHdl,gsLocalMax);
               
               for ArtNo := ArtMin to ArtMax do begin

                  ArtSize := 4096;
                  ArtText := Hamster.ArticleBase.ReadArticleSized( GrpHdl, ArtNo, ArtSize );

                  if ArtSize > 1 then begin // still available?
                  
                     if TMess.HasHeaderEnd( ArtText ) then begin
                        Art.FullText := ArtText;
                     end else begin
                        Art.FullText := Hamster.ArticleBase.ReadArticle( GrpHdl, ArtNo );
                     end;

                     if Art.GetReceivedDT >= dtStart then begin // a new one?
                        MessageID := Art.HeaderValueByNameSL( 'Message-ID:' );

                        // check if article was already reported
                        SplitXrefGroups( Art.HeaderValueByNameSL( 'Xref:' ), TS );
                        TS.Sort; // sort Xref-groups like GroupList-groups
                        for RefNo:=0 to TS.Count-1 do begin
                           // is current group the first selected group?
                           if AnsiCompareText( TS[RefNo], GroupList[GrpNo] ) = 0 then break;
                           // was a previous Xref-group selected?
                           if GroupMatchesPatterns( TS[RefNo], GroupPats ) then begin
                              MessageID := ''; // don't report again
                              break;
                           end;
                        end;

                        if MessageID<>'' then HWriteLnQ( MessageID );
                     end;
                  end;
               end;
            finally
               Hamster.ArticleBase.Close( GrpHdl );
            end;
         end;
      end;

      HWriteLn( '.' );

   finally
      Art.Free;
      GroupPats.Free;
      TS.Free;
   end;
end;

function TServerClientNNTP.Report_XHSearchResults( const Groupname: String;
                                                   const ArticleNo: LongWord;
                                                   const Article  : TMess ): Boolean;
begin
   try
      HWriteLnQ( Groupname
           + #9 + inttostr(ArticleNo)
           + #9 + Article.HeaderValueByNameSL('Message-ID') );
      Result := True;
   except
      Result := False;
   end;
end;

function TServerClientNNTP.Cmd_XHSEARCH( const Par: String ): Boolean;
// a) XHSEARCH groupregex field patterns
// b) XHSEARCH => multiline -> *(GRP groupregex|PAT field patterns)
var  SL: TStringList;
     NS: TNewsSearcher;
     i : Integer;
begin
   Result := True;

   Log( LOGID_WARN, 'Note: XHSEARCH command will be removed in next version!' ); // TODO

   if not UserMayXHSearch then begin
      HWriteLn( '502 XHSEARCH-permission denied.' );
      exit;
   end;

   if AutoGroupPends then AutoGroupCommit;

   SL := TStringList.Create;
   NS := TNewsSearcher.Create;

   try
      try
         if Par = '' then begin

            SL.Text := HRequestText(
               '388 Ok, send params (GRP groupregex|PAT field patterns)' );
            NS.SelectParamList( SL, GroupList );

         end else begin

            i := PosWhSpace( Par );
            if i = 0 then begin
               NS.SelectGroups( Par, GroupList );
               NS.LinesAdd( '' );
            end else begin
               NS.SelectGroups( copy( Par, 1, i-1 ), GroupList );
               NS.LinesAdd( copy( Par, i+1, MaxInt ) );
            end;

         end;

      except
         on E: Exception do begin
            HWriteLn( '488 Invalid params: ' + E.Message );
            exit;
         end;
      end;

      // some checks
      if NS.SelectedGroups.Count = 0 then begin
         HWriteLn( '488 No newsgroups selected' );
         exit;
      end;
      if NS.LinesCount = 0 then begin
         HWriteLn( '488 No search patterns given' );
         exit;
      end;

      // search
      HWriteLn( '288 Match list follows' );
      NS.Search( Report_XHSearchResults );
      HWriteLnQ( '.summary' + #9 + inttostr(NS.CountFound )
                             + #9 + inttostr(NS.CountTested) );
      HWriteLn( '.' );

   finally
      NS.Free;
      SL.Free;
   end;
end;

function TServerClientNNTP.Cmd_XHHELP( const Par: String ): Boolean;
begin
     Result := True;
     Log( LOGID_WARN, 'Note: XHHELP command will be removed in next version!' ); // TODO
     HWriteLn( '180 Implemented XH commands follow:' );
     HWriteLnQ( '    xhhelp [xhcommand]' );
     HWriteLnQ( '    xhsearch [groupregex field patterns]' );
     HWriteLn( '.' );
end;

function TServerClientNNTP.Cmd_HELP( const Par: String ): Boolean;
begin
     Result := True;
     HWriteLn( '100 Implemented commands follow:' );
     HWriteLnQ( '    authinfo user Name|pass Password' );
     HWriteLnQ( '    article [MessageID|Number]' );
     HWriteLnQ( '    body [MessageID|Number]' );
     HWriteLnQ( '    check MessageID' );
     HWriteLnQ( '    date' );
     HWriteLnQ( '    group newsgroup' );
     HWriteLnQ( '    head [MessageID|Number]' );
     HWriteLnQ( '    help' );
     HWriteLnQ( '    ihave MessageID' );
     HWriteLnQ( '    last' );
     HWriteLnQ( '    list [active|active.times|newsgroups|overview.fmt]' );
     HWriteLnQ( '    listgroup [newsgroup]' );
     HWriteLnQ( '    mode reader' );
     HWriteLnQ( '    newgroups [YY]yymmdd hhmmss ["GMT"|"UTC"]' );
     HWriteLnQ( '    newnews newsgroups [YY]yymmdd hhmmss ["GMT"|"UTC"]' );
     HWriteLnQ( '    next' );
     HWriteLnQ( '    post' );
     HWriteLnQ( '    quit' );
     HWriteLnQ( '    slave' );
     HWriteLnQ( '    stat [MessageID|Number]' );
     HWriteLnQ( '    takethis MessageID' );
     HWriteLnQ( '    xhdr header [range]' );
     HWriteLnQ( '    xpat header range pat [pat...]' );
     HWriteLnQ( '    xover [range]' );
     HWriteLn( '.' );
end;

function TServerClientNNTP.FormatErrorReply( const ErrType: TClientErrorType;
                                             const ErrMsg: String ): String;
var  e: Integer;
begin
   e := 503;

   case NntpMode of

      nmDefault:
         case ErrType of
            cetRefusedDontRetry, cetRefusedTemporary:
               Result := '502 ' + ErrMsg;
            cetLineTooLong, cetTextTooLarge:
               Result := '503 ' + ErrMsg;
            else
               Result := '503 ' + ErrMsg;
         end;

      nmIHAVE:
         e := 437;

      nmTAKETHIS:
         e := 439;
   end;

   Result := IntToStr(e) + ' ' + ErrMsg;
end;

procedure TServerClientNNTP.SendGreeting( var KeepConnection: Boolean;
                                          var Reason: String );

   procedure AutoLogin;
   var  account, s: String;
   begin
      // try to auto-login with account that is associated with connecting IP
      account := IPAccount;

      // if none is set, use 'nntpdefault' (=default of old versions)
      if account = '' then account := 'nntpdefault';

      // auto-login user if account exists
      if CurrentUserID = ACTID_INVALID then begin
         Log( LOGID_DEBUG, 'Try to auto-login account "' + account + '"' );
         if Hamster.Accounts.UserIDOf( account ) <> ACTID_INVALID then begin
            CurrentUserName := account;
            CurrentUserID   := ACTID_INVALID;
            s := LoginUser( '*' ); // only single-star-password will work
            Log( LOGID_DEBUG, 'Auto-Login "' + account + '": ' + s );
            ClientsChange;
         end;
      end;
   end;

begin
   KeepConnection := False;

   if (IPAccess and IPACC_ACCESS_RW)=IPACC_ACCESS_RW then begin

      HWriteLn( '200 Hamster-NNTP, '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo );
      AutoLogin;
      KeepConnection := True;

   end else if (IPAccess and IPACC_ACCESS_RO)=IPACC_ACCESS_RO then begin
   
      HWriteLn( '201 Hamster-NNTP, '
                + GetMyStringFileInfo('ProductName','Hamster') + ' '
                + GetMyVersionInfo );
      AutoLogin;
      KeepConnection := True;

   end else begin

      Reason := 'Permission denied - closing connection.';
      HWriteLn( FormatErrorReply( cetRefusedDontRetry, Reason ) );

   end;
end;

constructor TServerClientNNTP.Create( ACreateSuspended: Boolean );
var  i: Integer;
begin
     inherited Create( True );

     FLimitLineLen  := Hamster.Config.Settings.GetInt(hsLocalNntpLimitLineLen);
     FLimitTextSize := Hamster.Config.Settings.GetInt(hsLocalNntpLimitTextSize);

     i := Hamster.Config.Settings.GetInt( hsLocalNntpInactivitySec );
     if i > 0 then FInactivitySecs := i;

     CurrentUserID   := ACTID_INVALID;
     CurrentUserName := '';
     CurrentGroup    := -1;
     CurrentArtNo    := 0;
     UserMayPost     := False;
     UserMayFeed     := False;
     UserMayNewNews  := False;
     UserMayXHSearch := False;
     UserAutoSubMode := NEWSAUTOSUB_NONE;
     AutoGroupName   := '';
     AutoGroupPends  := False;
     RecommendedMID  := '';
     PermRead        := '';
     PermPost        := '';
     ScoreFileFeed   := nil;
     ScoreFilePost   := nil;
     NntpMode        := nmDefault;

     GroupList := TStringList.Create;
     GroupList.Sorted := True;
     GroupList.Duplicates := dupIgnore;

     if not ACreateSuspended then Resume;
end;

destructor TServerClientNNTP.Destroy;
begin
     SetCurrentGroup( '' );
     if Assigned( GroupList ) then FreeAndNil( GroupList );
     if Assigned( ScoreFileFeed ) then FreeAndNil( ScoreFileFeed );
     if Assigned( ScoreFilePost ) then FreeAndNil( ScoreFilePost );
     inherited;
end;

// ----------------------------------------------------------------------------

end.

