// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cScheduler;

interface

uses SysUtils, Classes, uDateTime, SyncObjs, cSchedulerCommon;

type
   TGlobalScheduler = class( TGlobalSchedulerCustom )

      protected
         FFilename: String;
         FActive  : Boolean;

         procedure SetActive( const NewActive: Boolean ); virtual;

      public
         property Active: Boolean read FActive write SetActive;

         procedure SetDefText( const NewDefText: String ); override;
         function  Check: TSchedulerEntryCustom; override;
         function  Trigger: Boolean; virtual;

         constructor Create( const AFilename: String; AActive: Boolean );

   end;


implementation

uses uConst, uTools, cLogFileHamster, cUserTasks, tUserTasks, tScript, uRasDyn;

// ----------------------------------------------------- TGlobalScheduler -----

procedure TGlobalScheduler.SetActive( const NewActive: Boolean );
var  i: Integer;
     s: String;
begin
   if NewActive then begin
      with FList.LockList do try
         for i := 0 to Count - 1 do begin
            s := TGlobalSchedulerEntry( Items[i] ).ToString;
            Log( LOGID_INFO, 'Scheduler: ' + s ); 
         end;
      finally FList.UnlockList end;
   end;

   FActive := NewActive;
end;

procedure TGlobalScheduler.SetDefText( const NewDefText: String );
var  SL: TStringList;
     OldActive: Boolean;
begin
   with FList.LockList do try

      OldActive := Active;
      Active := False;

      inherited;

      SL := TStringList.Create;
      try

         try
            SL.Text := self.GetDefText;
            SL.SaveToFile( FFilename );
         except
            on E: Exception do begin
               Log( LOGID_ERROR, FFilename + ': ' + E.Message );
            end;
         end;

      finally SL.Free end;

      Active := OldActive;

   finally FList.UnlockList end;
end;

function TGlobalScheduler.Check: TSchedulerEntryCustom;
begin
   if FActive then begin
      Result := inherited Check;
   end else begin
      Result := nil;
   end;
end;

function TGlobalScheduler.Trigger: Boolean;
var  Entry: TGlobalSchedulerEntry;
     s: String;
begin
   Result := False;
   if not FActive then exit;

   Entry := TGlobalSchedulerEntry( Check );
   if Entry = nil then exit;

   s := '  (next run at ' + DateTimeToStr( Entry.NextDT ) + ')';
   
   if Entry.ActionKey = '' then begin
      Log( LOGID_INFO, 'Scheduler starts ... nothing assigned!' + s );
   end else if Entry.IsUserTask then begin
      Log( LOGID_INFO, 'Scheduler starts user task "' + Entry.ActionKey + '"' + s );
      with TUserTaskThread.Create( Entry.ActionKey, True{silent} ) do Resume;
   end else begin
      Log( LOGID_INFO, 'Scheduler starts script "' + Entry.ActionKey + '"' + s );
      StartNewScript( Entry.ActionKey, '', False{don't wait}, True{silent} );
   end;
end;

constructor TGlobalScheduler.Create( const AFilename: String; AActive: Boolean );
var  SL: TStringList;
begin
   inherited Create;

   FFilename := AFilename;
   FActive   := AActive;

   if FileExists( FFilename ) then begin

      SL := TStringList.Create;
      try

         try
            SL.LoadFromFile( FFilename );
            SetDefText( SL.Text );
         except
            on E: Exception do begin
               Log( LOGID_ERROR, FFilename + ': ' + E.Message );
            end;
         end;

      finally SL.Free end;

   end;
end;

end.
