// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cResControl; // ressource controller

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows;

const
   RESID_HamsterGroup = 1;
   RESID_HscListEntry = 2;
   RESID_WinApiHandle = 3;
   RESID_GlobalMemory = 4;
   RESID_DynamicDll   = 5;
   RESID_StrNew       = 6;
   RESID_DelphiObject = 7;

type
   TResAbstract = class
      protected
         ResID    : LongWord;
         EntityInt: LongWord;
         EntityStr: String;
         EntityKey: String;
      public
         function AutoDestroy: String; virtual; abstract;
         constructor Create( aResID: LongWord; aEntityInt: LongWord ); overload;
         constructor Create( aResID: LongWord; aEntityStr: String   ); overload;
         constructor Create( aResID: LongWord; aEntityInt: LongWord;
                                               aEntityStr: String   ); overload;
   end;

   TResHamsterGroup = class( TResAbstract )
      // x := ArticleBase.Open( s );
      // RC.Add( TRes_HamsterGroup.Create(x) );
      // ...
      // RC.Remove( RESID_HamsterGroup, x );
      // ArticleBase.Close( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aGroupHandle: Integer );
   end;

   TResHscListEntry = class( TResAbstract )
      // x := FHscLists.ListAlloc(...)
      // RC.Add( TRes_HscListEntry.Create(Engine.FHscLists,x) );
      // ...
      // RC.Remove( RESID_HscListEntry, x );
      // FHscLists.ListFree( x );
      private
         HscLists: TObject;
      public
         function AutoDestroy: String; override;
         constructor Create( aHscLists: TObject; aHscListNo: Integer );
   end;

   TResWinApiHandle = class( TResAbstract )
      // x := ...
      // RC.Add( TRes_WinApiHandle.Create(x) );
      // ...
      // RC.Remove( RESID_WinApiHandle, x );
      // CloseHandle( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aHandle: LongWord );
   end;

   TResGlobalMemory = class( TResAbstract )
      // x := GlobalAlloc(...)
      // RC.Add( TRes_GlobalMemory.Create(x) );
      // ...
      // RC.Remove( RESID_GlobalMemory, x );
      // GlobalFree( x );
      public
         function AutoDestroy: String; override;
         constructor Create( aMemory: HGLOBAL );
   end;

   TResDynamicDll = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( aDll: HMODULE );
   end;

   TResStrNew = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( pStr: Pointer );
   end;

   TResDelphiObject = class( TResAbstract )
      public
         function AutoDestroy: String; override;
         constructor Create( obj: TObject );
   end;

   TResController = class
      // Engine:
      //    RC := TRessourceController.Create;
      //    ...
      //    s := RC.AutoDestroyAll;
      //    if s<>'' then Log( LOGID_WARN, s );
      //    RC.Free;
      private
         InUseList: TStringList;
      public
         procedure Add   ( ResObj: TResAbstract   );
         procedure Remove( ResID: LongWord; Entity: LongWord ); overload;
         procedure Remove( ResID: LongWord; Entity: String   ); overload;
         function AutoDestroyOne: String;
         function AutoDestroyAll: String;
         constructor Create;
         destructor Destroy; override;
   end;


implementation

uses cArtFiles, cHscEngine, uDynDll, cHamster;

function ResKey( ResID: LongWord; Entity: LongWord ): String; overload;
begin
   Result := inttostr( ResID ) + ':' + inttostr( Entity );
end;

function ResKey( ResID: LongWord; Entity: String ): String; overload;
begin
   Result := inttostr( ResID ) + ':' + LowerCase( Entity );
end;

function ResKey( ResID, EntityInt: LongWord; EntityStr: String ): String; overload;
begin
   Result := inttostr( ResID ) + ':' + LowerCase( EntityStr ) + ':' + inttostr( EntityInt );
end;

{ TResAbstract }

constructor TResAbstract.Create( aResID: LongWord; aEntityInt: LongWord );
begin
   inherited Create;
   ResID     := aResID;
   EntityInt := aEntityInt;
   EntityKey := ResKey( ResID, EntityInt );
end;

constructor TResAbstract.Create( aResID: LongWord; aEntityStr: String );
begin
   inherited Create;
   ResID     := aResID;
   EntityStr := aEntityStr;
   EntityKey := ResKey( ResID, EntityStr );
end;

constructor TResAbstract.Create( aResID, aEntityInt: LongWord; aEntityStr: String );
begin
   inherited Create;
   ResID     := aResID;
   EntityInt := aEntityInt;
   EntityStr := aEntityStr;
   EntityKey := ResKey( ResID, EntityInt, EntityStr );
end;

{ TResHamsterGroup }

function TResHamsterGroup.AutoDestroy: String;
begin
   Result := 'Group ' + inttostr(EntityInt)
           + ' (' + Hamster.ArticleBase.Name[EntityInt] + ')';
   Hamster.ArticleBase.Close( EntityInt );
end;

constructor TResHamsterGroup.Create(aGroupHandle: Integer);
begin
   inherited Create( RESID_HamsterGroup, aGroupHandle );
end;

{ TResHscList }

function TResHscListEntry.AutoDestroy: String;
begin
   Result := 'HSC-list ' + inttostr(EntityInt);
   THscLists( HscLists ).ListFree( EntityInt );
end;

constructor TResHscListEntry.Create(aHscLists: TObject; aHscListNo: Integer);
begin
   inherited Create( RESID_HscListEntry, aHscListNo );
   HscLists := aHscLists;
end;

{ TRes_ApiHandle }

function TResWinApiHandle.AutoDestroy: String;
begin
   Result := 'API-handle ' + inttostr(EntityInt);
   CloseHandle( EntityInt );
end;

constructor TResWinApiHandle.Create(aHandle: LongWord);
begin
   inherited Create( RESID_WinApiHandle, aHandle );
end;

{ TResGlobalMemory }

function TResGlobalMemory.AutoDestroy: String;
begin
   Result := 'Global memory ' + inttostr(EntityInt);
   GlobalFree( EntityInt );
end;

constructor TResGlobalMemory.Create(aMemory: HGLOBAL);
begin
   inherited Create( RESID_GlobalMemory, aMemory );
end;

{ TResDynamicDll }

function TResDynamicDll.AutoDestroy: String;
begin
   Result := 'Dynamic dll ' + inttostr(EntityInt);
   DynDllFree( EntityInt );
end;

constructor TResDynamicDll.Create(aDll: HMODULE);
begin
   inherited Create( RESID_DynamicDll, aDll );
end;

{ TResStrNew }

function TResStrNew.AutoDestroy: String;
begin
   Result := 'Allocated string ' + inttostr(EntityInt);
   StrDispose( PChar( EntityInt ) );
end;

constructor TResStrNew.Create(pStr: Pointer);
begin
   inherited Create( RESID_StrNew, LongWord( pStr ) );
end;

{ TResDelphiObject }

function TResDelphiObject.AutoDestroy: String;
begin
   Result := 'Object ' + TObject(EntityInt).ClassName
           + ' ' + inttohex(EntityInt,8);
   TObject( EntityInt ).Free;
end;

constructor TResDelphiObject.Create(obj: TObject);
begin
   inherited Create( RESID_DelphiObject, LongWord( obj ) );
end;

{ TResController }

constructor TResController.Create;
begin
   inherited Create;
   InUseList := TStringList.Create;
end;

destructor TResController.Destroy;
begin
   if Assigned(InUseList) then begin
      while InUseList.Count>0 do begin
         InUseList.Objects[ 0 ].Free;
         InUseList.Delete( 0 );
      end;
      InUseList.Free;
   end;
   inherited Destroy;
end;

procedure TResController.Remove( ResID: LongWord; Entity: LongWord );
var  Index: Integer;
begin
   Index := InUseList.IndexOf( ResKey( ResID, Entity ) );
   if Index >= 0 then begin
      InUseList.Objects[ Index ].Free;
      InUseList.Delete( Index );
   end;
end;

procedure TResController.Remove( ResID: LongWord; Entity: String );
var  Index: Integer;
begin
   Index := InUseList.IndexOf( ResKey( ResID, Entity ) );
   if Index >= 0 then begin
      InUseList.Objects[ Index ].Free;
      InUseList.Delete( Index );
   end;
end;

procedure TResController.Add( ResObj: TResAbstract );
begin
   InUseList.AddObject( ResObj.EntityKey, ResObj );
end;

function TResController.AutoDestroyOne: String;
var  Index: Integer;
     Key  : String;
begin
   Result := '';

   // identify ressource
   Index := InUseList.Count - 1;
   if Index < 0 then exit;
   Key := TResAbstract( InUseList.Objects[ Index ] ).EntityKey;

   // release ressource
   try
      Result := TResAbstract( InUseList.Objects[ Index ] ).AutoDestroy;
   except
      on E: Exception do Result := Result + 'AutoDestroy-Exception: ' + E.Message;
   end;

   // remove ressource from list if not already done by .AutoDestroy
   Index := InUseList.IndexOf( Key );
   if Index >= 0 then begin
      try
         TResAbstract( InUseList.Objects[ Index ] ).Free;
      except
         on E: Exception do Result := Result + 'AutoDestroy-Exception-2: ' + E.Message;
      end;

      try
         InUseList.Delete( Index );
      except
         on E: Exception do Result := Result + 'AutoDestroy-Exception-3: ' + E.Message;
      end;
   end;
end;

function TResController.AutoDestroyAll: String;
begin
   Result := '';
   while InUseList.Count > 0 do begin
      if Result <> '' then Result := Result + ', ';
      Result := Result + AutoDestroyOne;
   end;
end;

end.
