// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cRasDialer;

interface

uses Classes, uRasDyn, cPasswords;

type
   TRasDialer = class
      private
         FLastError: Integer;
         OurRasConnection: THRasConn;
         OurRasDialParams: TRasDialParams;
         OurRasConnStatus: TRasConnStatus;
         StatTime : TDateTime;
         StatByte : Int64;
         StatArts : Int64;
         StatMails: Int64;
         ConnID   : String;
         sDynamicIP: String;
         lDynamicIP: LongInt;
         FPasswords: TPasswords;
         
         function GetDynamicIP: LongInt;

      public
         property DynamicIP: LongInt read GetDynamicIP;
         property LastError: Integer read FLastError;

         function Dial( ConnectionID, UserName, Password: String ): Integer;
         function HangUp: Integer;
         function IsConnected: Boolean;
         function GetConnectionList( RasList: TStringList ): Boolean;
         function IsInstalled: Boolean;

         constructor Create( APasswords: TPasswords );
         destructor Destroy; override;
   end;


implementation

uses SysUtils, Windows, Winsock, uWinSock, uConst, uConstVar, uVar, uHamTools,
     cLogFileHamster, uDateTime;

// ----------------------------------------------------------- TRasDialer -----

function TRasDialer.Dial( ConnectionID, UserName, Password: String ): Integer;
var  i: Integer;
     u, p: String;
begin
   Result := -1;
   FLastError := -1;
   try
      if not RasDynInit then begin
         Log( LOGID_ERROR, 'Ras.Dial failed: RAS is not installed.' );
         exit;
      end;

      if OurRasConnection<>0 then Hangup;

      ConnID := ConnectionID;
      OurRasConnection := 0;
      OurRasDialParams.dwSize           := sizeof( OurRasDialParams );
      strpcopy( OurRasDialParams.szEntryName, ConnectionID );
      OurRasDialParams.szPhoneNumber    := '';
      OurRasDialParams.szCallbackNumber := '';

      if (UserName='') or (Password='') then begin
         if not FPasswords.UsePassword( ConnectionID, u, p ) then exit;
         if UserName='' then UserName:=u;
         if Password='' then Password:=p;
      end;
      strpcopy( OurRasDialParams.szUserName, UserName );
      strpcopy( OurRasDialParams.szPassword, Password );
      OurRasDialParams.szDomain         := '';

      Log( LOGID_INFO, 'Dialing ...');
      i := RasDynDial( nil, // pointer to function extensions data
                       nil, // pointer to full path and filename of phonebook file
                       OurRasDialParams, // pointer to calling parameters data
                       0, // specifies type of RasDial event handler
                       nil, // specifies a handler for RasDial events
                       OurRasConnection ); // pointer to variable to receive connection handle

      if i=0 then begin
         FLastError := 0;
         OurRasConnStatus.dwSize  := sizeof( OurRasConnStatus );
         OurRasConnStatus.dwError := 0;
         repeat
            i := RasDynGetConnectStatus( OurRasConnection, OurRasConnStatus );
            if i<>0 then begin
               FLastError := i;
               Result := i;
               Log( LOGID_ERROR, Format(
                  'Error.A %s on dialing RAS-connection.', [inttostr(i)] ) );
               RasDynHangup( OurRasConnection );
               OurRasConnection := 0;
               break;
            end;
            Log( LOGID_DEBUG,
               'RasConnState' + '=' + inttostr(OurRasConnStatus.rasconnstate) );
            if OurRasConnStatus.rasconnstate=RASCS_Connected then begin
               Result := 0;
               break;
            end;
            if OurRasConnStatus.dwError<>0 then begin
               FLastError := OurRasConnStatus.dwError;
               Result := OurRasConnStatus.dwError;
               Log( LOGID_ERROR, Format(
                  'Error.A %s on dialing RAS-connection.', [inttostr(OurRasConnStatus.dwError)] ) );
               RasDynHangup( OurRasConnection );
               OurRasConnection := 0;
               break;
            end;
            Sleep( 250 );
         until False;

         // stat-start
         StatTime := Now;
         CounterEnter;
         try
            StatByte := CounterByteIn + CounterByteOut;
            StatArts := CounterArtLoad;
            StatMails:= CounterMailNew;
         finally CounterLeave end;

         sDynamicIP := RasDynGetPppIp( OurRasConnection );
         if sDynamicIP='' then begin
            lDynamicIP := 0;
         end else begin
            lDynamicIP := inet_addr( PChar(sDynamicIP) );
            Log( LOGID_INFO, Format('RAS PPP-IP = %s (%s)',
                             [sDynamicIP, LookupHostName(lDynamicIP) ] ));
         end;

      end else begin
         // #676 bei besetzt
         FLastError := i;
         Result := i;
         Log( LOGID_ERROR, Format(
            'Error %s dialing RAS-connection.', [inttostr(i)]));
         u := RasDynErrTxt( i );
         if u<>'' then
            Log( LOGID_ERROR, 'RAS-Error'
               + ' ' + inttostr(i) + ': ' + u );
         if OurRasConnection<>0 then RasDynHangup( OurRasConnection );
         OurRasConnection := 0;
      end;
   except
      on E:Exception do begin
         if Result=0 then Result:=-2;
         if FLastError=0 then FLastError:=-2;
         Log( LOGID_ERROR, 'RasDynDial-Exception: ' + E.Message );
      end;
   end;
end;

function TRasDialer.GetDynamicIP: LongInt;
begin
   Result := lDynamicIP;
   if Result=0 then exit;
   if OurRasConnection=0 then begin Result:=0; exit; end;
   if RasDynGetConnectStatus( OurRasConnection, OurRasConnStatus )=0 then begin
      if OurRasConnStatus.rasconnstate<>RASCS_Connected then begin
         // not connected any more
         lDynamicIP := 0;
         Result := 0;
      end;
   end;
end;

function TRasDialer.HangUp: Integer;
const DT2SEC : Double = (24.0*60.0*60.0);
var  i, HangupRasConn: Integer;
     s: String;
     LogNow: TDateTime;
begin
   Result     := -1;
   sDynamicIP := '';
   lDynamicIP := 0;
   if not RasDynInit then exit;

   Result     := 0;
   if OurRasConnection<>0 then begin
      try
         // stat-end
         LogNow := Now;
         StatTime := ( LogNow - StatTime ) * DT2SEC;
         if StatTime<=0.01 then StatTime:=0.01;
         CounterEnter;
         try
            StatByte  := CounterByteIn + CounterByteOut - StatByte;
            StatArts  := CounterArtLoad - StatArts;
            StatMails := CounterMailNew - StatMails;
         finally CounterLeave end;
         s := ''
            + #9 + 'Secs' + '=' + FormatFloat('0',StatTime)
            + #9 + 'Mails'+ '=' + inttostr(StatMails)
            + #9 + 'Arts' + '=' + inttostr(StatArts)
            + #9 + 'A/s'  + '=' + FormatFloat('0.0',(StatArts+StatMails)/StatTime)
            + #9 + 'Bytes'+ '=' + inttostr(StatByte)
            + #9 + 'B/s'  + '=' + FormatFloat('0',StatByte/StatTime)
            + #9 + 'RAS'  + '=' + ConnID;
         Log( LOGID_INFO, 'Connection-stats:'
                        + StringReplace( s, #9, '  ', [rfReplaceAll] ) );
         HamFileAppendLine( AppSettings.GetStr(asPathLogs)
                            + LOGFILE_RASDIAL
                            + FormatDateTime( '"-"yyyy"-"mm', LogNow )
                            + LOGFILE_EXTENSION,
                            DateTimetoLogTime( LogNow ) + s );
      except
      end;
   end;

   // hangup
   HangupRasConn    := OurRasConnection;
   if HangupRasConn=0 then HangupRasConn := RasDynGetConn;
   OurRasConnection := 0;

   if HangupRasConn<>0 then begin
      try
         i := RasDynHangup( HangupRasConn );
         if i=0 then begin
            Log( LOGID_INFO, 'RAS-connection closed.' );
         end else begin
            FLastError := i;
            Result := i;
            Log( LOGID_ERROR, 'Error ' + inttostr(i) + ' closing RAS-connection.' );
         end;
      except
         on E:Exception do begin
            if Result=0 then Result:=-2;
            if FLastError=0 then FLastError:=-2;
            Log( LOGID_ERROR, 'RasDynHangup-Exception: ' + E.Message );
         end;
      end;
   end else begin
      Log( LOGID_INFO, 'RAS-hangup ignored; no connection' );
   end;
end;

function TRasDialer.IsConnected: Boolean;
begin
   Result := ( OurRasConnection <> 0 );
end;

function TRasDialer.GetConnectionList( RasList: TStringList ): Boolean;
begin
   Result := RasDynEnumPhonebookEntries( RasList );
end;

function TRasDialer.IsInstalled: Boolean;
begin
   RasDynInit;
   Result := ( RasDynDll <> 0 );
end;

constructor TRasDialer.Create( APasswords: TPasswords );
begin
   inherited Create;

   FPasswords := APasswords;
   OurRasConnection := 0;
   FLastError := 0;
   sDynamicIP := '';
   lDynamicIP := 0;
end;

destructor TRasDialer.Destroy;
begin
   // if OurRasConnection<>0 then // Hangup?
   inherited Destroy;
end;

end.
