// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cPasswords;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, SyncObjs;

type
  TPasswords = class
    private
      FLock   : TCriticalSection;
      FileList: TStringList;
      TempList: TStringList;

    public
      procedure SavePassword( const ToFile: Boolean;
                              const Identifier: String;
                              const Username, Password: String );
      function  LoadPassword( const Identifier: String;
                              out Username, Password: String ): Boolean;
      function  UsePassword ( const Identifier: String;
                              out Username, Password: String ): Boolean;

      function  PasswordInfo( const Identifier : String;
                              out   Username   : String;
                              out   HasUserPass: Boolean ): Boolean;

      constructor Create;
      destructor Destroy; override;
  end;

  
implementation

uses Windows, uConst, uConstVar, uVar, uTools, uCRC32, uBlowfish,
     cLogFileHamster, cHamster;

const
   PWCB_PC_OS_USER = 0;
   PWCB_PC_OS      = 1;
   PWCB_NONE       = 2;

function GetKeyword( const CodeBase: Integer ): String; overload;
var  p: array[0..256] of char;
     dw: DWord;
     s: String;
     V: TOSVersionInfoA;
begin
     s := '';

     if CodeBase in [PWCB_PC_OS_USER, PWCB_PC_OS] then begin
        dw := 256;
        if GetComputerName( p, dw ) then s := s + String( p );
     end else begin
        s := s + 'NoName';
     end;

     if CodeBase in [PWCB_PC_OS_USER, PWCB_PC_OS] then begin
        V.dwOSVersionInfoSize := sizeof(V);
        if GetVersionEx( V ) then begin
           s := s + inttohex(V.dwBuildNumber,1) + inttohex(V.dwMinorVersion,1)
                  + inttohex(V.dwPlatformId,1)  + inttohex(V.dwMajorVersion,1);
        end;
     end else begin
        s := s + 'NoVer';
     end;

     if CodeBase in [PWCB_PC_OS_USER] then begin
        dw := 256;
        if GetUserName( p, dw ) then s := s + String( p );
     end else begin
        s := s + 'NoName';
     end;

     s := s + 'Hamster';
     Result := s;

end;

function GetKeyword: String; overload;
begin
   Result := GetKeyword( Hamster.Config.Settings.GetInt(hsPasswordCodeBase) );
end;

function XRadix( l : LongInt ): String;
const ALPHABET = 'AbCd6efG5hij4kLmn3opqR2sTuv1Wxyz';
var  p : LongInt;
begin
     Result := '';
     repeat
        p := l and $1F;
        l := l shr 5;
        Result := Result + copy(ALPHABET,p+1,1);
     until l=0;
     while length(Result)<7 do Result:='i'+Result;
end;

procedure PWEncode( var buf: String );
var  i: Integer;
     s: String;
begin
     s := HamBlowfishEncipher( GetKeyword, buf );
     buf := '';
     for i:=1 to length(s) do begin
        buf := buf + inttohex( ord(s[i]), 2 );
     end;
end;

procedure PWDecode( var buf: String );
var  i: Integer;
     s: String;
begin
     s := '';
     for i:=1 to length(buf) div 2 do begin
        s := s + chr( strtoint( '$' + copy(buf,i*2-1,2) ) );
     end;
     buf := HamBlowfishDecipher( GetKeyword, s );
end;

function TPasswords.UsePassword( const Identifier: String;
                                 out Username, Password: String ): Boolean;
var  u, p: String;
     Save: Boolean;
begin
     Result   := False;
     Username := '';
     Password := '';

     u := '';
     p := '';
     if LoadPassword( Identifier, u, p ) then begin
        Save := False;
        if (u='?') then begin
           Log( LOGID_ERROR, 'Unsupported user "?" at ' + Identifier );
           // if not InputDlgStr( PChar(Identifier), 'Username:', u, 0 ) then exit;
           // Save := True;
           Result := False;
           exit;
        end;
        if (p='?') then begin
           Log( LOGID_ERROR, 'Unsupported password "?" at ' + Identifier );
           // if not InputDlgPwd( PChar(Identifier), 'Password:', p, 0 ) then exit;
           // Save := True;
           Result := False;
           exit;
        end;
        if Save then SavePassword( False, Identifier, u, p ); // save for current session

        Result   := True;
        Username := u;
        Password := p;
     end;
end;

function TPasswords.PasswordInfo( const Identifier : String;
                                  out   Username   : String;
                                  out   HasUserPass: Boolean ): Boolean;
var  p: String;
begin
   Result := LoadPassword( Identifier, Username, p );
   HasUserPass := Result and (Username<>'') and (p<>'');
end;

function TPasswords.LoadPassword( const Identifier: String;
                                  out Username, Password: String ): Boolean;
var  AccountKey: String;
     AccountRec: String;
     j         : Integer;
begin
     Log( LOGID_DEBUG, 'Loading username/password for ' + Identifier );

     Result   := False;
     Username := '';
     Password := '';
     AccountKey := XRadix( StrToCRC32( Identifier ) );

     FLock.Enter;
     try
        AccountRec := TempList.Values[ AccountKey ];
        if AccountRec='' then begin
           // Log( LOGID_DEBUG, '"' + Identifier + '" not in TempList, trying FileList.' );
           AccountRec := FileList.Values[ AccountKey ];
        end;
     finally FLock.Leave end;

     if AccountRec='' then begin
        // Log( LOGID_DEBUG, '"' + Identifier + '" not in FileList, assume "not needed".' );
        Result := True; // 1.3.17.1
        exit;
     end;

     // decode and split record
     PWDecode( AccountRec );
     if AccountRec <> '' then begin
        j := Pos( #9, AccountRec );
        if j > 0 then begin
           System.Delete( AccountRec, 1, j );
           j := Pos( #9, AccountRec );
           if j > 0 then begin
              Username := copy( AccountRec, 1, j-1 );
              Password := copy( AccountRec, j+1, 255 );
              Result := True;
           end;
        end;
     end;
     if not Result then begin
        Log( LOGID_WARN,
             'Password record "' + Identifier + '" could not be decoded! '
             + 'See "Passwords, Encryption" in helpfile for details!' );
     end;
end;

procedure TPasswords.SavePassword( const ToFile: Boolean;
                                   const Identifier: String;
                                   const Username, Password: String );
var  AccountKey: String;
     AccountRec: String;
begin
     AccountKey := XRadix( StrToCRC32( Identifier ) );
     AccountRec := Identifier + #9 + Username + #9 + Password;
     PWEncode( AccountRec );

     FLock.Enter;
     try
        if ToFile then begin
           FileList.Values[ AccountKey ] := AccountRec;
           FileList.SaveToFile( AppSettings.GetStr(asPathBase) + CFGFILE_PASS );
        end else begin
           TempList.Values[ AccountKey ] := AccountRec;
        end;
     finally FLock.Leave end;
end;

constructor TPasswords.Create;
begin
     inherited Create;
     FLock := TCriticalSection.Create;
     FileList := TStringList.Create;
     TempList := TStringList.Create;
     if FileExists( AppSettings.GetStr(asPathBase) + CFGFILE_PASS ) then begin
        FileList.LoadFromFile( AppSettings.GetStr(asPathBase) + CFGFILE_PASS );
     end;
end;

destructor TPasswords.Destroy;
begin
     FileList.Free;
     TempList.Free;
     FLock.Free;
     inherited Destroy;
end;

end.
