// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cNewsSearcher;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, uTools, cPCRE, cFiltersBase, cFiltersNews,
     cArticle, cArtFiles;

type
   TNewsSearcherPattern = class( TFilterPatternBase )
      IsSameField: Boolean;
      XFiltField : String;
   end;

   TNewsSearcherPatLine = class( TFilterLineBase )
      public
         XFiltField: String;
         function SetFilterLine( FilterLine: String ): Boolean; override;
         function AsString: String; override;

         function MatchesArticle( const RE: TPCRE;
                                  const Article: TMess ): Boolean;
   end;

   TNewsSearcherReportFunc = function ( const Groupname: String;
                                        const ArticleNo: LongWord;
                                        const Article  : TMess )
                                        : Boolean of object;

   TNewsSearcher = class( TFiltersBase )
      private
         FGroups : TStringList; // of ActiveName
         FCntTested, FCntFound: Integer;

      public
         property SelectedGroups: TStringList read FGroups;
         property CountTested: Integer read FCntTested;
         property CountFound : Integer read FCntFound;

         function  IsFilterLine( const Line: String ): Boolean; override;
         function  LinesAdd( const LineText: String ): Integer; override;
         procedure SelectSections( const SectionIdentifier: String ); override;

         procedure SelectGroups( const NewGroups: String;
                                 const ValidGroups: TStringList );
         procedure SelectParamList( const ParamList  : TStringList;
                                    const ValidGroups: TStringList );

         procedure Search( ReportTo: TNewsSearcherReportFunc );

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses uConst, cHamster;

{ TNewsSearcherPatLine }

function TNewsSearcherPatLine.AsString: String;
begin
   raise Exception.Create( 'Unsupported: TNewsSearcherPatLine.AsString' );
end;

function TNewsSearcherPatLine.MatchesArticle( const RE: TPCRE;
                                              const Article: TMess): Boolean;
const ONE=0; YES=1; NO=2;
var  Pat: TNewsSearcherPattern;
     TestStr, DefStr: String;
     NeedOneOf, HaveOneOf: Boolean;
     PatNo: Integer;
     Matches : Boolean;
begin
     Result := True;

     DefStr := GetXFiltVal( Article, 0, XFiltField, DoMimeDecode );

     NeedOneOf := False;
     HaveOneOf := False;
     Matches   := False;

     for PatNo:=0 to PatternCount-1 do begin
        Pat := TNewsSearcherPattern( PatternItem[ PatNo ] );

        if Pat.IsSameField then begin
           TestStr := DefStr;
        end else begin
           TestStr := GetXFiltVal( Article, 0, Pat.XFiltField, DoMimeDecode );
        end;

        if (Pat.SelectType<>' ') or not(HaveOneOf) then begin
           if Pat.IsRegex then begin
              try
                 RE.OptCompile := PCRE_CASELESS;
                 Matches := RE.Match( PChar(Pat.Pattern), PChar(TestStr) );
              except
                 on E: Exception do begin
                    Matches := False;
                 end;
              end;
           end else begin
              Matches := MatchSimple( TestStr, Pat.Pattern );
           end;

           case Pat.SelectType of
              '+': if not Matches then begin Result:=False; break; end;
              '-': if Matches     then begin Result:=False; break; end;
              ' ': begin
                      NeedOneOf := True;
                      if Matches then HaveOneOf:=True;
                   end;
           end;
        end;
     end;

     if NeedOneOf and not HaveOneOf then Result:=False;
end;

function TNewsSearcherPatLine.SetFilterLine( FilterLine: String ): Boolean;
var  i, k: Integer;
     s: String;
     SelectType: Char;
     IsRegex: Boolean;
     TempIsSameField: Boolean;
     TempXFiltField: String;
     Pattern: String;
     cEnd: Char;
     pat: TNewsSearcherPattern;
begin
   Result := False;
   LastSetError := 'invalid line';

   DoMimeDecode := False;
   XFiltField   := '';

   FilterLine := TrimWhSpace( FilterLine );
   if FilterLine='' then exit;

   // ['~'] defaultfield WHSP pattern [WHSP pattern ...]

   // Default-Field with optional MIME-Decode
   i := PosWhSpace( FilterLine );
   if i<2 then begin LastSetError:='missing: [~]default-field'; exit; end;
   s := copy( FilterLine, 1, i-1 );
   if (i=2) and (s[1]='~') then begin
      LastSetError:='missing default-field or invalid space between ~ and field';
      exit
   end;
   System.Delete( FilterLine, 1, i );
   FilterLine := TrimWhSpace( FilterLine );
   if Copy(s,1,1)='~' then begin
      DoMimeDecode := True;
      System.Delete( s, 1, 1 );
   end;
   s := LowerCase( s );
   if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 );
   XFiltField := s;

   // One or more patterns of the following forms:
   //    ['+'|'-'] ['@' fieldname ':'] '{' regex-pattern '}'
   //    ['+'|'-'] ['@' fieldname ':'] '"' simple-pattern '"'
   //    ['+'|'-'] ['@' fieldname ':'] simple-pattern without WHSP

   while FilterLine<>'' do begin
      SelectType := ' ';
      if FilterLine[1] in ['+','-'] then begin
         SelectType := FilterLine[1];
         System.Delete( FilterLine, 1, 1 );
      end;

      TempIsSameField := True;
      TempXFiltField  := XFiltField;

      if FilterLine[1]='@' then begin
         TempIsSameField := False;
         i := Pos( ':', FilterLine );
         if i<3 then begin LastSetError:='missing: "fieldname:" on "@"'; exit; end;
         s := copy( FilterLine, 2, i-2 );
         System.Delete( FilterLine, 1, i );
         s := LowerCase( s );
         if (s>'') and (s[length(s)]=':') then System.Delete( s, length(s), 1 );
         TempXFiltField := s;
      end;

      if FilterLine='' then begin LastSetError:='missing: pattern'; exit; end;
      Pattern := '';

      if FilterLine[1]='{' then begin
         IsRegex := True;
         System.Delete( FilterLine, 1, 1 );
         k := 1;
         while FilterLine<>'' do begin
            if FilterLine[1]='{' then inc(k);
            if FilterLine[1]='}' then begin
               dec(k);
               if k=0 then break;
            end;
            Pattern := Pattern + FilterLine[1];
            System.Delete( FilterLine, 1, 1 );
         end;
         if copy(FilterLine,1,1)='}' then System.Delete( FilterLine, 1, 1 );
      end else begin
         IsRegex := False;
         if FilterLine[1]='"' then begin
            cEnd := '"';
            System.Delete( FilterLine, 1, 1 );
            i := Pos( cEnd, FilterLine );
         end else begin
            cEnd := #32;
            if (FilterLine[1] = '%') then
               while ((FilterLine[3] = #32) or (FilterLine[3] = #9)) do
                  System.Delete( FilterLine, 3, 1 );
            i := PosWhSpace( FilterLine ); {WJ}
         end;
         // falls EOL:
         if (i<=0) then i := length(FilterLine) + 1;
         Pattern := copy( FilterLine, 1, i-1 );
         System.Delete( FilterLine, 1, i-1 );
         if copy(FilterLine,1,1)=cEnd then System.Delete( FilterLine, 1, 1 );
      end;

      if Pattern='' then begin LastSetError:='missing: pattern/-delimiter'; exit; end;

      pat := TNewsSearcherPattern.Create;
      pat.SelectType   := SelectType;
      pat.IsRegex      := IsRegex;
      pat.Pattern      := Pattern;
      pat.IsSameField  := TempIsSameField;
      pat.XFiltField   := TempXFiltField;
      PatternAdd( pat );

      FilterLine := TrimWhSpace( FilterLine );
      if FilterLine<>'' then begin
         if FilterLine[1] in ['#',';'] then FilterLine:=''; // rest of line is [valid] comment
      end;
   end;

   LastSetError := '';
   Result := True;
end;

{ TNewsSearcher }

constructor TNewsSearcher.Create;
begin
   inherited Create( '' );

   FGroups := TStringList.Create;
   FGroups.Sorted := True;
   FGroups.Duplicates := dupIgnore;
end;

destructor TNewsSearcher.Destroy;
begin
   if Assigned(FGroups) then FGroups.Free;
   inherited Destroy;
end;

procedure TNewsSearcher.SelectSections( const SectionIdentifier: String );
begin
   raise Exception.Create( 'Unsupported: TNewsSearcher.SelectSections' );
end;

procedure TNewsSearcher.SelectGroups( const NewGroups: String;
                                      const ValidGroups: TStringList );
var  SL: TStringList;
     ip, ig, j: Integer;
     sp, sg: String;
     del: Boolean;
begin
   // NewGroups := ["-"] GroupnameRegex *( WSP ["-"] GroupnameRegex )

   SL := TStringList.Create;

   try
      ArgsWhSpace( NewGroups, SL );

      for ip:=0 to SL.Count-1 do begin
         sp := SL[ip];
         if copy( sp, 1, 1 ) = '-' then begin
            del := True;
            System.Delete( sp, 1, 1 );
         end else begin
            del := False;
         end;

         RegExSection.Compile( PChar( sp ) );
         
         Hamster.Config.BeginRead;
         try
            for ig:=0 to Hamster.Config.Newsgroups.Count-1 do begin
               sg := Hamster.Config.Newsgroups.Name[ig];
               if RegExSection.Exec( PChar(sg), 0 ) then begin
                  if del then begin
                     j := FGroups.IndexOf( sg );
                     if j >= 0 then FGroups.Delete( j );
                  end else begin
                     if Assigned( ValidGroups ) then begin
                        if ValidGroups.IndexOf(sg) >= 0 then FGroups.Add( sg );
                     end else begin
                        FGroups.Add( sg );
                     end;
                  end;
               end;
            end;
         finally
            Hamster.Config.EndRead;
         end;
      end;

   finally
      SL.Free;
   end;
end;

procedure TNewsSearcher.SelectParamList( const ParamList  : TStringList;
                                         const ValidGroups: TStringList );
var  i, k: Integer;
     s: String;
begin
   for k:=0 to ParamList.Count-1 do begin

      i := PosWhSpace( ParamList[k] );
      if i = 0 then s := ''
               else s := lowercase( copy( ParamList[k], 1, i-1 ) );

      if s = 'grp' then begin
         SelectGroups( copy( ParamList[k], i+1, MaxInt ), ValidGroups );

      end else if s = 'pat' then begin
         LinesAdd( copy( ParamList[k], i+1, MaxInt ) );

      end else begin
         raise Exception.Create( 'NewsSearcher: Invalid param line '
                               + inttostr(k) + ': ' + ParamList[k] );
      end;

   end;
end;

function TNewsSearcher.IsFilterLine( const Line: String ): Boolean;
begin
   Result := ( Line > '' ) and ( Line[1] in [ '~', 'A'..'Z', 'a'..'z' ] )
end;

function TNewsSearcher.LinesAdd( const LineText: String ): Integer;
var  lin: TNewsSearcherPatLine;
begin
     lin := TNewsSearcherPatLine.Create;
     if lin.SetFilterLine( LineText ) then begin
        Result := fFilterLines.Add( lin );
     end else begin
        lin.Free;
        Result := -1;
     end;
end;

procedure TNewsSearcher.Search( ReportTo: TNewsSearcherReportFunc );
var  GrpIdx, GrpHdl, ArtMin, ArtMax, ArtNo, LineNo: Integer;
     GrpNam, ArtTxt: String;
     ArtObj: TMess;
     Line  : TNewsSearcherPatLine;
     ok: Boolean;
begin
   FCntTested := 0;
   FCntFound  := 0;

   ArtObj := TMess.Create;
   try

      for GrpIdx:=0 to FGroups.Count-1 do begin
         GrpNam := FGroups[ GrpIdx ];

         GrpHdl := Hamster.ArticleBase.Open( GrpNam );
         if GrpHdl >= 0 then try

            ArtMin := Hamster.ArticleBase.GetInt( GrpHdl, gsLocalMin );
            ArtMax := Hamster.ArticleBase.GetInt( GrpHdl, gsLocalMax );

            for ArtNo:=ArtMin to ArtMax do begin
               ArtTxt := Hamster.ArticleBase.ReadArticle( GrpHdl, ArtNo );
               if length( ArtTxt ) > 0 then begin
                  inc( FCntTested );
                  ArtObj.FullText := ArtTxt;

                  ok := True;
                  for LineNo:=0 to LinesCount-1 do begin
                     Line := TNewsSearcherPatLine( LinesItem[ LineNo ] );
                     ok := ok and Line.MatchesArticle( RegexFilter, ArtObj );
                     if not ok then break;
                  end;

                  if ok then begin
                     inc( FCntFound );
                     if not ReportTo( GrpNam, ArtNo, ArtObj ) then exit;
                  end;
               end;
            end;

         finally
            Hamster.ArticleBase.Close( GrpHdl );
         end;
      end;

   finally
      ArtObj.Free;
   end;
end;

end.
