// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cNewsJobs; // Job-based handling of news-transfers

// ----------------------------------------------------------------------------
// Container which handles and controls all news transfers based on a list of
// jobs.
// ----------------------------------------------------------------------------

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, uTools;
                           
const
   JOBTYPE_INVALID  = 0;
   JOBTYPE_SRVINFOS = 1;   JOBPRIO_SRVINFOS = MaxInt - 2;
   JOBTYPE_NEWSPOST = 2;   JOBPRIO_NEWSPOST = MaxInt - 3;
   JOBTYPE_GETBYMID = 3;   JOBPRIO_GETBYMID = MaxInt - 4;
   JOBTYPE_NEWSPULL = 4;   // Prio = No. of articles in local group
   JOBTYPE_NEWSFEED = 5;   JOBPRIO_NEWSFEED = MaxInt - 1; // highest priority!

type
   TNewsJob = class
      private
         FJobServer  : String;
         FJobType    : Integer;
         FJobPar     : String;
         FJobPriority: Integer;

      public
         property JobServer  : String  read FJobServer;
         property JobType    : Integer read FJobType;
         property JobPar     : String  read FJobPar;
         property JobPriority: Integer read FJobPriority write FJobPriority;

         constructor Create( const AJobServer  : String;
                             const AJobType    : Integer;
                             const AJobPar     : String;
                             const AJobPriority: Integer );
   end;

   TNewsJobList = class
      private
         FLock   : TRTLCriticalSection;
         FJobList: TList;
         FResort : Boolean;

         function  GetCount: Integer;

      public
         property  Count: Integer read GetCount;

         procedure Clear;
         procedure Sort;
         procedure Enter;
         procedure Leave;
         
         function  JobIndexSTP( const AJobServer  : String;
                                const AJobType    : Integer;
                                const AJobPar     : String  ): Integer;
         function  JobIndexST ( const AJobServer  : String;
                                const AJobType    : Integer ): Integer;
         function  JobIndexTP ( const AJobType    : Integer;
                                const AJobPar     : String  ): Integer;
         procedure JobSet     ( const AJobServer  : String;
                                const AJobType    : Integer;
                                const AJobPar     : String;
                                const AJobPriority: Integer );
         function  JobGet     ( const AJobServer  : String;
                                var   AJobType    : Integer;
                                var   AJobPar     : String  ): Boolean;
         procedure JobDelay   ( const AJobServer  : String;
                                const AJobType    : Integer;
                                const AJobPar     : String;
                                JobsToSkip        : Integer );
         function MaxThreadsFor( AJobServer: String ): Integer;

         constructor Create;
         destructor Destroy; override;
   end;

   TNewsJobs = class
      private
         FJobList    : TNewsJobList;
         FGroupPrios : TStringList;
         FPostBlocker: TExpireStrings;

         function  GetGroupPrio( const Groupname: String ): Integer;
         procedure SetGroupPrio( const Groupname: String; NewPriority: Integer );

      public
         property JobList: TNewsJobList read FJobList;
         property GroupPriority[ const Groupname:String ]: Integer read  GetGroupPrio
                                                                   write SetGroupPrio;

         function MaxThreadsFor( const Servername: String ): Integer;

         function Clear: Integer;

         function AddPullSrv( const Servername, reGrpSelect: String ): Integer;
         function AddPostSrv( const Servername, reGrpSelect, reMsgSelect: String ): Integer;
         function AddFeedSrv( const Servername, reGrpSelect: String ): Integer;

         function AddPullDef( const ServerList: String ): Integer;
         function AddPostDef( const ServerList: String ): Integer;

         function StartThreads( const ServerList: String;
                                const UIDList: TList;
                                WithPostOnly: Boolean = False ): Integer;

         constructor Create;
         destructor Destroy; override;
   end;

// ----------------------------------------------------------------------------

implementation

uses uConst, uConstVar, uVar, cPCRE, cArticle, tTransfer, cLogFileHamster,
     cHamster, Math;

function IsServerSelected( ServerList, TestSrv, TestPort: String ): Boolean;
begin
   if ServerList = '' then begin
      Result := True;
   end else begin
      Result := Pos( ';' + LowerCase(TestSrv)    + ';',
                     ';' + LowerCase(ServerList) + ';' ) > 0;
      if not Result then begin
         Result := Pos( ';' + LowerCase(TestSrv+','+TestPort) + ';',
                        ';' + LowerCase(ServerList)           + ';' ) > 0;
      end;
   end;
end;

// ------------------------------------------------------------- TNewsJob -----

constructor TNewsJob.Create( const AJobServer  : String;
                             const AJobType    : Integer;
                             const AJobPar     : String;
                             const AJobPriority: Integer );
begin
   inherited Create;
   FJobServer   := AJobServer;
   FJobType     := AJobType;
   FJobPar      := AJobPar;
   FJobPriority := AJobPriority;
end;


// --------------------------------------------------------- TNewsJobList -----

procedure TNewsJobList.Enter;
begin
   EnterCriticalSection( FLock );
end;

procedure TNewsJobList.Leave;
begin
   LeaveCriticalSection( FLock );
end;

function TNewsJobList.GetCount: Integer;
begin
   Enter;
   try
      Result := FJobList.Count;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.Clear;
begin
   Enter;
   try
      while FJobList.Count>0 do begin
         TNewsJob( FJobList[FJobList.Count-1] ).Free;
         FJobList.Delete( FJobList.Count-1 );
      end;
      FResort := False;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexSTP( const AJobServer: String;
                                   const AJobType  : Integer;
                                   const AJobPar   : String ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if ( CompareText( JobServer, AJobServer )=0 )
               and (JobType=AJobType) and (JobPar=AJobPar) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexST( const AJobServer  : String;
                                  const AJobType    : Integer ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if ( CompareText( JobServer, AJobServer )=0 )
               and (JobType=AJobType) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function TNewsJobList.JobIndexTP( const AJobType: Integer;
                                  const AJobPar : String ): Integer;
var  Index: Integer;
begin
   Result := -1;
   Enter;
   try
      for Index:=0 to FJobList.Count-1 do
         with TNewsJob( FJobList[Index] ) do
            if (JobType=AJobType) and (JobPar=AJobPar) then begin
               Result := Index;
               break;
            end;
   finally
      Leave;
   end;
end;

function SortByPriority( Item1, Item2: Pointer ): Integer;
var  A: TNewsJob absolute Item1;
     B: TNewsJob absolute Item2;
begin
   if      A.JobPriority < B.JobPriority then Result := +1
   else if A.JobPriority > B.JobPriority then Result := -1
   else                                       Result :=  0
end;

procedure TNewsJobList.Sort;
begin
   if FResort then begin
      Enter;
      try
         FJobList.Sort( SortByPriority );
         FResort := False;
      finally
         Leave;
      end;
   end;
end;

function TNewsJobList.JobGet( const AJobServer: String;
                              var   AJobType  : Integer;
                              var   AJobPar   : String ): Boolean;
var  Index: Integer;
begin
   Result   := False;
   AJobType := JOBTYPE_INVALID;

   Enter;
   try
      if FResort then Sort;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               AJobType := JobType;
               AJobPar  := JobPar;
               Result   := True;
               TNewsJob( FJobList[Index] ).Free;
               FJobList.Delete( Index );
               break;
            end;
         end;
      end;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.JobSet( const AJobServer  : String;
                               const AJobType    : Integer;
                               const AJobPar     : String;
                               const AJobPriority: Integer );
var  Index: Integer;
begin
   Enter;
   try
      Index := JobIndexSTP( AJobServer, AJobType, AJobPar );
      if Index>=0 then begin
         TNewsJob( FJobList[Index] ).JobPriority := AJobPriority;
      end else begin
         FJobList.Add(
            TNewsJob.Create( AJobServer, AJobType, AJobPar, AJobPriority )
         );
      end;
      FResort := True;
   finally
      Leave;
   end;
end;

procedure TNewsJobList.JobDelay( const AJobServer: String;
                                 const AJobType  : Integer;
                                 const AJobPar   : String;
                                 JobsToSkip      : Integer );
var  Index, AJobPriority: Integer;
begin
   Enter;
   try
      AJobPriority := -1;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               if AJobType <> JOBTYPE_NEWSFEED then AJobPriority := JobPriority - 1;
               dec( JobsToSkip );
               if JobsToSkip<=0 then break;
            end;
         end;
      end;
      JobSet( AJobServer, AJobType, AJobPar, AJobPriority );
   finally
      Leave;
   end;
end;

function TNewsJobList.MaxThreadsFor( AJobServer: String ): Integer;
var  Index   : Integer;
     HaveJobs: Boolean;
begin
   Result := 0;
   Enter;
   try
      HaveJobs := False;
      for Index:=0 to FJobList.Count-1 do begin
         with TNewsJob( FJobList[Index] ) do begin
            if CompareText( JobServer, AJobServer )=0 then begin
               HaveJobs := True;
               if JobType=JOBTYPE_NEWSPULL then begin
                  inc( Result );
                  if Result>=4 then break;
               end;
            end;
         end;
      end;
      if HaveJobs and (Result=0) then Result:=1;
   finally
      Leave;
   end;
end;

constructor TNewsJobList.Create;
begin
   inherited Create;
   InitializeCriticalSection( FLock );
   FJobList := TList.Create;
   FResort := False;
end;

destructor TNewsJobList.Destroy;
begin
   if Assigned( FJobList ) then begin Clear; FJobList.Free end;
   DeleteCriticalSection( FLock );
   inherited Destroy;
end;


// ------------------------------------------------------------ TNewsJobs -----

function TNewsJobs.GetGroupPrio( const Groupname: String ): Integer;
var  i: Integer;
begin
   FJobList.Enter;
   try
      i := FGroupPrios.IndexOf( GroupName );
      if i<0 then Result := 0
             else Result := Integer( FGroupPrios.Objects[i] );
   finally
      FJobList.Leave;
   end;
end;

procedure TNewsJobs.SetGroupPrio( const Groupname: String; NewPriority: Integer );
var  i: Integer;
begin
   FJobList.Enter;
   try
      i := FGroupPrios.IndexOf( GroupName );
      if i<0 then FGroupPrios.AddObject( Groupname, Pointer(NewPriority) )
             else FGroupPrios.Objects[i] := Pointer(NewPriority);
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.MaxThreadsFor( const Servername: String ): Integer;
begin
   Result := FJobList.MaxThreadsFor( Servername );
end;

function TNewsJobs.Clear: Integer;
begin
   Result := 0;
   FJobList.Enter;
   try
      FJobList.Clear;
   finally
      FJobList.Leave;
   end;
end;

function TNewsJobs.AddPullSrv( const Servername, reGrpSelect: String ): Integer;
var  Groupname: String;
     indexServer, i: Integer;
begin
   Result := 0;
   Hamster.Config.BeginRead;
   FJobList.Enter;
   try
      // skip disabled server
      indexServer := Hamster.Config.NntpServers.IndexOfAlias( Servername, true );
      if indexServer >= 0 then begin
         if Hamster.Config.NntpServers.IsDisabled[ indexServer ] then exit;
      end;

      // get server infos (initial group-list, new groups, ...)
      FJobList.JobSet( Servername, JOBTYPE_SRVINFOS, '', JOBPRIO_SRVINFOS );

      // get list of Message-IDs
      if FileExists( AppSettings.GetStr(asPathServer) + Servername + '\' + SRVFILE_GETMIDLIST ) then begin
         FJobList.JobSet( Servername, JOBTYPE_GETBYMID, '', JOBPRIO_GETBYMID );
      end;

      // pull newsgroups
      for i:=0 to Hamster.Config.NewsPulls.Count-1 do begin
         if CompareText( Hamster.Config.NewsPulls.Server[i], Servername )=0 then begin
            Groupname := Hamster.Config.NewsPulls.Group[i];
            if (reGrpSelect='')
            or RE_Match( PChar(Groupname), PChar(reGrpSelect), PCRE_CASELESS ) then begin
               FJobList.JobSet( Servername, JOBTYPE_NEWSPULL,
                                Groupname, GroupPriority[Groupname] );
               inc( Result );
            end;
         end;
      end;
   finally
      FJobList.Leave;
      Hamster.Config.EndRead;
   end;
end;

function TNewsJobs.AddPullDef( const ServerList: String ): Integer;

var  LfdServer : Integer;
     PullServer: String;
begin
   Result := 0;
   Hamster.Config.BeginRead;
   FJobList.Enter;
   try
      for LfdServer:=0 to Hamster.Config.NntpServers.Count-1 do begin
         if Hamster.Config.NntpServers.IsDisabled[ LfdServer ] then continue;
         PullServer := Hamster.Config.NntpServers.AliasName[LfdServer];
         if IsServerSelected( ServerList, PullServer,
                              Hamster.Config.NntpServers.SrvPort[LfdServer] ) then begin
            // refresh Pull-List for server
            inc( Result, AddPullSrv( PullServer, '' ) );
         end;
      end;

   finally
      FJobList.Leave;
      Hamster.Config.EndRead;
   end;
end;

function TNewsJobs.AddPostSrv( const Servername, reGrpSelect, reMsgSelect: String ): Integer;
var  Art     : TMess;
     Parser  : TParser;
     iFindRes, iGroup, iLine: Integer;
     PostSR  : TSearchRec;
     PostFile, PostServer, PostGroup, sLine: String;
begin
   Result := 0;
   Art    := TMess.Create;
   Parser := TParser.Create;

   CS_LOCK_NEWSOUT.Enter;
   Hamster.Config.BeginRead;
   FJobList.Enter;
   try
      // check all messages waiting to be posted
      iFindRes := SysUtils.FindFirst( AppSettings.GetStr(asPathNewsOut) + '*.msg', faAnyFile, PostSR );

      while iFindRes=0 do begin

         PostFile := AppSettings.GetStr(asPathNewsOut) + PostSR.Name;
         Art.FullText := '';

         // load message, if it's not already assigned to any server
         if FJobList.JobIndexTP( JOBTYPE_NEWSPOST, PostFile ) < 0 then begin
            try Art.LoadFromFile(PostFile) except Art.FullText:='' end;
         end;

         // if message is not assigned yet and could be loaded: check msg&server
         if length(Art.HeaderText) > 0 then begin
            Log( LOGID_INFO, 'Check article ' + PostFile );
            PostServer := '';

            // check newsgroup-selection
            sLine := Art.HeaderValueByNameSL('Newsgroups:');
            if (reGrpSelect='')
            or RE_Match( PChar(sLine), PChar(reGrpSelect), PCRE_CASELESS ) then begin

               // check, if given server is a valid post-server for any group
               Parser.Parse( Art.HeaderValueByNameSL('Newsgroups:'), ',' );
               iGroup := 0;
               repeat
                  PostGroup  := TrimWhSpace( Parser.sPart( iGroup, '' ) );
                  if PostGroup='' then break;
                  if Hamster.Config.NewsPulls.IsPostServerFor( Servername, PostGroup ) then begin
                     PostServer := Servername;
                     break;
                  end;
                  inc( iGroup );
               until PostServer<>'';

            end;

            // check, if any header matches the given regular expression
            if (PostServer <> '') and (reMsgSelect <> '') then begin
               PostServer := '';
               for iLine := 0 to Art.HeaderLineCountSL - 1 do begin
                  sLine := Art.HeaderLineSL[iLine];
                  if RE_Match( PChar(sLine), PChar(reMsgSelect), PCRE_CASELESS ) then begin
                     PostServer := Servername;
                     break;
                  end;
               end;
            end;

            // check, if its an recently assigned POST that may already run
            if PostServer <> '' then begin
               if not FPostBlocker.Add( PostFile ) then begin
                  Log( LOGID_INFO, 'POST postponed because blocking time (60s)'
                                 + ' has not expired yet: ' + PostFile );
                  PostServer := ''; // ignore until blocking time has expired
               end;
            end;

            // create post-job for the message if a postserver was found for it
            if PostServer <> '' then begin
               Log( LOGID_INFO, PostFile + ' accepted; will post to: ' + PostServer );
               FJobList.JobSet( PostServer, JOBTYPE_SRVINFOS, '',       JOBPRIO_SRVINFOS );
               FJobList.JobSet( PostServer, JOBTYPE_NEWSPOST, PostFile, JOBPRIO_NEWSPOST );
               inc( Result );
            end;
         end;

         iFindRes := SysUtils.FindNext( PostSR );

      end;

      SysUtils.FindClose( PostSR );

   finally
      FJobList.Leave;
      Hamster.Config.EndRead;
      CS_LOCK_NEWSOUT.Leave;
      Parser.Free;
      Art.Free;
   end;
end;

function TNewsJobs.AddPostDef( const ServerList: String ): Integer;
var  Art     : TMess;
     Parser  : TParser;
     iFindRes, iGroup, iServer, PostSrvIdx: Integer;
     PostSR  : TSearchRec;
     PostFile, PostServer, PostGroup, TestServer: String;
begin
   Result := 0;
   Art    := TMess.Create;
   Parser := TParser.Create;

   CS_LOCK_NEWSOUT.Enter;
   Hamster.Config.BeginRead;
   FJobList.Enter;
   try
      // check all messages waiting to be posted
      iFindRes := SysUtils.FindFirst( AppSettings.GetStr(asPathNewsOut) + '*.msg', faAnyFile, PostSR );

      while iFindRes=0 do begin

         PostFile := AppSettings.GetStr(asPathNewsOut) + PostSR.Name;
         Art.FullText := '';

         // load message, if it's not already assigned to any server
         if FJobList.JobIndexTP( JOBTYPE_NEWSPOST, PostFile ) < 0 then begin
            try Art.LoadFromFile(PostFile) except Art.FullText:='' end;
         end;

         // if message is not assigned yet and could be loaded: find post-server
         if length(Art.HeaderText) > 0 then begin

            Log( LOGID_INFO, 'Check article ' + PostFile );

            // get postserver-name based on newsgroup-names
            Parser.Parse( Art.HeaderValueByNameSL('Newsgroups:'), ',' );
            iGroup := 0;

            repeat
               PostServer := '';
               PostGroup  := TrimWhSpace( Parser.sPart( iGroup, '' ) );
               if PostGroup = '' then break;

               // use preferred postserver, if configured and selected
               PostServer := Hamster.Config.NewsPulls.GetPostServer( PostGroup );
               PostSrvIdx := Hamster.Config.NntpServers.IndexOfAlias( PostServer );
               if not IsServerSelected( ServerList, PostServer,
                  Hamster.Config.NntpServers.SrvPort[PostSrvIdx] ) then PostServer:='';

               // look for alternative, selected postservers
               if PostServer = '' then begin
                  for iServer := 0 to Hamster.Config.NntpServers.Count - 1 do begin
                     if Hamster.Config.NntpServers.IsDisabled[iServer] then continue;
                     TestServer := Hamster.Config.NntpServers.AliasName[iServer];
                     if Hamster.Config.NewsPulls.IsPostServerFor( TestServer, PostGroup ) then begin
                        // alternative server found, but is it selected?
                        if IsServerSelected( ServerList, TestServer,
                           Hamster.Config.NntpServers.SrvPort[iServer] ) then begin
                           PostServer := TestServer;
                           break;
                        end;
                     end;
                  end;
               end;

               inc( iGroup );
            until PostServer <> '';

            // check, if its an recently assigned POST that may already run
            if PostServer <> '' then begin
               if not FPostBlocker.Add( PostFile ) then begin
                  Log( LOGID_DEBUG, 'POST postponed because blocking time '
                                  + 'has not expired yet: ' + PostFile );
                  PostServer := ''; // ignore until blocking time has expired
               end;
            end;

            // create post-job for the message if a postserver was found for it
            if PostServer <> '' then begin
               Log( LOGID_INFO, PostFile + ' accepted; will post to: ' + PostServer );
               FJobList.JobSet( PostServer, JOBTYPE_SRVINFOS, '',       JOBPRIO_SRVINFOS );
               FJobList.JobSet( PostServer, JOBTYPE_NEWSPOST, PostFile, JOBPRIO_NEWSPOST );
               inc( Result );
            end;
         end;

         iFindRes := SysUtils.FindNext( PostSR );
      end;

      SysUtils.FindClose( PostSR );

   finally
      FJobList.Leave;
      Hamster.Config.EndRead;
      CS_LOCK_NEWSOUT.Leave;
      Parser.Free;
      Art.Free;
   end;
end;

function TNewsJobs.AddFeedSrv( const Servername, reGrpSelect: String ): Integer;
var  Groupname: String;
     indexServer, GroupHandle, ArtNoMax, FeedNo, i: Integer;
     skip: Boolean;
begin
   Result := 0;
   Hamster.Config.BeginRead;
   FJobList.Enter;
   try
      // skip disabled server
      indexServer := Hamster.Config.NntpServers.IndexOfAlias( Servername, true );
      if indexServer >= 0 then begin
         if Hamster.Config.NntpServers.IsDisabled[ indexServer ] then exit;
      end;

      // feed newsgroups
      for i := 0 to Hamster.Config.Newsgroups.Count - 1 do begin

         Groupname := Hamster.Config.Newsgroups.Name[i];
         if (reGrpSelect='')
         or RE_Match( PChar(Groupname), PChar(reGrpSelect), PCRE_CASELESS ) then begin

            skip := False;
            
            GroupHandle := Hamster.ArticleBase.Open( GroupName );
            if GroupHandle >= 0 then try
               ArtNoMax := Hamster.ArticleBase.GetInt( GroupHandle, gsLocalMax );
               FeedNo   := Hamster.ArticleBase.FeederLast[ GroupHandle, Servername ];
               skip     := ( FeedNo >= ArtNoMax ); // already fed
            finally
               Hamster.ArticleBase.Close( GroupHandle );
            end;

            if skip then begin
               Log( LOGID_DETAIL, 'Feeding skipped, no new articles for '
                                + Servername + ' in ' + Groupname );
            end else begin
               FJobList.JobSet( Servername, JOBTYPE_NEWSFEED,
                                Groupname, GroupPriority[Groupname] );
               inc( Result );
            end;

         end;
      end;

   finally
      FJobList.Leave;
      Hamster.Config.EndRead;
   end;
end;

function TNewsJobs.StartThreads( const ServerList: String;
                                 const UIDList: TList;
                                 WithPostOnly: Boolean = False ): Integer;
var  LfdServer, ThreadCnt, i: Integer;
     ServerName: String;
     TS: TStringList;
     OK, NoneLeft: Boolean;
begin
   Result := 0;

   if Assigned( UIDList ) then UIDList.Clear;

   TS := TStringList.Create;
   try
      Hamster.Config.BeginRead;
      try
         // create list of threads to start
         for LfdServer := 0 to Hamster.Config.NntpServers.Count - 1 do begin
            ServerName := Hamster.Config.NntpServers.AliasName[LfdServer];
            if IsServerSelected( ServerList, ServerName,
                                 Hamster.Config.NntpServers.SrvPort[LfdServer] ) then begin
               OK := True;
               if WithPostOnly then begin
                  if Hamster.NewsJobs.JobList.JobIndexST(ServerName,JOBTYPE_NEWSPOST)<0 then OK:=False;
               end;
               if OK then begin
                  ThreadCnt := Hamster.Config.NntpServers.PullThreads( LfdServer, True );
                  if ThreadCnt>0 then TS.AddObject( ServerName, Pointer(ThreadCnt) )
               end;
            end;
         end;

      finally
         Hamster.Config.EndRead;
      end;

      // Start one thread for each server and repeat this until all threads
      // have been started. This minimizes number of concurrent connections
      // to a specific server if task-limit is reached and further threads
      // have to wait for running ones to finish.
      repeat
         NoneLeft := True;
         for i := TS.Count - 1 downto 0 do begin
            ThreadCnt := Integer( TS.Objects[i] );
            if ThreadCnt > 0 then begin
               NoneLeft   := False;
               ServerName := TS[i];
               TS.Objects[i] := Pointer( ThreadCnt - 1 );
               with TThreadNewsJobs.Create( ServerName ) do begin
                  if Assigned( UIDList ) then UIDList.Add( Pointer(UniqueID) );
                  Resume;
               end;
               inc( Result );
               Sleep( 100 );
             end;
         end;
      until NoneLeft;

   finally
      TS.Free;
   end;
end;

constructor TNewsJobs.Create;
begin
   inherited Create;

   FJobList := TNewsJobList.Create;

   FGroupPrios            := TStringList.Create;
   FGroupPrios.Sorted     := True;
   FGroupPrios.Duplicates := dupIgnore;

   // every assigned POST job is blocked to avoid unwanted multiple injection
   FPostBlocker := TExpireStrings.Create( 60 ); // 60 seconds
end;

destructor TNewsJobs.Destroy;
begin
   if Assigned(FPostBlocker) then FPostBlocker.Free;
   if Assigned(FGroupPrios)  then FGroupPrios.Free;
   if Assigned(FJobList)     then FJobList.Free;
   inherited Destroy;
end;

// ----------------------------------------------------------------------------

end.
