// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cMailUsers;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes;

type
   TMailUserTypes = ( muMissing,      // missing address
                      muLocal,        // local user
                      muLocalUnknown, // unknown local user
                      muLocalInvalid, // invalid address 
                      muRemote );     // remote mail address

   TMailUserCustom = class    
      // Base class for sender and recipients, which are either local users
      // with mailboxes or are identified by a (remote) mail address.

      protected
         FMailUserType: TMailUserTypes;
         FOrgEnvAddr  : String;
         FMailAddress : String;
         FUserID      : Integer;

         procedure SetUserID( const NewUserID: Integer );
         function  GetEnvelopeAddr: String;
         procedure SetEnvelopeAddr( const NewEnvelopeAddr: String );

      public
         property MailUserType: TMailUserTypes read FMailUserType;
         property MailAddress : String  read FMailAddress;
         property EnvelopeAddr: String  read GetEnvelopeAddr write SetEnvelopeAddr;
         property OrgEnvAddr  : String  read FOrgEnvAddr;
         property UserID      : Integer read FUserID write SetUserID;

         function IsLocal : Boolean;
         function IsRemote: Boolean;
         function IsEqual( const Other: TMailUserCustom ): Boolean;

         procedure Clear; virtual;

         constructor Create( const AEnvelopeAddr: String  ); overload;
         constructor Create( const AUserID      : Integer ); overload;
   end;

   TMailSender = class( TMailUserCustom )
      protected
         FAuthUserID: Integer;
         FHELOStr   : String;
         FIPAddr    : Cardinal;

      public
         property AuthUserID: Integer read FAuthUserID write FAuthUserID;
         property HELOStr: String  read FHELOStr write FHELOStr;
         property IPAddr: Cardinal read FIPAddr  write FIPAddr;

         procedure Clear; override;
   end;

   TMailDeliveryResults = (
      mdrPending,
      mdrSuccess,
      mdrFailed_NoSender,
      mdrFailed_NoRecipient,
      mdrFailed_NoHandler,
      mdrFailed_LocalUnknown,
      mdrFailed_LocalInvalid,
      mdrFailed_SaveMailboxErr,
      mdrFailed_SaveMailOutErr,
      mdrFailed_Loop,
      mdrFailed_ForwardLoop,
      mdrFailed_Exception
   );

const
   MailDeliveryResultTexts: array[ TMailDeliveryResults ] of String = (
      'Still pending',
      'Successful',
      'Missing sender (MAIL FROM)',
      'Missing recipient (RCPT TO)',
      'No Handler could process mail',
      'Unknown local recipient.',
      'Invalid recipient.',
      'Could not save in user''s mailbox',
      'Could not save in Mail.Out',
      'Mail loop detected',
      'Local forward loop detected - no final recipient found',
      'Exception'
   );

type
   TMailRecipient = class( TMailUserCustom )
      protected
         FDeliveryResult: TMailDeliveryResults;

         procedure SetDeliveryResult( const mdr: TMailDeliveryResults );

      public
         property DeliveryResult: TMailDeliveryResults read  FDeliveryResult
                                                       write SetDeliveryResult;
         function IsPending: Boolean;
         function HasFailed: Boolean;

         procedure Clear; override;
   end;

   TMailRecipients = class
      protected
         FList: TList;

         function GetCount: Integer;
         function GetItem( Index: Integer ): TMailRecipient;

      public
         property Count: Integer read GetCount;
         property Items[ Index: Integer ]: TMailRecipient read GetItem; default;

         function Add( const AEnvelopeAddr: String  ): Integer; overload;
         function Add( const AUserID      : Integer ): Integer; overload;
         procedure Delete( Index: Integer );

         function AnyLocal : Boolean;
         function AnyRemote: Boolean;
         function AnyFailed: Boolean;
         function AllFailed: Boolean;

         function AlreadyContains( const Destination: String ): Boolean;
         
         procedure Clear;

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses uConst, uTools, cHamster, cLogFileHamster;

{ TMailUserCustom }

constructor TMailUserCustom.Create( const AEnvelopeAddr: String );
begin
   inherited Create;
   SetEnvelopeAddr( AEnvelopeAddr );
end;

constructor TMailUserCustom.Create( const AUserID: Integer );
begin
   inherited Create;
   SetUserID( AUserID );
end;

procedure TMailUserCustom.Clear;
begin
   FMailUserType := muLocalInvalid;
   FMailAddress  := '';
   FOrgEnvAddr   := '';
   FUserID       := ACTID_INVALID;
end;

function TMailUserCustom.IsLocal: Boolean;
begin
   Result := ( FMailUserType = muLocal );
end;

function TMailUserCustom.IsRemote: Boolean;
begin
   Result := ( FMailUserType = muRemote );
end;

function TMailUserCustom.GetEnvelopeAddr: String;
var  addr, domain: String;
begin
   if IsRemote or (MailAddress <> '') then begin
      addr := MailAddress;
      if pos( '@', addr ) = 0 then begin
         domain := Hamster.Config.Settings.GetStr(hsFQDNforMID);
         if length(domain) = 0 then domain := 'localhost';
         addr := addr + '@' + domain;
      end;
      Result := '<' + addr + '>';

   end else if IsLocal then begin
      Result := '<' + Hamster.Accounts.Value[ FUserID, apUsername ] + '>';

   end else begin
      Result := '<>';
   end;
end;

procedure TMailUserCustom.SetEnvelopeAddr( const NewEnvelopeAddr: String );
var  MailType: Integer;
begin
   Clear;
   FOrgEnvAddr := NewEnvelopeAddr;

   try
      FMailAddress := ExtractMailAddr( FOrgEnvAddr );

      if ( FMailAddress = '' ) and ( FOrgEnvAddr <> '<>' ) then begin

         FMailUserType := muMissing;

      end else begin

         if Hamster.Accounts.IsLocalMailbox( FMailAddress,
                                             FUserID, MailType ) then begin

            case MailType of
               LOCALMAILTYPE_NORMAL: begin
                  FMailUserType := muLocal;
                  FMailAddress  := Hamster.Accounts.Value[ FUserID, apUsername ];
               end;
               LOCALMAILTYPE_UNKNOWN:
                  FMailUserType := muLocalUnknown;
               else
                  FMailUserType := muLocalInvalid;
            end;

         end else begin

            FMailUserType := muRemote;
            FUserID := ACTID_INVALID;

         end;

      end;
      
   except end;
end;

procedure TMailUserCustom.SetUserID( const NewUserID: Integer );
begin
   Clear;
   FMailUserType := muLocal;
   FUserID       := NewUserID;
   FMailAddress  := Hamster.Accounts.Value[ FUserID, apUsername ];
   FOrgEnvAddr   := FMailAddress;
end;


function TMailUserCustom.IsEqual( const Other: TMailUserCustom ): Boolean;
begin
   Result := False;

   if IsLocal then begin
      if not Other.IsLocal then exit;
      if UserID <> Other.UserID then exit;
   end else if IsRemote then begin
      if not Other.IsRemote then exit;
      if AnsiCompareText( MailAddress, Other.MailAddress ) <> 0 then exit;
   end else begin
      if Other.IsLocal or Other.IsRemote then exit;
      if AnsiCompareText( EnvelopeAddr, Other.EnvelopeAddr ) <> 0 then exit;
   end;

   Result := True;
end;


{ TMailSender }

procedure TMailSender.Clear;
begin
   inherited Clear;
   FAuthUserID := ACTID_INVALID;
   // FHELOStr := '';
   // FIPAddr  := 0;
end;


{ TMailRecipient }

procedure TMailRecipient.Clear;
begin
   inherited Clear;
   FDeliveryResult := mdrPending;
end;

procedure TMailRecipient.SetDeliveryResult( const mdr: TMailDeliveryResults );
var  s: String;
begin
   FDeliveryResult := mdr;
   if not( mdr in [ mdrPending, mdrSuccess ] ) then begin
      s := OrgEnvAddr;
      if s = '' then s := EnvelopeAddr;
      Log( LOGID_DETAIL, 'Mail recipient ' + s + ' failed: '
                       + MailDeliveryResultTexts[mdr] );
   end;
end;

function TMailRecipient.IsPending: Boolean;
begin
   Result := ( DeliveryResult = mdrPending );
end;

function TMailRecipient.HasFailed: Boolean;
begin
   Result := ( DeliveryResult <> mdrSuccess );
end;


{ TMailRecipients }

constructor TMailRecipients.Create;
begin
   inherited Create;
   FList := TList.Create;
end;

destructor TMailRecipients.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

procedure TMailRecipients.Clear;
begin
   while GetCount > 0 do begin
      GetItem( 0 ).Free;
      FList.Delete( 0 );
   end;
end;

function TMailRecipients.GetCount: Integer;
begin
   Result := FList.Count;
end;

function TMailRecipients.GetItem( Index: Integer ): TMailRecipient;
begin
   Result := FList[ Index ];
end;

function TMailRecipients.Add( const AEnvelopeAddr: String ): Integer;
begin
   Result := FList.Add( TMailRecipient.Create( AEnvelopeAddr ) );
end;

function TMailRecipients.Add( const AUserID: Integer ): Integer;
begin
   Result := FList.Add( TMailRecipient.Create( AUserID ) );
end;

procedure TMailRecipients.Delete( Index: Integer );
begin
   FList.Delete( Index );
end;

function TMailRecipients.AnyLocal: Boolean;
var  Index: Integer;
begin
   Result := False;
   for Index := 0 to GetCount - 1 do begin
      if GetItem(Index).IsLocal then begin Result := True; break end;
   end;
end;

function TMailRecipients.AnyRemote: Boolean;
var  Index: Integer;
begin
   Result := False;
   for Index := 0 to GetCount - 1 do begin
      if GetItem(Index).IsRemote then begin Result := True; break end;
   end;
end;

function TMailRecipients.AnyFailed: Boolean;
var  Index: Integer;
begin
   Result := False;
   for Index := 0 to GetCount - 1 do begin
      if GetItem(Index).HasFailed then begin Result := True; break end;
   end;
end;

function TMailRecipients.AllFailed: Boolean;
var  Index: Integer;
begin
   Result := True;
   for Index := 0 to GetCount - 1 do begin
      if not GetItem(Index).HasFailed then begin Result := False; break end;
   end;
end;

function TMailRecipients.AlreadyContains( const Destination: String ): Boolean;
var  DestUser: TMailUserCustom;
     Index: Integer;
begin
   Result := False;

   DestUser := TMailUserCustom.Create( Destination );
   try
      for Index := 0 to GetCount - 1 do begin
         if GetItem(Index).IsEqual( DestUser ) then begin
            Result := True;
            break;
         end;
      end;
   finally DestUser.Free end;
end;

end.
