// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cMailTrap;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, cSyncObjects, cArticle, cTraps;

type
   // manages list of mail traps stored in file "MailTrap.hst"
   TMailTrap = class( TMailTrapCustom )

      protected
         procedure NotifyHit( const HeaderValues: TStringList;
                              const MatchIndex: Integer;
                              const TrapItem: TMailTrapItem ); override;
      
      public
         procedure Reload; override;

   end;

implementation

uses uTools, uConst, uConstVar, uVar, cLogFileHamster, uDateTime, uHamTools,
     cHamster;

// ------------------------------------------------------------ TMailTrap -----

procedure TMailTrap.NotifyHit( const HeaderValues: TStringList;
                               const MatchIndex: Integer;
                               const TrapItem: TMailTrapItem );
// Called whenever a trap has matched. "TrapItem" is the trap that matched,
// "HeaderValues[MatchIndex]" is the matching line of the message.
var  LogFile, s: String;
     LogNow: TDateTime;
begin
   (*
   h := HeaderValues[ MatchIndex ];
   h := StringReplace( h,  #9, ' ', [rfReplaceAll] );
   h := StringReplace( h, #10, ' ', [rfReplaceAll] );
   h := StringReplace( h, #13, ' ', [rfReplaceAll] );
   *)

   LogNow := Now;

   s := DateTimeToLogTime( LogNow )
      + #9 + inttostr( ord( TrapItem.Action ) )
      + #9 + iif( TrapItem.ScoreValue>=0, '+' ) + inttostr(TrapItem.ScoreValue)
      + #9 + TrapItem.AsInfo;

   LogFile := AppSettings.GetStr( asPathLogs )
            + LOGFILE_MAILTRAPHITS
            + FormatDateTime( '"-"yyyy"-"mm', LogNow )
            + LOGFILE_EXTENSION;
   HamFileAppendLine( LogFile, s );
end;

procedure TMailTrap.Reload;
// Reload mail traps stored in file "MailTrap.hst".
var  SL: TStringList;
     i: Integer;
     Filename: String;
     Item: TMailTrapItem;
begin
   SL := TStringList.Create;
   
   FLock.BeginWrite;
   try
      try
         Clear;

         // refresh score limits
         FAcceptScore := Hamster.Config.Settings.GetInt( hsMailTrapAcceptScore );
         FDeleteScore := Hamster.Config.Settings.GetInt( hsMailTrapDeleteScore );

         // reload definitions in file "MailTrap.hst"
         Filename := AppSettings.GetStr(asPathBase) + CFGFILE_MAILTRAP;
         if not FileExists( Filename ) then exit;
         
         SL.LoadFromFile( Filename );

         for i := 0 to SL.Count - 1 do begin

            if Trim( SL[i] ) <> '' then begin

               Item := TMailTrapItem.Create( SL[i] );

               if Item.CheckAndFixTrap then begin
                  // only add valid and enabled traps
                  if not Item.Disabled then FList.Add( Item );
               end else begin
                  // Should not happen as HControl does checks before saving.
                  Log( LOGID_ERROR, 'Invalid Mail Trap: ' + Item.AsInfo );
                  try Clear except end; // disable all traps to accept all
               end;

            end;

         end;

      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'Error loading ' + CFGFILE_MAILTRAP + ': ' + E.Message );
            try Clear except end; // disable all traps to accept all
         end;
      end;

   finally
      FLock.EndWrite;
      SL.Free;
   end;
end;


end.
