// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cMailLists;
                                                          
interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, SyncObjs, IniFiles, cMailItem;
                                  
const
   FILEEXT_MAILLIST = '.hml'; // file extension of "Hamster Mail Lists"

type
   TMailList = class
      private
         FPath   : String;
         FName   : String;
         FInit   : Boolean;
         FChanged: Boolean;
         FProps  : TStringList;
         FMembers: TStringList;

         procedure DoInitialize;
         procedure DoLoadFromFile;
         procedure DoSaveToFile;

         function  GetDescription : String;
         function  GetOwnerAccount: String;
         procedure SetOwnerAccount( const NewAccount: String );
         function  GetRestrictFrom: Integer;
         function  GetMembers     : TStringList;

      public
         property Name        : String      read  FName;
         property Description : String      read  GetDescription;
         property OwnerAccount: String      read  GetOwnerAccount
                                            write SetOwnerAccount;
         property RestrictFrom: Integer     read  GetRestrictFrom;
         property Members     : TStringList read  GetMembers;

         function  IsMember( const Name: String ): Boolean;
         procedure DelMember( const Name: String );

         procedure ListSetData( const ListData: TStringList );
         procedure ListGetData( const ListData: TStringList );

         constructor Create( const APath, AListname: String );
         destructor Destroy; override;
   end;

   TMailLists = class
      protected
         FLock: TCriticalSection;
         FList: TStringList;
         FPath: String;

         procedure DoClear;
         procedure DoDel( const Index: Integer );
         function  DoAdd( const ListName: String ): Integer;
         function  DoIndexOf( const ListName: String ): Integer;

      public
         procedure Refresh;

         function  MailListOwner( const ListName: String ): String;
         function  MailListMembers( const ListName: String;
                                    const ListAddr: TStrings ): Boolean;
         function  MailListExists( const ListName: String ): Boolean;

         procedure MailListDir( const ListNames: TStrings );
         function  MailListGet( const ListName: String;
                                const ListData: TStringList ): Boolean;
         function  MailListSet( const ListName: String;
                                const ListData: TStringList ): Boolean;
         function  MailListDel( const ListName: String ): Boolean;
         function  MailListReport( const ListName: String ): String;

         function  MailListAccept( const ListName: String;
                                   const MailItem: TMailItem ): Boolean;

         procedure RemoveAccountFromAllLists( const AccountName: String );
         function  NotifyOwner( const ListName, Owner, Subject, Body: String ): Boolean;

         constructor Create( const APath: String );
         destructor Destroy; override;
   end;

implementation

uses uTools, uConst, cLogFileHamster, uHamTools, cHamster, cMailDispatcher;


// ------------------------------------------------------------ TMailList -----

constructor TMailList.Create( const APath, AListname: String );
begin
   inherited Create;

   FPath    := APath;
   FName    := AListname;
   FInit    := False;
   FChanged := False;
   FProps   := TStringList.Create;
   FMembers := TStringList.Create;
end;

destructor TMailList.Destroy;
begin
   if FChanged then DoSaveToFile;
   if Assigned( FMembers ) then FMembers.Free;
   if Assigned( FProps   ) then FProps.Free;
   
   inherited Destroy;
end;

procedure TMailList.DoInitialize;
begin
   if not FInit then DoLoadFromFile;
end;

procedure TMailList.DoLoadFromFile;
var  SL: TStringList;
     FN: String;
begin
   FMembers.Clear;
   FProps.Clear;

   FN := FPath + FName + FILEEXT_MAILLIST; 
   if not FileExists( FN ) then exit;

   SL := TStringList.Create;
   try
      try
         SL.LoadFromFile( FN );
         ListSetData( SL );
         FChanged := False;
      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error reading distribution list file %s: %s',
                                     [ FN, E.Message ] ) );
      end;

   finally SL.Free end;
end;

procedure TMailList.DoSaveToFile;
var  SL: TStringList;
     FN: String;
begin
   DoInitialize;

   // make sure that we have description and owner
   GetDescription;
   GetOwnerAccount;

   SL := TStringList.Create;
   try
   
      ListGetData( SL );
      FN := FPath + FName + FILEEXT_MAILLIST;

      try
         SL.SaveToFile( FN );
         FChanged := False;
      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error writing distribution list file %s: %s',
                                     [ FN, E.Message ] ) );
      end;
      
   finally SL.Free end;
end;

procedure TMailList.ListSetData( const ListData: TStringList );
var  i : Integer;
     s : String;
begin
   FProps.Clear;
   FMembers.Clear;

   for i := 0 to ListData.Count - 1 do begin
      s := TrimWhSpace( ListData[i] );
      if copy( s, 1, 1 ) = '!' then begin
         FProps.Add( s );
      end else begin
         if length(s) > 0 then FMembers.Add( s );
      end;
   end;

   FInit    := True;
   FChanged := True;
end;

procedure TMailList.ListGetData( const ListData: TStringList );
var  i : Integer;
begin
   DoInitialize;
   ListData.Clear;
   for i := 0 to FProps.Count - 1 do ListData.Add( FProps[i]  );
   ListData.Add( '' );
   for i := 0 to FMembers.Count - 1 do ListData.Add( FMembers[i] );
end;

function TMailList.GetDescription: String;
begin
   DoInitialize;
   Result := FProps.Values[ mlpDescription ]; 
   if Result = '' then begin
      Result := 'List ' + FName;
      FProps.Values[ mlpDescription ] := Result;
      FChanged := True;
   end;
end;

function TMailList.GetOwnerAccount: String;
begin
   DoInitialize;
   Result := FProps.Values[ mlpOwnerAccount ];
   if Hamster.Accounts.UserIDOf( Result ) = ACTID_INVALID then begin
      Result := 'admin';
      FProps.Values[ mlpOwnerAccount ] := Result;
      FChanged := True;
   end;
end;

procedure TMailList.SetOwnerAccount( const NewAccount: String );
begin
   DoInitialize;
   FProps.Values[ mlpOwnerAccount ] := NewAccount;
   GetOwnerAccount; // validity check
   FChanged := True;
end;

function TMailList.GetRestrictFrom: Integer;
begin
   DoInitialize;
   Result := strtointdef( FProps.Values[ mlpRestrictFrom ], mlprfNoRestrict );
end;

function TMailList.GetMembers: TStringList;
begin
   DoInitialize;
   Result := FMembers;
end;

function TMailList.IsMember( const Name: String ): Boolean;
var  i: Integer;
begin
   DoInitialize;

   Result := False;
   for i := 0 to FMembers.Count - 1 do begin
      if Name = FMembers[i] then begin
         Result := True;
         exit;
      end;
   end;
end;

procedure TMailList.DelMember( const Name: String );
var  i: Integer;
begin
   DoInitialize;

   for i := 0 to FMembers.Count - 1 do begin
      if Name = FMembers[i] then begin
         FMembers.Delete( i );
         FChanged := True;
         exit;
      end;
   end;
end;

// ----------------------------------------------------------- TMailLists -----

constructor TMailLists.Create( const APath: String );
begin
   inherited Create;
   FLock := TCriticalSection.Create;
   FList := TStringList.Create;
   FPath := APath;
   Refresh;
end;

destructor TMailLists.Destroy;
begin
   DoClear;
   FList.Free;
   FLock.Free;
   inherited Destroy;
end;

procedure TMailLists.DoClear;
var  Index: Integer;
begin
   for Index := FList.Count - 1 downto 0 do DoDel( Index );
end;

function TMailLists.DoIndexOf( const ListName: String ): Integer;
var  i: Integer;
begin
   Result := -1;
   for i := 0 to FList.Count - 1 do begin
      if CompareText( FList[i], ListName ) = 0 then begin
         Result := i;
         break;
      end;
   end;
end;

procedure TMailLists.DoDel( const Index: Integer );
begin
   TMailList( FList.Objects[Index] ).Free;
   FList.Delete( Index );
end;

function TMailLists.DoAdd( const ListName: String ): Integer;
begin
   Result := FList.AddObject( ListName, TMailList.Create( FPath, ListName ) );
end;

procedure TMailLists.Refresh;
// refreshes internal list of all mail lists
var  SR: TSearchRec;
     s : String;
     i : Integer;
begin
   FLock.Enter;
   try
   
      try
         DoClear;

         if FindFirst(FPath + '*' + FILEEXT_MAILLIST, faAnyFile, SR) = 0 then try
            repeat
               if (SR.Attr and faDirectory) = 0 then begin
                  s := SR.Name;
                  i := LastDelimiter( '.', s );
                  if i > 0 then SetLength( s, i-1 );
                  DoAdd( s );
               end;
            until FindNext( SR ) <> 0;
         finally FindClose( SR ) end;

         FList.Sort;

      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error reading distribution lists: %s',
                                     [ E.Message ] ) );
      end;

   finally FLock.Leave end;
end;

procedure TMailLists.RemoveAccountFromAllLists( const AccountName: String );
var  Index: Integer;
     Old: String;
begin
   FLock.Enter;
   try
      for Index := 0 to FList.Count - 1 do begin
         with TMailList( FList.Objects[Index] ) do begin

            if OwnerAccount = AccountName then begin

               Old := OwnerAccount;
               OwnerAccount := 'admin';

               NotifyOwner( Name, OwnerAccount,
                   Format( '[%s] Distribution list owner removed!', [Name] ),
                   Format( 'Former owner "%s" of list "%s" was removed, so '
                         + 'ownership was assigned to "%s"!'#13#10#13#10'%s',
                           [ Old, Name, OwnerAccount, MailListReport(Name) ] ));

            end;

            if IsMember( AccountName ) then begin

               DelMember( AccountName );

               NotifyOwner( Name, OwnerAccount,
                  Format( '[%s] List member was removed!', [Name] ),
                  Format( 'Account "%s" was removed, so it was removed from '
                        + 'list "%s" as well!'#13#10#13#10'%s',
                          [ AccountName, Name, MailListReport(Name) ] ) );

            end;
            
         end;
      end;
   finally FLock.Leave end;
end;

function TMailLists.MailListOwner( const ListName: String ): String;
var  Index: Integer;
begin
   Result := '';
   FLock.Enter;
   try
      Index := DoIndexOf( ListName );
      if Index >= 0 then begin
         Result := TMailList( FList.Objects[Index] ).OwnerAccount;
      end;
   finally FLock.Leave end;
end;

function TMailLists.MailListMembers( const ListName: String;
                                     const ListAddr: TStrings ): Boolean;
// returns True if given list exists and returns the list members
var  Index: Integer;
begin
   FLock.Enter;
   try
      Result := False;
      ListAddr.Clear;

      Index := DoIndexOf( ListName );
      if Index >= 0 then begin
         Result := True;
         ListAddr.Assign( TMailList( FList.Objects[Index] ).Members );
      end;

   finally FLock.Leave end;
end;

function TMailLists.MailListExists( const ListName: String ): Boolean;
// returns True if given list exists
begin
   FLock.Enter;
   try
      Result := ( DoIndexOf(ListName) >= 0 );
   finally FLock.Leave end;
end;

function TMailLists.MailListGet( const ListName: String;
                                 const ListData: TStringList ): Boolean;
// returns data of given list
var  Index: Integer;
begin
   FLock.Enter;
   try

      Result := False;
      try

         ListData.Clear;

         Index := DoIndexOf( ListName );

         if Index >= 0 then begin
            TMailList( FList.Objects[Index] ).ListGetData( ListData );
            Result := True;
         end;

      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error getting distribution list %s: %s',
                                     [ ListName, E.Message ] ) );
      end;

   finally FLock.Leave end;
end;

function TMailLists.MailListSet( const ListName: String;
                                 const ListData: TStringList ): Boolean;
// changes data of given list
var  Index: Integer;
     IsNew: Boolean;
begin
   IsNew := False;
   
   FLock.Enter;
   try

      Result := False;
      try

         Index := DoIndexOf( ListName );

         if Index < 0 then begin
            Index := DoAdd( ListName );
            IsNew := True;
         end;

         if Index >= 0 then begin
            TMailList( FList.Objects[Index] ).ListSetData( ListData );
            TMailList( FList.Objects[Index] ).DoSaveToFile;
         end;

         if IsNew then Refresh;

         Result := True;

      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error setting distribution list %s: %s',
                                     [ ListName, E.Message ] ) );
      end;

   finally FLock.Leave end;

   if IsNew then begin
      NotifyOwner( ListName, '',
                   Format( '[%s] Distribution list was created!', [ListName] ),
                   Format( 'Distribution list "%s" was created!'#13#10#13#10'%s',
                           [ ListName, MailListReport(ListName) ] ) );
   end else begin
      NotifyOwner( ListName, '',
                   Format( '[%s] Distribution list was modified!', [ListName] ),
                   Format( 'Distribution list "%s" was modified!'#13#10#13#10'%s',
                           [ ListName, MailListReport(ListName) ] ) );
   end;
end;

function TMailLists.MailListDel( const ListName: String ): Boolean;
// deletes the given list
var  Index : Integer;
     IsDel : Boolean;
     Filename, Owner, Report: String;
begin
   IsDel  := False;
   Owner  := '';
   Report := '';

   FLock.Enter;
   try

      Result := False;
      try

         Index := DoIndexOf( ListName );
         IsDel := False;

         if Index >= 0 then begin
            Owner  := TMailList( FList.Objects[Index] ).OwnerAccount;
            Report := MailListReport( ListName );
            DoDel( Index );
            Filename := FPath + ListName + FILEEXT_MAILLIST;
            if FileExists( Filename ) then DeleteFile( Filename );
            IsDel := True;
         end;

         if IsDel then Refresh;

         Result := True;

      except
         on E:Exception do
            Log( LOGID_WARN, Format( 'Error deleting distribution list %s: %s',
                                     [ ListName, E.Message ] ) );
      end;

   finally FLock.Leave end;

   if IsDel and (Owner<>'') then begin
      NotifyOwner( ListName, Owner,
                   Format( '[%s] Distribution list was deleted!', [ListName] ),
                   Format( 'Distribution list "%s" was deleted!'#13#10#13#10'%s',
                           [ ListName, Report ] ) );
   end;
end;

procedure TMailLists.MailListDir( const ListNames: TStrings );
// return all list names
begin
   FLock.Enter;
   try
      Refresh;
      ListNames.Assign( FList );
   finally FLock.Leave end;
end;

function TMailLists.NotifyOwner( const ListName, Owner, Subject, Body: String ): Boolean;
// send local info mail to list owner
var  Index: Integer;
     DestAcc: String;
begin
   Result := False;

   DestAcc := Owner;
   if DestAcc = '' then begin
      FLock.Enter;
      try
         Index := DoIndexOf( ListName );
         if Index >= 0 then DestAcc := TMailList(FList.Objects[Index]).OwnerAccount;
      finally FLock.Leave end;
   end;

   if DestAcc <> '' then begin
      Log( LOGID_DEBUG, 'Notify owner of list "' + DestAcc + '": ' + Subject );
      Result := SendLocalInfoMail( DestAcc, Subject, Body );
   end;
end;

function TMailLists.MailListReport( const ListName: String ): String;
var  SL: TStringList;
begin
   SL := TStringList.Create;
   try
      if MailListGet( ListName, SL ) then begin
         Result := 'Settings of distribution list "' + ListName + '" follow:'
                 + #13#10#13#10 + SL.Text;
      end else begin
         Result := 'Mail list "' + ListName + '" is unknown!';
      end;
   finally SL.Free end;
end;

function TMailLists.MailListAccept( const ListName: String;
                                    const MailItem: TMailItem ): Boolean;
var  Index, rf, i, t: Integer;
     s: String;
begin
   FLock.Enter;
   try
      Result := False;
      Index  := DoIndexOf( ListName );

      if Index >= 0 then begin

         rf := TMailList( FList.Objects[Index] ).RestrictFrom;

         if rf = mlprfNoRestrict then begin
            // accept from any sender
            Result := True;
         end;

         if (not Result) and (rf = mlprfMemberFrom) then begin
            // from known 'From:' of member-account only
            s := MailItem.MailText.HeaderValueByNameSL( 'From:' );
            if Hamster.Accounts.IsLocalMailbox( s, i, t ) then begin
               if t = LOCALMAILTYPE_NORMAL then begin
                  s := Hamster.Accounts.Value[ i, apUsername ];
                  Result := TMailList(FList.Objects[Index]).IsMember( s );
               end;
            end;
            rf := mlprfMemberAuth; // if failed, try auth. members
         end;

         if (not Result) and (rf = mlprfMemberAuth) then begin
            // from authenticated member-account only
            if MailItem.Sender.AuthUserID <> ACTID_INVALID then begin
               s := Hamster.Accounts.Value[ MailItem.Sender.AuthUserID, apUsername ];
               Result := TMailList(FList.Objects[Index]).IsMember( s );
               rf := mlprfOwnerAuth; // if failed, try auth. owner
            end;
         end;

         if (not Result) and (rf = mlprfOwnerAuth) then begin
            // from authenticated owner-account only
            if MailItem.Sender.AuthUserID <> ACTID_INVALID then begin
               s := Hamster.Accounts.Value[ MailItem.Sender.AuthUserID, apUsername ];
               Result := (s = TMailList(FList.Objects[Index]).OwnerAccount);
            end;
         end;

      end;

   finally FLock.Leave end;

end;

end.
