// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cMailItem;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, cMailUsers, cArticle;

type 
   TMailOriginTypes = ( moSMTP, moESMTP, // SMTP server
                        moScript,        // Script engine
                        moPOP3,          // POP3 client
                        moInternal,      // internal info mail
                        moNewsToMail     // moderated/gateway group
                        // Don't change above items, as sequence is used with
                        // ord() to report origin of the mails in actions!
                      );

   TMailItem = class
      // Temporary container for a mail, that was received by SMTP, by a
      // script (HamNewMail command) or was fetched by POP3.

      protected
         FOrigin    : TMailOriginTypes;
         FSender    : TMailSender;
         FRecipients: TMailRecipients;
         FMailText  : TMess;
         FMailUIDL  : String;
         FRcvdDone  : Boolean;
         FRPathDone : Boolean;

         function  GetMailUIDL: String;

      public
         property Origin    : TMailOriginTypes read FOrigin write FOrigin;
         property Sender    : TMailSender      read FSender;
         property Recipients: TMailRecipients  read FRecipients;
         property MailText  : TMess            read FMailText;
         property MailUIDL  : String           read GetMailUIDL;

         procedure Clear;

         procedure AddRequiredHeaders;
         procedure AddHeaderMessageID( const ForceMID: Boolean = False );
         procedure AddHeaderUserAgent;
         procedure AddHeaderReceivedComment( Comment: String );
         procedure AddHeaderReceived;
         procedure AddHeaderReturnPath;
         procedure AddHeaderXHamsterInfo;

         constructor Create( const AOrigin: TMailOriginTypes );
         destructor Destroy; override;
   end;


implementation

uses uConst, uHamTools, uTools, uWinsock, uDateTime, uVar, cHamster,
     cLogFileHamster;

const 
   MailOriginTypeNames: array[ TMailOriginTypes ] of String =
      ( 'smtp', 'esmtp', 'script', 'pop3', 'internal', 'news-to-mail' );

   MailAddUserAgentSet    = [ moSMTP, moESMTP, moScript        , moInternal, moNewsToMail ];
   MailAddReceivedSet     = [ moSMTP, moESMTP, moScript                    , moNewsToMail ];
   MailAddMessageIDSet    = [ moSMTP, moESMTP, moScript, moPOP3, moInternal, moNewsToMail ];
   MailAddReturnPathSet   = [ moSMTP, moESMTP, moScript                    , moNewsToMail ];
   MailAddXHamsterInfoSet = [ moSMTP, moESMTP, moScript        , moInternal               ];

{ TMailItem }

constructor TMailItem.Create( const AOrigin: TMailOriginTypes );
begin
   inherited Create;

   FOrigin     := AOrigin;
   FSender     := TMailSender.Create;
   FRecipients := TMailRecipients.Create;
   FMailText   := TMess.Create;

   Clear;
end;

destructor TMailItem.Destroy;
begin
   FSender.Free;
   FRecipients.Free;
   FMailText.Free;

   inherited Destroy;
end;

procedure TMailItem.Clear;
begin
   FSender.Clear;
   FRecipients.Clear;
   FMailText.FullText := '';
   FMailUIDL := '';
   FRcvdDone := False;
   FRPathDone := False;
end;

function TMailItem.GetMailUIDL: String;
begin
   if length( FMailUIDL ) = 0 then FMailUIDL := UIDLGenerator;
   Result := FMailUIDL;
end;

procedure TMailItem.AddRequiredHeaders;
begin
   if MailText.IndexOfHeader( 'Date:' ) < 0 then begin
      MailText.AddHeaderSL(
         HDR_NAME_DATE, DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone )
      );
   end;

   if MailText.IndexOfHeader( HDR_NAME_FROM ) < 0 then begin
      MailText.AddHeaderSL( HDR_NAME_FROM, Sender.EnvelopeAddr );
   end;
end;

procedure TMailItem.AddHeaderMessageID( const ForceMID: Boolean = False );
begin
   if MailText.IndexOfHeader( 'Message-ID:' ) >= 0 then exit;

   if not ForceMID then begin
      if not( Origin in MailAddMessageIDSet ) then exit;
      if not( Hamster.Config.Settings.GetBoo(hsGenerateMailMID) ) then exit;
      if Hamster.Config.Settings.GetStr(hsFQDNforMID) = '' then exit;
   end;

   MailText.AddHeaderSL(
      HDR_NAME_MESSAGE_ID, 
      MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) )
   );
end;

procedure TMailItem.AddHeaderUserAgent;
var  i: Integer;
     s: String;
begin
   if not( Origin in MailAddUserAgentSet ) then exit;
   if not Hamster.Config.Settings.GetBoo(hsMailAddXHamster) then exit;

   i := MailText.IndexOfHeader( HDR_NAME_USER_AGENT );
   if i < 0 then begin // add
      MailText.AddHeaderSL( HDR_NAME_USER_AGENT, OUR_VERINFO );
   end else begin // expand
      s := MailText.HeaderLineSL[i];
      if Pos(OUR_VERINFO,s)=0 then MailText.HeaderLineSL[i]:=s+' '+OUR_VERINFO;
   end;
end;

procedure TMailItem.AddHeaderReceivedComment( Comment: String );
var  s: String;
begin
   try
   
      s := '(hservice ' + Comment + ')'
         + ' ; ' + DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone );

      MailText.InsertHeaderSL( 0, HDR_NAME_RECEIVED, s );

   except end;
end;

procedure TMailItem.AddHeaderReceived;
var  n, s: String;
begin
   if FRcvdDone then exit;
   if not( Origin in MailAddReceivedSet ) then exit;
   if not Hamster.Config.Settings.GetBoo(hsMailAddReceived) then exit;
   
   try

      if Sender.IPAddr <> 0 then n := LookupHostName( Sender.IPAddr )
                            else n := 'unknown';
      // h := '';
      // if (Sender.HELOStr <> '') and (Sender.HELOStr <> n) then
      //    h := '(HELO ' + Sender.HELOStr + ')' + ' ';
      s := 'from ' + n + ' ' // + h
                    + '([' + nAddrToStr(Sender.IPAddr) + '])'
         + ' by '   + LookupLocalHostName + ' '
                    + '(' + nAddrToStr(LookupLocalHostAddr) + ')'
         + CRLF + #9
         + ' with ' + MailOriginTypeNames[ Origin ]
         + ' ; '    + DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone );

      MailText.InsertHeaderSL( 0, HDR_NAME_RECEIVED, s );
      FRcvdDone := True

   except end;
end;

procedure TMailItem.AddHeaderReturnPath;
var  s: String;
begin
   if FRPathDone then exit;
   if not( Origin in MailAddReturnPathSet ) then exit;

   s := ExtractMailAddr( Sender.OrgEnvAddr );
   if length(s) = 0 then s := Sender.EnvelopeAddr;

   MailText.DelHeaderML( HDR_NAME_RETURN_PATH );
   MailText.InsertHeaderSL( 0, HDR_NAME_RETURN_PATH, s );
   
   FRPathDone := True;
end;

procedure TMailItem.AddHeaderXHamsterInfo;
var  s: String;
begin
   if not( Origin in MailAddXHamsterInfoSet ) then exit;

   s := 'UIDL=' + MailUIDL
      + ' ' + 'Received=' + DateTimeToTimeStamp(Now)
      + ' ' + 'UID=' + IntToStr( GetUID(0) );
   MailText.SetHeaderSL( HDR_NAME_X_HAMSTER_INFO, s, '' );
end;

end.
