// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cMailDispatcher;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, uType, cMailUsers, cMailItem, cArticle;

type
   TMailDispatcher = class
      protected
         procedure CheckPrepareMessage    ( MailItem: TMailItem );
         procedure CheckPrepareRecipients ( MailItem: TMailItem );
         procedure SendRemoteToMailOut    ( MailItem: TMailItem );
         procedure SendLocalToMailbox     ( MailItem: TMailItem );
         procedure SendFailureNotification( MailItem: TMailItem );

         procedure SetPendingOnesTo( MailItem: TMailItem;
                                     MDResult: TMailDeliveryResults );

      public
         function Process( MailItem: TMailItem ): TMailDeliveryResults;
   end;


function SendLocalInfoMail( const LocalRecipient, Subject, MailBody: String ): Boolean;
function ScriptNewMail( const MailFrom, RcptToList, MailText: String ): Boolean;

function ProcessAction_MailInHeader( const Origin: TMailOriginTypes;
                                     const MailMsg: TMess;
                                     out   ModifiedHeaders: String ): Boolean;

implementation

uses uConst, uVar, uTools, uDateTime, uHamTools, cLogFileHamster, cHamster,
     uCharsets, cHscAction;

var
   MailLoopDetectionCounter: Int64 = 0;

// ----------------------------------------------------------------------------

function SaveMailFile( const DestFile, MsgText: String ): Boolean;
// Saves mail text in given path, using a unique filename.
// Note: DestPath has to be locked before calling!
var  TF: TextFile;
begin
   Result := False;

   try
      AssignFile( TF, DestFile );
      Rewrite( TF );
      Write( TF, MsgText );
      CloseFile( TF );
      Result := True;

   except
      on E: Exception do begin
         Log( LOGID_WARN, 'Error saving mail to file '
                        + DestFile + ': ' + E.Message
                        + ' (.SaveMailFile)' );
      end;
   end;
end;  

function SaveMailFile_MailOut( const MailText: TMess;
                               const AuthUserID: Integer ): Boolean;
// Save mail text in Mail.Out.
var  FN, TN, Path, AuthUserName: String;
     ok: Boolean;
begin
   Result := False;
   ok := False;

   Path := AppSettings.GetStr(asPathMailOut);
   FN := GetUniqueMsgFilename( Path, 'mail' );
   AuthUserName := Hamster.Accounts.Value[ AuthUserID, apUsername ];

   // handle 'MailOut' action
   if Hamster.HscActions.Execute(
         actMailOut, FN + CRLF + inttostr( Integer(MailText) )
                        + CRLF + AuthUserName
      ) then begin

      if not MailText.IsValid then begin
         Log( LOGID_INFO, 'Note: Action "MailOut" invalidated message.' );
         Result := True; // ok, this is allowed
         exit;
      end;

   end;

   // handle 'MailOutFile' action
   if Hamster.HscActions.IsAssigned( actMailOutFile ) then begin

      // save temporary mail with different extension
      TN := FN + '.tmp';
      MailText.SaveToFile( TN );

      // execute action on temporary mailfile
      Hamster.HscActions.Execute(
         actMailOutFile, TN + CRLF + AuthUserName );

      // update internal/real mail from temporary file
      if FileExists( TN ) then begin
         MailText.LoadFromFile( TN );
         DeleteFile( TN );
      end else begin
         MailText.FullText := '';
      end;

      // is modified mail still valid?
      if not MailText.IsValid then begin
         Log( LOGID_INFO, 'Note: Action "MailOutFile" invalidated message.' );
         Result := True; // ok, this is allowed
         exit;
      end;

   end;

   CS_LOCK_MAILOUT_ADD.Enter;
   try

      // save mail 'ready to be sent'
      try
         Result := SaveMailFile( FN, MailText.FullText );
         ok := Result;
      except
         on E: Exception do begin
            Log( LOGID_WARN, 'Error saving mail: ' + E.Message
                           + ' (.SaveMailFile_MailOut)' );
         end;
      end;

   finally CS_LOCK_MAILOUT_ADD.Leave end;

   if ok then begin
      Hamster.HscActions.Execute(
         actMailWaiting, FN + CRLF + inttostr( Integer(MailText) ) );
      Hamster.HscActions.Execute(
         actMsgWaiting,  FN + CRLF + inttostr( Integer(MailText) ) );
   end;
end;

function SaveMailFile_Mailbox( const UserID  : Integer;
                               const MailText: TMess;
                               const Origin  : TMailOriginTypes ): Boolean;
// Save mail text in user's mailbox.
var  FN, TN, Path, User: String;
     TempText: TMess;
begin
   Result := False;

   TempText := TMess.Create;
   try
      TempText.FullText := MailText.FullText;

      User := Hamster.Accounts.Value[ UserID, apUsername ];
      Path := Hamster.Accounts.MailboxPath( UserID );
      if Path = '' then Path := Hamster.Accounts.MailboxPath( ACTID_ADMIN );
      FN := GetUniqueMsgFilename( Path, 'mail' );

      // handle 'MailIn' action
      if Hamster.HscActions.Execute(
            actMailIn, FN + CRLF + inttostr( Integer(TempText) )
                          + CRLF + User
                          + CRLF + inttostr( ord(Origin) )
         ) then begin

         if not TempText.IsValid then begin
            Log( LOGID_INFO, 'Note: Action "MailIn" invalidated message'
                           + ' for user "' + User + '".' );
            Result := True; // ok, this is allowed
            exit;
         end;

      end;

      // handle 'MailInFile' action
      if Hamster.HscActions.IsAssigned( actMailInFile ) then begin

         // save temporary mail with different extension
         TN := FN + '.tmp';
         TempText.SaveToFile( TN );

         // execute action on temporary mailfile
         Hamster.HscActions.Execute(
            actMailInFile,
            TN + CRLF + User + CRLF + inttostr( ord(Origin) )
         );

         // update internal/real mail from temporary file
         if FileExists( TN ) then begin
            TempText.LoadFromFile( TN );
            DeleteFile( TN );
         end else begin
            TempText.FullText := '';
         end;

         // is modified mail still valid?
         if not TempText.IsValid then begin
            Log( LOGID_INFO, 'Note: Action "MailInFile" invalidated message'
                           + ' for user "' + User + '".' );
            Result := True; // ok, this is allowed
            exit;
         end;

      end;

      CS_LOCK_MAILBOX_ALL.Enter;
      try

         // save mail 'ready to be fetched by user'
         try
            // HDR_NAME_X_HAMSTER_LOG
            Result := SaveMailFile( FN, TempText.FullText );
         except
            on E: Exception do begin
               Log( LOGID_WARN, Format(
                     'Couldn''t save mailfile: %s %s',
                     [E.Message, '(.SaveMailFile_Mailbox)' ] ) );
            end;
         end;

      finally CS_LOCK_MAILBOX_ALL.Leave end;

   finally TempText.Free end;
end;

function ProcessAction_MailInHeader( const Origin: TMailOriginTypes;
                                     const MailMsg: TMess;
                                     out   ModifiedHeaders: String ): Boolean;
// Process actions "MailInHeader" and "MailInHeaderFile".
// Returns 'True' to continue, 'False' to delete or reject the mail.
var  TempMsg: TMess;
     UserId: Integer;
     FN, TN, Path, User: String;
begin
   Result := True; // = continue
   ModifiedHeaders := '';

   TempMsg := TMess.Create;
   try

      TempMsg.HeaderText := MailMsg.HeaderText;

      UserId := ACTID_ADMIN;
      User := Hamster.Accounts.Value[ UserID, apUsername ];
      Path := Hamster.Accounts.MailboxPath( UserID );
      if Path = '' then Path := Hamster.Accounts.MailboxPath( ACTID_ADMIN );
      FN := GetUniqueMsgFilename( Path, 'mail' );

      // handle 'MailInHeader' action
      if Hamster.HscActions.Execute(
            actMailInHeader, FN + CRLF + inttostr( Integer(TempMsg) )
                                + CRLF + User
                                + CRLF + inttostr( ord(Origin) )
         ) then begin

         if TempMsg.IsValid then begin
            if TempMsg.HeaderText <> MailMsg.HeaderText then begin
               ModifiedHeaders := TempMsg.HeaderText;
            end;
         end else begin
            Log( LOGID_INFO, 'Note: Action "MailInHeader" invalidated message.' );
            Result := False; // = delete/reject mail
            exit;
         end;

      end;

      // handle 'MailInHeaderFile' action
      if Hamster.HscActions.IsAssigned( actMailInHeaderFile ) then begin

         // save temporary mail with different extension
         TN := FN + '.tmp';
         TempMsg.SaveToFile( TN );

         // execute action on temporary mailfile
         Hamster.HscActions.Execute(
            actMailInHeaderFile,
            TN + CRLF + User + CRLF + inttostr( ord(Origin) )
         );

         // update internal/real mail from temporary file
         if FileExists( TN ) then begin
            TempMsg.LoadFromFile( TN );
            DeleteFile( TN );
         end else begin
            TempMsg.FullText := '';
         end;

         // is modified mail still valid?
         if TempMsg.IsValid then begin
            if TempMsg.HeaderText <> MailMsg.HeaderText then begin
               ModifiedHeaders := TempMsg.HeaderText;
            end;
         end else begin
            Log( LOGID_INFO, 'Note: Action "MailInHeaderFile" invalidated message.' );
            Result := False; // = delete/reject mail
            exit;
         end;

      end;

   finally TempMsg.Free end;
end;

procedure BuildLocalInfoMail( const MailText: TMess;
                              const LocalRecipient, RcptName, Subject, MailBody: String );
begin
   with MailText do begin
      FullText := '';
      AddHeaderSL( HDR_NAME_FROM, '"Hamster-Info" <local-hamster-info@hamster.invalid>' );
      AddHeaderSL( HDR_NAME_TO, '"' + LocalRecipient + '" <' + RcptName+ '@hamster.invalid>' );
      AddHeaderSL( HDR_NAME_DATE, DateTimeGMTToRfcDateTime( NowGMT, NowRfcTimezone ) );
      AddHeaderSL( HDR_NAME_SUBJECT, Subject );
      AddHeaderSL( HDR_NAME_MESSAGE_ID, MidGenerator( Hamster.Config.Settings.GetStr(hsFQDNforMID) ) );
      AddHeaderSL( HDR_NAME_MIME_VERSION, '1.0' );
      AddHeaderSL( HDR_NAME_CONTENT_TYPE, 'text/plain; charset="'
                 + Charsets.WindowsACPHandler.PreferredName + '"' );
      AddHeaderSL( HDR_NAME_CONTENT_TRANSFER_ENCODING, '8bit' );
      BodyText := MailBody;
   end;
end;

function SendLocalInfoMail( const LocalRecipient, Subject, MailBody: String ): Boolean;
// Send an internal info mail to a local recipient.
var  RcptID, RcptType: LongInt;
     RcptName: String;
     MailItem: TMailItem;
begin
   RcptID := Hamster.Accounts.UserIDOf( LocalRecipient );
   if RcptID <> ACTID_INVALID then begin
      if not Hamster.Accounts.IsLocalMailbox( LocalRecipient, RcptID, RcptType ) then begin
         RcptID := ACTID_INVALID;
      end;
   end;
   if RcptID = ACTID_INVALID then RcptID := ACTID_ADMIN;
   RcptName := Hamster.Accounts.Value[ RcptID, apUsername ];

   MailItem := TMailItem.Create( moInternal );
   with MailItem do try

      Sender.UserID := ACTID_ADMIN;
      Recipients.Add( RcptID );
      BuildLocalInfoMail( MailText, LocalRecipient, RcptName, Subject, MailBody );
      Result := ( Hamster.MailDispatcher.Process( MailItem ) = mdrSuccess );

   finally MailItem.Free end;
end;

function ScriptNewMail( const MailFrom, RcptToList, MailText: String ): Boolean;
// Send a mail by script (HamNewMail command).
var  MailItem: TMailItem;
     tmp, s: String;
     mdr: TMailDeliveryResults;
begin
   Result := False;

   MailItem := TMailItem.Create( moScript );
   try
      MailItem.Sender.EnvelopeAddr := MailFrom;

      tmp := RcptToList;
      repeat
         s := NextSepPart( tmp, ';' );
         if s = '' then break;
         if s <> '' then MailItem.Recipients.Add( s );
      until False;

      MailItem.MailText.FullText := MailText;

      try
         mdr := Hamster.MailDispatcher.Process( MailItem );
         s   := MailDeliveryResultTexts[ mdr ];
         Result := ( mdr = mdrSuccess );
         if Result then Log( LOGID_DETAIL, 'ScriptNewMail: ' + s )
                   else Log( LOGID_WARN,   'ScriptNewMail: ' + s );
      except
         on E: Exception do s := 'ScriptNewMail-Error: ' + E.Message;
      end;

   finally MailItem.Free end;
end;

// ------------------------------------------------------ TMailDispatcher -----

function TMailDispatcher.Process( MailItem: TMailItem ): TMailDeliveryResults;
var  i: Integer;
begin
   Result := mdrFailed_Loop;
   CounterInc( MailLoopDetectionCounter );
   try

      try
         // make sure that we are not in an endless (forward/notification) loop
         if CounterGet( MailLoopDetectionCounter ) >= 100 then begin
            Log( LOGID_DEBUG, 'Dispatcher loop detected and stopped.' );
            Result := mdrFailed_Loop;
            SetPendingOnesTo( MailItem, mdrFailed_Loop );
            exit;
         end;
         
         // check mail to be delivered
         Log( LOGID_DEBUG, 'MailDispatcher (1/7): Check and prepare message ...' );
         CheckPrepareMessage( MailItem );

         // prepare mail recipients
         Log( LOGID_DEBUG, 'MailDispatcher (2/7): Check recipients ...' );
         CheckPrepareRecipients( MailItem );

         // deliver mail to remote recipients
         Log( LOGID_DEBUG, 'MailDispatcher (3/7): Send remote to MailOut ...' );
         SendRemoteToMailOut( MailItem );

         // deliver mail to local recipients
         Log( LOGID_DEBUG, 'MailDispatcher (4/7): Send local to mailbox ...' );
         SendLocalToMailbox( MailItem );

         // mark still pending recipients as failed (should not happen)
         Log( LOGID_DEBUG, 'MailDispatcher (5/7): SetPendingOnesTo(NoHandler) ...' );
         SetPendingOnesTo( MailItem, mdrFailed_NoHandler );

         // send notification for failed recipients
         Log( LOGID_DEBUG, 'MailDispatcher (6/7): Send failure notification ...' );
         SendFailureNotification( MailItem );

         // Common result is mdrSuccess if at least one recipient was handled
         // successfully. Otherwise first failure reason is returned.
         Log( LOGID_DEBUG, 'MailDispatcher (7/7): Determine final result ...' );
         Result := mdrFailed_NoRecipient;
         for i := MailItem.Recipients.Count - 1 downto 0 do begin
            Result := MailItem.Recipients.Items[i].DeliveryResult;
            if Result = mdrSuccess then break;
         end;

      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'MailDispatcher-Error: ' + E.Message );
            Result := mdrFailed_Exception;
         end;
      end;

   finally
      CounterDec( MailLoopDetectionCounter );
      Log( LOGID_DEBUG, 'MailDispatcher''s final result: '
                      + MailDeliveryResultTexts[Result] );
   end;
end;

procedure TMailDispatcher.CheckPrepareMessage( MailItem: TMailItem );
const Received = 'Received:';
var  i, n: Integer;
begin
   with MailItem do begin

      // Loop detection (max. 100 Received headers)
      n := 0;
      for i := 0 to MailText.HeaderLineCountSL - 1 do begin
         if CompareText(
               copy( MailText.HeaderLineSL[i], 1, length(Received) ),
               Received
            ) = 0 then inc( n );
      end;
      if n > 100 then begin
         Log( LOGID_DEBUG, 'Mail loop detected and stopped.' );
         SetPendingOnesTo( MailItem, mdrFailed_Loop );
      end;

      // Check/add required and server generated headers
      AddRequiredHeaders; // check/add 'Date:'/'From:'
      AddHeaderMessageID; // check/add 'Message-ID:'
      AddHeaderUserAgent; // check/add 'User-Agent:'
      AddHeaderReceived;  // check/add 'Received:'

      // Check sender of message ('MAIL FROM:')
      if Origin in [ moSMTP, moESMTP, moScript ] then begin
         if Sender.MailUserType = muMissing then begin
            Log( LOGID_DEBUG, 'Missing MAIL FROM detected.' );
            SetPendingOnesTo( MailItem, mdrFailed_NoSender );
            exit;
         end;
      end;

   end;
end;

procedure TMailDispatcher.CheckPrepareRecipients( MailItem: TMailItem );
var  i, k: Integer;
     FwdReport, FwdAccount, FwdAddress, FwdMailList, ForwardUIDs, s: String;
     FwdKeepCopy, NeedFinalRecipient, HaveFinalRecipient: Boolean;
     SL: TStringList;
     rcptIndex: String;
begin
   with MailItem do begin

      // Check recipients of message ('RCPT TO:') and resolve any forwards

      ForwardUIDs := '';
      NeedFinalRecipient := False;
      HaveFinalRecipient := False;

      i := 0;
      while i < Recipients.Count do with Recipients.Items[i] do begin

         if IsPending then begin // only pending ones

            rcptIndex := '(rcpt ' + inttostr(i+1) + '/' + inttostr(Recipients.Count) + ') ';
            s := Recipients.Items[i].OrgEnvAddr
                      + ' (ID ' + inttostr( Recipients.Items[i].UserID )
                      + ', ' + Recipients.Items[i].EnvelopeAddr + ')';
            Log( LOGID_DEBUG, rcptIndex + 'Process recipient ' + s );

            case MailUserType of

               // Remote recipient
               muRemote:
                  begin
                     Log( LOGID_DEBUG, rcptIndex + 'Is remote recipient' );
                     HaveFinalRecipient := True;
                  end;

               // Mark recipients as failed, which were recognized as being
               // unknown or which had an obviously invalid address.
               muMissing:
                  begin
                     Log( LOGID_DEBUG, rcptIndex + 'Is no recipient' );
                     DeliveryResult := mdrFailed_NoRecipient;
                  end;
               muLocalUnknown:
                  begin
                     Log( LOGID_DEBUG, rcptIndex + 'Is unknown local recipient' );
                     DeliveryResult := mdrFailed_LocalUnknown;
                  end;
               muLocalInvalid:
                  begin
                     Log( LOGID_DEBUG, rcptIndex + 'Is invalid local recipient' );
                     DeliveryResult := mdrFailed_LocalInvalid;
                  end;

               // Resolve any forward settings of a known local account
               muLocal:
                  begin
                  Log( LOGID_DEBUG, rcptIndex + 'Is known local recipient' );
                  if not Hamster.Accounts.HasMailForward(
                        UserID, FwdAccount, FwdAddress, FwdMailList, FwdKeepCopy
                     ) then begin

                     HaveFinalRecipient := True;

                  end else begin

                     Log( LOGID_DEBUG, rcptIndex + 'Has forward settings!' );
                     ForwardUIDs := ForwardUIDs + ' ' + inttostr(UserID);
                     FwdReport   := 'Forward report for "' + MailAddress + '"';

                     // forward to account
                     if FwdAccount <> '' then begin
                        Log( LOGID_DEBUG, rcptIndex + 'Forward to account "' + FwdAccount + '"' );
                        if not Recipients.AlreadyContains( FwdAccount ) then begin
                           FwdReport := FwdReport + ', Local:' + FwdAccount;
                           Recipients.Add( FwdAccount );
                        end;
                     end;

                     // forward to address
                     if FwdAddress <> '' then begin
                        Log( LOGID_DEBUG, rcptIndex + 'Forward to remote address "' + FwdAddress + '"' );
                        if not Recipients.AlreadyContains( FwdAddress ) then begin
                           FwdReport := FwdReport + ', Remote:' + FwdAddress;
                           Recipients.Add( FwdAddress );
                        end;
                     end;

                     // forward to mail list
                     if FwdMailList <> '' then begin
                        Log( LOGID_DEBUG, rcptIndex + 'Request forward to mail list "' + FwdMailList + '"' );
                        if Hamster.MailLists.MailListAccept( FwdMailList, MailItem ) then begin

                           // sender ok, forward to members
                           Log( LOGID_DEBUG, rcptIndex + 'Forward to mail list "' + FwdMailList + '" granted' );
                           SL := TStringList.Create;
                           try
                              FwdReport := FwdReport + ', List:' + FwdMailList;

                              Hamster.MailLists.MailListMembers( FwdMailList, SL );
                              for k := 0 to SL.Count - 1 do begin
                                 Log( LOGID_DEBUG, rcptIndex + 'Forward to list member "' + SL[k] + '"' );
                                 if not Recipients.AlreadyContains( SL[k] ) then begin
                                    FwdReport := FwdReport + ', Member:' + SL[k];
                                    Recipients.Add( SL[k] );
                                 end;
                              end;
                           finally SL.Free end;

                        end else begin

                           // invalid sender, notify and forward to owner
                           Log( LOGID_DEBUG, rcptIndex + 'Forward to mail list "' + FwdMailList + '" rejected' );
                           FwdReport := FwdReport + ', Forward to list rejected: ' + FwdMailList;

                           // notify owner
                           s := 'A mail for mail list "' + FwdMailList + '" was not sent to the'#13#10
                              + 'members of the list because the sender of the mail'#13#10
                              + 'did not meet the given list restrictions. The mail'#13#10
                              + 'was forwarded to the owner of the list (YOU) instead.'#13#10
                              + #13#10
                              + 'Headers and first 2 KB of mail follow:'#13#10
                              + #13#10
                              + MailItem.MailText.HeaderText + #13#10
                              + copy( MailItem.MailText.FullText, 1, 2048 );
                           Hamster.MailLists.NotifyOwner( FwdMailList, '',
                              Format( '[%s] Mail for restricted list rejected!',
                                      [FwdMailList] ), s );

                           // forward to owner
                           s := Hamster.MailLists.MailListOwner( FwdMailList );
                           if not Recipients.AlreadyContains( s ) then begin
                              Log( LOGID_DEBUG, rcptIndex + 'Forward to list owner "' + s + '"' );
                              FwdReport := FwdReport + ', Forward to list owner: ' + s;
                              Recipients.Add( s );
                           end;

                           // mail did not reach members, so don't keep it here
                           FwdKeepCopy := False;
                           
                        end;

                     end;

                     // mark account as done if no copy should be kept here
                     NeedFinalRecipient := True;
                     if FwdKeepCopy then begin
                        FwdReport := FwdReport + ', KeepCopy:Yes';
                        HaveFinalRecipient := True;
                     end else begin
                        FwdReport := FwdReport + ', KeepCopy:No';
                        DeliveryResult := mdrSuccess;
                     end;

                     Log( LOGID_FULL, rcptIndex + FwdReport );

                  end;
                  end; // muLocal: if Hamster.Accounts.HasMailForward(...

            end; // case MailUserType

         end; // if IsPending ...

         inc( i );

      end; // while i < Recipients.Count ...

      if ForwardUIDs <> '' then begin
         // Mark any forwarded mails with an additional 'Received:' header so
         // it will trigger the loop detection at .CheckPrepareMessage when
         // the mail is sent out and comes back over and over again.
         AddHeaderReceivedComment( 'forward by uid' + ForwardUIDs );
      end;

      if NeedFinalRecipient and not HaveFinalRecipient then begin
         // Misconfigured forward settings have left no valid recipient for
         // the mail (local forward loop: A -> B -> ... -> A -> B -> ...).
         // Revoke any prior success results and mark all with error instead.
         Log( LOGID_DEBUG, 'Forward loop detected and stopped.' );
         for i := 0 to Recipients.Count-1 do with Recipients.Items[i] do begin
            if DeliveryResult = mdrSuccess then DeliveryResult := mdrPending;
         end;
         SetPendingOnesTo( MailItem, mdrFailed_ForwardLoop );
      end;
   
   end; // with MailItem ...
end;

procedure TMailDispatcher.SendRemoteToMailOut( MailItem: TMailItem );
// deliver to remote recipients, i.e. save mail in Mail.Out
var  i: Integer;
     s: String;
     mdr : TMailDeliveryResults;
     MT: TMess;
begin
   with MailItem do begin

      if not Recipients.AnyRemote then exit;

      MT := TMess.Create;
      try

         // create Mail.Out format
         // (=message with leading envelope headers, each preceded by '!')
         // s := '!MAIL FROM: ' + Sender.EnvelopeAddr + CRLF;
         s := '!MAIL FROM: <' + ExtractMailAddr( Sender.OrgEnvAddr ) + '>' + CRLF;
         if Pos( '@', s ) = 0 then s := '!MAIL FROM: ' + Sender.EnvelopeAddr + CRLF;
         for i := 0 to Recipients.Count - 1 do begin
            with Recipients.Items[i] do begin
               if IsPending and IsRemote then begin
                  Log( LOGID_DEBUG, 'Send to: ' + EnvelopeAddr );
                  s := s + '!RCPT TO: ' + EnvelopeAddr + CRLF;
               end;
            end;
         end;
         MT.FullText := s + MailText.FullText;

         // store mail in Mail.Out
         if SaveMailFile_MailOut( MT, Sender.AuthUserID ) then begin
            mdr := mdrSuccess;
            CounterInc( CounterOutboxM   );
            CounterInc( CounterOutboxChk );
         end else begin
            mdr := mdrFailed_SaveMailOutErr;
         end;

         // set result of recipients
         for i := 0 to Recipients.Count - 1 do begin
            with Recipients.Items[i] do begin
               if IsPending and IsRemote then DeliveryResult := mdr;
            end;
         end;

      finally MT.Free end;
   end;
end;

procedure TMailDispatcher.SendLocalToMailbox( MailItem: TMailItem );
// deliver to local recipients, i. e. save mail in user's mailbox
var  i, userId: Integer;
     s: String;
begin
   with MailItem do begin

      if not Recipients.AnyLocal then exit;

      for i := 0 to Recipients.Count - 1 do begin
         if Recipients.Items[i].IsPending and
            Recipients.Items[i].IsLocal then begin

            userId := Recipients.Items[i].UserID;
            s := 'ID ' + inttostr( userID );
            try
               s := s + '=' + Hamster.Accounts.Value[ userID, apUsername ];
            except
               on E: Exception do s := s + ' ERROR: ' + E.Message;
            end;
            Log( LOGID_DEBUG, 'Store in mailbox: ' + s );

            // precede message with (local only) info-lines
            AddHeaderReturnPath;
            AddHeaderXHamsterInfo;

            // store mail in local mailbox
            if SaveMailFile_Mailbox( userID, MailText, MailItem.Origin ) then begin
               Recipients.Items[i].DeliveryResult := mdrSuccess;
            end else begin
               Recipients.Items[i].DeliveryResult := mdrFailed_SaveMailboxErr;
            end;

         end;
      end;

   end;
end;

procedure TMailDispatcher.SendFailureNotification( MailItem: TMailItem );
// Sends notification for failed recipients, currently to 'admin' only.
var  i: Integer;
     Err, s: String;
     AdminFailed: Boolean;
     TempMail: TMess;
begin
   with MailItem do begin

      if not Recipients.AnyFailed then exit;

      for i := 0 to Recipients.Count - 1 do begin
         with Recipients.Items[i] do begin
            if DeliveryResult <> mdrSuccess then begin
               Log( LOGID_DEBUG, 'Failed: ' + Recipients.Items[i].EnvelopeAddr );
            end;
         end;
      end;

      // create error text
      Err := '|| [Hamster]' + CRLF
           + '||' + CRLF
           + '|| Message could not be delivered to all recipients:' + CRLF;
      for i := 0 to Recipients.Count - 1 do begin
         with Recipients.Items[i] do begin
            s := OrgEnvAddr;
            if s = '' then s := EnvelopeAddr;
            // if DeliveryResult <> mdrSuccess then s := '*** ' + s;
            Err := Err
                 + '||    ' + s + ': '
                            + MailDeliveryResultTexts[DeliveryResult] + CRLF;
         end;
      end;
      Err := Err
           + '||' + CRLF
           + '|| Message follows (up to 10 KB):' + CRLF
           + CRLF
           + copy( MailItem.MailText.FullText, 1, 1024*10 );

      // check, if admin itself had problems (e. g. caused by forward loop)
      AdminFailed := False;
      for i := 0 to Recipients.Count - 1 do begin
         with Recipients.Items[i] do begin
            if IsLocal and (UserId = ACTID_ADMIN) then begin
               if DeliveryResult <> mdrSuccess then AdminFailed := True;
            end;
         end;
      end;

      // deliver error mail to admin
      if AdminFailed then begin

         // store mail directly in admin's mailbox
         TempMail := TMess.Create;
         try
            BuildLocalInfoMail( TempMail, 'admin', 'admin',
                                '[Hamster] Mail failure notice', Err );
            SaveMailFile_Mailbox( ACTID_ADMIN, TempMail, moInternal );
         finally TempMail.Free end;

      end else begin

         // send mail the normal way (may be forwarded)
         SendLocalInfoMail( 'admin', '[Hamster] Mail failure notice', Err );
         
      end;

   end;
end;

procedure TMailDispatcher.SetPendingOnesTo( MailItem: TMailItem;
                                            MDResult: TMailDeliveryResults );
// sets result of all pending recipients to the given value
var  i: Integer;
begin
   with MailItem.Recipients do begin
      for i := 0 to Count - 1 do begin
          if Items[i].DeliveryResult = mdrPending then begin
             Items[i].DeliveryResult := MDResult;
          end;
      end;
   end;
end;

end.
