// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cLogFileHamster; // Logfile-object and functions.

interface

{$INCLUDE Compiler.inc}

uses SysUtils, SyncObjs, Classes, cLogFile;

type
   TLogFileHamster = class( TLogFile )
      protected
         procedure DoAddLogView( const ID: Integer; const Msg: String ); override;
   end;

   TUidMapper = class
      private
         FRWLock: TMultiReadExclusiveWriteSynchronizer;
         FListUid, FListTid: TList;

      public
         procedure Add( const UID, ThreadID: Integer ); overload;
         procedure Add( const UID: Integer ); overload;
         function  GetUid( const ThreadID: Integer ): Integer; overload;
         function  GetUid: Integer; overload;
         procedure RemoveUid( const UID: Integer );

         constructor Create;
         destructor Destroy; override;
   end;

var
   LogFile  : TLogFileHamster = nil;
   UidMapper: TUidMapper = nil;

function GetUID( const UID: Integer ): Integer;
procedure Log( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
procedure LogConsole( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
procedure SafeLog( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );


implementation

uses Windows, uConst, uHamTools, cHamster;

function GetUID( const UID: Integer ): Integer;
begin
   if UID <> 0 then begin Result := UID; exit end;
   if not Assigned(UidMapper) then begin Result := UID; exit end;
   Result := UidMapper.GetUid;
end;

procedure Log( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
begin
   if Assigned( LogFile ) then begin
      LogFile.Add( ID, OrgMsg, GetUID(UID) );
   end else begin
      WriteLn( 'LogFile used but not initialized yet!' );
      WriteLn( 'Logtype: ', inttohex(ID,4) );
      WriteLn( 'Message: ', OrgMsg );
      Sleep( 5000 );
      Halt( 1 );
   end;
end;

procedure LogConsole( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
var  s: String;
begin
   s := '<CONSOLE> ' + OrgMsg;
   try if Assigned( LogFile ) then Log( ID, s, GetUID(UID) ) except end;
   try if not Assigned( Hamster ) then WriteLn( s ) except end;
end;

procedure SafeLog( const ID: Integer; const OrgMsg: String; const UID: Integer = 0 );
begin
   try
      if Assigned( LogFile ) then begin
         LogFile.Add( ID, OrgMsg, GetUID(UID) );
      end else begin
         WriteLn( 'SafeLog: ', OrgMsg );
      end;
   except end;
end;


{ TLogFileHamster }

procedure TLogFileHamster.DoAddLogView( const ID: Integer; const Msg: String );
begin
   try
      if Assigned( Hamster ) then begin
         if Assigned( Hamster.LiveServer ) then begin
            // add to queue of all live clients
            Hamster.LiveServer.Broadcast(
               LMINF_LOGFILE_ADD, inttohex(ID,4) + '=' + Msg );
         end;
      end;
   except end;
end;


{ TUidMapper }

constructor TUidMapper.Create;
begin
   inherited Create;
   FRWLock  := TMultiReadExclusiveWriteSynchronizer.Create;
   FListUid := TList.Create;
   FListTid := TList.Create;
end;

destructor TUidMapper.Destroy;
begin
   FRWLock.BeginWrite;
   FListUid.Free;
   FListTid.Free;
   FRWLock.Free;
   inherited Destroy;
end;

procedure TUidMapper.Add( const UID, ThreadID: Integer );
var  Index: Integer;
begin
   FRWLock.BeginWrite;
   try
      Index := FListTid.IndexOf( Pointer(ThreadID) );
      if Index >= 0 then begin
         FListUid[Index] := Pointer(UID);
      end else begin
         Index := FListUid.IndexOf( Pointer(UID) );
         if Index >= 0 then begin
            FListTid[Index] := Pointer(ThreadID);
         end else begin
            FListTid.Add( Pointer(ThreadID) );
            FListUid.Add( Pointer(UID     ) );
         end;
      end;
   finally FRWLock.EndWrite end;
end;

procedure TUidMapper.Add( const UID: Integer );
begin
   Add( UID, GetCurrentThreadId );
end;

function TUidMapper.GetUid( const ThreadID: Integer ): Integer;
var  Index: Integer;
begin
   FRWLock.BeginRead;
   try
      Index := FListTid.IndexOf( Pointer(ThreadID) );
      if Index >= 0 then begin
         Result := Integer( FListUid[Index] );
         exit;
      end else begin
         Result := NewUniqueID;
      end;
   finally FRWLock.EndRead end;

   if Index < 0 then Add( Result, ThreadID );
end;

function TUidMapper.GetUid: Integer;
begin
   Result := GetUid( GetCurrentThreadId );
end;

procedure TUidMapper.RemoveUid( const UID: Integer );
var  Index: Integer;
begin
   FRWLock.BeginWrite;
   try
      Index := FListUid.IndexOf( Pointer(UID) );
      if Index >= 0 then begin
         FListTid.Delete( Index );
         FListUid.Delete( Index );
      end;
   finally FRWLock.EndWrite end;
end;

initialization
   LogFile   := nil;
   UidMapper := nil;

finalization
   if Assigned( LogFile ) then FreeAndNil( LogFile );
   if Assigned( UidMapper ) then FreeAndNil( UidMapper );

end.
