// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cLiveServer;

interface

uses SysUtils, Classes, cLiveMsg, cLiveQueue;

type
   TLiveServer = class;

   TLiveClient = class
      protected
         FLiveServer: TLiveServer;
         FLiveQueue : TLiveQueue;
         FSessionKey: String;

      public
         property LiveServer: TLiveServer read FLiveServer;
         property LiveQueue : TLiveQueue  read FLiveQueue;    
         property SessionKey: String      read FSessionKey;

         procedure ReceiveMsg( NewMsg: TLiveMsg );

         constructor Create( ALiveServer: TLiveServer;
                             AQueueSize : Integer;
                             ASessionKey: String ); overload;
         constructor Create( ALiveServer: TLiveServer;
                             AQueueSize : Integer ); overload;
         destructor Destroy; override;
   end;

   TLiveClientQueueHandlerThread = class( TThread ) 
      protected
         FLiveClient: TLiveClient;

         procedure HandleMessage( var LiveMsg: TLiveMsg ); virtual; abstract;
         procedure Execute; override;

      public
         constructor Create( ACreateSuspended: Boolean;
                             ALiveClient     : TLiveClient );
         destructor Destroy; override;
   end;

   TLiveDelayedRequestThread = class( TLiveClientQueueHandlerThread )
      protected
         FLiveServer: TLiveServer;
         procedure HandleMessage( var LiveMsg: TLiveMsg ); override;
         function GetRequestQueue: TLiveQueue;
      public
         property RequestQueue: TLiveQueue read GetRequestQueue;

         constructor Create( ACreateSuspended: Boolean;
                             ALiveServer     : TLiveServer );
         destructor Destroy; override;
   end;

   TLiveServer = class
      protected
         FClients: TThreadList;
         FLiveDelayedRequestThread: TLiveDelayedRequestThread;

      public
         procedure Broadcast( const AMsgType: Word;
                              const AMsgData: String );

         function ReplyFor( const Request: TLiveMsg ): TLiveMsg;

         procedure ClientRequest( const Client  : TLiveClient;
                                  const Request : TLiveMsg ); overload;
         procedure ClientRequest( const Client  : TLiveClient;  
                                  const MsgType : Word;
                                  const MsgData : String ); overload;
         procedure ClientRequest( const Client  : TLiveClient;
                                  const TransferStr: String ); overload;

         procedure DelayedRequest( const AMsg: TLiveMsg );

         procedure ClientConnect   ( Client: TLiveClient );
         procedure ClientDisconnect( Client: TLiveClient ); 

         constructor Create;
         destructor Destroy; override;
   end;

implementation

uses uType, uConst, uConstVar, uVar, uTools, cHamster, cLogFileHamster,
     uHamTools, cHscFiles, cSettings, cHamsterCfg, uRasDyn, cIPAccess,
     cServerRECO, Windows, tBase, tScript, tMaintenance, tTransfer, tReports,
     tUserTasks;

// ---------------------------------------------------------- TLiveClient -----

constructor TLiveClient.Create( ALiveServer: TLiveServer;
                                AQueueSize : Integer;
                                ASessionKey: String );
begin
   inherited Create;
   FLiveServer := ALiveServer;
   FLiveQueue  := TLiveQueue.Create( AQueueSize );
   FSessionKey := ASessionKey;
   if Assigned( FLiveServer ) then FLiveServer.ClientConnect( Self );
end;

constructor TLiveClient.Create( ALiveServer: TLiveServer;
                                AQueueSize : Integer );
begin
   Create( ALiveServer, AQueueSize, '' );
end;

destructor TLiveClient.Destroy;
begin
   if Assigned( FLiveServer ) then FLiveServer.ClientDisconnect( Self );
   FreeAndNil( FLiveQueue );
   inherited Destroy;
end;

procedure TLiveClient.ReceiveMsg( NewMsg: TLiveMsg );
begin
   FLiveQueue.Add( NewMsg );
end;


// ---------------------------------------- TLiveClientQueueHandlerThread -----

procedure TLiveClientQueueHandlerThread.Execute;
var  LiveMsg: TLiveMsg;
begin
   // Note: Be careful with new Log() entries within this procedure, as
   //       they would also be placed in the queue, which could cause new
   //       Log() entries, which could cause new queue entries ...

   while not Terminated do try

      // wait for new queue entries
      FLiveClient.LiveQueue.WaitFor( INFINITE );
      if Terminated then break;

      // handle and remove all queue entries
      while FLiveClient.LiveQueue.Get( LiveMsg ) do try // get/remove

         if Terminated then break;

         if Assigned( LiveMsg ) then try // send

            LiveMsg.SessionKey := FLiveClient.SessionKey;
            HandleMessage( LiveMsg );

         except end;

      finally
         if Assigned( LiveMsg ) then LiveMsg.Free;
      end;

   except
      on E: Exception do begin
         Log( LOGID_ERROR, 'TLiveClientQueueHandlerThread-Exception: ' + E.Message );
         Terminate;
      end;
   end;
end;

constructor TLiveClientQueueHandlerThread.Create( ACreateSuspended: Boolean;
                                                  ALiveClient: TLiveClient );
begin
   inherited Create( True );
   FLiveClient := ALiveClient;
   if not ACreateSuspended then Resume;
end;

destructor TLiveClientQueueHandlerThread.Destroy;
begin
   inherited Destroy;
end;

// -------------------------------------------- TLiveDelayedRequestThread -----

procedure TLiveDelayedRequestThread.HandleMessage( var LiveMsg: TLiveMsg );
begin
   // execute requests, ignore all other types
   if (LiveMsg.MsgType and LMREQ) = LMREQ then begin
      Sleep( 1000 );
      try
         FLiveServer.ClientRequest( nil, LiveMsg );
      finally
         LiveMsg := nil; // freed by .ClientRequest
      end;
   end;
end;

function TLiveDelayedRequestThread.GetRequestQueue: TLiveQueue;
begin
   Result := FLiveClient.LiveQueue;
end;

constructor TLiveDelayedRequestThread.Create( ACreateSuspended: Boolean;
                                              ALiveServer     : TLiveServer );
begin
   inherited Create( True, TLiveClient.Create( ALiveServer, 1024 ) );
   FLiveServer := ALiveServer;
   Priority := tpLower;
   if not ACreateSuspended then Resume;
end;

destructor TLiveDelayedRequestThread.Destroy;
begin
   FLiveClient.Free;
   inherited Destroy;
end;


// ---------------------------------------------------------- TLiveServer -----

constructor TLiveServer.Create;
begin
   inherited Create;
   FClients := TThreadList.Create;
   FLiveDelayedRequestThread := TLiveDelayedRequestThread.Create( False, Self );
end;

destructor TLiveServer.Destroy;
var  i: Integer;
begin
   FLiveDelayedRequestThread.Terminate;
   FLiveDelayedRequestThread.RequestQueue.Add( nil ); // wake up
   FLiveDelayedRequestThread.WaitFor;
   FLiveDelayedRequestThread.Free;

   with FClients.LockList do try
      for i := Count-1 downto 0 do TLiveClient( Items[i] ).Free;
   finally
      FClients.UnlockList;
   end;
   FreeAndNil( FClients );

   inherited Destroy;
end;

procedure TLiveServer.ClientConnect( Client: TLiveClient );
begin
   FClients.Add( Client );
end;

procedure TLiveServer.ClientDisconnect( Client: TLiveClient );
begin
   FClients.Remove( Client );
end;

procedure TLiveServer.Broadcast( const AMsgType: Word;
                                 const AMsgData: String );
var  i: Integer;
begin
   with FClients.LockList do try
      for i := 0 to Count - 1 do try
            TLiveClient( Items[i] ).ReceiveMsg(
               TLiveMsg.Create( AMsgType, AMsgData )
            );
      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'ServerBroadcast-Error: ' + E.Message );
            Delete( i );
            break;
         end;
      end;
   finally
      FClients.UnlockList;
   end;
end;

function TLiveServer.ReplyFor( const Request: TLiveMsg ): TLiveMsg;

   procedure SettingsSet( const Reply: TLiveMsg;
                          const Settings: TSettingsPlain;
                          const SetList: String );
   var  SL: TStringList;
        i, j, ID: Integer;
        Value: String;
   begin
      SL := TStringList.Create;
      try
         SL.Text := SetList;
         for i := 0 to SL.Count - 1 do begin
            j := Pos( '=', SL[i] );
            if j > 0 then begin
               ID := strtointdef( copy( SL[i], 1, j-1 ), 0 );
               Value := copy( SL[i], j+1, MaxInt );
               if ID > 0 then Settings.SetStr( ID, Value );
            end;
         end;
         Settings.Flush;
         Reply.MsgType := LMREP_OK;
      finally SL.Free end;
   end;

   procedure GrpSrvSettingsSet( const Reply: TLiveMsg;
                                const GrpHdl: Integer;
                                const Server: String;
                                const SetList: String );
   var  SL: TStringList;
        i, j, ID: Integer;
        Value: String;
   begin
      SL := TStringList.Create;
      try
         SL.Text := SetList;
         for i := 0 to SL.Count - 1 do begin
            j := Pos( '=', SL[i] );
            if j > 0 then begin
               ID := strtointdef( copy( SL[i], 1, j-1 ), 0 );
               Value := copy( SL[i], j+1, MaxInt );
               if ID > 0 then Hamster.ArticleBase.ServerSetStr( GrpHdl, Server, ID, Value );
            end;
         end;
         Reply.MsgType := LMREP_OK;
      finally SL.Free end;
   end;

   procedure ServerList( const Reply: TLiveMsg;
                         const Servers: TConfigServers );
   var  s: String;
        i: Integer;
   begin
      Hamster.Config.BeginRead;
      try
         s := '';
         for i:=0 to Servers.Count-1 do begin
            s := s + Servers.AliasName[i] + ',' + Servers.SrvPort[i] + CRLF
         end;
      finally Hamster.Config.EndRead end;

      Reply.MsgType := LMREP_OK;
      Reply.MsgData := s;
   end;

   procedure PasswordGet( Identifier: String;
                          const Reply: TLiveMsg );
   var  b2: Boolean;
        nam: String;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      if Hamster.Passwords.PasswordInfo( Identifier, nam, b2 ) then begin
         Reply.MsgData := DQuoteStr( Identifier ) + TAB
                        + DQuoteStr( nam ) + TAB
                        + iif( b2, '1', '0' );
         Reply.MsgType := LMREP_OK;
         exit;
      end;
   end;

   procedure PasswordList( const Reply: TLiveMsg );
   var  s, id, nam: String;
        i, n: Integer;
        b1, b2: Boolean;
        SL: TStringList;
   begin
      s := '';

      // available RAS-connections
      SL := TStringList.Create;
      try
         if RasDynEnumPhonebookEntries( SL ) then begin
            for i := 0 to SL.Count - 1 do begin
               id := SL[i];
               b1 := Hamster.Passwords.PasswordInfo( id, nam, b2 );
               s := s + DQuoteStr( id ) + TAB
                      + DQuoteStr( 'RAS: ' + id ) + TAB
                      + iif( b1 and b2, '1', '0' ) + CRLF;
            end;
         end;
      finally SL.Free end;

      // multi purpose passwords
      n := 30; // former version's default
      i :=  1;
      while i <= n do begin
         id := '$' + inttostr( i );
         b1 := Hamster.Passwords.PasswordInfo( id, nam, b2 );
         if b1 or (i<=30) then begin
            s := s + id + TAB
                   + DQuoteStr( id + ': ' + nam ) + TAB
                   + iif( b1 and b2, '1', '0' ) + CRLF;
            if b1 and b2 and ( i + 10 > n ) then n := i + 10;
         end;
         inc( i );
         if i > 100 then break;
      end;

      Reply.MsgData := s;
      Reply.MsgType := LMREP_OK;
   end;

   procedure AccountsGet( const Reply: TLiveMsg;
                          const Account: String );
   var  s, pval: String;
        ID, uid: Integer;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      uid := Hamster.Accounts.UserIDOf( Account );
      if uid = ACTID_INVALID then exit;

      s := '';

      with SettingsDef_Accounts do begin
         for ID := MinID to MaxID do begin
            if IDExists( ID ) then begin
               pval := Hamster.Accounts.Value[ uid, ID ];
               if ID = apPassword then begin
                  s := s + inttostr( ID ) + '='
                         + iif( (pval<>ACTPW_NOTNEEDED) and
                                (pval<>ACTPW_NOACCESS ), '1', '0' )
                         + CRLF;
               end else begin
                  s := s + inttostr( ID ) + '=' + pval + CRLF;
               end;
            end;
         end;
      end;

      Reply.MsgData := s;
      Reply.MsgType := LMREP_OK;
   end;

   procedure AccountsSet( const Reply: TLiveMsg;
                          const Account, SetList: String );
   var  SL: TStringList;
        i, j, ID, uid: Integer;
        Value: String;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      uid := Hamster.Accounts.UserIDOf( Account );
      if uid = ACTID_INVALID then exit;

      SL := TStringList.Create;
      try
         SL.Text := SetList;
         for i := 0 to SL.Count - 1 do begin
            j := Pos( '=', SL[i] );
            if j > 0 then begin
               ID := strtointdef( copy( SL[i], 1, j-1 ), 0 );
               Value := copy( SL[i], j+1, MaxInt );
               if ID > 0 then Hamster.Accounts.Value[ uid, ID ] := Value;
            end;
         end;
         Reply.MsgType := LMREP_OK;
      finally SL.Free end;
   end;

   function FilePathOf( const Name: String ): String;
   var  SR: TSearchRec;
   begin
      Result := '';
      
      if ( CompareText( Name, CFGFILE_SCORES     ) = 0 ) or
         ( CompareText( Name, CFGFILE_MAILFILT   ) = 0 ) or
         ( CompareText( Name, CFGFILE_MAILTRAP   ) = 0 ) or
         ( CompareText( Name, CFGFILE_IPACCESS   ) = 0 ) or
         ( CompareText( Name, CFGFILE_RCPROFILES ) = 0 ) or
         ( CompareText( Name, CFGFILE_HSCACTIONS ) = 0 ) or
         ( CompareText( Name, CFGFILE_SMTPROUTER ) = 0 ) or
         ( CompareText( Name, CFGFILE_SCHEDULER  ) = 0 )
         then begin
         Result := AppSettings.GetStr(asPathBase);
         exit;
      end;

      if ( CompareText( Name, GRPFILE_KILLSLOG ) = 0 ) then begin
         Result := AppSettings.GetStr(asPathGroups);
         exit;
      end;

      if Pos( LOGFILE_EXTENSION, Name ) > 0 then begin // .log files in logs dir.
         if SysUtils.FindFirst( AppSettings.GetStr(asPathLogs) + '*' + LOGFILE_EXTENSION,
                                faAnyFile, SR ) = 0 then begin
            repeat
               if (SR.Attr and faDirectory) = 0 then begin
                  if CompareText( SR.Name, Name ) = 0 then begin
                     Result := AppSettings.GetStr(asPathLogs);
                     break;
                  end;
               end;
            until SysUtils.FindNext( SR ) <> 0;
            SysUtils.FindClose( SR );
         end;
      end;

      if ( CompareText( Name, REPFILE_REPORTTITLES ) = 0 ) then begin
         Result := AppSettings.GetStr(asPathReports);
         exit;
      end;

      if Pos( REPFILE_EXTENSION, Name ) > 0 then begin // .htm files in reports dir.
         if SysUtils.FindFirst( AppSettings.GetStr(asPathReports) + '*' + REPFILE_EXTENSION,
                                faAnyFile, SR ) = 0 then begin
            repeat
               if (SR.Attr and faDirectory) = 0 then begin
                  if CompareText( SR.Name, Name ) = 0 then begin
                     Result := AppSettings.GetStr(asPathReports);
                     break;
                  end;
               end;
            until SysUtils.FindNext( SR ) <> 0;
            SysUtils.FindClose( SR );
         end;
      end;
   end;

   procedure FileGet( const Request, Reply: TLiveMsg );
   var  Data, Filepath, Filename: String;
        SL: TStringList;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      Data          := Request.MsgData;
      Filename      := UnDQuoteStr( NextTopLine( Data ) );
      Filepath      := FilePathOf( Filename );
      if Filepath = '' then exit;

      if Filename = CFGFILE_SCHEDULER then begin
         Reply.MsgData := Hamster.Scheduler.GetDefText;
         Reply.MsgType := LMREP_OK;
         exit;
      end;

      SL := TStringList.Create;
      try
         if Filename = CFGFILE_SCHEDULER then begin
            SL.Text := Hamster.Scheduler.GetDefText;
         end else begin
            if FileExists( Filepath + Filename ) then begin
               SL.LoadFromFile( Filepath + Filename );
            end;
         end;
         Reply.MsgData := SL.Text;
         Reply.MsgType := LMREP_OK;
      finally SL.Free end;
   end;

   procedure FileSet( const Request, Reply: TLiveMsg );
   var  Data, Filepath, Filename: String;
        SL: TStringList;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      Data          := Request.MsgData;
      Filename      := UnDQuoteStr( NextTopLine( Data ) );
      Filepath      := FilePathOf( Filename );
      if Filepath = '' then exit;

      if Filename = CFGFILE_SCHEDULER then begin
         Hamster.Scheduler.SetDefText( Data );
         Reply.MsgType := LMREP_OK;
         exit;
      end;

      SL := TStringList.Create;
      try
         SL.Text := Data;
         SL.SaveToFile( Filepath + Filename );
         Reply.MsgType := LMREP_OK;

         if CompareText( Filename, CFGFILE_IPACCESS   ) = 0 then
            Hamster.IPAccess.WantReload := True;
         if CompareText( Filename, CFGFILE_HSCACTIONS ) = 0 then
            Hamster.HscActions.Reload;
         if CompareText( Filename, CFGFILE_SMTPROUTER ) = 0 then
            Hamster.SmtpRouter.Refresh;
         if CompareText( Filename, CFGFILE_MAILTRAP ) = 0 then
            Hamster.MailTrap.Reload;

      finally SL.Free end;
   end;

   procedure ReportsList( const Request, Reply: TLiveMsg );
   var  SL: TStringList;
        SR: TSearchRec;
   begin
      Reply.MsgType := LMREP_UNKNOWN_ITEM;
      SL := TStringList.Create;
      try
         if SysUtils.FindFirst( AppSettings.GetStr(asPathReports) + '*.htm',
                                faAnyFile, SR ) = 0 then begin
            repeat
               if (SR.Attr and faDirectory) = 0 then begin
                  SL.Add( SR.Name );
               end;
            until SysUtils.FindNext( SR ) <> 0;
            SysUtils.FindClose( SR );
         end;
         Reply.MsgData := SL.Text;
         Reply.MsgType := LMREP_OK;
      finally SL.Free end;
   end;

var  Reply: TLiveMsg;
     st: TServerTypes;
     SL: TStringList;
     TL: TList;
     i, i1, i2, GrpHdl: Integer;
     PathSub, Filename, ScriptParams, Server, Group, Data: String;
     s, s1, s2, s3, s4, s5, s6: String;
     HscFile: THscFile;
     ScriptThread: TThreadExecuteScript;
     Ok: Boolean;
begin
   Reply := nil;

   try
      Reply := TLiveMsg.Create( LMREP_FAILED, '' );

      try
         case Request.MsgType of

            LMREQ_APP_SETTING_GET: begin // (none)
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := AppSettings.GetAll;
            end;
            LMREQ_APP_SETTING_SET: begin // *( id '=' value CRLF )
               SettingsSet( Reply, AppSettings, Request.MsgData );
            end;

            LMREQ_HAM_SETTING_GET: begin // (none)
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := Hamster.Config.Settings.GetAll;
            end;
            LMREQ_HAM_SETTING_SET: begin // *( id '=' value CRLF )
               SettingsSet( Reply, Hamster.Config.Settings, Request.MsgData );
            end;

            LMREQ_LOCALSERVER_CONTROL: begin // ("*"/servertype) CRLF servercontrol
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               i2   := strtointdef( NextTopLine( Data ), -1 );
               if s1 = '*' then begin
                  Reply.MsgData := '';
                  for st := stNNTP to stRECO do begin
                     Reply.MsgData := Reply.MsgData + inttostr( Hamster.ServerControl( st, TServerControls(i2) ) ) + CRLF;
                  end;
               end else begin
                  i1 := strtointdef( s1, -1 );
                  Reply.MsgData := inttostr( Hamster.ServerControl( TServerTypes(i1), TServerControls(i2) ) );
               end;
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_LOCALSERVER_CLIENTS: begin // (none)  
               Reply.MsgData := Hamster.ServerClients;
               Reply.MsgType := LMREP_OK;
            end;

            LMREQ_COUNTERS_LIST: begin // (none)
               Reply.MsgData := CountersList;
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_COUNTERS_RESET: begin // (none)
               CountersReset;
               Reply.MsgType := LMREP_OK;
            end;

            LMREQ_TASKS_LIST: begin // (none)  
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := Hamster.ActiveThreads.ListAll;
            end;
            LMREQ_TASKS_STOPALL: begin // (none)
               Data := Request.MsgData;
               i1 := strtointdef( NextTopLine(Data), -1 );
               if i1 = -1 then
                  Hamster.ActiveThreads.StopAll
               else
                  Hamster.ActiveThreads.StopAllByType( TActiveThreadTypes(i1) );
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_TASKS_STOPBYID: begin // id                          
               Data := Request.MsgData;
               Reply.MsgType := LMREP_OK;
               Hamster.ActiveThreads.StopByID( strtointdef( NextTopLine(Data), 0 ) );
            end;

            LMREQ_LOGFILE_LIST: begin // size
               Data := Request.MsgData;
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := LogFile.LastLines( strtointdef( NextTopLine(Data), 8 ) );
            end;
            LMREQ_LOGFILE_ROTATE: begin // (none)
               Reply.MsgType := LMREP_OK;
               LogFile.RotateLog;
            end;
            LMREQ_LOGFILE_LISTERRORS: begin // (none)
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := LogFile.ErrorBuffer;
            end;
            LMREQ_LOGFILE_FIND: begin // Pattern CRLF Regex 0/1 CRLF FromAgo
                                      // CRLF TilAgo CRLF MaxLines
               Data := Request.MsgData;
               s1 := NextTopLine( Data );
               s2 := NextTopLine( Data );
               s3 := NextTopLine( Data );
               s4 := NextTopLine( Data );
               s5 := NextTopLine( Data );
               s := LogFile.FindLines( s1, s2 = '1',
                                       StrToIntDef( s3, 0 ),
                                       StrToIntDef( s4, 0 ),
                                       StrToIntDef( s5, 100 ) );
               Reply.MsgType := LMREP_OK;
               Reply.MsgData := s;
            end;

            LMREQ_GROUPS_LIST: begin // [ selection ]
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               Reply.MsgData := Hamster.Config.Newsgroups.GetList( s1 );
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_GROUPS_ADD: begin // group
               Data := Request.MsgData;
               if Hamster.Config.Newsgroups.Add( NextTopLine(Data) ) then
                  Reply.MsgType := LMREP_OK;
            end;

            LMREQ_GROUPS_DELETE: begin // group
               Data := Request.MsgData;
               if Hamster.Config.Newsgroups.Del( NextTopLine(Data) ) then
                  Reply.MsgType := LMREP_OK;
            end;
            LMREQ_GROUPS_SETTING_GET: begin // Group
               Data := Request.MsgData;
               Group  := NextTopLine(Data); 
               GrpHdl := Hamster.ArticleBase.Open( Group );
               if GrpHdl >= 0 then try
                  Reply.MsgType := LMREP_OK;
                  Reply.MsgData := Hamster.ArticleBase.Settings( GrpHdl ).GetAll;
               finally Hamster.ArticleBase.Close( GrpHdl ) end;
            end;
            LMREQ_GROUPS_SETTING_SET: begin // group CRLF *( id '=' value CRLF )
               Data   := Request.MsgData;
               Group  := NextTopLine( Data );
               GrpHdl := Hamster.ArticleBase.Open( Group );
               if GrpHdl >= 0 then try
                  SettingsSet( Reply, Hamster.ArticleBase.Settings( GrpHdl ), Data );
               finally Hamster.ArticleBase.Close( GrpHdl ) end;
            end;

            LMREQ_PULLS_LIST: begin // (none)
               SL := TStringList.Create;
               try
                  Hamster.Config.BeginRead;
                  try
                     for i:=0 to Hamster.Config.NewsPulls.Count-1 do
                        SL.Add( Hamster.Config.NewsPulls.Group[i] + TAB
                              + Hamster.Config.NewsPulls.Server[i] );
                  finally Hamster.Config.EndRead end;
                  Reply.MsgType := LMREP_OK;
                  Reply.MsgData := SL.Text;
               finally SL.Free end;
            end;
            LMREQ_PULLS_ADD: begin // group TAB server
               Data := Request.MsgData;
               s := NextTopLine( Data );
               i := Pos( TAB, s );
               if i > 0 then begin
                  Group  := copy( s, 1, i-1 );
                  Server := copy( s, i+1, MaxInt );
                  if Hamster.Config.NntpServers.IndexOfAlias(Server) >= 0 then begin
                     if Hamster.Config.NewsPulls.Add( Server, Group ) then
                        Reply.MsgType := LMREP_OK;
                  end;
               end;
            end;
            LMREQ_PULLS_DELETE: begin // group TAB server
               Data := Request.MsgData;
               s := NextTopLine( Data );
               i := Pos( TAB, s );
               if i > 0 then begin
                  Group  := copy( s, 1, i-1 );
                  Server := copy( s, i+1, MaxInt );
                  if Hamster.Config.NntpServers.IndexOfAlias(Server) >= 0 then begin
                     if Hamster.Config.NewsPulls.Del( Server, Group ) then
                        Reply.MsgType := LMREP_OK;
                  end;
               end;
            end;
            LMREQ_PULLS_SETTING_GET: begin // group TAB server
               Data := Request.MsgData;
               s := NextTopLine( Data );
               i := Pos( TAB, s );
               if i > 0 then begin
                  Group  := copy( s, 1, i-1 );
                  Server := copy( s, i+1, MaxInt );
                  GrpHdl := Hamster.ArticleBase.Open( Group );
                  if GrpHdl >= 0 then try
                     Reply.MsgType := LMREP_OK;
                     Reply.MsgData := Hamster.ArticleBase.ServerSettingsAll( GrpHdl, Server );
                  finally Hamster.ArticleBase.Close( GrpHdl ) end;
               end;
            end;
            LMREQ_PULLS_SETTING_SET: begin // group TAB server CRLF *( id '=' value CRLF )
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               i := Pos( TAB, s1 );
               if i > 0 then begin
                  Group  := copy( s1, 1, i-1 );
                  Server := copy( s1, i+1, MaxInt );
                  GrpHdl := Hamster.ArticleBase.Open( Group );
                  if GrpHdl >= 0 then try
                     GrpSrvSettingsSet( Reply, GrpHdl, Server, Data );
                  finally Hamster.ArticleBase.Close( GrpHdl ) end;
               end;
            end;

            LMREQ_SRVNNTP_LIST: // (none)
               ServerList( Reply, Hamster.Config.NntpServers );
            LMREQ_SRVPOP3_LIST: // (none)
               ServerList( Reply, Hamster.Config.Pop3Servers );
            LMREQ_SRVSMTP_LIST: // (none)
               ServerList( Reply, Hamster.Config.SmtpServers );

            LMREQ_SRVNNTP_ADD: begin // servername ["," port]
               Data := Request.MsgData;
               if Hamster.Config.NntpServers.SrvAdd( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVPOP3_ADD: begin // servername ["," port]
               Data := Request.MsgData;
               if Hamster.Config.Pop3Servers.SrvAdd( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVSMTP_ADD: begin // servername ["," port]
               Data := Request.MsgData;
               if Hamster.Config.SmtpServers.SrvAdd( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;

            LMREQ_SRVNNTP_DEL: begin // servername "," port
               Data := Request.MsgData;
               if Hamster.Config.NntpServers.SrvDel( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVPOP3_DEL: begin // servername "," port
               Data := Request.MsgData;
               if Hamster.Config.Pop3Servers.SrvDel( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVSMTP_DEL: begin // servername "," port
               Data := Request.MsgData;
               if Hamster.Config.SmtpServers.SrvDel( NextTopLine(Data) ) then Reply.MsgType := LMREP_OK;
            end;

            LMREQ_SRVNNTP_SETTING_GET: begin // Server
               Data := Request.MsgData;
               Reply.MsgData := Hamster.Config.NntpServers.Settings( NextTopLine(Data) ).GetAll;
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVPOP3_SETTING_GET: begin // Server
               Data := Request.MsgData;
               Reply.MsgData := Hamster.Config.Pop3Servers.Settings( NextTopLine(Data) ).GetAll;
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_SRVSMTP_SETTING_GET: begin // Server
               Data := Request.MsgData;
               Reply.MsgData := Hamster.Config.SmtpServers.Settings( NextTopLine(Data) ).GetAll;
               Reply.MsgType := LMREP_OK;
            end;

            LMREQ_SRVNNTP_SETTING_SET: begin // server CRLF *( id '=' value CRLF )
               Data   := Request.MsgData;
               Server := NextTopLine( Data );
               SettingsSet( Reply, Hamster.Config.NntpServers.Settings( Server ), Data );
            end;
            LMREQ_SRVPOP3_SETTING_SET: begin // server CRLF *( id '=' value CRLF )
               Data   := Request.MsgData;
               Server := NextTopLine( Data );
               SettingsSet( Reply, Hamster.Config.Pop3Servers.Settings( Server ), Data );
            end;
            LMREQ_SRVSMTP_SETTING_SET: begin // server CRLF *( id '=' value CRLF )
               Data   := Request.MsgData;
               Server := NextTopLine( Data );
               SettingsSet( Reply, Hamster.Config.SmtpServers.Settings( Server ), Data );
            end;

            LMREQ_SCRIPT_DIR: begin // (none)
               Hamster.HscFiles.Refresh;
               SL := TStringList.Create;
               try
                  Reply.MsgType := LMREP_OK;
                  Hamster.HscFiles.List( hfsAnyFile, hlftFileInfo, SL );
                  Reply.MsgData := SL.Text;
               finally SL.Free end;
            end;
            LMREQ_SCRIPT_GET: begin // filename
               Data := Request.MsgData;
               if HscFileCheckSplitName( UnDQuoteStr( NextTopLine(Data) ),
                                         PathSub, Filename ) then begin
                  HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsAnyFile );
                  if Assigned( HscFile ) then begin
                     SL := TStringList.Create;
                     try
                        try
                           Hamster.HscFiles.Load( HscFile, SL );
                           Reply.MsgType := LMREP_OK;
                           Reply.MsgData := SL.Text;
                        except on E: Exception do Log( LOGID_WARN, E.Message ) end;
                     finally SL.Free end;
                  end;
               end;
            end;
            LMREQ_SCRIPT_PUT: begin // filename CRLF scriptlines...
               SL := TStringList.Create;
               try
                  SL.Text := Request.MsgData; // 1st line=filename
                  if SL.Count >= 2 then begin
                     if HscFileCheckSplitName( UnDQuoteStr( SL[0] ),
                                               PathSub, Filename ) then begin
                        SL.Delete( 0 ); // remove filename, rest is script
                        try
                           Hamster.HscFiles.Save( PathSub, Filename, SL );
                           Reply.MsgType := LMREP_OK;
                        except on E: Exception do Log( LOGID_WARN, E.Message ) end;
                     end;
                  end;
               finally SL.Free end;
            end;
            LMREQ_SCRIPT_DELETE: begin // filename
               Data := Request.MsgData;
               if HscFileCheckSplitName( UnDQuoteStr( NextTopLine(Data) ),
                                         PathSub, Filename ) then begin
                  HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsAnyFile );
                  if Assigned( HscFile ) then begin
                     if Hamster.HscFiles.Delete( HscFile ) then begin
                        Reply.MsgType := LMREP_OK;
                     end;
                  end;
               end;
            end;
            LMREQ_SCRIPT_START: begin // filename CRLF params...
               SL := TStringList.Create;
               try
                  SL.Text := Request.MsgData; // 1st line=filename, 2nd=params
                  if SL.Count >= 1 then begin
                     if HscFileCheckSplitName( UnDQuoteStr( SL[0] ),
                                               PathSub, Filename ) then begin
                        HscFile := Hamster.HscFiles.Find( PathSub, Filename, hfsAnyFile );
                        if Assigned( HscFile ) then begin
                           ScriptParams := '';
                           if SL.Count >= 2 then begin
                              ScriptParams := StringReplace( SL[1], #9, #13#10, [rfReplaceAll] );
                           end;
                           i := Hamster.HscFiles.Start(
                                   HscFile, ScriptParams, False, ScriptThread,
                                   tftFreeOnTerminate, False
                                );
                           if i = 0 then begin
                              Reply.MsgType := LMREP_OK;
                              Reply.MsgData := inttostr( ScriptThread.UniqueId );
                           end;
                        end;
                     end;
                  end;
               finally SL.Free end;
            end;

            LMREQ_PASSWORD_LIST: // (none)
               PasswordList( Reply );
            LMREQ_PASSWORD_GET: begin // identifier
               Data := Request.MsgData;
               PasswordGet( NextTopLine(Data), Reply );
            end;
            LMREQ_PASSWORD_SET: begin // identifier CRLF username CRLF password 
               Data := Request.MsgData;
               s1 := NextTopLine( Data );
               s2 := NextTopLine( Data );
               s3 := NextTopLine( Data );
               Hamster.Passwords.SavePassword( True, s1, s2, s3 );
               Reply.MsgType := LMREP_OK;
            end;

            LMREQ_ACCOUNTS_LIST: begin // (none)
               Reply.MsgData := Hamster.Accounts.ListAll;
               Reply.MsgType := LMREP_OK;
            end;
            LMREQ_ACCOUNTS_ADD: begin // account [ CRLF template-account ]
               Data := Request.MsgData;
               s1 := NextTopLine( Data ); // new
               s2 := NextTopLine( Data ); // template
               i := Hamster.Accounts.UserIDOf( s1 );
               if i = ACTID_INVALID then begin
                  i := Hamster.Accounts.UserIDOf( s2 );
                  i := Hamster.Accounts.AddClone( s1, i );
                  if i <> ACTID_INVALID then begin
                     Hamster.Accounts.Value[ i, apUsername ] := s1;
                     Hamster.Accounts.Value[ i, apFullname ] := s1;
                     Reply.MsgType := LMREP_OK;
                  end;
               end;
            end;
            LMREQ_ACCOUNTS_DEL: begin // account
               Data := Request.MsgData;
               Reply.MsgType := LMREP_UNKNOWN_ITEM;
               i := Hamster.Accounts.UserIDOf( NextTopLine(Data) );
               if i <> ACTID_INVALID then begin
                  Hamster.Accounts.Delete( i );
                  Reply.MsgType := LMREP_OK;
               end;
            end;
            LMREQ_ACCOUNTS_GET: begin // account
               Data := Request.MsgData;
               AccountsGet( Reply, NextTopLine(Data) );
            end;
            LMREQ_ACCOUNTS_SET: begin // account CRLF *( id '=' value CRLF )
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               AccountsSet( Reply, s1, Data );
            end;

            LMREQ_MAILLISTS_LIST: begin // (none)
               SL := TStringList.Create;
               try
                  Reply.MsgType := LMREP_OK;
                  Hamster.MailLists.MailListDir( SL );
                  Reply.MsgData := SL.Text;
               finally SL.Free end;
            end;
            LMREQ_MAILLISTS_DEL: begin // listname
               Data := Request.MsgData;
               Reply.MsgType := LMREP_UNKNOWN_ITEM;
               if Hamster.MailLists.MailListDel( NextTopLine(Data) ) then begin
                  Reply.MsgType := LMREP_OK;
               end;
            end;
            LMREQ_MAILLISTS_GET: begin // listname
               Data := Request.MsgData;
               Reply.MsgType := LMREP_UNKNOWN_ITEM;
               SL := TStringList.Create;
               try
                  if Hamster.MailLists.MailListGet( NextTopLine(Data), SL ) then begin
                     Reply.MsgType := LMREP_OK;
                     Reply.MsgData := SL.Text;
                  end;
               finally SL.Free end;
            end;
            LMREQ_MAILLISTS_SET: begin // listname CRLF *( line CRLF )
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               SL := TStringList.Create;
               try
                  SL.Text := Data;
                  if Hamster.MailLists.MailListSet( s1, SL ) then begin
                     Reply.MsgType := LMREP_OK;
                  end;
               finally SL.Free end;
            end;

            LMREQ_USERTASKS_LIST: begin // (none)
               SL := TStringList.Create;
               try
                  Reply.MsgType := LMREP_OK;
                  TUserTaskThread.UserTasks_Lst( SL );
                  Reply.MsgData := SL.Text;
               finally SL.Free end;
            end;
            LMREQ_USERTASKS_DEL: begin // name
               Data := Request.MsgData;
               Reply.MsgType := LMREP_UNKNOWN_ITEM;
               if TUserTaskThread.UserTasks_Del( NextTopLine(Data) ) then begin
                  Reply.MsgType := LMREP_OK;
               end;
            end;
            LMREQ_USERTASKS_GET: begin // name
               Data := Request.MsgData;
               Reply.MsgType := LMREP_UNKNOWN_ITEM;
               SL := TStringList.Create;
               try
                  if TUserTaskThread.UserTasks_Get( NextTopLine(Data), SL ) then begin
                     Reply.MsgType := LMREP_OK;
                     Reply.MsgData := SL.Text;
                  end;
               finally SL.Free end;
            end;
            LMREQ_USERTASKS_SET: begin // name CRLF *( line CRLF )
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               SL := TStringList.Create;
               try
                  SL.Text := Data;
                  if TUserTaskThread.UserTasks_Set( s1, SL ) then begin
                     Reply.MsgType := LMREP_OK;
                  end;
               finally SL.Free end;
            end;

            LMREQ_RCPROFILES_LIST: begin // none
               with TReCoProfiles.Create do try
                  Reply.MsgData := GetList;
                  Reply.MsgType := LMREP_OK;
               finally Free end;
            end;

            LMREQ_FILE_GET: // filename
               FileGet( Request, Reply );
            LMREQ_FILE_SET: // filename CRLF *( line CRLF )
               FileSet( Request, Reply );

            LMREQ_REPORTS_LIST: // (none)
               ReportsList( Request, Reply );

            LMREQ_RUN_PURGE: begin // bits CRLF [group]
               Data := Request.MsgData;
               i1 := strtointdef( NextTopLine( Data ), 0 );
               s2 := NextTopLine( Data );
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadPurge.Create( i1, s2, tftFreeOnTerminate ) do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;
            end;
            LMREQ_RUN_REBUILDHISTORY: // (none)
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadHistoryRebuild.Create do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;
            LMREQ_RUN_REBUILDLISTS: // (none)
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadRebuildGlobalLists.Create( tftFreeOnTerminate ) do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;
            LMREQ_RUN_STATISTICS: // (none)
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadStatistics.Create( tftFreeOnTerminate ) do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;
            LMREQ_RUN_AUTOUNSUBSCRIBE: // (none)
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadAutoUnsubscribe.Create( tftFreeOnTerminate ) do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;
            LMREQ_RUN_REPORTS: // (none)
               if Hamster.ActiveThreads.CountActiveTasks <= 0 then begin
                  with TThreadReports.Create( tftFreeOnTerminate ) do try
                     Reply.MsgData := inttostr( UniqueId );
                     Reply.MsgType := LMREP_OK;
                  finally Resume end;
               end;

            LMREQ_RUN_RASDIAL: begin // name [ CRLF user [ CRLF pass ] ]
               Data := Request.MsgData;
               s1   := NextTopLine( Data );
               s2   := NextTopLine( Data );
               s3   := NextTopLine( Data );
               Ok := True;
               if ( copy(s2,1,1) = '$' ) and ( s3 = '' ) then begin
                  s := s2;
                  if not Hamster.Passwords.UsePassword( s, s2, s3 ) then Ok := False;
               end;
               if Ok then begin
                  i1 := Hamster.RasDialer.Dial( s1, s2, s3 );
                  if i1 = 0 then Reply.MsgType := LMREP_OK;
               end;
            end;
            LMREQ_RUN_RASHANGUP: begin // (none)
               if Hamster.RasDialer.HangUp = 0 then Reply.MsgType := LMREP_OK;
            end;

            LMREQ_RUN_FETCHMAIL: begin // srv [ CRLF user [ CRLF pass [ ...
               Data := Request.MsgData;
               s1   := NextTopLine( Data ); // srv
               s2   := NextTopLine( Data ); // user
               s3   := NextTopLine( Data ); // pass
               s4   := NextTopLine( Data ); // destuser
               s5   := NextTopLine( Data ); // filtsect
               s6   := NextTopLine( Data ); // leave
               with TThreadPop3Fetch.Create( s1, '', s2, s3, s4, s5, s6 ) do try
                  Reply.MsgData := inttostr( UniqueId );
                  Reply.MsgType := LMREP_OK;
               finally Resume end;
            end;
            LMREQ_RUN_SENDMAIL: begin // srv [ CRLF user [ CRLF pass [ ...
               Data := Request.MsgData;
               s1   := NextTopLine( Data ); // srv
               s2   := NextTopLine( Data ); // user
               s3   := NextTopLine( Data ); // pass
               s4   := NextTopLine( Data ); // fromsel
               s5   := NextTopLine( Data ); // tosel
               with TThreadSmtpSend.Create( False, s1, '', s2, s3, s4, s5 ) do try
                  Reply.MsgData := inttostr( UniqueId );
                  Reply.MsgType := LMREP_OK;
               finally Resume end;
            end;
            LMREQ_RUN_SENDMAILMX: begin // none
               with TThreadSmtpSend.Create( True, '', '', '', '', '', '' ) do try
                  Reply.MsgData := inttostr( UniqueId );
                  Reply.MsgType := LMREP_OK;
               finally Resume end;
            end;

            LMREQ_RUN_NEWSJOBS: begin // srv
               Data := Request.MsgData;
               s1   := NextTopLine( Data ); // srv
               Hamster.NewsJobs.AddPostDef( s1 );
               Hamster.NewsJobs.AddPullDef( s1 );
               TL := TList.Create;
               try
                  Hamster.NewsJobs.StartThreads( s1, nil );
                  Reply.MsgData := '';
                  for i1 := 0 to TL.Count-1 do begin
                     Reply.MsgData := Reply.MsgData
                                    + iif( length(Reply.MsgData) > 0, TAB, '' )
                                    + inttostr( Integer( TL[i1] ) );
                  end;
                  Reply.MsgType := LMREP_OK;
               finally TL.Free end;
            end;

            LMREQ_RUN_USERTASK: begin // name
               Data := Request.MsgData;
               s1 := NextTopLine( Data ); // name
               with TUserTaskThread.Create( s1, False ) do try
                  Reply.MsgData := inttostr( UniqueId );
                  Reply.MsgType := LMREP_OK;
               finally Resume end;
            end;

            else begin
               Reply.MsgType := LMREP_UNKNOWN_MESSAGE;
               Reply.MsgData := 'Unknown message ($'
                              + inttohex( Request.MsgType, 4 ) + ')';
            end;

         end;

      except
         on E:Exception do begin
            Log( LOGID_ERROR, 'TLiveServer.ReplyFor: ' + E.Message
                            + ' .MsgType=' + inttohex(Request.MsgType,1)
                            + ' .MsgData="' + Request.MsgData + '"' );
            Reply.MsgType := LMREP_EXCEPTION;
            Reply.MsgData := 'Exception: ' + E.Message;
         end;
      end;

   finally
      Request.Free;
      Result := Reply;
   end;
end;

procedure TLiveServer.ClientRequest( const Client : TLiveClient;
                                     const Request: TLiveMsg );
var  Reply: TLiveMsg;
begin
   try
      Reply := ReplyFor( Request );
      if Assigned( Reply ) then begin
         if Assigned( Client ) then Client.ReceiveMsg( Reply )
                               else Reply.Free;
      end;

   except
      on E:Exception do Log( LOGID_ERROR,
                             'TLiveServer.ClientRequest: ' + E.Message );
   end;
end;

procedure TLiveServer.ClientRequest( const Client     : TLiveClient;
                                     const TransferStr: String );
begin
   ClientRequest( Client, TLiveMsg.Create( TransferStr ) );
end;

procedure TLiveServer.ClientRequest( const Client: TLiveClient;
                                     const MsgType: Word;
                                     const MsgData: String );
begin
   ClientRequest( Client, TLiveMsg.Create( MsgType, MsgData ) );
end;

procedure TLiveServer.DelayedRequest(const AMsg: TLiveMsg);
begin
   FLiveDelayedRequestThread.RequestQueue.Add( AMsg );
end;

end.
