// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cIndexRc;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, uTools, cFileStream64;

type
  TIndexedRecords = class
    private
      CritSect  : TRTLCriticalSection;
      FMemPtr   : PChar;
      FMemSize  : LongInt;
      FUseSize  : LongInt;
      FRecSize  : LongInt;
      FSorted   : Boolean;
      FChanged  : Boolean;
      FFilename : String;
      FSidx     : TFileStream64;
      FFileMode : Word;

      procedure QuickSort( CompareOffset, L, R: Integer );
      procedure SetSorted( NewSorted: Boolean );

      procedure RecIns( Index: Integer );
      procedure RecInsert( Index: Integer; const Data );

    protected
      procedure RecSet( Index: Integer; const Data );
      procedure RecKeyFindPos( const Data;
                               KeySize: LongInt;
                               WantInsertPos: Boolean;
                               out Index: Integer );
      function  GetCount: LongInt;
      procedure RecGet( Index: Integer; var Data );
      function  RecKeyInsPosOf( KeySize: LongInt; const Data ): Integer;

    public
      property  Filename: String read FFilename;
      property  Count  : LongInt read GetCount;
      property  Sorted : Boolean read FSorted  write SetSorted;
      property  Changed: Boolean read FChanged write FChanged;
      property  RecSize: LongInt read FRecSize;

      function  RecKeyIndexOf ( KeySize: LongInt; const Data ): Integer;

      procedure RecDelete( Index: Integer );
      procedure RecAdd( out Index: Integer; const Data );
      function  RecPtr( Index: Integer ): PChar;

      procedure Add   ( const Data );
      procedure Remove( const Data );
      procedure RemoveKey( const Data; KeySize: LongInt );
      function  ContainsKey( const Data; KeySize: LongInt ): Boolean;

      procedure Clear;
      procedure Pack;
      procedure Sort; overload;
      procedure Sort( CompareOffset: Integer ); overload;

      procedure Enter;
      procedure Leave;

      procedure LoadFromFile;
      procedure SaveToFile;
      procedure FlushToFile;

      constructor Create( AFilename : String;
                          AFileMode : Word;
                          ARecSize  : LongInt;
                          ASorted   : Boolean );
      destructor Destroy; override;
  end;

implementation

// ---------------------------------------------------------------- Tools -----

function CompareUInt32Ptr( p1, p2: Pointer ): Integer;
begin
     if PLongWord(p1)^ < PLongWord(p2)^ then begin
        Result := -1;
     end else begin
        if PLongWord(p1)^ > PLongWord(p2)^ then begin
           Result := +1;
        end else begin
           Result := 0;
        end;
     end;
end;

// ------------------------------------------------------ TIndexedRecords -----

procedure TIndexedRecords.QuickSort( CompareOffset, L, R: Integer );
var  I, J, P: Integer;
     DI, DJ : Pointer;
begin
     GetMem( DI, FRecSize );
     GetMem( DJ, FRecSize );

     repeat
        I := L;
        J := R;
        P := (L+R) div 2; 
        repeat
           while CompareUInt32Ptr( RecPtr(I) + CompareOffset,
                                   RecPtr(P) + CompareOffset ) < 0 do inc( I );
           while CompareUInt32Ptr( RecPtr(J) + CompareOffset,
                                   RecPtr(P) + CompareOffset ) > 0 do dec( J );
           if I <= J then begin
              RecGet( I, DI^ );
              RecGet( J, DJ^ );
              RecSet( J, DI^ );
              RecSet( I, DJ^ );
              if P = I then P := J
                       else if P = J then P := I;
              inc( I );
              dec( J );
           end;
        until I > J;
        if L < J then QuickSort( CompareOffset, L, J );
        L := I;
     until I >= R;

     FreeMem( DI, FRecSize );
     FreeMem( DJ, FRecSize );
end;

procedure TIndexedRecords.Sort;
begin
     Sort( 0 );
end;

procedure TIndexedRecords.Sort( CompareOffset: Integer );
begin
     Enter;
     try
        if FUseSize>0 then QuickSort( CompareOffset, 0, Count-1 );
     finally Leave end;
end;

procedure TIndexedRecords.SetSorted( NewSorted: Boolean );
begin
     Enter;
     try
        if NewSorted and not FSorted then Sort;
        FSorted := NewSorted;
     finally Leave end;
end;

function TIndexedRecords.GetCount: LongInt;
begin
     Enter;
     try
        if FRecSize>0 then Result := FUseSize div FRecSize
                      else Result := 0;
     finally Leave end;
end;

procedure TIndexedRecords.Clear;
begin
     Enter;
     try
        FUseSize := 0;
        FChanged := True;
     finally Leave end;
end;

procedure TIndexedRecords.Pack;
begin
     Enter;
     try
        if FUseSize<FMemSize then begin
           FMemSize := FUseSize;
           ReallocMem( FMemPtr, FMemSize );
        end;
     finally Leave end;
end;

function TIndexedRecords.RecPtr( Index: Integer ): PChar;
begin
     Enter;
     try
        Result := FMemPtr + Index * FRecSize;
     finally Leave end;
end;

procedure TIndexedRecords.RecGet( Index: Integer; var Data );
begin
     Enter;
     try
        if (Index>=0) and (Index<Count) then begin
           Move( RecPtr(Index)^, Data, FRecSize );
        end;
     finally Leave end;
end;

procedure TIndexedRecords.RecSet( Index: Integer; const Data );
begin
     Enter;
     try
        if (Index>=0) and (Index<Count) then begin
           Move( Data, RecPtr(Index)^, FRecSize );
           FChanged := True;
        end;
     finally Leave end;
end;

procedure TIndexedRecords.RecAdd( out Index: Integer; const Data );
begin
     Enter;
     try
        if FSorted then begin
           Index := RecKeyInsPosOf( RecSize, Data );
        end else begin
           Index := Count;
        end;
        RecInsert( Index, Data );
     finally Leave end;
end;

procedure TIndexedRecords.RecIns( Index: Integer );
var  reserved: LongInt;
begin
     Enter;
     try
        if (Index>=0) and (Index<=Count) then begin
           inc( FUseSize, FRecSize );
           if FUseSize>FMemSize then begin
              reserved := Count div 100; // pre-alloc for 1% of current count
              if reserved<1 then reserved:=1;
              FMemSize := FUseSize + FRecSize * reserved;
              ReallocMem( FMemPtr, FMemSize );
           end;
           if Index<>Count-1 then begin
              Move( RecPtr(Index)^,
                    RecPtr(Index+1)^,
                    FRecSize*(Count-Index-1) );
           end;
           FChanged := True;
        end;
     finally Leave end;
end;

procedure TIndexedRecords.RecInsert( Index: Integer; const Data );
begin
     Enter;
     try
        if (Index>=0) and (Index<=Count) then begin
           RecIns( Index );
           RecSet( Index, Data );
        end;
     finally Leave end;
end;

procedure TIndexedRecords.RecDelete( Index: Integer );
begin
     Enter;
     try
        if (Index>=0) and (Index<Count) then begin
           if Index<>Count-1 then begin
              Move( RecPtr(Index+1)^,
                    RecPtr(Index)^,
                    FRecSize*(Count-Index-1) );
           end;
           dec( FUseSize, FRecSize );
           FChanged := True;
        end;
     finally Leave end;
end;

procedure TIndexedRecords.RecKeyFindPos( const Data;
                                         KeySize: LongInt;
                                         WantInsertPos: Boolean;
                                         out   Index: Integer );
var  i, res, min, max, InsPos: Integer;
     P: PChar;
begin
     if FSorted then begin

        Index  := -1;
        InsPos := 0;

        min := 0;
        max := Count-1;

        repeat
           if min>max then break;
           i := ( min + max ) div 2;

           // res := MemCompare( RecPtr(i), @Data, KeySize );
           res := CompareUInt32Ptr( RecPtr(i), @Data );

           if res<0 then begin
              min := i+1;
              InsPos := min; //=compared-pos.+1
           end else begin
              if res>0 then begin
                 InsPos := i;
                 max := i-1;
              end else begin
                 InsPos := i; //=compared-pos.
                 Index  := i; // = already in list
                 break;
              end;
           end;

        until False;

        if WantInsertPos then Index:=InsPos;

     end else begin

        if WantInsertPos then begin
           Index := Count; // append
        end else begin
           Index := -1;
           P := RecPtr( 0 );
           for i:=0 to Count-1 do begin
              if CompareMem( P, @Data, KeySize ) then begin
                 Index := i;
                 break;
              end;
              inc( P, FRecSize );
           end;
        end;

     end;

end;

function TIndexedRecords.RecKeyInsPosOf( KeySize: LongInt; const Data ): Integer;
begin
     Enter;
     try
        RecKeyFindPos( Data, KeySize, True, Result );
     finally Leave end;
end;

function TIndexedRecords.RecKeyIndexOf( KeySize: LongInt; const Data ): Integer;
begin
     Enter;
     try
        RecKeyFindPos( Data, KeySize, False, Result );
     finally Leave end;
end;

procedure TIndexedRecords.Add( const Data );
var  Index: Integer;
begin
     Enter;
     try
        RecAdd( Index, Data );
     finally Leave end;
end;

procedure TIndexedRecords.RemoveKey( const Data; KeySize: LongInt );
var  Index: Integer;
begin
     Enter;
     try
        Index := RecKeyIndexOf( KeySize, Data );
        if Index>=0 then RecDelete( Index );
     finally Leave end;
end;

procedure TIndexedRecords.Remove( const Data );
begin
     RemoveKey( Data, RecSize );
end;

function TIndexedRecords.ContainsKey( const Data; KeySize: LongInt ): Boolean;
begin
     Enter;
     try
        Result := ( RecKeyIndexOf( KeySize, Data ) >= 0 );
     finally Leave end;
end;

procedure TIndexedRecords.LoadFromFile;

   function IsSorted: Boolean;
   var  i: Integer;
        p0, p1: PChar;
   begin
      Result := True;
      p1 := RecPtr( 0 );
      for i:=1 to Count-1 do begin
         p0 := p1;
         p1 := RecPtr( i );
         if CompareUInt32Ptr( p0, p1 ) > 0 then begin
            Result := False;
            break;
         end;
      end;
   end;

begin
   Enter;
   try
      Clear;
      if FileExists( Filename ) then begin

         if not Assigned( FSidx ) then begin
            FSidx := TFileStream64.Create( Filename, FFileMode );
         end;

         FreeMem( FMemPtr, FMemSize );
         FUseSize := ( ( FSidx.Size + FRecSize - 1 ) div FRecSize ) * FRecSize;
         FMemSize := FUseSize;
         FMemPtr  := AllocMem( FMemSize );

         FSidx.Seek( 0 );
         if FUseSize > 0 then FSidx.Read( FMemPtr^, FUseSize );

      end;

      FChanged := False;
      if FSorted and not IsSorted then Sort;

   finally Leave end;
end;

procedure TIndexedRecords.SaveToFile;
begin
   if not FChanged then exit;

   Enter;
   try
      if not Assigned( FSidx ) then begin
         FSidx := TFileStream64.Create( Filename, FFileMode );
      end;

      FSidx.Seek( 0 );
      if FUseSize > 0 then FSidx.Write( FMemPtr^, FUseSize );
      
      FSidx.SetEndOfFile;
      FSidx.Flush;

      FChanged := False;

   finally Leave end;
end;

procedure TIndexedRecords.FlushToFile;
begin
     Enter;
     try
        if Changed then SaveToFile;
     finally Leave end;
end;

procedure TIndexedRecords.Enter;    
begin
     EnterCriticalSection( CritSect );
end;

procedure TIndexedRecords.Leave;
begin
     LeaveCriticalSection( CritSect );
end;

constructor TIndexedRecords.Create( AFilename: String;
                                    AFileMode: Word;
                                    ARecSize : LongInt;
                                    ASorted  : Boolean );
begin
     inherited Create;

     InitializeCriticalSection( CritSect );

     FRecSize  := ARecSize;
     FSorted   := ASorted;
     FChanged  := False;
     FFilename := AFilename;
     FFileMode := AFileMode;
     FSidx     := nil;

     FUseSize := 0;
     FMemSize := FRecSize;
     FMemPtr  := AllocMem( FMemSize );
end;

destructor TIndexedRecords.Destroy;
begin
     Enter;
     try
        if Assigned( FSidx ) then try FreeAndNil( FSidx ); except end;
        try FreeMem( FMemPtr, FMemSize ); except end;
     finally Leave end;

     DeleteCriticalSection( CritSect );

     inherited Destroy;
end;

end.

