// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cIPAccess;

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, uTools, cSyncObjects;

const
   IPACC_SCOPE_NONE = $0000;
   IPACC_SCOPE_NNTP = $0001;
   IPACC_SCOPE_POP3 = $0010;
   IPACC_SCOPE_SMTP = $0100;
   IPACC_SCOPE_MAIL = $0110;
   IPACC_SCOPE_RECO = $1000;
   IPACC_SCOPE_ALL  = $FFFF;

   IPACC_ACCESS_NA  = $0000;
   IPACC_ACCESS_RO  = $0001;
   IPACC_ACCESS_WO  = $0002;
   IPACC_ACCESS_RW  = $0003;
   IPACC_ACCESS_ALL = $FFFF;

type
   TIPAccess = class

      private
         FLock        : TReaderWriterLock;
         FWantReload  : Boolean;
         FIpAccessList: TList;
         FReplacement_for_LocalIPs: LongInt;

         procedure LoadRecs;
         procedure FreeRecs;
         procedure Reload;

      public
         property WantReload: Boolean write FWantReload;

         function GetAccess( nChkIP: LongInt; ChkScope: LongInt;
                             out Account: String ): LongInt;

         constructor Create;
         destructor Destroy; override;

   end;

function IPAccessScopeStr( Scope: LongInt ): String;
function IPAccessAccessStr( Access: LongInt ): String;


implementation

uses uConst, uConstVar, uVar, WinSock, uWinSock, cLogFileHamster;

type
   TIPAccessItem = class
      public
         IpMin, IpMax, Scope, Access: LongInt;
         IpName: String;
         IpAccount: String;
   end;

function IPAccessScopeStr( Scope: LongInt ): String;
begin
   case Scope of
      IPACC_SCOPE_NONE: Result := 'NONE';
      IPACC_SCOPE_NNTP: Result := 'NNTP';
      IPACC_SCOPE_POP3: Result := 'POP3';
      IPACC_SCOPE_SMTP: Result := 'SMTP';
      IPACC_SCOPE_MAIL: Result := 'MAIL';
      IPACC_SCOPE_RECO: Result := 'RECO';
      IPACC_SCOPE_ALL : Result := 'ALL';
      else              Result := '?'+inttohex(Scope,8)+'?';
   end;
end;

function IPAccessAccessStr( Access: LongInt ): String;
begin
   case Access of
      IPACC_ACCESS_NA : Result := 'NA';
      IPACC_ACCESS_RO : Result := 'RO';
      IPACC_ACCESS_WO : Result := 'WO';
      IPACC_ACCESS_RW : Result := 'RW';
      IPACC_ACCESS_ALL: Result := 'ALL';
      else              Result := '?'+inttohex(Access,8)+'?';
   end;
end;

function TIPAccess.GetAccess( nChkIP: LongInt; ChkScope: LongInt; out Account: String ): LongInt;

   procedure LogGrant( const item: TIPAccessItem; const reason: String );
   begin
      Log( LOGID_DETAIL, 'IP-Access of IP ' + nAddrToStr( nChkIP )
                       + ' for '            + IPAccessScopeStr( ChkScope )
                       + ' is '             + IPAccessAccessStr( item.Access )
                       + ' because '        + reason
                       + iif( item.IpAccount <> '', ' (auto-login account is '
                                                    + item.IpAccount + ')' ) );
   end;

var  hChkIP, ipMin, ipMax, i: Integer;
     ipName: String;
     item: TIPAccessItem;
begin
   // default: no access
   Result  := IPACC_ACCESS_NA;
   Account := '';

   // reload file if changed by HControl
   if FWantReload then Reload;

   FLock.BeginRead;
   try
      // IPs to compare are in host byte order
      hChkIP := ntohl( nChkIP );

      // replace IP with given LOCAL replacement
      if FReplacement_for_LocalIPs <> 0 then begin
         if FReplacement_for_LocalIPs <> hChkIP then begin

            if IsLocalHost( nChkIp ) then begin
               Log( LOGID_INFO, 'IP-Access: Treat local IP ' + hAddrToStr(hChkIP)
                              + ' like ' + hAddrToStr(FReplacement_for_LocalIPs) );
               hChkIP := FReplacement_for_LocalIPs;
            end;
            
         end;
      end;

      // loop through list and use first matching entry
      for i := 0 to FIpAccessList.Count - 1 do begin

         item := TIPAccessItem( FIpAccessList[i] );

         if (ChkScope and item.Scope) = ChkScope then begin // wanted scope?
         
            if length( item.IpName ) = 0 then begin

               // compare with given IP range
               ipMin := item.IpMin;
               ipMax := item.IpMax;
               if (inttohex(hChkIP,8) >= inttohex(ipMin,8)) and
                  (inttohex(hChkIP,8) <= inttohex(ipMax,8)) then begin
                  LogGrant( item, 'it is in range ' + hAddrToStr(ipMin)
                                              + '-' + hAddrToStr(ipMax) );
                  Result  := item.Access;
                  Account := item.IpAccount;
                  break;
               end;

            end else begin

               if copy( item.IpName, 1, 1 ) = '?' then begin

                  // compare with current IP of given name
                  ipName := copy( item.IpName, 2, MaxInt );
                  ipMin  := ntohl( LookupHostAddr( ipName ) );
                  if hChkIP = ipMin then begin
                     LogGrant( item, 'it is current IP of ' + ipName );
                     Result  := item.Access;
                     Account := item.IpAccount;
                     break;
                  end;

               end else begin

                  // compare with given IP name
                  ipName := LookupHostName( nChkIP );
                  if CompareText( ipName, item.IpName ) = 0 then begin
                     LogGrant( item, 'it is ' + item.IpName );
                     Result  := item.Access;
                     Account := item.IpAccount;
                     break;
                  end;
                  if length( ipName ) > length( item.IpName ) then begin
                     ipName := copy( ipName,
                                     length(ipName) - length(item.IpName) - 1,
                                     length(item.IpName) + 1 );
                     if CompareText( ipName, '.' + item.IpName ) = 0 then begin
                        LogGrant( item, 'it is ' + ipName
                        + ' (sub-domain of ' + item.IpName + ')' );
                        Result  := item.Access;
                        Account := item.IpAccount;
                        break;
                     end;
                  end;
                  
               end;
               
            end;

         end;

      end;

   finally
      FLock.EndRead;
   end;
end;

procedure TIPAccess.LoadRecs;
var TS: TStringList;
    i : Integer;

   procedure AddEntry( const IpMin, IpMax, Scope, Access: LongInt;
                       const IpName, IpAccount: String );
   var  item: TIPAccessItem;
   begin
      item := TIPAccessItem.Create;
      item.IpMin     := IpMin;
      item.IpMax     := IpMax;
      item.Scope     := Scope;
      item.Access    := Access;
      item.IpName    := IpName;
      item.IpAccount := IpAccount;
      FIpAccessList.Add( item );
   end;

   function AddLine( DefLine: String ): Boolean;
   var  j: Integer;
        P: TParser;
        p1, p2, p3, p4, p5: String;
        s, s1, s2, account: String;
        ip1, ip2, sc, ac: LongInt;
   begin
      // assume a valid comment-only line
      Result := True;

      // remove trailing comments
      j := Pos(';',DefLine); if j>0 then DefLine := copy(DefLine,1,j-1);
      j := Pos('#',DefLine); if j>0 then DefLine := copy(DefLine,1,j-1);
      DefLine := TrimWhSpace( DefLine );
      if DefLine='' then exit;

      // assume invalid line
      Result := False;

      // parse line
      P := TParser.Create;
      try
         // (ALL|NNTP|POP3|SMTP), (NA|RO|WO|RW), <ip-from>, <ip-to>
         // (ALL|NNTP|POP3|SMTP), (NA|RO|WO|RW), "LOCAL", <ip>
         // (ALL|NNTP|POP3|SMTP), (NA|RO|WO|RW), <domain>
         // (ALL|NNTP|POP3|SMTP), (NA|RO|WO|RW), "?"<domain>
         P.Parse( DefLine, ',' );
         p1 := TrimWhSpace( P.sPart( 0, '' ) );
         p2 := TrimWhSpace( P.sPart( 1, '' ) );
         p3 := TrimWhSpace( P.sPart( 2, '' ) );
         p4 := TrimWhSpace( P.sPart( 3, '' ) );
         p5 := TrimWhSpace( P.sPart( 4, '' ) );

         // 1st value: Scope
         s := UpperCase( p1 );
         if      s = 'NNTP' then sc := IPACC_SCOPE_NNTP
         else if s = 'POP3' then sc := IPACC_SCOPE_POP3
         else if s = 'SMTP' then sc := IPACC_SCOPE_SMTP
         else if s = 'MAIL' then sc := IPACC_SCOPE_MAIL
         else if s = 'RECO' then sc := IPACC_SCOPE_RECO
         else if s = 'ALL'  then sc := IPACC_SCOPE_ALL
         else                    exit;

         // 2nd value: Access
         s := UpperCase( p2 );
         if      s = 'NA' then ac := IPACC_ACCESS_NA
         else if s = 'RO' then ac := IPACC_ACCESS_RO
         else if s = 'WO' then ac := IPACC_ACCESS_WO
         else if s = 'RW' then ac := IPACC_ACCESS_RW
         else                  exit;

         // 3rd value: (IP from) or ('LOCAL') or (IP name)
         s1 := p3;
         if s1 = '' then exit;
         ip1 := 0;
         if s1[1] in ['0'..'9'] then ip1 := StrTohAddr( s1 );

         // 4th value: (IP to) or (IP) or (./.)
         s2 := p4;
         ip2 := ip1;
         if (s2 <> '') and (s2[1] in ['0'..'9']) then ip2 := StrTohAddr( s2 );

         // 5th value: default account
         account := p5;

         // add item
         if UpperCase( s1 ) = 'LOCAL' then begin

            // keyword 'LOCAL' instead of first IP
            if ip2 = 0 then begin
               Log( LOGID_WARN, 'Invalid line in IPAccess.hst ignored: "'
                              + DefLine + '"  Missing 4th parameter!'  );
               exit;
            end else begin
               FReplacement_for_LocalIPs := ip2;
            end;

         end else if s1[1] in ['0'..'9'] then begin

            // IP range
            if inttohex( ip1, 8 ) > inttohex( ip2, 8 ) then exit;
            AddEntry( ip1, ip2, sc, ac, '', account );

         end else begin

            // IP name
            if p4 <> '' then begin
               Log( LOGID_WARN, 'Invalid line in IPAccess.hst ignored: "'
                              + DefLine + '"  Unexpected 4th parameter!'  );
               exit;
            end else begin
               AddEntry( 0, 0, sc, ac, s1, account );
            end;

         end;

         Result := True;

      finally P.Free end;
   end;

begin
   FReplacement_for_LocalIPs := 0;

   // load settings of IpAccess.hst
   if FileExists( AppSettings.GetStr(asPathBase) + CFGFILE_IPACCESS ) then begin

      TS := TStringList.Create;
      try

         TS.LoadFromFile( AppSettings.GetStr(asPathBase) + CFGFILE_IPACCESS );

         for i := 0 to TS.Count - 1 do begin
            if not AddLine( TS[i] ) then begin
               Log( LOGID_WARN, 'Invalid line in IPAccess.hst: ' + TS[i] );
               FreeRecs; // ignore invalid file and reset to safe defaults
            end;
         end;

      finally TS.Free end;

   end;

   // add defaults if no (valid) lines are given
   if FIpAccessList.Count = 0 then begin

      AddLine( 'ALL,NA,LOCAL,127.0.0.1' );
      AddLine( 'ALL,RW,127.0.0.1' );
      AddLine( 'ALL,RW,192.168.0.0,192.168.255.255' );
      AddLine( 'ALL,NA,0.0.0.0,255.255.255.255' );

   end;
end;

procedure TIPAccess.FreeRecs;
var  i: Integer;
begin
   for i := 0 to FIpAccessList.Count - 1 do begin
      TIPAccessItem( FIpAccessList[i] ).Free;
      FIpAccessList[i] := nil;
   end;
   FIpAccessList.Clear;
end;

procedure TIPAccess.Reload;
begin
   FLock.BeginWrite;
   try
      FreeRecs;
      LoadRecs;
      FWantReload := False;
   finally
      FLock.EndWrite;
   end;
end;

constructor TIPAccess.Create;
begin
   inherited Create;

   FLock := TReaderWriterLock.Create;
   FIpAccessList := TList.Create;
   FReplacement_for_LocalIPs := 0;
   FWantReload := True;
   Reload;
end;

destructor TIPAccess.Destroy;
begin
   FLock.BeginWrite;
   FreeRecs;
   FIpAccessList.Free;
   FLock.Free;

   inherited Destroy;
end;

end.
