// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHscHelpers; // Helper classes and functions for script-engine

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, Windows, SyncObjs, uTools;

const
   HSCERR_NOERROR           = 0;
   HSCERR_STOPPED           = 1;
   HSCERR_ENGINEEXCEPTION   = 3;
   HSCERR_INVALIDEXPRESSION = 4;
   HSCERR_UNSUPPORTED       = 5;
   HSCERR_VARNOTINITIALIZED = 6;
   HSCERR_SYNTAX            = 7;
   HSCERR_LABELNOTFOUND     = 8;
   HSCERR_LOOPSTACK         = 9;
   HSCERR_IFTHENELSE        = 10;
   HSCERR_LOOPNOTFOUND      = 11;
   HSCERR_LIMITEXCEEDED     = 12;
   HSCERR_UNKNOWNVAR        = 13;
   HSCERR_INVALID           = 14;
   HSCERR_INITIALIZEFAILED  = 15;
   HSCERR_DUPLICATEVAR      = 16;
   HSCERR_USERDEFINED       = 1000;

   SMARKER_MASK_LINENO = $00FFFFFF;
   SMARKER_MASK_MARKER = $FF000000;

   SMARKER_CHECKED   = $80000000;
   SMARKER_NOSPECIAL = SMARKER_CHECKED;
   SMARKER_IF        = SMARKER_CHECKED or $01000000;
   SMARKER_ELSE      = SMARKER_CHECKED or $02000000;
   SMARKER_ENDIF     = SMARKER_CHECKED or $03000000;
   SMARKER_DO        = SMARKER_CHECKED or $04000000;
   SMARKER_LOOP      = SMARKER_CHECKED or $05000000;
   SMARKER_WHILE     = SMARKER_CHECKED or $06000000;
   SMARKER_ENDWHILE  = SMARKER_CHECKED or $07000000;
   SMARKER_REPEAT    = SMARKER_CHECKED or $08000000;
   SMARKER_UNTIL     = SMARKER_CHECKED or $09000000;
   SMARKER_FOR       = SMARKER_CHECKED or $0A000000;
   SMARKER_ENDFOR    = SMARKER_CHECKED or $0B000000;
   SMARKER_SUB       = SMARKER_CHECKED or $0C000000;
   SMARKER_ENDSUB    = SMARKER_CHECKED or $0D000000;
   SMARKER_LABEL     = SMARKER_CHECKED or $0E000000;
   SMARKER_ELSEIF    = SMARKER_CHECKED or $0F000000;

type
   THscVarTypes = ( hvtEmpty, hvtInteger, hvtString );

   EHscVarError = class(Exception);

   THscVariant = class
      private
         FValTyp: THscVarTypes;
         FValInt: Integer;
         FValStr: String;
         FConst : Boolean;

         function  GetAsVar: Variant;
         procedure SetAsVar( const NewValue: Variant );
         function  GetAsInt: Integer;
         procedure SetAsInt( const NewValue: Integer );
         function  GetAsStr: String;
         procedure SetAsStr( const NewValue: String );
         function  GetAsPtr: Integer;

      public
         property TypOf: THscVarTypes read FValTyp;
         property AsVar: Variant read GetAsVar write SetAsVar;
         property AsInt: Integer read GetAsInt write SetAsInt;
         property AsStr: String  read GetAsStr write SetAsStr;
         property AsPtr: Integer read GetAsPtr;
         property IsConst: Boolean read FConst write FConst;

         procedure Assign( const HV: THscVariant );
         function  Unassigned: Boolean;
         procedure Unassign;
         procedure ToInt;
         procedure ToStr;

         constructor Create;                          overload;
         constructor Create( const AValue: Integer;
                             const AConst: Boolean ); overload;
         constructor Create( const AValue: String;
                             const AConst: Boolean ); overload;
         constructor Create( const AValue: THscVariant;
                             const AConst: Boolean ); overload;
   end;

   THscVariantArray = class
      private
         FCount: Integer;
         FArray: array of THscVariant;

         function GetItem( const Index: Integer ): THscVariant;

      public
         property Count: Integer read FCount;
         property Item[ const Index: Integer]: THscVariant
                                               read GetItem; default;

         constructor Create( const ACount: Integer );
         destructor Destroy; override;
   end;

   THscVariableReference = class
      protected
         FContext: Integer;
         FName   : String;
         FValue  : THscVariant;

      public
         property Context: Integer     read FContext;
         property Name   : String      read FName;
         property Value  : THscVariant read FValue;

         constructor Create( const AContext: Integer;
                             const AName   : String;
                             const AValue  : THscVariant );
   end;

   THscVariable = class( THscVariableReference )
      public
         constructor Create( const AContext: Integer;
                             const AName   : String;
                             const AValue  : THscVariant;
                             const AConst  : Boolean );
         destructor Destroy; override;
   end;

   THscExpTypes = (
      hetINT, // Number
      hetSTR, // String
      hetVAR, // Variable
      hetFUN, // Function
      hetOP0, // "||"
      hetOP1, // "&&"
      hetOP2, // "|"
      hetOP3, // "^"
      hetOP4, // "&"
      hetOP5, // "==" / "!=" [ / "=" / "!" / "<>"
      hetOP6, // "<" / ">" / "<=" / ">="
      hetOP7, // "<<" / ">>"
      hetOP8, // "+" / "-"
      hetOP9, // "*" / "/" / "%"
      hetUOP, // "+" / "-" / "!" / "~" (unary)
      hetGON, // "("
      hetGOF, // ")"
      hetERR  // error-marker
   );

   THscExpressionPart = class
      public
         EText : String;
         EType : THscExpTypes;
         EValue: THscVariant;

         constructor Create( const AExpPat: THscExpressionPart ); overload;
         constructor Create( const AText  : String;
                             const AType  : THscExpTypes;
                             const AValue : Integer            ); overload;
         constructor Create( const AText  : String;
                             const AType  : THscExpTypes;
                             const AValue : String             ); overload;
         destructor Destroy; override;
   end;

   THscExpressionParts = class
      private
         FList: TList;

         function GetCount: Integer;
         function GetItem( const Index: Integer ): THscExpressionPart;

      public
         property Count: Integer read GetCount;
         property Item[ const Index: Integer ]: THscExpressionPart
                                                read GetItem; default;

         procedure Add( const AText : String;
                        const AType : THscExpTypes;
                        const AValue: Integer      ); overload;
         procedure Add( const AText : String;
                        const AType : THscExpTypes;
                        const AValue: String       ); overload;
         procedure Clear;
         procedure Delete( const Index: Integer );
         procedure Assign( const EP: THscExpressionParts );

         constructor Create;
         destructor Destroy; override;
   end;

   THscProfilerItem = class
      Counter            : Int64;
      TimeWithChildren   : Int64;
      TimeWithoutChildren: Int64;
   end;

   THscProfiler = class
      private
         FAddNested: Boolean;
         FItems   : TStringList;
         FActive  : TList;
         StartTime, StopTime, ResultTime: Int64;
         CorrTime : array[0..10] of Int64;
         procedure DoEstimateCorrTime;
         procedure DoAllStop;
         procedure DoAllStart;
      public
         property  AddNested: Boolean read FAddNested write FAddNested;
         procedure Start( ID: String );
         procedure Stop;
         procedure Clear;
         function Report: String;
         procedure SaveReport( Filename: String );
         constructor Create( AAddNested: Boolean = True );
         destructor Destroy; override;
   end;

   PLoopStackEntry = ^TLoopStackEntry;
   TLoopStackEntry = record
      LoopContext: Integer;
      LoopMarker : LongWord;
      LoopPos    : Integer;
      ForVar     : THscVariant;
      ForStep    : Integer;
      ForLimit   : Integer;
   end;

   EHscGlobalError = class(Exception);

   THscGlobal = class
      private
         FLock   : TCriticalSection;
         FChanged: TEvent;
         FLocks  : TStringList_NoAnsi;
         FValues : TStringList_NoAnsi;

         procedure CheckName( const Name: String;
                              allowSystem: Boolean );

      public
         procedure ValueSet   ( const Name: String;
                                const Value: THscVariant;
                                allowSystem: Boolean = False );
         procedure ValueGet   ( const Name: String;
                                const Value, Default: THscVariant;
                                allowSystem: Boolean = False );
         function  ValueAdd   ( const Name: String;
                                const AddVal: Integer;
                                allowSystem: Boolean = False ): Integer;
         function  ValueAppend( const Name: String;
                                const AddVal: String;
                                allowSystem: Boolean = False ): String;
         function  ValueWait  ( const Name: String;
                                const WaitVal: THscVariant;
                                const StopEvent: TEvent;
                                allowSystem: Boolean = False ): Boolean;

         function  Enter( const Name: String;
                          const StopEvent: TEvent;
                                allowSystem: Boolean = False ): Boolean;
         function  Leave( const Name: String;
                          allowSystem: Boolean = False ): Boolean;
         procedure LeaveAll( const ThreadID: Cardinal );

         function  Once( const Name: String;
                         const Minutes: Integer;
                         allowSystem: Boolean = False ): Boolean;

         constructor Create;
         destructor Destroy; override;
   end;

var
   HscGlobal: THscGlobal;

function HscMarkerInArray( const Marker: LongWord;
                           const MarkerArray: array of LongWord ): Boolean;
function HscCmdToMarker( const Cmd: String ): LongWord;
function HscMarkerToCmd( const Marker: LongWord ): String;
function HscReadLineCmd( const s: String ): String;

function IsHscDigitChar     ( const c: Char ): Boolean;
function IsHscLetterChar    ( const c: Char ): Boolean;
function IsHscHexDigitChar  ( const c: Char ): Boolean;
function IsHscIdentifierChar( const c: Char ): Boolean;

function IsHscScalar  ( const het: THscExpTypes ): Boolean;
function IsHscOperator( const het: THscExpTypes ): Boolean;

function IsHscSimpleNumber( const s: String ): Boolean;
function IsHscSimpleString( const s: String ): Boolean;
function IsHscIdentifier  ( const s: String ): Boolean;
function IsHscVariable    ( const s: String ): Boolean;

procedure HscSkipEmbeddedComment( var pch: PChar );
function  HscTrimWSPEC( const s : String ) : String;
function HscTrimWSPC( const s: String ): String;
procedure HscSkipWSPC( var pch: PChar );

function VarToHscVarType( const V: Variant ): THscVarTypes;

function HscExecuteProcess( CommandLine     : String;
                            WorkingDir      : String;
                            SWShowWindow    : Integer;
                            WaitForEnd      : Boolean;
                            var WaitExitCode: Integer;
                            StopEvent       : TEvent ): Integer;

implementation

uses uConst, cLogFileHamster, uVar, IniFiles, uDateTime
     {$IFDEF H_NEED_VARIANTS} , Variants {$ENDIF} ;

function HscMarkerInArray( const Marker: LongWord;
                           const MarkerArray: array of LongWord ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=0 to High(MarkerArray) do begin
      if MarkerArray[i]=Marker then begin Result:=True; exit end;
   end;
end;

function HscCmdToMarker( const Cmd: String ): LongWord;
begin
   if pos( ','+Cmd+',',
           ',if,else,endif,do,loop,while,endwhile,repeat,until,for,endfor,sub,endsub,label,elseif,' ) = 0 then begin
      Result := SMARKER_NOSPECIAL;
      exit;
   end;

   if      Cmd='if'       then Result:=SMARKER_IF
   else if Cmd='else'     then Result:=SMARKER_ELSE
   else if Cmd='endif'    then Result:=SMARKER_ENDIF
   else if Cmd='do'       then Result:=SMARKER_DO
   else if Cmd='loop'     then Result:=SMARKER_LOOP
   else if Cmd='while'    then Result:=SMARKER_WHILE
   else if Cmd='endwhile' then Result:=SMARKER_ENDWHILE
   else if Cmd='repeat'   then Result:=SMARKER_REPEAT
   else if Cmd='until'    then Result:=SMARKER_UNTIL
   else if Cmd='for'      then Result:=SMARKER_FOR
   else if Cmd='endfor'   then Result:=SMARKER_ENDFOR
   else if Cmd='sub'      then Result:=SMARKER_SUB
   else if Cmd='endsub'   then Result:=SMARKER_ENDSUB
   else if Cmd='label'    then Result:=SMARKER_LABEL
   else if Cmd='elseif'   then Result:=SMARKER_ELSEIF
   else                        Result:=SMARKER_NOSPECIAL;
end;

function HscMarkerToCmd( const Marker: LongWord ): String;
begin
   case Marker of
      SMARKER_IF      : Result:='if';
      SMARKER_ELSE    : Result:='else';
      SMARKER_ENDIF   : Result:='endif';
      SMARKER_DO      : Result:='do';
      SMARKER_LOOP    : Result:='loop';
      SMARKER_WHILE   : Result:='while';
      SMARKER_ENDWHILE: Result:='endwhile';
      SMARKER_REPEAT  : Result:='repeat';
      SMARKER_UNTIL   : Result:='until';
      SMARKER_FOR     : Result:='for';
      SMARKER_ENDFOR  : Result:='endfor';
      SMARKER_SUB     : Result:='sub';
      SMARKER_ENDSUB  : Result:='endsub';
      SMARKER_LABEL   : Result:='label';
      SMARKER_ELSEIF  : Result:='elseif';
      else              Result:='';
   end;
end;

function HscReadLineCmd( const s: String ): String;
var  i: Integer;
begin
   Result := LowerCase( s );
   i := PosWhSpace( Result ); if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '(', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '#', Result );   if i>0 then Result:=copy(Result,1,i-1);
   i := Pos( '{', Result );   if i>0 then Result:=copy(Result,1,i-1);
   if copy(Result,1,1)='$' then Result:='set';
end;

function IsHscDigitChar( const c: Char ): Boolean;
const CSET_DIGITS = ['0'..'9'];
begin
   Result := c in CSET_DIGITS;
end;

function IsHscLetterChar( const c: Char ): Boolean;
const CSET_LETTERS = ['A'..'Z','a'..'z'];
begin
   Result := c in CSET_LETTERS;
end;

function IsHscHexDigitChar( const c: Char ): Boolean;
const CSET_HEXDIGITS = ['0'..'9','A'..'F','a'..'f'];
begin
   Result := c in CSET_HEXDIGITS;
end;

function IsHscIdentifierChar( const c: Char ): Boolean;
const CSET_IDENTIFIER = ['a'..'z','A'..'Z','0'..'9','_'];
begin
   Result := c in CSET_IDENTIFIER;
end;

function IsHscScalar( const het: THscExpTypes ): Boolean;
begin
   Result := ( het in [ hetINT, hetSTR ] );
end;

function IsHscOperator( const het: THscExpTypes ): Boolean;
begin
   Result := ( het in [ hetOP0..hetOP9 ] );
end;

function IsHscSimpleNumber( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   for i:=1 to length(s) do begin
      if not IsHscDigitChar( s[i] ) then exit;
   end;
   if length(s) > 0 then Result := True;
end;

function IsHscSimpleString( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if s='' then exit;
   if s[1]<>'"' then exit;
   if s[length(s)]<>'"' then exit;
   for i:=2 to length(s)-1 do begin
      if s[i]='"' then exit;
   end;
   Result := True;
end;

function IsHscIdentifier( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if s='' then exit;
   if not IsHscLetterChar( s[1] ) then exit;
   for i:=2 to length(s) do begin
      if not IsHscIdentifierChar( s[i] ) then exit;
   end;
   Result := True;
end;

function IsHscVariable( const s: String ): Boolean;
var  i: Integer;
begin
   Result := False;
   if length(s)<2 then exit;
   if s[1]<>'$' then exit;
   if not IsHscLetterChar( s[2] ) then exit;
   for i:=3 to length(s) do begin
      if not IsHscIdentifierChar( s[i] ) then exit;
   end;
   Result := True;
end;

procedure HscSkipEmbeddedComment( var pch: PChar );
begin
   if pch^<>'{' then exit;
   inc( pch );
   while pch^<>#0 do begin
      if pch^='}' then begin inc(pch); break; end;
      inc( pch );
   end;
end;

function HscTrimWSPEC( const s : String ) : String;
var  pb, pe: PChar;
begin
   SetLength( Result, 0 );

   pb := PChar( s );
   repeat
      case pb^ of
         #9, ' ': inc( pb );
         '{'    : begin
                     while not( pb^ in [#0,'}'] ) do inc( pb );
                     if pb^=#0 then exit else inc(pb);
                  end;
         else     break;
      end;
   until False;
   if pb^=#0 then exit;

   pe := strend( pb ) - 1;
   repeat
      case pe^ of
         #9, ' ': dec( pe );
         '}'    : begin
                     while (pe>pb) and (pe^<>'{') do dec( pe );
                     if pb=pe then exit else dec(pe);
                  end;
         else     break;
      end;
   until False;

   // Result := copy( pb, 1, pe-pb+1 );
   SetString( Result, pb, pe-pb+1 );
end;

function HscTrimWSPC( const s: String ): String;
var  i: Integer;
begin
   Result := HscTrimWSPEC( s );
   i := Pos( '#', s );
   if i > 0 then begin
      SetLength( Result, i-1 );
      Result := TrimWhSpace( Result );
   end;
end;

procedure HscSkipWSPC( var pch: PChar );
begin
   while pch^<>#0 do begin
      case pch^ of
        ' ', #9: inc( pch );
        '#'    : begin pch:=strend(pch); break end;
        '{'    : begin
                    while not( pch^ in [#0,'}'] ) do inc(pch);
                    if pch^=#0 then break else inc(pch);
                 end;
        else     break;
      end;
   end;
end;


function VarToHscVarType( const V: Variant ): THscVarTypes;
begin
   case VarType( V ) of
      varInteger, varByte, varSmallInt, varBoolean:
         Result := hvtInteger;
      varString, varOleStr:
         Result := hvtString;
      else
         Result := hvtEmpty;
   end;
end;

function HscExecuteProcess( CommandLine     : String;
                            WorkingDir      : String;
                            SWShowWindow    : Integer;
                            WaitForEnd      : Boolean;
                            var WaitExitCode: Integer;
                            StopEvent       : TEvent ): Integer;
var  StartupInfo : TStartUpInfo;
     ProcessInfo : TProcessInformation;
     WaitObjects : TWOHandleArray;
     WaitObjCount: Integer;
     pWorkingDir : PChar;
     res: DWord;
begin
     Result := 0;
     WaitExitCode := -1;

     if WorkingDir='' then pWorkingDir:=nil
                      else pWorkingDir:=PChar(WorkingDir);

     FillChar( StartUpInfo, sizeof(StartUpInfo), 0 );
     StartupInfo.cb := Sizeof( StartupInfo );
     StartupInfo.dwFlags     := STARTF_USESHOWWINDOW;
     StartupInfo.wShowWindow := SWShowWindow;

     FillChar( ProcessInfo, sizeof(ProcessInfo), 0 );

     if CreateProcess( nil, PChar(CommandLine), nil, nil, False,
                       NORMAL_PRIORITY_CLASS, nil, pWorkingDir,
                       StartUpInfo, ProcessInfo ) then begin

        if WaitForEnd then begin
           WaitObjects[0] := ProcessInfo.hProcess;
           WaitObjCount := 1;
           if Assigned( StopEvent ) then begin
              WaitObjects[1] := StopEvent.Handle;
              WaitObjCount := 2;
           end;

           res := WaitForMultipleObjects( WaitObjCount, @WaitObjects,
                                          False, INFINITE );
           case res of
              WAIT_OBJECT_0  : Result :=  0; // ok
              WAIT_OBJECT_0+1: Result := -1; // stop-event
              WAIT_FAILED    : Result := GetLastError; // error
           end;

           if Result=0 then begin
              GetExitCodeProcess( ProcessInfo.hProcess, res );
              WaitExitCode := res
           end

        end;

        CloseHandle( ProcessInfo.hThread  );
        CloseHandle( ProcessInfo.hProcess );

     end else begin
        Result := GetLastError;
        if Result=0 then Result:=-1;
     end;
end;


// ---------------------------------------------------------- THscVariant -----

const
   SHInvalidVarCast = 'Invalid use of unassigned value!';
   SHConstChanged   = 'Constants can''t be changed!';

procedure THscVariant.Assign( const HV: THscVariant );
begin
   if IsConst then raise EHscVarError.Create( SHConstChanged );

   if Assigned( HV ) then begin
      FValTyp := HV.FValTyp;
      case FValTyp of
         hvtInteger: FValInt := HV.FValInt;
         hvtString : FValStr := HV.FValStr;
      end;
   end else begin
      Unassign;
   end;
end;

procedure THscVariant.Unassign;
begin
   FValTyp := hvtEmpty;
end;

function THscVariant.Unassigned: Boolean;
begin
   Result := ( FValTyp = hvtEmpty );
end;

procedure THscVariant.ToInt;
begin
   case FValTyp of
      hvtString: AsInt := SysUtils.strtoint(FValStr);
      hvtEmpty : raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.ToStr;
begin
   case FValTyp of
      hvtInteger: AsStr := inttostr(FValInt);
      hvtEmpty  : raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

function THscVariant.GetAsVar: Variant;
begin
   case FValTyp of
      hvtEmpty  : Result := Unassigned;
      hvtInteger: Result := AsInt;
      hvtString : Result := AsStr;
   end;
end;

procedure THscVariant.SetAsVar( const NewValue: Variant );
begin
   if IsConst then raise EHscVarError.Create( SHConstChanged );

   case VarToHscVarType( NewValue ) of
      hvtEmpty  : Unassign;
      hvtInteger: AsInt := NewValue;
      hvtString : AsStr := NewValue;
   end;
end;

function THscVariant.GetAsInt: Integer;
begin
   case FValTyp of
      hvtInteger: Result := FValInt;
      hvtString : Result := SysUtils.strtoint(FValStr);
      else        raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.SetAsInt(const NewValue: Integer);
begin
   if IsConst then raise EHscVarError.Create( SHConstChanged );

   FValTyp := hvtInteger;
   FValInt := NewValue;
end;

function THscVariant.GetAsStr: String;
begin
   case FValTyp of
      hvtString : Result := FValStr;
      hvtInteger: Result := inttostr(FValInt);
      else        raise EHscVarError.Create( SHInvalidVarCast );
   end;
end;

procedure THscVariant.SetAsStr(const NewValue: String);
begin
   if IsConst then raise EHscVarError.Create( SHConstChanged );

   FValTyp := hvtString;
   FValStr := NewValue;
end;

function THscVariant.GetAsPtr: Integer;
begin
   case FValTyp of
      hvtInteger: Result := Integer(@FValInt);
      hvtString : Result := Integer(PChar(FValStr));
      else        Result := 0;
   end;
end;

constructor THscVariant.Create;
begin
   inherited Create;
   Unassign;
end;

constructor THscVariant.Create( const AValue: Integer; const AConst: Boolean );
begin
   inherited Create;
   AsInt  := AValue;
   FConst := AConst;
end;

constructor THscVariant.Create( const AValue: String; const AConst: Boolean );
begin
   inherited Create;
   AsStr  := AValue;
   FConst := AConst;
end;

constructor THscVariant.Create( const AValue: THscVariant; const AConst: Boolean );
begin
   inherited Create;
   Assign( AValue );
   FConst := AConst;
end;


// ----------------------------------------------------- THscVariantArray -----

constructor THscVariantArray.Create( const ACount: Integer );
var  i: Integer;
begin
   inherited Create;
   FCount := ACount;
   SetLength( FArray, FCount );
   for i:=0 to FCount-1 do FArray[i] := THscVariant.Create;
end;

destructor THscVariantArray.Destroy;
var  i: Integer;
begin
   for i:=0 to FCount-1 do FArray[i].Free;
   FArray := nil;
   inherited Destroy;
end;

function THscVariantArray.GetItem( const Index: Integer ): THscVariant;
begin
   Result := FArray[ Index ];
end;


// ------------------------------------------------ THscVariableReference -----

constructor THscVariableReference.Create( const AContext: Integer;
                                          const AName   : String;
                                          const AValue  : THscVariant );
begin
   inherited Create;
   FContext := AContext;
   FName    := AName;
   FValue   := AValue;
end;

// --------------------------------------------------------- THscVariable -----

constructor THscVariable.Create( const AContext: Integer;
                                 const AName   : String;
                                 const AValue  : THscVariant;
                                 const AConst  : Boolean );
var  value: THscVariant;
begin
   value := THscVariant.Create( AValue, AConst );
   inherited Create( AContext, AName, value );
end;

destructor THscVariable.Destroy;
begin
   FValue.Free;
   inherited Destroy;
end;

// --------------------------------------------------- THscExpressionPart -----

constructor THscExpressionPart.Create( const AExpPat: THscExpressionPart );
begin
   inherited Create;
   EText  := AExpPat.EText;
   EType  := AExpPat.EType;
   EValue := THscVariant.Create( AExpPat.EValue, False );
end;

constructor THscExpressionPart.Create( const AText : String;
                                       const AType : THscExpTypes;
                                       const AValue: Integer );
begin
   inherited Create;
   EText  := AText;
   EType  := AType;
   EValue := THscVariant.Create( AValue, False );
end;

constructor THscExpressionPart.Create( const AText : String;
                                       const AType : THscExpTypes;
                                       const AValue: String );
begin
   inherited Create;
   EText  := AText;
   EType  := AType;
   EValue := THscVariant.Create( AValue, False );
end;

destructor THscExpressionPart.Destroy;
begin
   EValue.Free;
   inherited Destroy;
end;


// -------------------------------------------------- THscExpressionParts -----

procedure THscExpressionParts.Add( const AText : String;
                                   const AType : THscExpTypes;
                                   const AValue: Integer );
begin
   FList.Add( THscExpressionPart.Create( AText, AType, AValue ) );
end;

procedure THscExpressionParts.Add( const AText : String;
                                   const AType : THscExpTypes;
                                   const AValue: String );
begin
   FList.Add( THscExpressionPart.Create( AText, AType, AValue ) );
end;

procedure THscExpressionParts.Assign( const EP: THscExpressionParts );
var  i: Integer;
begin
   Clear;
   for i:=0 to EP.Count-1 do FList.Add( THscExpressionPart.Create( EP[i] ) );
end;

procedure THscExpressionParts.Clear;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do THscExpressionPart( FList[ i ] ).Free;
   FList.Clear;
end;

constructor THscExpressionParts.Create;
begin
   inherited Create;
   FList := TList.Create;
end;

procedure THscExpressionParts.Delete( const Index: Integer );
begin
   THscExpressionPart( FList[ Index ] ).Free;
   FList.Delete( Index );
end;

destructor THscExpressionParts.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

function THscExpressionParts.GetCount: Integer;
begin
   Result := FList.Count;
end;

function THscExpressionParts.GetItem(const Index: Integer): THscExpressionPart;
begin
   Result := THscExpressionPart( FList[ Index ] );
end;


// --------------------------------------------------------- THscProfiler -----

constructor THscProfiler.Create;
begin
   inherited Create;
   FAddNested := AAddNested;
   FItems := TStringList.Create;
   FActive := TList.Create;
   DoEstimateCorrTime;
end;

destructor THscProfiler.Destroy;
begin
   Clear;
   FActive.Free;
   FItems.Free;
   inherited;
end;

procedure THscProfiler.Clear;
begin
   while FItems.Count>0 do begin
      THscProfilerItem( FItems.Objects[0] ).Free;
      FItems.Delete( 0 );
   end;
   FItems.Clear;
   FActive.Clear;
end;

function SortByObjects( List: TStringList; Index1, Index2: Integer ): Integer;
begin
   Result := Integer( List.Objects[Index2] ) - Integer( List.Objects[Index1] );
end;

procedure THscProfiler.SaveReport( Filename: String );
var  TS: TStringList;
begin
   TS := TStringList.Create;
   TS.Text := Report;
   try if TS.Count>0 then TS.SaveToFile( Filename ) except end;
   TS.Free;
end;

function THscProfiler.Report: String;
var  TS, RS: TStringList;
     i: Integer;
     F: int64;
     cF, cD, cX, Sum, Percent: Currency;
     s: String;
begin
   while FActive.Count > 0 do Stop;

   Result := '';
   if FItems.Count=0 then exit;

   QueryPerformanceFrequency( F );
   cF := F;

   Sum := 0;
   for i:=0 to FItems.Count-1 do begin
      with THscProfilerItem( FItems.Objects[i] ) do begin
         Sum := Sum + TimeWithoutChildren;
      end;
   end;
   Sum := Sum / cF;

   TS := TStringList.Create;
   RS := TStringList.Create;

   TS.Add( 'Times including nested children:' );
   TS.Add( Format( '%8s  %8.4s  %8.4s  %8.2s   %s',
           [ 'n', 't', 't/n', '%t', 'ID' ] ) );
   RS.Clear;
   for i:=0 to FItems.Count-1 do begin
      with THscProfilerItem( FItems.Objects[i] ) do begin
         try
            cD := ( 1.0 * TimeWithChildren / cF );
            if Counter=0 then cX := 0 else cX := ( cD / Counter );

            Percent := cD * 100.0 / Sum;

            RS.AddObject( Format( '%8d  %8.4f  %8.4f  %8.2f   %s',
                          [ Counter, cD, cX, Percent, s+FItems[i]] ),
                          Pointer( Trunc(Percent*10000) ) );
         except
            on E: Exception do TS.Add( E.Message );
         end;
      end;
   end;
   RS.CustomSort( SortByObjects );
   TS.Text := TS.Text + RS.Text + #13#10;

   TS.Add( 'Times excluding nested children:' );
   TS.Add( Format( '%8s  %8.4s  %8.4s  %8.2s   %s',
           [ 'n', 't', 't/n', '%t', 'ID' ] ) );
   RS.Clear;
   for i:=0 to FItems.Count-1 do begin
      with THscProfilerItem( FItems.Objects[i] ) do begin
         try
            cD := ( 1.0 * TimeWithoutChildren / cF );
            if Counter=0 then cX := 0 else cX := ( cD / Counter );

            Percent := cD * 100.0 / Sum;

            RS.AddObject( Format( '%8d  %8.4f  %8.4f  %8.2f   %s',
                          [ Counter, cD, cX, Percent, s+FItems[i]] ),
                          Pointer( Trunc(Percent*10000) ) );
         except
            on E: Exception do TS.Add( E.Message );
         end;
      end;
   end;
   RS.CustomSort( SortByObjects );
   TS.Text := TS.Text + RS.Text;

   Result := TS.Text;

   RS.Free;
   TS.Free;
end;

procedure THscProfiler.DoAllStart;
begin
   QueryPerformanceCounter( StartTime );
end;

procedure THscProfiler.DoAllStop;
var  i, Index: Integer;
begin
   QueryPerformanceCounter( StopTime );
   if FActive.Count = 0 then exit;

   Index := FActive.Count;
   if Index > 10 then Index := 10;
   ResultTime := StopTime - StartTime - CorrTime[Index];
   if ResultTime < 0 then exit;

   for i := 0 to FActive.Count-1 do begin
      Index := Integer( FActive.Items[i] );
      with THscProfilerItem( FItems.Objects[Index] ) do inc( TimeWithChildren, ResultTime );
   end;

   Index := Integer( FActive.Items[ FActive.Count-1 ] );
   with THscProfilerItem( FItems.Objects[Index] ) do inc( TimeWithoutChildren, ResultTime );
end;

procedure THscProfiler.Start(ID: String);
var  Index: Integer;
     FullID: String;
begin
   DoAllStop;

   FullID := ID;
   if FAddNested and ( FActive.Count > 0 ) then begin
      Index := Integer( FActive.Items[FActive.Count-1] );
      FullID := FullID + ', ' + FItems[Index];
   end;

   Index := FItems.IndexOf( FullID );
   if Index < 0 then Index := FItems.AddObject( FullID, THscProfilerItem.Create );
   FActive.Add( Pointer(Index) );

   DoAllStart;
end;

procedure THscProfiler.Stop;
var  Index: Integer;
begin
   DoAllStop;
   if FActive.Count=0 then exit;

   Index := Integer( FActive.Items[ FActive.Count - 1 ] );
   FActive.Delete( FActive.Count - 1 );

   with THscProfilerItem( FItems.Objects[Index] ) do inc( Counter );

   DoAllStart;
end;

procedure THscProfiler.DoEstimateCorrTime;
var  CT: array[0..64] of Int64;
     ic, i: Integer;
begin
   Clear;
   for iC := 0 to 10 do begin CT[ic]:=0; CorrTime[iC]:=0 end;
   for iC := 0 to 10 do begin
      CT[iC] := 0;
      for i:=1 to 10 do begin
         Start( 'Test' );
         Stop;
         CT[iC] := CT[iC] + ResultTime;
      end;
      Start( 'Dummy' + inttostr(iC) );
   end;
   for iC := 0 to 10 do CorrTime[iC] := CT[iC] div 10;
   Clear;
end;


// ----------------------------------------------------------- THscGlobal -----

constructor THscGlobal.Create;
begin
   inherited Create;

   FLock := TCriticalSection.Create;

   FChanged := TEvent.Create( nil, True, False, '' );

   FLocks := TStringList_NoAnsi.Create;
   FLocks.Sorted := True;
   FLocks.Duplicates := dupAccept;

   FValues := TStringList_NoAnsi.Create;
   FValues.Sorted := True;
   FValues.Duplicates := dupIgnore;
end;

destructor THscGlobal.Destroy;
var  i: Integer;
begin
   for i := 0 to FValues.Count - 1 do THscVariant( FValues.Objects[i] ).Free;
   FValues.Free;
   FLocks.Free;
   FChanged.Free;
   FLock.Free;
   inherited Destroy;
end;

procedure THscGlobal.CheckName( const Name: String;
                                allowSystem: Boolean );
begin
   if length( Name ) = 0 then begin
      raise EHscGlobalError.Create( 'Forbidden global name: ""' );
   end;

   if not allowSystem then begin
      if Pos( 'system', LowerCase(Name) ) = 1 then begin
         raise EHscGlobalError.Create( 'Forbidden global name: ' + Name );
      end;
   end;
end;

function THscGlobal.Enter( const Name: String;
                           const StopEvent: TEvent;
                           allowSystem: Boolean = False ): Boolean;
// add lock for current thread

   function TryLock: Boolean;
   var  Index: Integer;
   begin
      Result := False;
      FLock.Enter;
      try
         Index := FLocks.IndexOf( Name );
         if Index < 0 then begin
            // not locked by any thread -> add lock
            FLocks.AddObject( Name, Pointer(GetCurrentThreadID) );
            Result := True;
         end else begin
            if Cardinal(FLocks.Objects[Index]) = GetCurrentThreadID then begin
               // already locked by same thread -> add additional/nested lock
               FLocks.AddObject( Name, Pointer(GetCurrentThreadID) );
               Result := True;
            end;
         end;
      finally FLock.Leave end;
   end;

var  WaitObjects : TWOHandleArray;
     WaitObjCount: Integer;
begin
   CheckName( Name, allowSystem );

   WaitObjects[0] := FChanged.Handle;
   WaitObjCount := 1;
   if Assigned( StopEvent ) then begin
      WaitObjects[1] := StopEvent.Handle;
      WaitObjCount := 2;
   end;

   repeat
   
      Result := TryLock;
      if Result then exit;

      // Note: There's a risk, that we miss an event pulse, so we don't wait
      //       INFINITE here but retry at least every few seconds.
      case WaitForMultipleObjects( WaitObjCount, @WaitObjects, False, 5000 ) of
         WAIT_OBJECT_0, WAIT_TIMEOUT: ; // try again
         else exit;
      end;

   until False;
end;

function THscGlobal.Leave( const Name: String;
                           allowSystem: Boolean = False ): Boolean;
// remove lock for current thread
var  i: Integer;
begin
   CheckName( Name, allowSystem );
   
   Result := False;
   FLock.Enter;
   try
      i := FLocks.IndexOf( Name );
      if i >= 0 then begin
         if Cardinal( FLocks.Objects[i] ) = GetCurrentThreadID then begin
            FLocks.Delete( i );
            Windows.PulseEvent( FChanged.Handle );
            Sleep( 100 ); // give other threads a better chance to enter
            Result := True;
         end;
      end else begin
         Log( LOGID_DEBUG, 'GlobalUnlock("' + Name + '") was used but "'
                         + Name + '" is not locked! Misspelled?' );
      end;
   finally FLock.Leave end;
end;

procedure THscGlobal.LeaveAll( const ThreadID: Cardinal );
// remove all locks of thread
var  i: Integer;
begin
   FLock.Enter;
   try
      for i := FLocks.Count - 1 downto 0 do begin
         if Cardinal( FLocks.Objects[i] ) = ThreadID then begin
            FLocks.Delete( i );
            Windows.PulseEvent( FChanged.Handle );
         end;
      end;
   finally FLock.Leave end;
end;

procedure THscGlobal.ValueGet( const Name: String;
                               const Value, Default: THscVariant;
                               allowSystem: Boolean = False );
// get global value; create with Default if it doesn't exist yet
var  Index: Integer;
begin
   FLock.Enter;
   try
      Index := FValues.IndexOf( Name );
      if Index < 0 then begin
         Value.Assign( Default );
         ValueSet( Name, Value );
      end else begin
         Value.Assign( THscVariant( FValues.Objects[Index] ) );
         if Value.Unassigned then begin
            Value.Assign( Default );
            ValueSet( Name, Value );
         end;
      end;
   finally FLock.Leave end;
end;

procedure THscGlobal.ValueSet( const Name: String; const Value: THscVariant;
                               allowSystem: Boolean = False );
// set global value; create it if it doesn't exist yet
var  Index: Integer;
begin
   CheckName( Name, allowSystem );

   FLock.Enter;
   try
      Index := FValues.IndexOf( Name );

      if Index < 0 then begin
         FValues.AddObject( Name, THscVariant.Create( Value, False ) );
      end else begin
         THscVariant( FValues.Objects[Index] ).Assign( Value );
      end;

      Windows.PulseEvent( FChanged.Handle );

   finally FLock.Leave end;
end;

function THscGlobal.ValueAdd( const Name: String;
                              const AddVal: Integer;
                              allowSystem: Boolean = False ): Integer;
var  TempVal: THscVariant;
begin
   FLock.Enter;
   try
      TempVal := THscVariant.Create( 0, False );
      try
         ValueGet( Name, TempVal, TempVal );
         TempVal.AsInt := TempVal.AsInt + AddVal;
         Result := TempVal.AsInt;
         ValueSet( Name, TempVal );
      finally TempVal.Free end;
   finally FLock.Leave end;
end;

function THscGlobal.ValueAppend( const Name: String;
                                 const AddVal: String;
                                 allowSystem: Boolean = False ): String;
var  TempVal: THscVariant;
begin
   FLock.Enter;
   try
      TempVal := THscVariant.Create( '', False );
      try
         ValueGet( Name, TempVal, TempVal );
         TempVal.AsStr := TempVal.AsStr + AddVal;
         Result := TempVal.AsStr;
         ValueSet( Name, TempVal );
      finally TempVal.Free end;
   finally FLock.Leave end;
end;

function THscGlobal.ValueWait( const Name: String;
                               const WaitVal: THscVariant;
                               const StopEvent: TEvent;
                               allowSystem: Boolean = False ): Boolean;
// wait until global value becomes the given one
var  Index : Integer;
     CurrVal: THscVariant;
     WaitObjects : TWOHandleArray;
     WaitObjCount: Integer;
begin
   Result := False;

   WaitObjects[0] := FChanged.Handle;
   WaitObjCount := 1;
   if Assigned( StopEvent ) then begin
      WaitObjects[1] := StopEvent.Handle;
      WaitObjCount := 2;
   end;

   repeat

      FLock.Enter;
      try
         Index := FValues.IndexOf( Name );
         if Index >= 0 then begin
            CurrVal := THscVariant( FValues.Objects[Index] );
            if not CurrVal.Unassigned then begin
               case WaitVal.TypOf of
                  hvtInteger:  Result := ( WaitVal.AsInt = CurrVal.AsInt );
                  hvtString :  Result := ( WaitVal.AsStr = CurrVal.AsStr );
               end;
               if Result then exit;
            end;
         end;
      finally FLock.Leave end;

      // Note: There's a risk, that we miss an event pulse, so we don't wait
      //       INFINITE here but retry at least every few seconds.
      case WaitForMultipleObjects( WaitObjCount, @WaitObjects, False, 5000 ) of
         WAIT_OBJECT_0, WAIT_TIMEOUT: ; // try again
         else exit;
      end;

   until False;
end;


function THscGlobal.Once( const Name: String; const Minutes: Integer;
                          allowSystem: Boolean = False ): Boolean;
// returns true if <Name> was not used within <Minutes> minutes
var  dt, dtEmpty: TDateTime;
     diffMin: Integer;
begin
   CheckName( Name, allowSystem );

   FLock.Enter;
   try
      with TIniFile.Create( AppSettings.GetStr(asPathBase) + 'Hamster.ini' ) do try

         dtEmpty := EncodeDate( 2000, 1, 1 );
         dt := ReadDateTime( 'GlobalOnce', Name, dtEmpty );

         if dt = dtEmpty then begin
            Result := True;
         end else begin
            diffMin := abs( DateTimeDiffToMinutes( Now - dt ) );
            Result := ( diffMin >= Minutes );
         end;

         if Result then WriteDateTime( 'GlobalOnce', Name, Now );

      finally Free end;
   finally FLock.Leave end;
end;

initialization
   HscGlobal := THscGlobal.Create;

finalization
   try HscGlobal.Free except end;

end.
