// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHscFiles; // helper-object for available script files

interface

{$INCLUDE Compiler.inc}

uses SysUtils, Classes, SyncObjs, uType, tScript;

type
   THscFile = class
      private
         FPathBase: String;
         FPathSub : String;
         FFileName: String;
         FFileTime: TDateTime;
         FFileSize: Integer;
         FIsScript: Boolean;

      protected
         function  ExpandedFilename: String;
         procedure LoadText( TextList: TStrings );
         procedure SaveText( TextList: TStrings );

      public
         property PathBase: String    read FPathBase;
         property PathSub : String    read FPathSub;
         property Filename: String    read FFilename;
         property FileTime: TDateTime read FFileTime;
         property FileSize: Integer   read FFileSize;
         property IsScript: Boolean   read FIsScript;

         constructor Create( const APathBase, APathSub, AFilename: String;
                             const AIsScript: Boolean;
                             const AFileTime: TDateTime;
                             const AFileSize: Integer );
   end;

   THscFileList = class
      private
         FList: TStringList;
         FHscPath, FHsmPath: String;

         function  GetCount: Integer;
         function  GetItem( Index: Integer ): THscFile;
         procedure DoScan( const PathBase, FileExt: String;
                           IsScript: Boolean;
                           Recursed: Boolean );

      protected
         property Count: Integer read GetCount;
         property Items[ Index: Integer ]: THscFile read GetItem;

         function IndexOf( const PathSub, FileName: String ): Integer; overload;
         function IndexOf( const HscFile: THscFile ): Integer; overload;

         function  Add( HscFile: THscFile ): Integer;
         procedure Delete( const Index: Integer );
         procedure Clear;
         procedure Refresh;

      public
         constructor Create( const AHscPath, AHsmPath: String );
         destructor Destroy; override;
   end;

   THscFileSelections = ( hfsScripts, hfsModules, hfsAnyFile );
   THscListFilesTypes = ( hlftFile, hlftPathFile, hlftFileInfo );

   THscFiles = class
      private
         FLock    : TCriticalSection;
         FHscPath, FHsmPath: String;
         FFileList: THscFileList;

      public
         procedure Refresh;

         function  Find( const PathSub, FileName: String;
                         Selection: THscFileSelections ): THscFile;
         procedure List( Selection: THscFileSelections;
                         ListFileType: THscListFilesTypes;
                         List: TStrings );

         procedure Load( const HscFile: THscFile;
                         TextList: TStrings );
         procedure Save( const HscFile: THscFile;
                         TextList: TStrings ); overload;
         procedure Save( const PathSub, FileName: String;
                         TextList: TStrings ); overload;
         function  Delete( const HscFile: THscFile ): Boolean;

         function  Start( const HscFile: THscFile;
                          const ScriptParams: String;
                          const WaitForEnd: Boolean;
                          out   ScriptThread: TThreadExecuteScript;
                          const FreeType: TThreadFreeTypes;
                          const DoBufferOutput: Boolean ): Integer;

         constructor Create( const AHscPath, AHsmPath: String );
         destructor Destroy; override;
   end;

function HscFileCheckSplitName( const PathNameStr: String;
                                out   PathSub, Filename: String ): Boolean;
   
implementation

uses uConst, uConstVar, uVar, cLogFileHamster;

function HscFileCheckSplitName( const PathNameStr: String;
                                out   PathSub, Filename: String ): Boolean;
var  sn, h: String;
     i: Integer;
begin
   Result   := False;
   PathSub  := '';
   Filename := '';
   sn := PathNameStr;

   // add/check extension
   h := LowerCase( ExtractFileExt( sn ) );
   if ( h = '' ) then begin h := '.hsc'; sn := sn + h end;
   if ( h <> '.hsc' ) and ( h <> '.hsm' ) then exit;

   // also accept "/" as path separator but convert it to "\"
   sn := StringReplace( sn, '/', '\', [rfReplaceAll] );

   // check for parts known to be dangerous
   if Pos( '\\', sn ) > 0 then exit; // reject UNC
   if Pos( ':',  sn ) > 0 then exit; // reject drive
   if Pos( '..', sn ) > 0 then exit; // reject parent dir
   if Pos( '%',  sn ) > 0 then exit; // reject OS variables

   // check for unwanted parts
   if Pos( '.\', sn ) > 0 then exit; // unwanted dir-spec.
   if Pos( '\',  sn ) = 1 then exit; // unwanted root
   if Pos( '.',  sn ) <> length(sn)-3 then exit; // only accepted in ext.
   for i:=1 to length(sn) do begin
      case sn[i] of
         'a'..'z', 'A'..'Z', '0'..'9', '_', '-', '!', ' ':
            {ok}; // allowed chars for path/file names
         '\', '.':
            {ok}; // allowed after the checks above
         else
            exit; // unwanted character
      end;
   end;

   // split into sub-dir and filename, check filename
   i := LastDelimiter( '\', sn );
   if i = 0 then begin
      PathSub  := '';
      Filename := sn;
   end else begin
      PathSub  := copy( sn, 1, i );
      Filename := copy( sn, i+1, MaxInt );
   end;
   if length( Filename ) < 5 then exit; // at least 1 char + extension

   Result := True;
end;

{ TScriptFileItem }

constructor THscFile.Create( const APathBase, APathSub, AFilename: String;
                             const AIsScript: Boolean;
                             const AFileTime: TDateTime;
                             const AFileSize: Integer );
begin
   inherited Create;
   FPathBase := APathBase;
   FPathSub  := APathSub;
   FFilename := AFilename;
   FIsScript := AIsScript;
   FFileTime := AFileTime;
   FFileSize := AFileSize;
end;

function THscFile.ExpandedFilename: String;
begin
   Result := FPathBase + FPathSub + FFileName;
end;

procedure THscFile.LoadText( TextList: TStrings );
begin
   TextList.Clear;
   TextList.LoadFromFile( ExpandedFilename );
end;

procedure THscFile.SaveText( TextList: TStrings );
var  SR: TSearchRec;
begin
   TextList.SaveToFile( ExpandedFilename );
   if FindFirst( ExpandedFilename, faAnyFile, SR ) = 0 then try
      FFileTime := FileDateToDateTime( SR.Time );
      FFileSize := SR.Size;
   finally
      FindClose( SR );
   end;
end;


{ THscFileList }

constructor THscFileList.Create( const AHscPath, AHsmPath: String );
begin
   inherited Create;
   FHscPath := AHscPath;
   FHsmPath := AHsmPath;
   FList := TStringList.Create;
   FList.Sorted := True;
   Refresh;
end;

destructor THscFileList.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

function THscFileList.Add( HscFile: THscFile ): Integer;
var  FullFilename: String;
begin
   FullFilename := HscFile.PathBase + HscFile.PathSub + HscFile.Filename;
   Result := FList.AddObject( FullFilename, HscFile );
end;

procedure THscFileList.Delete( const Index: Integer );
begin
   THscFile( FList.Objects[Index] ).Free;
   FList.Delete( Index );
end;

procedure THscFileList.Clear;
var  i: Integer;
begin
   for i:=0 to FList.Count-1 do THscFile( FList.Objects[i] ).Free;
   FList.Clear;
end;

procedure THscFileList.DoScan( const PathBase, FileExt: String;
                               IsScript: Boolean;
                               Recursed: Boolean );

   procedure ScanDir( const PathSub: String );
   var  SR: TSearchRec;
   begin
      // 1.) files
      if FindFirst( PathBase + PathSub + '*.*', faAnyFile, SR ) = 0 then try
         repeat
            if ( SR.Attr and faDirectory ) = 0 then begin
               if AnsiCompareText( ExtractFileExt(SR.Name), FileExt ) = 0 then
                  Add( THscFile.Create(
                     PathBase, PathSub, SR.Name, IsScript,
                     FileDateToDateTime( SR.Time ), SR.Size
                  ) );
            end;
         until FindNext( SR ) <> 0;
      finally FindClose( SR ) end;

      // 2.) subdirs
      if FindFirst( PathBase + PathSub + '*.*', faAnyFile, SR ) = 0 then try
         repeat
            if ( SR.Attr and faDirectory ) <> 0 then begin
               if Recursed and ( SR.Name[1] <> '.' ) then begin
                  ScanDir( PathSub + SR.Name + '\' );
               end;
            end;
         until FindNext( SR ) <> 0;
      finally FindClose( SR ) end;
   end;

begin
   ScanDir( '' );
end;

procedure THscFileList.Refresh;
begin
   Clear;
   DoScan(  // add scripts
      FHscPath, '.hsc', True,
      AppSettings.GetStr( asPathScripts ) <> AppSettings.GetStr(asPathBase)
   );
   DoScan( // add modules
      FHsmPath, '.hsm', False,
      False
   );
end;

function THscFileList.GetCount: Integer;
begin
   Result := FList.Count;
end;

function THscFileList.GetItem(Index: Integer): THscFile;
begin
   Result := THscFile( FList.Objects[ Index ] );
end;

function THscFileList.IndexOf( const PathSub, FileName: String ): Integer;
var  i: Integer;
begin
   Result := -1;
   for i:=0 to FList.Count-1 do begin
      if ( AnsiCompareText( GetItem(i).PathSub,  PathSub  ) = 0 ) and
         ( AnsiCompareText( GetItem(i).FileName, FileName ) = 0 ) then begin
         Result := i;
         break;
      end;
   end;
end;

function THscFileList.IndexOf( const HscFile: THscFile ): Integer;
var  i: Integer;
begin
   Result := -1;
   for i:=0 to FList.Count-1 do begin
      if GetItem(i) = HscFile then begin
         Result := i;
         break;
      end;
   end;
end;

{ THscFiles }

constructor THscFiles.Create( const AHscPath, AHsmPath: String );
begin
   inherited Create;
   FLock     := TCriticalSection.Create;
   FHscPath  := AHscPath;
   FHsmPath  := AHsmPath;
   FFileList := THscFileList.Create( FHscPath, FHsmPath );
end;

destructor THscFiles.Destroy;
begin
   FreeAndNil( FFileList );
   FreeAndNil( FLock );
   inherited Destroy;
end;

procedure THscFiles.Refresh;
begin
   FLock.Enter;
   try
      FFileList.Refresh;
   finally
      FLock.Leave;
   end;
end;

function THscFiles.Find( const PathSub, FileName: String;
                         Selection: THscFileSelections ): THscFile;
var  Index  : Integer;
     HscFile: THscFile;
begin
   FLock.Enter;
   try
      Result := nil;
      Index  := FFileList.IndexOf( PathSub, Filename );
      if Index >= 0 then begin
         HscFile := FFileList.Items[ Index ];
         case Selection of
            hfsScripts: if HscFile.IsScript then Result := HscFile;
            hfsModules: if not HscFile.IsScript then Result := HscFile;
            hfsAnyFile: Result := HscFile;
         end;
      end;
   finally
      FLock.Leave;
   end;
end;

function THscFiles.Delete( const HscFile: THscFile ): Boolean;
var  Index: Integer;
begin
   FLock.Enter;
   try
      Result := False;
      Index  := FFileList.IndexOf( HscFile );
      if Index >= 0 then begin
         Result := SysUtils.DeleteFile( HscFile.ExpandedFilename );
         if Result then FFileList.Delete( Index );
         Refresh;
      end;
   finally
      FLock.Leave;
   end;
end;

procedure THscFiles.Load( const HscFile: THscFile;
                          TextList: TStrings );
var  Index: Integer;
begin
   FLock.Enter;
   try
      TextList.Clear;
      Index := FFileList.IndexOf( HscFile );
      if Index >= 0 then HscFile.LoadText( TextList );
   finally
      FLock.Leave;
   end;
end;

procedure THscFiles.Save( const HscFile: THscFile;
                          TextList: TStrings );
var  Index: Integer;
begin
   FLock.Enter;
   try
      Index := FFileList.IndexOf( HscFile );
      if Index >= 0 then HscFile.SaveText( TextList );
   finally
      FLock.Leave;
   end;
end;

procedure THscFiles.Save( const PathSub, FileName: String;
                          TextList: TStrings);
var  HscFile : THscFile;
     IsScript: Boolean;
     BasePath: String;
begin
   FLock.Enter;
   try
      HscFile := Find( PathSub, Filename, hfsAnyFile );
      if Assigned( HscFile ) then begin
         // known file
         Save( HscFile, TextList );
      end else begin
         // new file
         if LowerCase( ExtractFileExt( Filename ) ) = '.hsm' then begin
            IsScript := False;
            BasePath := FHsmPath;
         end else begin
            IsScript := True;
            BasePath := FHscPath;
         end;
         HscFile := THscFile.Create( BasePath, PathSub, Filename,
                                     IsScript, 0, 0 );
         FFileList.Add( HscFile );
         Save( HscFile, TextList );
         Refresh; 
      end;
   finally
      FLock.Leave;
   end;
end;

procedure THscFiles.List( Selection: THscFileSelections;
                          ListFileType: THscListFilesTypes;
                          List: TStrings );
var  Index  : Integer;
     HscFile: THscFile;
     s      : String;
begin
   FLock.Enter;
   try
      List.Clear;

      for Index := 0 to FFileList.Count-1 do begin
         HscFile := FFileList.Items[ Index ];
         case Selection of
            hfsScripts: if not HscFile.IsScript then HscFile := nil;
            hfsModules: if HscFile.IsScript     then HscFile := nil;
         end;

         if Assigned( HscFile ) then begin
            with HscFile do begin
               case ListFileType of
                  hlftFile:
                     s := Filename;
                  hlftPathFile:
                     s := PathSub + Filename;
                  hlftFileInfo:
                     s := FormatDateTime( 'yyyy"-"mm"-"dd hh":"nn":"ss', FileTime )
                          + TAB + inttostr( FileSize )
                          + TAB + PathSub + Filename
               end;
               List.AddObject( s, FFileList.Items[Index] );
            end;
         end;
      end;

   finally
      FLock.Leave;
   end;
end;

function THscFiles.Start( const HscFile: THscFile;
                          const ScriptParams: String;
                          const WaitForEnd: Boolean;
                          out   ScriptThread: TThreadExecuteScript;
                          const FreeType: TThreadFreeTypes;
                          const DoBufferOutput: Boolean ): Integer;
var  Index : Integer;
     ExFile: String;
begin
   Result := -1;
   ExFile := '';

   FLock.Enter;
   try
      Index := FFileList.IndexOf( HscFile );
      if Index < 0 then exit;
      if not HscFile.IsScript then exit;
      ExFile := HscFile.ExpandedFilename;
   finally
      FLock.Leave;
   end;

   Result := StartNewScript( ExFile, ScriptParams, WaitForEnd,
                             ScriptThread, FreeType, False, DoBufferOutput );
end;

end.
