// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
// See file License.txt for details.
// ============================================================================

unit cHscActions; // Script and User Task Actions

interface

uses SysUtils, Classes, Windows, SyncObjs, uType, uConst, cHscAction;

type
   THscActions = class
      private
         FPath: String;
         FList: TList;
         FLock: TCriticalSection;
         FLockId: DWORD;
         FReleased: TEvent;

         procedure Clear;

         function  enterLock( Action: THscActionTypes ): Boolean; overload;
         function  enterLock: Boolean; overload;
         procedure leaveLock;

      public
         function IsAssigned( const Action: THscActionTypes ): Boolean;
         function Execute( const Action: THscActionTypes;
                           const Params: String ): Boolean;
         procedure Reload;

         constructor Create( const APath: String );
         destructor Destroy; override;
   end;

implementation

uses IniFiles, uVar, uTools, tScript, cLogFileHamster, cUserTasks, tUserTasks;


{ THscActions }

constructor THscActions.Create( const APath: String );
begin
   inherited Create;
   FPath := APath;
   FList := TList.Create;
   FLock := TCriticalSection.Create;
   FReleased := TEvent.Create( nil, True, False, '' );
   FLockId := 0;
   Reload;
end;

destructor THscActions.Destroy;
begin
   Clear;
   FList.Free;
   FLock.Free;
   FReleased.Free;
   inherited Destroy;
end;

function THscActions.enterLock: Boolean;
begin
   Result := enterLock( actIgnore );
end;

function THscActions.enterLock( Action: THscActionTypes ): Boolean;
const tryTimeMax = 5 * 60 * 1000;
      tryTimeOne = 100;
var   tryTime: Integer;
begin
   Result := False;
   tryTime := 0;

   repeat

      FLock.Enter;
      try
         if FLockId = 0 then begin
            FLockId := GetCurrentThreadId;
            Result := true;
            exit;
         end else begin
            if FLockId = GetCurrentThreadId then begin
               Result := true;
               exit;
            end;
         end;
      finally
         FLock.Leave;
      end;

      inc( tryTime, tryTimeOne );
      FReleased.WaitFor( tryTimeOne );

   until tryTime >= tryTimeMax;

   if Action = actIgnore then begin
      Log( LOGID_ERROR, 'Actions could not be accessed within '
                      + inttostr( tryTimeMax div 1000 )
                      + ' seconds - they are already locked by another task!' );
   end else begin
      Log( LOGID_ERROR, 'Nested action "' + THscActionNames[Action]
                      + '" could not be started within '
                      + inttostr( tryTimeMax div 1000 )
                      + ' seconds! Presumably locked by another task!' );
   end;
   Log( LOGID_ERROR, 'Such errors are most likely be caused by actions with '
                   + '"[X] Lock" or "[X] Wait" options set!' );
end;

procedure THscActions.leaveLock;
begin
   FLock.Enter;
   try
      if FLockId = GetCurrentThreadId then begin
         FLockId := 0;
         Windows.PulseEvent( FReleased.Handle );
      end;

   finally
      FLock.Leave;
   end;
end;

procedure THscActions.Clear;
var  i: Integer;
begin
   if not enterLock then exit;
   try
      for i := 0 to FList.Count - 1 do begin
         if Assigned( FList[i] ) then begin
            THscAction( FList[i] ).Free;
            FList[i] := nil;
         end;
      end;
      FList.Clear;
   finally leaveLock end;
end;

procedure THscActions.Reload;
var  act: THscActionTypes;
     n, s: String;
     w, l, q: Boolean;
begin
   if not enterLock then exit;

   try
      Clear;

      with TIniFile.Create( FPath + CFGFILE_HSCACTIONS ) do try

         for act := Low( THscActionNames ) to High( THscActionNames ) do begin

            FList.Add( nil ); // ord(Action) = FList-Index, nil = unassigned

            n := THscActionNames[act];
            s := ReadString( n, 'Script', ''    );
            w := ReadBool  ( n, 'Wait',   False );
            l := ReadBool  ( n, 'Lock',   False );
            q := ReadBool  ( n, 'Silent', False );

            if s <> '' then begin
               // if a script is given, assign it to action
               FList[ ord(act) ] := THscAction.Create( n, s, w, l, q );
            end;
            
         end;

      finally Free end;

   finally leaveLock end;
end;

function THscActions.IsAssigned( const Action: THscActionTypes ): Boolean;
var  HscAction: THscAction;
begin
   Result := false;
   if Action = actIgnore then exit;
   try
      if not Assigned( FList[ ord(Action) ] ) then exit;
   except end;

   if not enterLock then exit;
   try
      HscAction := FList[ ord(Action) ];
      Result := Assigned( HscAction );
   finally leaveLock end;
end;

function THscActions.Execute( const Action: THscActionTypes;
                              const Params: String ): Boolean;
var  HscAction: THscAction;
     n, s: String;
     w, l, q, u: Boolean;
begin
   Result := False;
   if Action = actIgnore then exit;
   try
      if not Assigned( FList[ ord(Action) ] ) then exit;
   except end;

   // check/get action's definition
   if not enterLock( Action ) then exit;
   try
      HscAction := FList[ ord(Action) ];
      if Assigned( HscAction ) then with HscAction do begin
         n := ActionName;
         s := ActionFile;
         w := WaitForEnd;
         l := LockedExec;
         q := ExecSilent;
         u := IsUserTask;
      end else begin
         n := '';
         s := '';
         w := False;
         l := False;
         q := False;
         u := False;
      end;
   finally leaveLock end;

   // is an action file assigned?
   if length( n ) = 0 then exit;

   // execute action file
   if( (not q) or (Action <> actNewsIn) ) then begin
      Log( LOGID_DEBUG, 'Execute action "' + n + '" ...' );
   end;
   if l then begin
      if not enterLock( Action ) then exit;
   end;
   try
      try
         if u then begin
            with TUserTaskThread.Create( s, q ) do Resume;
         end else begin
            StartNewScript( s, Trim( n + CRLF + Params ), w, q );
         end;
         Result := True;
      except
         on E: Exception do begin
            Log( LOGID_ERROR, 'Error executing action: ' + E.Message );
         end;
      end;
   finally if l then leaveLock end;
end;

end.
